/**
 ** define.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

#define DEFINE_MODULE

#include "core.h"

#include <queue.h>

/*-------------------------------------------------------------------------*/
/* Private Variables                                                       */

Int define__block_id;
long define__high_water_mark, define__low_water_mark;

/*-------------------------------------------------------------------------*/
/* Public Variables                                                        */

long  current_stack_size = 0;
Int   current_stack_item= 1;
long  current_global_location = 2;
Int   define_debug= 0;

Type  **stack_type= 0;
long  *stack_size= 0;
Int   define_lexical_level;

Queue define__arg_name_queue;

/*-------------------------------------------------------------------------*/
/* Public Functions                                                        */

void define_init(void)
{
    static Int initted= 0;
    if (!initted) {
	initted= 1;
	stack_type= malloc(sizeof(stack_type[0]) * MAXSTACK);
	stack_size= malloc(sizeof(stack_size[0]) * MAXSTACK);
	queue_init(&define__arg_name_queue, 50, sizeof(Symbol*));
    }
    current_stack_size= 0;
    current_stack_item= 1;
    current_global_location= 2;
    stack_size[0]= 0;
}

void define_set_globals_addr(long a)
{
    current_global_location= a;
}

void define_start_pass(void)
{
    define__block_id= 0;
    define_lexical_level= 0;
    symbol_reset_temp();
    if (define_debug) printf("vvvvvvvvvvvv define_start_pass vvvvvvvvvvv\n");
}

/****************/
/* Push and pop */
/****************/

Type *define_pop(void)
{
    current_stack_item--;
    if (current_stack_item < 1)
      die(("Negative stack item %ld in define_pop", current_stack_item));
    
    current_stack_size = stack_size[current_stack_item-1];
    
    if (current_stack_size < 0)
      die(("Negative stack size %ld in define_pop", current_stack_size));
    
    if (define_debug) printf("define pop to %ld\n", current_stack_size);
    return stack_type[current_stack_item];
}

void define_push(Type *type)
{
    Int size= type_sizeof(type);

    
    if (current_stack_item >= MAXSTACK-1)
      die(("Expression or function too complex.  (increase MAXSTACK)"));

    current_stack_size += size;

    if (current_stack_size > define__high_water_mark)
      define__high_water_mark= current_stack_size;
    
    stack_size[current_stack_item]= current_stack_size;
    stack_type[current_stack_item]= type;

    current_stack_item++;
    if (define_debug) printf("define push <%s> to %ld\n", type_name(type), current_stack_size);
}

void define_set_low_water_mark(void)
{
    define__low_water_mark= define__high_water_mark= current_stack_size;
}

long define_get_high_water_mark(void)
{
    return define__high_water_mark - define__low_water_mark;
}

Type *define_stack_type(Int depth)   /* 0 = top of stack, 1 is below... */
{
    return stack_type[current_stack_item - depth - 1];
}

long define_stack_offset(Int depth)   /* 0 = top of stack, 1 is below... */
{
    return current_stack_size - stack_size[current_stack_item - depth - 1];
}

Int define_stack_items(void)
{
    return current_stack_item - 1;
}

/***********************/
/* Begin and end block */
/***********************/

void define_begin_block(void)
{
    define_lexical_level++;
    symtab_begin_block();
    define_push(type_Block());
    define_declare_local(symbol_create("current block"));
}

void define_begin_breakable_block(void)
{
    symtab_begin_block();

    symtab_define_symbol(symbol_create("current breakable block exit"),
			 define_temp_symbol());

    define_push(type_Block());
    define_declare_local(symbol_create("current breakable block"));
}

void define_end_block(void)
{
    define_lexical_level--;
    symtab_end_block();
    while (type_id(define_stack_type(0)) != block_id) code_pop();
    define_pop();
}

void define_end_block_no_pop(void)
{
    define_lexical_level--;
    symtab_end_block();
    while (type_id(define_stack_type(0)) != block_id) define_pop();
    define_pop();
}

void define_end_breakable_block(void)
{
    spew_label_definition
      (symtab_get(symbol_create("current breakable block exit")));
    symtab_end_block();
    while (type_id(define_stack_type(0)) != block_id) code_pop();
    define_pop();
}

void define_break(Int levels)
{
    Value *current_block= symtab_get(symbol_create("current breakable block"));
    long exit_stack_size;

    if (levels != 0) {
	user_error(("Exit through multiple levels not yet supported"));
    }
    if (!current_block) {
	if (pass == 1) {
	    user_error(("Attempt to use 'break' not inside a loop"));
	}
    } else {
	exit_stack_size= value_address(current_block);
	if (!code_inline)
	  {
	    spew_pop_op((Int) (current_stack_size - exit_stack_size));
	    spew_jump(symtab_get(symbol_create("current breakable block exit")));
	  }
	else			/* do the native thang */
	  {
	    code_pop_n((Int)(current_stack_size - exit_stack_size));
	    code_jump(symtab_get(symbol_create("current breakable block exit")));
	  }
    }
}

/********************/
/* Define Procedure */
/********************/

void define_procedure_argument(Symbol *name)
{
    queue_add_tail(&define__arg_name_queue, &name);
}

void define_procedure(Type *ret_type, Symbol *sym)
{
    Type *proc_type;
    Int  nargs, i;

    if (pass > 0) {   /* functions declared on pass 0 only */
	proc_type= value_type(symtab_get(sym));
    } else {
	/* pass = 0 */
	/* The local scope contains the arguments to the procedure */
	
	for (nargs= 0; type_id(define_stack_type(nargs)) != block_id; nargs++);
	
	proc_type= type_Func(ret_type, pcode_calling_convention, nargs);
	
	for (i= 0; i< nargs; i++) {
	    type_Func_set_arg_type(proc_type, i, define_stack_type(nargs - i - 1));
	    type_Func_set_arg_name(proc_type, i,
				   *(Symbol**)queue_remove_head(&define__arg_name_queue));
	}
	proc_type= find_original_type(proc_type);

	{
	    Value *v= value_create(proc_type, global_location, 0);
	    value_set_module(v, spew_module);
    
	    if (!symtab_define_global_value(sym, v)) {
		user_error(("%s already defined", symbol_name(sym)));
		free(v);
	    }
	}
    }
    spew_label_definition(sym);
    symtab_define_type(symbol_create("current procedure"), proc_type);
}

/* Right now all assembly routines take an Int and return an Int.
   Sorry about that. */

void define_binary_subroutine(Symbol *sym)
{
    if (pass == 0) {
	Type *proc_type= type_Func(type_Int(), ml_calling_convention, 1);
	Value *v;
	type_Func_set_arg_type(proc_type, 0, type_Int());
	type_Func_set_arg_name(proc_type, 0, symbol_create("arg"));
	proc_type= find_original_type(proc_type);
	v= value_create(proc_type, global_location, 0);
	if (!symtab_define_global_value(sym, v)) {
	    user_error(("%s already defined", symbol_name(sym)));
	}
    }
}

/* Right now all assembly routines variables are 2-byte ints.
   Sorry about that. */

void define_binary_variable(Symbol *sym)
{
    if (pass == 0) {
	Value *v= value_create(type_Int(), global_location, 0);
	if (!symtab_define_global_value(sym, v)) {
	    user_error(("%s already defined", symbol_name(sym)));
	}
    }
}

void define_declare_global(Symbol *sym)
/* TODO: make this deal with initializations when necessary */
{
    Type *type= define_pop();
    if (pass > 0) {
	while (current_stack_item > 1) define_pop();
	return;  /* globals are declared on pass 0 only */
    }
    if (define_debug) printf("global %s, ", type_name(type));
    current_global_location -= type_sizeof(type);
    while (current_stack_item > 1) {
	Type *to_pop= define_pop();
	current_global_location -= type_sizeof(to_pop);
	if (define_debug) printf(", also %s ", type_name(to_pop));
    }
    if (define_debug) printf(" to %04lX\n",current_global_location);
	
    if (!symtab_define_value(sym, value_create(type, global_location,
					       current_global_location))) {
	if (pass == 0)
	  user_error(("%s already defined", symbol_name(sym)));
    }
}

void define_declare_local(Symbol *sym)
{
    Type *type= define_stack_type(0);
    if (!symtab_define_value(sym, value_create(type, stack_location,
					       current_stack_size))) {
	if (pass == 0)
	  user_error(("Attempt to redefine local symbol %s", symbol_name(sym)));
    }
}

void define_declare_constant(Symbol *sym, Type *type)
{
    if (!symtab_define_value(sym, value_create(type, constant, 0)))
      {
	if (pass == 0)
	  user_error(("Attempt to redefine local symbol %s", symbol_name(sym)));
      }
}

void define_get_symbol_addr(Symbol *sym)
{
    Value *value= symtab_get(sym);
    if (!value) {
	if (pass == 1)
	  user_error(("Undeclared symbol %s", symbol_name(sym)));
	define_push(type_Undef());
    } else {
	switch(value_loctype(value)) {
	  case global_location:
	  case zero_page_global_location:
	    spew_op(Ppush2);
	    spew_label_reference(sym);
	    define_push(type_Pointer(value_type(value)));
	    break;
	  case stack_location:
	    spew_sprel(current_stack_size - value_address(value));
	    define_push(type_Pointer(value_type(value)));
	    break;
	  default:
	    die(("Illegal location_type in define_get_symbol"));
	}
    }
}
    
void define_set_symbol_addr(Symbol *sym, long addr)
{
    Value *value= symtab_get(sym);
    debug(define, ("Defining label %s to location %ld\n",
		   symbol_name(sym), addr));
    if (!value)
      die(("Tried to assign location to undeclared symbol %s", symbol_name(sym)));

    switch(value_loctype(value)) {
      case zero_page_global_location:
	if (addr & 0xffL != addr)
	  die(("Symbol %s (%04lX) isn't really in the zero page!",
	       symbol_name(sym), addr));
      case global_location:
	value_set_address(value, addr);
	break;
      case stack_location:
	die(("Tried to assign location of a stack variable %s", symbol_name(sym)));
	break;
      default:
	die(("Illegal location_type in define_set_symbol_addr"));
    }
}

Symbol *define_temp_symbol(void)
{
    Symbol *temp= symbol_temp();
    if (pass == 0)
      symtab_define_global_value(temp, value_create(type_Label(), global_location, 0));
    return temp;
}

void define_cast (Type *new_type)
{
    Type *top_of_stack= define_pop();

    if (type_sizeof(top_of_stack) != type_sizeof(new_type)) {
	die(("Attempt to cast type %s to type %s\n",
	     type_name(top_of_stack), type_name(new_type)));
    }

    define_push(new_type);
}

void define_coerce (Type *new_type)
{
    Type *top_of_stack= define_pop();

    if (type_compatible(top_of_stack, new_type)) {
	/* no problem; do nothing */
    }
    else if (type_id(top_of_stack) == int_id &&
	     type_id(new_type) == float_id) {
	spew_op(Pint2fl);
    }
    else if (type_id(top_of_stack) == float_id &&
	     type_id(new_type) == int_id) {
	spew_op(Pfl2int);
    }
    else if (type_id(top_of_stack) == long_id &&
	     type_id(new_type) == float_id) {
	spew_op(Plng2fl);
    }
    else if (type_id(top_of_stack) == float_id &&
	     type_id(new_type) == long_id) {
	spew_op(Pfl2lng);
    }
    else if (type_id(top_of_stack) == long_id &&
	     type_id(new_type) == int_id) {
	spew_op(Ppop2);
    }
    else if (type_id(top_of_stack) == int_id &&
	     type_id(new_type) == long_id) {
        spew_op(Pint2lng);
    }
    else if (type_id(top_of_stack) == pointer_id &&
             type_id(new_type) == int_id) {
    }
    else if (type_id(top_of_stack) == int_id &&
             type_id(new_type) == pointer_id) {
         /* Only coerce integers to pointer if you are wizard */
         if (ic_wizard) {
         }
         else {
             user_error(("Attempt to coerce type %s to type %s (must use ic -wizard to do this\n", type_name(top_of_stack), type_name(new_type)));
         }
    }
    else {
	user_error(("Attempt to coerce type %s to type %s\n",
		    type_name(top_of_stack), type_name(new_type)));
    }
    define_push(new_type);
}

#if 0 /* TODO */
void define_dump_labels(out) FILE *out;
{
    
}
#endif

Symbol *symbol_from_pc(long pc)
{
    Symbol *sym, *ret= 0;
    Value *val;
    long   dist= 0x10000;
    if (define_debug) printf("looking for pc %04lX\n", pc);
    for (sym= symtab_get_binding(&val, 1); sym; sym= symtab_get_binding(&val, 0)) {
	if (type_id(value_type(val)) == func_id) {
	    long loc= value_address(val);
	    if (define_debug) printf("found %s at pc %04lX\n", symbol_name(sym), loc);
	    if (pc - loc < dist && pc >= loc) {
		dist= pc-loc;
		ret= sym;
	    }
	}
    }
    return ret;
}
