/**
 ** ic.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* TODO: do a CRC of code downloaded to verify current code is in fact in the
   board before debugging a stopped program?  realize that sometimes a machine
   will halt and need to be debugged even though the person has not yet
   started ci */

#define CI_MODULE
#define PCODE_DEFINE_ERRORS /* make pcode.h define the system errors globally */

#include "core.h"

#include <stringlb.h>
#include <proc.h>
#include <iob.h>
#include <cmdlineo.h>
#include <parsargs.h>
#include <scan.h>

#include <time.h>

#include "board.h"
#include "pcodesim.h"

#ifdef MAC
#include <Memory.h>
#include <mac-args.h>
#endif

#ifdef PC
#pragma optimize("q", off)
#endif

Int old_pcode= 0;
Int ic_profile= 0;
Int ic_run= 0;
char ic_outfile[100]="";
Int ic_debug= 0;
Int ic_wizard= 0;

char *ccc="Copyright 1990, 1991 Randy Sargent.";

char editor[]= MAKE_CONFIGURATION(DEFAULT_EDITOR,
				  "Default editor");

#include <ctype.h>

extern char *date;

Int ci_major_version= 1;
Int ci_minor_version= 1;
Int ci_under_construction= 1;
Int ci_no_library= 0;

Int board= 0;

long ci_pcode_origin= PCODE_ORIGIN;
long ci_globals_origin= GLOBALS_ORIGIN;

void 	ci_reset_compiler      (void);
void 	ci_init                (void);
Int 	ci_load                (char *filename);
void 	ci_mainloop            (void);
void    ci_show_process_errors (void);
void    ci_execute_expression  (char *expression, Int display);
void    ci_spewprocdata        (Int addr);
void    ci_peekbyte            (Int addr);
void    ci_peekword            (Int addr);
Int     ci_link                (void);
void 	ic_handle_errors       (void);
Int     ci_load                (char *filename);
Int     ci_unload              (char *filename);

static IOStream *serial_stream= 0;
static IOStream *keyb_stream;

#ifndef IC_LIB_DIRECTORY
#define IC_LIB_DIRECTORY LIB_DIRECTORY "ic/"
#endif

char lib_directory_config[]=
  MAKE_CONFIGURATION(IC_LIB_DIRECTORY,
		     "Directory where libraries reside");
#define lib_directory GET_CONFIGURATION(lib_directory_config)

char lib_file_config[]=
  MAKE_CONFIGURATION(IC_LIB_DIRECTORY "lib_r22.lis",
		     "Default library file (blank for none)");
#define lib_file GET_CONFIGURATION(lib_file_config)

char portname_config[]=MAKE_CONFIGURATION(DEFAULT_SERIAL_PORT, "Default serial device name (in the form /dev/*)");
#define portname GET_CONFIGURATION(portname_config)

Int ic_sim= 0;

Int config= 0;
char bootstrapfile[200];

void ic_version(void)
{
    printf("ic %ld.%02ld\n", ci_major_version, ci_minor_version);
    exit(0);
}

Arg_descriptor ad[] = {
    {"port", "s", portname, "Serial device name (in the form /dev/*)"},
    {"bootstrap", "s", bootstrapfile,
                             "Filename (.s19 format) to use instead of \n"
			     "standard bootstrap."},
    {"libdir", "s", lib_directory, "Library directory"},
    {"lib", "s", lib_file, "Library file"},
    {"ansi", "n", &cmd_ansi, "do not assume ANSI (VT100 and xterm) control\n"
	                     "sequences. This disables parenthesis balancing\n"
			     "on the commandline.  If you get garbage when\n"
			     "you type parens, use this option."},
    {"nolib", "b", &ci_no_library, "do not load the library file"},
    {"sim", "b", &ic_sim,  "Use simulator (not board)"},
    {"delay", "i", &io_delay_factor, "Multiply communication timeouts by N"},
    {"hex", "i", &io_hex_on_serial, "Use hexadecimal serial protocol (useless)"},
    {"benchmark", "b", &ic_profile, "Load library a few times and exit"},
    {"config", "b", &config, "Reconfigure executable (executable must be in current directory)"},
    {"board_debug", "b", &board_debug, "Board module debugging"},
    {"define_debug", "b", &define_debug, "Define module debugging"},
    {"io_debug", "b", &io_debug, "IO module debugging"},
    {"lexer_debug", "b", &lexer_debug, "Lexer module debugging"},
    {"module_debug", "b", &module_debug, "Module debugging"},
    {"parser_debug", "b", &parser_debug, "Parser module debugging"},
    {"pcodesim_debug", "b", &pcodesim_debug, "Pcodesim module debugging"},
    /*{"queue_debug", "b", &queue_debug, "Queue module debugging"},*/
    {"ref_debug", "b", &ref_debug, "Ref module debugging"},
    {"spew_debug", "b", &spew_debug, "Spew module debugging"},
    {"symtab_debug", "b", &symtab_debug, "Symtab module debugging"},
    {"run",        "b", &ic_run, "Run main and exit"},
    {"outfile",    "s", ic_outfile, "Send simulator printing to file"},
    {"version",    "p", (void*) ic_version, "Print version number"},
    {"loopback", "b", &board_loopback, "Expect board to echo characters (rev 1.5)"},
    {"wizard",     "b", &ic_wizard, "Allow dangerous code to be written"},
    {"native", "b", &native_flag, "Use native HC11 codes"},
    LAST_ARG_DESCRIPTOR
};

void ic_usage(char **argv)
{
    usage(ad, argv, "[files to load]");
    exit(1);
}

void ic_help(char **argv)
{
	printf("Arguments are:\n");
	printf("-port printer   or   -port modem     Set which serial port to use\n");
	printf("-libdir directory                    Set which library folder to use\n");
	printf("-lib filename                        Set which library file to load\n");
	printf("-nolib                               Don't load a library file on startup\n");
	printf("-sim                                 Use simulator only\n");
	printf("Examples:\n");
	printf("-nolib -sim                 Runs simulator and doesn't load library\n");
	printf("-libdir :my-favorite-libs:  Find libraries in a folder called 'my-favorite-libs'\n");
}	

#ifdef MAC
void main()
#else
void main(int argc, char **argv)
#endif
{
    char **args;
    Int i;

    UNUSED(argc);

#ifdef MAC
    mac_get_args(ic_help);
#endif

    args= parse_args(ad, argv+1);

#ifdef PC    
    if (ic_sim) {
       printf("Sorry, simulator not implemented for PC version\n");
       ic_sim= 0;
    }
#endif       

    keyb_stream= io_open_stdin();
    io_make_stdin_unbuffered();
    
    printf("       Interactive C Native for 6811.  Version %ld.%02ld%s (%s)\n",
	   ci_major_version, ci_minor_version,
	   ci_under_construction ? " BETA" : " ",
	   date ? date : "");
    printf("\n");
    printf("IC written by Randy Sargent and Anne Wright.  Copyright 1994.\n");
    printf("ICn additions by Kurt Konolige.  Copyright 1995.\n");
    if (!ic_sim) printf("    (uses board pcode by R. Sargent, F. Martin, and A. Wright)\n");

    if (ci_under_construction) {
	printf("WARNING: this version is under construction!\n\n");
    }
    printf("\n");
    printf("This program is freeware and unsupported.  It is provided as a service to\n");
    printf("hobbyists and educators.  Type 'about' for information about support\n");
    printf("and obtaining newer versions of IC.\n");
    printf("\n");

    for (i= 0; args[i]; i++) {
	if (ic_debug) printf("didn't parse %s\n", args[i]);
	if (args[i][0] == '-') {
	    printf("Didn't understand flag %s\n", args[i]);
	    ic_usage(argv);
	}
    }

    if (config) {
	configure_executable(argv[0], keyb_stream, stdout);
	exit(0);
    }

    if (ic_sim) pcodesim_init();
        
    if (ic_sim ||
	!*portname ||
	!stricmp(portname, "none")) {
	serial_stream= 0;
    }
    else {
	serial_stream= io_open_serial(portname);
	if (!serial_stream) {
	    printf("Cannot open serial port %s\n", portname);
	    ic_usage(argv);
	}
    }
#if 0
    /* debugging code here */
    printf("console stream is %ld, serial stream is %ld\n", keyb_stream, serial_stream);
    io_serial_init(9600L, serial_stream);
    { Int q;  for (q= 0; q< 10; q++) { io_putchar('s', serial_stream);  }}
    io_flush(serial_stream);
    exit(0);
#endif    
    board= board_set_streams(serial_stream, keyb_stream);
    if (ic_outfile[0]) {
	FILE *outfile= fopen(ic_outfile, "w");
	if (!outfile) {
	    printf("Could not open %s for writing\n", ic_outfile);
	    exit(1);
	}
	board_set_output(outfile);
    }
    else {
	board_set_output(stdout);
    }
    if (board) {
	Int board_major, board_minor;
	if (serial_stream) {
	    printf("Attempting to link to board on port %s\n", portname);
	}
	else {
	    printf("Resetting simulator\n");
	}
	board_reset(1);
	board_get_version(&board_major, &board_minor);
	spew_msb_first= board_get_msb_first();
	spew_float_type= board_get_float_type();
	printf("Pcode version %ld.%02ld present on board\n",
	       board_major, board_minor);
	if ((board_major*10+board_minor/10) < 40 ||
	    (board_major*10+board_minor/10) > 50) {
	    printf("Incompatible pcode version, or pcode corrupt\n");
	    printf("Please use \"dl\" to reload pcode.\n");
	    exit(0);
	}
	if ((board_major * 10 + board_minor/10) == 26) {
	    /* version 2.6 */
	    old_pcode= 1;
	} else {
	    old_pcode= 0;
	}
	printf("Loading jump table\n");
	load_jumptable(); 
	printf(" done\n");

    }
    else {
	printf("No serial port, so ignoring board\n");
    }

    ci_init();
    {
	Int loaded= 0;
	if (!ci_no_library) {
	    if (ci_load(lib_file)) loaded++;
	}
	for (i= 0; args[i]; i++) {
	    if (ic_debug) printf("looking at arg %s\n", args[i]);
	    if (!ci_load(args[i])) {
		ic_usage(argv);
	    }
	    loaded++;
	}
	if (loaded) {
	    if (!ci_link()) {
		printf("Errors, unloading files\n");
		for (i= 0; args[i]; i++) {
		    ci_unload(args[i]);
		}
		ci_link();
	    }
	}
    }

    free(args);
	    
    ci_mainloop();
}

void ci_restart_compiler(void)
{
    parser_init();
    spew_init();
    define_init();
    spew_set_code_origin(ci_pcode_origin);
    define_set_globals_addr(ci_globals_origin);
    ref_clear();
}

void ci_reset_compiler(void)
{
    symtab_init();
    ci_restart_compiler();
}

void ci_init(void)
{
    type_init_module();
    spew_init_module();
    spew_file_name= 0;
/*    yydebug= 0; */
    ci_reset_compiler();
}

char *file_to_string(FILE *in)
{
    Int c;
    Growbuf g;
    growbuf_init(&g);
    while ((c= getc(in)) != EOF) {
	GROWBUF_ADD_CHAR(&g, c);
    }
    growbuf_add_char(&g, 0);
    return GROWBUF_DATA(&g);
}
    
Int ci_unload(char *filename)
{
    if (module_exists(filename)) {
	printf("Unloading %s.\n", filename);
	module_unload(filename);
	return 1;
    }
    else if (strrchr(filename, '.') &&
	     !strcmp(strrchr(filename, '.'), ".lis")) {
	FILE *source= fopen_here_or_there_with_format(lib_directory,
						      filename, "r",
						      "Unloading %s.\n");
	if (!source) return 0;
	{
	    char *f, **filenames;
	    Scan s;
	    Int i;
	    Int ret= 0;
	    f= file_to_string(source);
	    scan_init(&s, f, strlen(f));
	    filenames= scan_get_strings(&s, ",\n\r \t");
	    for (i= 0; filenames[i]; i++) {
		ret |= ci_unload(filenames[i]);
	    }
	    free(f);
	    free(filenames);
	    return ret;
	}
    }
    else {
	printf("File \"%s\" not loaded.\n", filename);
	return 0;
    }
}

void board_kill_interrupts()
{
    Int system_status, halted;

    printf("Initializing interrupts\n");
    board_write_ram(UI_PROCESS_BUFFER, Pinitint*2);
    board_write_ram(UI_PROCESS_BUFFER+1, Phaltnotify*2);
    board_write_ram_word(PROCESS_TABLE + P_PC, UI_PROCESS_BUFFER);
    board_clear_and_set_ram(PCODE_SYSTEM_STATUS, SYSSTAT_HALTNOTIFY, 0);
    board_write_ram(PROCESS_TABLE + P_STATUS, PSTAT_RUNNING);
    halted= !board_process_wait();

    system_status= board_read_mem(PCODE_SYSTEM_STATUS);

    if (system_status & SYSSTAT_ERRORHALT) ic_handle_errors();
}

/* 1 if successful */
Int ci_link(void)
{
    ci_reset_compiler();
    compile(all_modules);
    if (!errors) {
	if (board) {
	    board_kill_all_processes();
	    board_kill_interrupts();
	    board_download(ci_pcode_origin,
			   code_current - ci_pcode_origin, code, 1);
	    ci_execute_expression("__ci_statement _init_globals();", 0);
	    printf("Globals initialized.\n");
	}
	return 1;
    }
    return 0;
}
	
Int ci_load(char *filename)
{
    FILE *source=
      fopen_here_or_there_with_format(lib_directory, filename, "r",
				 module_exists(filename) ?
				 "Reloading %s.\n" : "Loading %s.\n");
    char *suffix= strrchr(filename, '.');
    Int ret;
    if (!source) {
	return 0;
    }
    if (suffix) {
	if (!strcmp(suffix, ".c")) {
	    if (module_exists(filename)) module_unload(filename);
	    lexer_reset();
	    lexer_push_stream(source);
	    module_load_c(filename);
	    ret= 1;
	}
	else if (!strcmp(suffix, ".icb")) {
	    if (module_exists(filename)) module_unload(filename);
	    module_load_icb(filename, source);
	    ret= 1;
	}
	else if (!strcmp(suffix, ".lis")) {
	    char *f, **filenames;
	    Scan s;
	    Int i;
	    f= file_to_string(source);
	    scan_init(&s, f, strlen(f));
	    filenames= scan_get_strings(&s, ",\n\r \t");
	    ret= 0;
	    for (i= 0; filenames[i]; i++) {
		ret |= ci_load(filenames[i]);
	    }
	    free(f);
	    free(filenames);
	}
	else {
	    printf("Unknown file format (suffix %s)\n", suffix);
	    return 0;
	}
    } else {
	printf("Unknown file format (no suffix)\n");
	return 0;
    }
	
    fclose(source);
    return ret;
}

void ci_set(char *var, Int val)
{
    if (!strcmp(var, "board_debug")) board_debug= val;
    else if (!strcmp(var, "io_debug")) io_debug= val;
    else if (!strcmp(var, "ref_debug")) ref_debug= val;
    else if (!strcmp(var, "define_debug")) define_debug= val;
    else if (!strcmp(var, "lexer_debug")) lexer_debug= val;
    else if (!strcmp(var, "parser_debug")) parser_debug= val;
    else if (!strcmp(var, "symtab_debug")) symtab_debug= val;
    else if (!strcmp(var, "spew_debug")) spew_debug= val;
    else if (!strcmp(var, "pcodesim_debug")) pcodesim_debug= val;
/*    else if (!strcmp(var, "queue_debug")) queue_debug= val;*/
    else if (!strcmp(var, "module_debug")) module_debug= val;
#ifdef MALLOC_DEBUG    
    else if (!strcmp(var, "dmalloc_debug")) dmalloc_debug= val;
#endif    
    else printf("Illegal variable %s\n", var);
}
     
void ci_print(Int pointer, Type *type)
{
    switch (type_id(type)) {
      case char_id:
	{
	    Int val= board_read_mem(pointer);
	    printf("<char> %ld", val);
	    if (32 <= val && val <= 127) printf(" '%c'", (int)val);
	}
	break;
      case int_id:
	{
	    Int val= board_read_mem_word(pointer);
	    if (val > 32767) val -= 65536;
	    printf("<int> %ld", val);
	}
	break;
      case long_id:
	{
	    long val= board_read_mem_long(pointer);

	    printf("<long> %ld", val);
	}
	break;
      case void_id:
	printf("<void>");
	break;
      case float_id:
	{
	    long val= board_read_mem_long(pointer);
	    
	    printf("<float> %lf", spew_read_float(val));
	}
	break;
      case pointer_id:
	{
	    Int obj= board_read_mem_word(pointer);
	    printf("pointer (%04lX) to ", obj);
	    ci_print(obj, type_Pointer_deref(type));
	}
	break;
      case array_id:
	{
	    Int bound= board_read_mem_word(pointer);
	    Type *elemtype= type_Array_elemtype(type);
	    Int i;

	    pointer += 2;

	    printf("array[%ld] {", bound);
	    for (i= 0; i< bound; i++) {
		ci_print(pointer, elemtype);
		if (i < bound-1) printf(", ");
		pointer += type_sizeof(elemtype);
	    }
	    printf("}");
	}
	break;
      default:
	printf("<unknown type>");
	break;
    }
}

void ic_handle_errors(void)
{
    ci_show_process_errors();
    board_clear_and_set_ram(PCODE_SYSTEM_STATUS, SYSSTAT_ERRORHALT, 0);
    board_kill_all_processes();
    return;
}
    
void ci_execute_expression(char *expression, Int display)
{
    Int initial_sp, final_sp, stack_size;
    Int system_status, halted;
    Module *m;
    
    ci_restart_compiler();
    symtab_set_temp_prefix("citemp");
    spew_main= 0;
    spew_set_code_origin(UI_PROCESS_BUFFER);

    lexer_reset();
    lexer_push_string(expression);
    m= module_create_c("user interaction");
    
    compile(m);
    
    symtab_set_temp_prefix("temp");
    spew_main= 1;
    if (errors) {
	symtab_backup_to_global_level();
	return;
    }

    {
	long bytes_over= code_current - UI_PROCESS_BUFFER_END + 1;
	if (bytes_over > 0) {
	    printf("Expression %ld bytes too long to download into board\n",
		   bytes_over);
	    return;
	}
    }
    
    if (!board) return;
    board_download(UI_PROCESS_BUFFER, code_current - UI_PROCESS_BUFFER,
		   code, display);
    board_write_ram(code_current, Phaltnotify * 2);

    initial_sp= board_read_mem_word(PROCESS_TABLE + P_SP);
    
    board_write_ram_word(PROCESS_TABLE + P_PC, UI_PROCESS_BUFFER);
    board_clear_and_set_ram(PCODE_SYSTEM_STATUS, SYSSTAT_HALTNOTIFY, 0);
    board_write_ram(PROCESS_TABLE + P_STATUS, PSTAT_RUNNING);

    halted= !board_process_wait();

    system_status= board_read_mem(PCODE_SYSTEM_STATUS);

    if (system_status & SYSSTAT_ERRORHALT) {
	ic_handle_errors();
	board_write_ram_word(PROCESS_TABLE + P_SP, initial_sp);
	return;
    }
    
    if (halted) {
	printf("Halting execution\n");
	board_write_ram(PROCESS_TABLE + P_STATUS, PSTAT_HALTED);
	board_write_ram_word(PROCESS_TABLE + P_SP, initial_sp);
	return;
    }

    final_sp= board_read_mem_word(PROCESS_TABLE + P_SP);

    if (define_stack_items() > 0) {
	Type *ret_type= define_pop();
	
	stack_size= type_sizeof(ret_type);
	if (display) {
	    printf("Returned ");
	    ci_print(final_sp, ret_type);
	    printf("\n");
	}
    } else {
	stack_size= 0;
    }

/*    printf("stack size %d , final sp %d , initial_sp %d\n",
	   stack_size, final_sp, initial_sp);*/
    
    if (stack_size + final_sp - initial_sp) {
	printf("COMPILER ERROR!: %ld extra bytes left on board stack\n",
	       stack_size + initial_sp - final_sp);
    }
    board_write_ram_word(PROCESS_TABLE + P_SP, initial_sp);
}
void wait_for_key()
{
   printf("Press any key to continue...");
   fflush(stdout);
   io_getchar(keyb_stream, -1);
   printf("\n");
}

void ic_about()
{
   printf("Thanks for using IC!\n");
   printf("The IC software system was written by Randy Sargent (rsargent@media.mit.edu)\n");
   printf("and Anne Wright (anarch@media.mit.edu)\n");
   printf("It makes use of a pcode interpreter and drivers that live on the\n");
   printf("robot board, which was written by Randy Sargent, Fred Martin,\n");
   printf("and Anne Wright\n");
   
   printf("\n");
   printf("This version of IC is freeware and unsupported.  It is provided as a service\n");
   printf("to hobbyists and educators.\n");
   printf("\n");
   printf("This version does continue to be updated and have reported bugs fixed,\n");
   printf("but most of our effort now goes into a newer IC (version 3.0+), which\n");
   printf("includes features such as structs, multidimensional arrays, and a\n");
   printf("built-in preprocessor.  Versions 3.0 and above are sold by Anne and\n");
   printf("Randy out of our apartment in hopes we can support ourselves doing cool\n");
   printf("robotics hardware and software; it is not freeware (sorry), but at\n");
   printf("least it does come with support :)\n");
   printf("\n");
   wait_for_key();
   printf("This free version IC is still very functional, having been used for years\n");
   printf("by the MIT LEGO Robot Design contest.  Updates to free IC (versions < 3.0)");
   printf("are available by anonymous FTP to cher.media.mit.edu in pub/interactive-c\n");
   printf("\n");
   printf("Version 3.0 of IC is available from Randy Sargent and Anne Wright\n");
   printf("for $50 + $5 S/H at the following address:\n");
   printf("Randy Sargent\n");
   printf("Imachinations\n");
   printf("2 Newton St.  Apt. 1\n");
   printf("Cambridge, MA 02139\n");
   printf("(617) 547-6664\n");
   printf("(Checks, Mastercard, and Visa accepted.)\n");
   printf("International orders please pay $15 S/H\n");
   printf("PLEASE SPECIFY -one- of IBM PC, Mac, SPARC, DECstation, or NeXT 68K\n");
   printf("Otherwise, IBM PC will be assumed.\n");
   printf("(Available 9/1/94)\n");
   printf("\n");
   wait_for_key();
   printf("Recommendations on hardware:\n");
   printf("This software was designed to be compatible with the 6811-based Rev 2.x\n");
   printf("robot controller designed at MIT for the LEGO Robot Design Competition.\n");
   printf("(designed by Fred Martin and Randy Sargent). It also works with the Rug\n");
   printf("Warrior from Mobile Robots:Inspiration to Implementation, by Joe Jones\n");
   printf("and Anita Flynn (highly recommended reading!)\n");
   printf("\n");
   printf("Our experience founding the MIT LEGO Robot Design Competition\n");
   printf("suggests that only about 50% of boards assembled from kits work\n");
   printf("the first time.  For this reason, Anne and Randy strongly recommend\n");
   printf("that you buy completed and tested hardware.\n");
   printf("We get lots of messages from folks saying \"Nothing works.  What's wrong?\"\n");
   printf("Future versions of the software may come with more diagnostics (currently\n");
   printf("there's only testbrd.c), but we regret that we really can't help people\n");
   printf("debug their hardware remotely.  Talk to the folks who sold you the kit.\n");
   printf("\n");
   wait_for_key();
   printf("We have had some requests from people who are desperate to get their boards\n");
   printf("up, and they have sent us their boards and had them repaired, for a fee.\n");
   printf("Hopefully other people (maybe students) can offer this service cheaper\n");
   printf("than we can; but if you want us to fix your board, you can send it,\n");
   printf("along with $100 for Rev 2 + expansion board, or $75 or Rev 2 without\n");
   printf("expansion, or Rug Warrior, to the above address.  I really wish I could do\n");
   printf("it cheaper, but it takes some time and I have to pay rent :(\n");
   printf("\n");
   wait_for_key();
   printf("We hope you are successful in playing with robotics, and hope\n");
   printf("that IC is useful to you.\n");
   printf("Randy Sargent and Anne Wright\n");
   printf("July 29, 1994");
}
    
void ci_help(char *topic)
{
    UNUSED(topic); /* TODO: implement topic */
    printf("Help:\n");
    printf("\n");
    printf("You may type a C expression at the prompt.  Most C expressions end\n");
    printf("with ; and others may end with }\n");
    printf("\n");
    printf("You may also type one of the following commands:\n");
    printf("\n");
    printf("about                 Display useful info about IC\n");
    printf("load filename         Load filename.c into the board\n");
    printf("unload filename       Unload filename.c from the board\n");
    printf("ps                    Print the status of currently executing processes\n");
    printf("kill_all              Kill all processes\n");
    printf("quit                  Quit\n");
    printf("help                  Display this message\n");
    /*printf("shell                 Run an inferior shell\n");*/
    printf("list files            List all files\n");
    printf("list functions        List all functions\n");
    printf("list globals          List all global variables\n");
    printf("pb                    Peek byte at address\n");
    printf("pw                    Peek word at address\n");
#ifdef MAC
    printf("mem                   Show free memory on host\n");
#endif
#ifdef PC 
    printf("mem                   Show free memory on host\n");
#endif
#ifdef MALLOC_DEBUG
    printf("print_malloc          Print malloc entries\n");
#endif
}

#if 0
Int ci_reset()
{
    char data[400];
    Int  addr, len, i;
    Int  current_addr= 0;

    FILE *in= fopen_here_or_there(LIB_DIRECTORY, "pboot.icb", "r");
    if (!in) return 0;
    printf("Downloading pboot.icb into zero page\n");

    for (i= 0; i< 256; i++) data[i]= 0;

    while (s19_get_line(in, &addr, &len, data+current_addr)) {
	if (addr != current_addr) {
	    printf("Bad bootstrap .s19 file, aborting\n");
	    die(("foo"));
	    return 0;
	}
	current_addr += len;
    }
    printf("Sending 256 bytes (%d bytes data, %d bytes filler)\n",
	   current_addr, 256 - current_addr);
    if (!board_special_download(data)) return 0;

    printf("Finished\n");
    fclose(in);
    
    if (!ci_load("pcode.s19")) return 0;

    return 1;
}
#endif

Int s19_get_line(FILE *in, Int *addr, Int *len, unsigned char *data)
{
    Int c, i, byte;
    while (isspace(c= getc(in)));
    if (c != 's' && c != 'S') die(("foo")); /*goto error;*/

    c= getc(in);
    if (c == '9') return 0;
    if (c != '1') die(("foo")); /*goto error;*/

    if (fscanf(in, "%2lx%4lx", len, addr) != 2) die(("foo")); /*goto error;*/
    (*len) -= 3;
    for (i= 0; i< *len; i++) {
	if (fscanf(in, "%2lx", &byte) != 1) die(("foo")); /*goto error;*/
	data[i]= (char) byte;
    }
    if (fscanf(in, "%2lx", &byte) != 1) die(("foo"));

    printf("returning addr %ld, len %ld\n", *addr, *len);
    return 1;
#if 0
    printf("Illegal .s19 format, aborted\n");
    die(("foo"));
    return 0;
#endif    
}

char *ci_name_from_pc(long pc)
{
    Symbol *sym= symbol_from_pc(pc);
    char *name= sym ? symbol_name(sym) : "<unknown>";
    if (pc >= UI_PROCESS_BUFFER && pc <= UI_PROCESS_BUFFER_END) {
	name= "<command line interaction>";
    }
    return name;
}

char *ci_error_name(Int stat)
{
    Int i;
    for (i= 0; system_errors[i].name; i++) {
	if (system_errors[i].id == stat) {
	    return system_errors[i].name;
	}
    }
    printf("unknown error status %ld\n", stat);
    return "!!Unknown error!!";
}

void ci_show_process_errors(void)
{
    Int process_ptr= PROCESS_TABLE;

    if (board) {
	do {
	    Int stat= board_read_mem(process_ptr + P_STATUS);
	    
	    if (stat != 0 && stat != 20) {
		
		printf("%s while executing %s (process # %ld)\n", 
		       ci_error_name(stat),
		       ci_name_from_pc((long) (unsigned Int) board_read_mem_word(process_ptr +
								  P_PC)),
		       board_read_mem_word(process_ptr + P_ID));
	    }
	    
	    process_ptr= board_read_mem_word(process_ptr + P_NEXT);
	} while (process_ptr != PROCESS_TABLE);
    }
}
    
/*  prints PC, current SP, and beginning SP of all running processes */
void ci_ps(void)
{
    Int process_ptr= PROCESS_TABLE;

    printf("  PID Stat   PC   SP  ORG Ticks   Executing\n");

    if (board) {
	do {
	    long pc= (long) (unsigned Int) board_read_mem_word(process_ptr + P_PC);
	    char *name= ci_name_from_pc(pc);
	    
	    printf("%5ld   %02lx %04lx %04lx %04lx   %3ld   %s\n",
		   board_read_mem_word(process_ptr + P_ID),
		   board_read_mem(process_ptr + P_STATUS),
		   0xffffl & pc,
		   board_read_mem_word(process_ptr + P_SP),
		   board_read_mem_word(process_ptr + P_STACK_ORG),
		   board_read_mem(process_ptr + P_TICKS),
		   name
		   );
	    
	    process_ptr= board_read_mem_word(process_ptr + P_NEXT);
	} while (process_ptr != PROCESS_TABLE);
    }
}

void ci_peekbyte(Int addr)
{
    if (board) {
	Int byte= board_read_mem(addr);

	printf("%04lx is %02lx\n", addr, byte);
    }
}

void ci_peekword(Int addr)
{
    if (board) {
	Int word= board_read_mem_word(addr);

	printf("%04lx:%02lx is %04lx\n", addr, (addr+1) % 256, word);
    }
}

void ci_spewprocdata(Int addr)
{
    if (board) {
	Int pc= board_read_mem_word(addr+P_PC);
	Int sp= board_read_mem_word(addr+P_SP);
	Int sporg= board_read_mem_word(addr+P_STACK_ORG);
	Int splim= board_read_mem_word(addr+P_STACK_LIM);
	Int ticks= board_read_mem(addr+P_TICKS);
	Int status= board_read_mem(addr+P_STATUS);
	Int pid= board_read_mem_word(addr+P_ID);
	Int prev= board_read_mem_word(addr+P_PREV);
	Int next= board_read_mem_word(addr+P_NEXT);
	
	printf("PC   SP   ORG  LIM  TICK STAT PID  PREV NEXT\n");
	printf("%04lx %04lx %04lx %04lx %02lx   %02lx   %04lx %04lx %04lx\n", 
	       pc, sp, sporg, splim, ticks, status, pid, prev, next);
    }
}

Int hextoi(char *hex)
{
    Int ret= 0;
    sscanf(hex, "%ld", &ret);
    return ret;
}

void ic_list_functions(void)
{
    Symbol *sym;
    Value *val;

    for (sym= symtab_get_binding(&val, 1);
	 sym;
	 sym= symtab_get_binding(&val, 0)) {

	if (value_type(val) &&
	    type_id(value_type(val)) == func_id) {
	    printf("%s\n", symbol_name(sym));
	}
    }
}

void ic_list_globals(void)
{
    Symbol *sym;
    Value *val;

    for (sym= symtab_get_binding(&val, 1);
	 sym;
	 sym= symtab_get_binding(&val, 0)) {

	if (value_type(val) &&
	    type_id(value_type(val)) != func_id &&
	    type_id(value_type(val)) != label_id) {
	    printf("%s\n", symbol_name(sym));
	}
    }
}

void ic_list_files(void)
{
    Module *m;
    for (m= all_modules; m; m=m->next) {
	printf("%s\n", m->name);
    }
}

#ifdef PC
long mem_free(void)
{
   int size=10000;
   char **ptr= NULL;
   long memfree= 0;

   while (size >= 100) {
      char **new= malloc(size);
      if (!new) {
	 size /= 10;
	 continue;
      }
      new[0]= (char*) ptr;
      ptr= new;
      memfree += size;
   }

   while (ptr) {
      char **next= (char**) (ptr[0]);
      free(ptr);
      ptr= next;
   }

   return memfree;
}
#endif

void ci_mainloop(void)
{
    char command_line[1000], interpret_buffer[1000];
    char **args, *command_name;
    Int nargs, i;
    Int arg1x= 0;
    Scan scan;
    int mem_warning= 0;

    if (ic_run) {
	printf("Running main\n");
	ci_execute_expression("__ci_statement main();", 1);
	printf("Done\n");
	exit(0);
    }
    if (ic_profile) {
	printf("Linking library 3 more times for profiling\n");
	for (i= 0; i< 3; i++) {
	    ci_link();
	}
	exit(0);
    }
    printf("Type 'help' for help\n");
    while (1) {
#ifdef PC
       {
	  long mem= mem_free();
	  if (mem < 60000 && !mem_warning) {
	     printf("IC is now dangerously low on memory.  You should make sure your\n");
	     printf("PC has at least 550K-600k of lower memory free before starting\n");
	     printf("(You can check with the dos 'mem' command).\n");
	     mem_warning= 1;
	  }
	  if (mem < 20000) {
	     printf("Sorry, IC is out of memory.  Press any key to quit\n");
	     while (io_getchar(keyb_stream, 0) != EOF);
	     io_getchar(keyb_stream, -1);
	     exit(0);
	  }
       }
#endif       
	         
       command_line[0]= 0;
       cmd_get_line("C> ", keyb_stream, stdout, command_line);
       printf("\n");
       
       /* Is it a C expression? */
       for (i= strlen(command_line)-1; i; i--)
	  if (command_line[i] != ' ') break;
       if (command_line[i] == ';' ||
	   command_line[i] == '}') {
	  sprintf(interpret_buffer, "__ci_statement %s", command_line);
	  ci_execute_expression(interpret_buffer, 1);
	  /* Yow!  This was a big bug!!! */
	  /*free(args);*/
	  continue;
       }
       
       /* Nope.  Try to treat it like a command. */
       scan_init(&scan, command_line, strlen(command_line));
       args= scan_get_strings(&scan, " \t,");
       for (nargs= 0; args[nargs]; nargs++);
       nargs--;
       if (nargs < 0) {
	  free(args);
	  continue;
       }
       command_name= args[0];
       if (nargs >= 1) {
	  arg1x= hextoi(args[1]);
       }
       if (!stricmp(command_name, "about")) {
	  ic_about();
       } else
       if (!stricmp(command_name, "help")) {
	  ci_help( nargs == 0 ? 0 : args[1]);
       }
       else if (!stricmp(command_name, "quit") ||
		!stricmp(command_name, "bye")  ||
		!stricmp(command_name, "exit")) {
	  io_sigint_interrupt();
	    exit(0);
	}
#ifdef MAC
	else if (!stricmp(command_name, "mem")) {
	    long contig_size, total_size;
	    PurgeSpace(&total_size, &contig_size);
	    printf("Total space is %ldK (max contiguous block is %ldK)\n",
	    	total_size/1024, contig_size/1024);
	}	
#endif
#ifdef PC
	else if (!stricmp(command_name, "mem")) {
	   printf("%ld bytes on host\n", mem_free());
	}	
#endif
#ifdef MALLOC_DEBUG
	else if (!stricmp(command_name, "print_malloc")) {
	  char fname[100];
	  FILE *logfile;
	  sprintf(fname, "/tmp/malloc.out.%d", debug_malloc_lognum);
	  logfile= fopen(fname, "w");
	  /*debug_malloc_print(stdout);*/
	  printf("Logging to file %s\n",fname);
	  debug_malloc_print(logfile);
	  fclose(logfile);
	}	
#endif
	else if (!stricmp(command_name, "list")) {
	    if (nargs > 0 && !stricmp(args[1], "functions")) {
		ic_list_functions();
	    }
	    else if (nargs > 0 && !stricmp(args[1], "globals")) {
		ic_list_globals();
	    }
	    else if (nargs > 0 && !stricmp(args[1], "files")) {
		ic_list_files();
	    }
	    else {
		printf("Use  list files,  list functions,  or list globals\n");
	    }
	}
	else if (!stricmp(command_name, "load")) {
	    Int relink= 0;
	    for (i= 0; i< nargs; i++) {
		relink |= ci_load(args[i+1]);
	    }
	    if (relink) {
		if (!ci_link()) {
		    for (i= 0; i< nargs; i++) {
			ci_unload(args[i+1]);
		    }
		}
	    }
	}
	else if (!stricmp(command_name, "unload")) {
	   Int relink= 0;
	   for (i= 0; i< nargs; i++) {
	      relink |= ci_unload(args[i+1]);
	   }
	   if (relink) ci_link();
	}
	else if (!stricmp(command_name, "link")) {
	   if (nargs != 0) ci_help("link");
	   else ci_link();
	}
	else if (!stricmp(command_name, "set")) {
	   if (nargs == 1)
	      ci_set(args[1], 1);
	   else
	      ci_set(args[1], atoi(args[2]));
	}
	else if (!stricmp(command_name, "procdata")) {
	   ci_spewprocdata(arg1x);
	}
#if 0	
        else if (!stricmp(command_name, "reset")) {
           ci_reset();
	}
#endif	
	else if (!stricmp(command_name, "kill_all")) {
	   board_kill_all_processes();
	}
	else if (!stricmp(command_name, "ps")) {
	   ci_ps();
	}
	else if (!stricmp(command_name, "pb")) {
	   ci_peekbyte(arg1x);
	}
	else if (!stricmp(command_name, "pw")) {
	   ci_peekword(arg1x);
	}
	else {
	   printf("Illegal command '%s'.  Type help for help.\n", command_name);
	}
        free(args);
    }
}


