/**
 ** module.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include "core.h"

#include "cdef.h"
#include <ctype.h>

/*-------------------------------------------------------------------------*/
/* Private Variables                                                       */

void *module__tokens;

/*-------------------------------------------------------------------------*/
/* Public Variables                                                        */

Module *module_current_module;
Module *module_last_module;
Module *all_modules= NULL;

Int module_debug= 0;

/*-------------------------------------------------------------------------*/
/* Public Functions                                                        */

Module *module_error_occurred(void)
{
    if (module_current_module) return module_current_module;
    if (module_last_module) return module_last_module;
    die(("No module in module_error_occurred\n"));
}

void module_start_pass(void)
{
    if (module_debug) printf("Module start pass\n");
    module_last_module= NULL;
}

Int module_name_cmp(char *specified_name, char *module_name, Int fuzziness)
{
    if (!stricmp(specified_name, module_name)) return 1;
    if (fuzziness) {
	Int ret;
	char *filename= string_remove_pathname(module_name);
	ret= !stricmp(specified_name, filename);
	free(filename);
	return ret;
    }
    return 0;
}
	
Int module_exists(char *name)
{
    Module *m;
    Int fuzziness;
    for (fuzziness= 0; fuzziness <= 1; fuzziness++) {
	for (m= all_modules; m; m= m->next)
	  if (module_name_cmp(name, m->name, fuzziness)) return 1;
    }
    return 0;
}

void module_set_input(Module *m)
{
    if (module_debug) {
        printf("Module input set to %s\n", 
            m ? m->name : "NULL");
    }
    lineno= 1;
    module_current_module= m;
    if (m) module__tokens=growbuf_data(&m->tokens);
}
    
void module_destroy(Module *m)
{
    Int type;
    m->next= NULL;
    module_set_input(m);
    while ( (type= module_get_token()) ) {
	switch (type) {
	  case STRING:
	    free(yylval.string);
	    break;
	  default:
	    break;
	}
    }
    if (m->binrec1) {
      binrec_destroy(m->binrec1);
      m->binrec1= NULL;
    }
    if (m->binrec2) {
      binrec_destroy(m->binrec2);
      m->binrec2= NULL;
    }
    growbuf_term(&m->tokens);
    module_set_input(NULL);
    free(m->name);
    m->name= NULL;
    free(m);
}
	
void module_unload(char *name)
{
    Module **mhandle;
    Module *m;
    Int fuzziness;

    for (fuzziness= 0; fuzziness <= 1; fuzziness++) {
	for (mhandle= &all_modules; *mhandle; mhandle= &((*mhandle)->next)) {
	    if (module_name_cmp(name, (*mhandle)->name, fuzziness)) {
		/* Got it */
		m= *mhandle;
		*mhandle= m->next;
		goto gotcha;
	    }
	}
    }
    /* Did not find it */
    die(("no such module %s in module_unload", name));
  gotcha:
    module_destroy(m);
    /* TODO: uncompile here */
}

void module_add(Module *m)
{
    Module **mhandle;
    Module *next_module;
    
    /* Stick at end of modules.  This is to keep order of compilation same
       as order of load, which means we might be able to compile a new file
       without recompiling old if there were no previous missing references. */
    /* Ha ha ha!  Never mind.  Let's stick all the icb's at the end, because
       their use of the stack in evaluating things messes up our persistents */

    for (mhandle= &all_modules; 1; mhandle= &((*mhandle)->next)) {
	if (!*mhandle) break;  /* stick it in the last one */
	if ((*mhandle)->type == module_type_icb &&
	    m->type == module_type_c) break; /* stick a c before an icb */
    }

    next_module= *mhandle;
    *mhandle= m;
    m->next= next_module;
}

void module_load_icb(char *name, FILE *in)
{
    Module *m;
    
    if (module_exists(name)) die(("module %s already loaded\n", name));

    m= module_create_icb(name, in);
    module_add(m);
}

void module_load_c(char *name)
{
    Module *m;
    
    if (module_exists(name)) die(("module %s already loaded\n", name));

    m= module_create_c(name);
    module_add(m);
}

Module *module_create_c(char *name)
{
    Module *m;
    Int type;
    Int my_line_number= 1;
    
    m= malloc(sizeof(Module));
    m->type= module_type_c;
    m->name= string_copy(name);
    growbuf_init(&m->tokens);
    m->next= NULL;

    m->binrec1= NULL;
    m->binrec2= NULL;
    lineno= 1;
    
    while ( (type= yylex() )) {
	while (my_line_number < lineno) {
	    growbuf_add_int(&m->tokens, NEWLINE);
	    my_line_number++;
	}
	growbuf_add_int(&m->tokens, type);
	switch (type) {
	  case INTEGER:
	    growbuf_add_int(&m->tokens, yylval.i);
	    break;
	  case LONG_INT:
	    growbuf_add_long(&m->tokens, yylval.l);
	    break;
	  case FLOATING_POINT:
	    growbuf_add_float(&m->tokens, (float) yylval.d);
	    break;
	  case STRING:
	    growbuf_add_ptr(&m->tokens, yylval.string);
	    break;
	  case SYMBOL:
	    growbuf_add_ptr(&m->tokens, yylval.s);
	    break;
	  default:
	    break;
	}
    }
    growbuf_add_int(&m->tokens, FILE_SEPARATOR);
    growbuf_add_int(&m->tokens, 0);
    return m;
}

Int module_init_number= 0;

Module *module_create_icb(char *name, FILE *in)
{
    Module *m;
    Int addr;
    char buffer[150], symname[100];
    char init_name[40];
    init_name[0]=0;
    
    m= malloc(sizeof(Module));
    m->type= module_type_icb;
    m->name= string_copy(name);
    growbuf_init(&m->tokens);
    m->next= NULL;

    growbuf_add_int(&m->tokens, BINARY_MODULE);
    growbuf_add_int(&m->tokens, SEMICOLON);

    m->binrec1= read_contiguous_s19_file(in);
    m->binrec2= read_contiguous_s19_file(in);

    if (m->binrec1->len != m->binrec2->len) {
	die(("incompatible first and second half in .icb file"));
    }

    while (fgets(buffer, 150, in)) {
	char *buf= buffer;
	while (isspace(*buf)) buf++;
	
	if (!strnicmp(buf, "subroutine_", 11)) {
	    /* defining a subroutine */
	    sscanf(buf, "%s %lx", symname, &addr);
	    growbuf_add_int(&m->tokens, BINARY_SUBROUTINE_DEF);
	    growbuf_add_int(&m->tokens, SYMBOL);

	    if (!strcmp(symname+11, "initialize_module")) {
		sprintf(init_name, "__initialize_module_%ld", module_init_number++);
		growbuf_add_ptr(&m->tokens, symbol_create(init_name));
	    }
	    else {
		growbuf_add_ptr(&m->tokens, symbol_create(symname+11));
	    }
	    growbuf_add_int(&m->tokens, INTEGER);
	    growbuf_add_int(&m->tokens, addr);
	    growbuf_add_int(&m->tokens, SEMICOLON);
	}
	if (!strnicmp(buf, "variable_", 9)) {
	    /* defining a variable */
	    sscanf(buf, "%s %lx", symname, &addr);
	    growbuf_add_int(&m->tokens, BINARY_VARIABLE_DEF);
	    growbuf_add_int(&m->tokens, SYMBOL);
	    growbuf_add_ptr(&m->tokens, symbol_create(symname+9));
	    growbuf_add_int(&m->tokens, INTEGER);
	    growbuf_add_int(&m->tokens, addr);
	    growbuf_add_int(&m->tokens, SEMICOLON);
	}
    }
    if (init_name[0]) {
	growbuf_add_int(&m->tokens, EXECUTE_ON_STARTUP);
	growbuf_add_int(&m->tokens, SYMBOL);
	growbuf_add_ptr(&m->tokens, symbol_create(init_name));
	growbuf_add_int(&m->tokens, LEFTPAREN);
	growbuf_add_int(&m->tokens, INTEGER);
	growbuf_add_int(&m->tokens, 0);
	growbuf_add_int(&m->tokens, RIGHTPAREN);
	growbuf_add_int(&m->tokens, SEMICOLON);
    }
    growbuf_add_int(&m->tokens, FILE_SEPARATOR);
    growbuf_add_int(&m->tokens, 0);
    return m;
}

Int module_get_token(void)
{
    Int type;

    if (module_current_module) module_last_module= module_current_module;
    
  try_again:
    while (! (type= *(Int*)module__tokens)) {
	lineno=1;
	/* Ran out of tokens.  Go to next module. */
	if (module_current_module)
	  module_current_module= module_current_module->next;
	if (!module_current_module) {
            if (module_debug) printf("Module EOF\n");
            return 0;
        }
	module__tokens=growbuf_data(&module_current_module->tokens);
    }
    
    {
        /* ((Int*)module__tokens)++; */
    	Int *ptr= module__tokens;
    	ptr++;
    	module__tokens= ptr;
    }
    switch (type) {
      case NEWLINE:
	lineno++;
	goto try_again;
      case INTEGER:
        {
      	  /* yylval.i= *((Int*)module__tokens)++; */
      	  Int *ptr;
      	  ptr= module__tokens;
      	  yylval.i = *ptr++;
      	  module__tokens= ptr;
        }
	    break;
      case LONG_INT:
        {
      	  /* yylval.l= *((long*)module__tokens)++; */
      	  long *ptr;
      	  ptr= module__tokens;
      	  yylval.l = *ptr++;
      	  module__tokens= ptr;
	    }
	    break;
      case FLOATING_POINT:
        {
      	  /* yylval.d= (double) *((float*)module__tokens)++; */
      	  float *ptr;
      	  ptr= module__tokens;
      	  yylval.d = (double) *ptr++;
      	  module__tokens= ptr;
	    }
	    break;
      case STRING:
		{
			/* yylval.string= *((char**)module__tokens)++; */
			char **ptr;
			ptr=module__tokens;
			yylval.string=(char *)*ptr++;
			module__tokens=ptr;
		}
		break;
      case SYMBOL:
		{
			/*yylval.s= *((Symbol**)module__tokens)++;*/
			Symbol **ptr;
			ptr=module__tokens;
			yylval.s=(Symbol *)*ptr++;
			module__tokens=ptr;
		}
		break;
      default:
		break;
    }
    if (module_debug) {
        printf("Module: returning token type %ld\n", type);
    }
    return type;
}
	    

