/**
 ** spew.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* 1147 no opt */
/* 1135 global peek opt */
/* 1045 global peek + local peek opt */

/* 1247 global peek + local peek opt */
/* 1244 poke_nopop opt */

/* optimizations to do:
   {poke constant, speek} -> {peeki}
   {push pop} -> {}
   {peek pop} -> {pop}
   */

#define SPEW_MODULE

#include "core.h"
#include <math.h>
#include <board.h>

#ifdef PC
#pragma optimize("q", off)
#endif

void 	    spew_push_op      (Int size);

/*-------------------------------------------------------------------------*/
/* Private Constants                                                       */

#define INITIAL_CODE_SIZE 8192
#define CODE_SIZE_GROW 4096

/*-------------------------------------------------------------------------*/
/* Private Variables                                                       */

Opcode *spew__int_binops=0;
Opcode *spew__int_unops=0;
Opcode *spew__long_binops=0;
Opcode *spew__long_unops=0;
Opcode *spew__float_binops=0;
Opcode *spew__float_unops=0;
FILE   *spew__out_file;
long   spew__check_stack_pc;

Opcode spew__last_op;
long   spew__last_op_loc;
long   spew__last_sprel_offset;
long   spew__code_size= 0;

long   spew__module_start;

/*-------------------------------------------------------------------------*/
/* Public Variables                                                        */

Int spew_module= 0;
Int spew_msb_first= 1;
Int spew_float_type= float_6811;

unsigned char *code= 0;
long code_origin;
long code_current= 0;
Int  spew_codegen= 1;
Int  spew_debug= 0;
Int  spew_bitbucket_code;
Int  debug_spew_float= 0;
char *spew_file_name= "out.bin";
Int  spew_main= 1;

Int  spew_optimize_read_global= 1;
Int  spew_optimize_read_local= 1;
Int  spew_optimize_poke_nopop= 1;

/*-------------------------------------------------------------------------*/
/* Public Functions                                                        */

/*
 * Fill pcode addresses
 */

Int jumptable[128];

Opcode name_to_opcode(char *name);

void
load_jumptable()
{
  Opdef *op = opcodes;
  Int i;

  for (i= 0; opcodes[i].name; i++) {
    opcodes[i].opcode= name_to_opcode(opcodes[i].name);
  }

  while (op->name)
    {
      op->addr = board_read_mem_word(JUMPTABLE + 2*op->opcode);
      jumptable[op->opcode] = op->addr;
      printf(".");  fflush(stdout);
      op++;
    }
}




void spew_optimize(void)
{
    spew_optimize_read_global= 1;
    spew_optimize_read_local= 1;
}

void spew_init(void)
{
    static Int done= 0;
    if (!done) {
	Int i;
	done= 1;
	spew__code_size= INITIAL_CODE_SIZE;
	code= (unsigned char *) malloc(sizeof(code[0]) * INITIAL_CODE_SIZE);
	spew__int_binops   = (Opcode*) malloc(128*sizeof(Opcode));
	spew__int_unops    = (Opcode*) malloc(128*sizeof(Opcode));
	spew__long_binops  = (Opcode*) malloc(128*sizeof(Opcode));
	spew__long_unops   = (Opcode*) malloc(128*sizeof(Opcode));
	spew__float_binops = (Opcode*) malloc(128*sizeof(Opcode));
	spew__float_unops  = (Opcode*) malloc(128*sizeof(Opcode));

	for(i= 0; i< 128; i++) {
	    spew__int_binops[i]= spew__int_unops[i]= Pundefined;
	    spew__long_binops[i]= spew__long_unops[i]= Pundefined;
	    spew__float_binops[i]= spew__float_unops[i]= Pundefined;
	}

	for (i= 0; opcodes[i].name; i++) {
	    Opcode *op_array= 0;
	    switch(opcodes[i].argtype) {
	      case int_id:
		switch(opcodes[i].mode) {
		  case BINOP: op_array= spew__int_binops; break;
		  case UNOP:  op_array= spew__int_unops;  break;
		}
		break;
	      case long_id:
		switch(opcodes[i].mode) {
		  case BINOP: op_array= spew__long_binops; break;
		  case UNOP:  op_array= spew__long_unops;  break;
		}
		break;
	      case float_id:
		switch(opcodes[i].mode) {
		  case BINOP: op_array= spew__float_binops; break;
		  case UNOP:  op_array= spew__float_unops;  break;
		}
		break;
	      default:
		break;
	    }
	    if (op_array) op_array[opcodes[i].id]= opcodes[i].opcode;
	}
    }
    code_origin= PCODE_ORIGIN;
}

void spew_start_pass(void)
{
    native_flag = 0;
    code_inline = 0;
    code_current= code_origin;
    spew__last_op= -1;

    if (spew_debug) printf("\nvvvvvvvvvvvv spew_start_pass %d vvvvvvvvvvv\n",pass);
    if (pass == 1 && spew_debug) {
	spew__out_file= fopen("out.bin", "w");
    } else {
	spew__out_file= NULL;
    }
    if (spew__out_file) printf("\n Outputting to out.bin\n");
    spew_bitbucket_code= 0;

    if (spew_main) {
	Symbol *init_sym= symbol_create("_init_globals");
	Symbol *main_sym= symbol_create("main");
	Symbol *temp= define_temp_symbol();

	if (pass != 0) {
	    Value *main_value= symtab_get(main_sym);
	    
	    spew_byte(0x01);
	    if (main_value && type_id(value_type(main_value)) == func_id) {
		spew_begin_start_process(main_sym);
		spew_continue_start_process(main_sym);
		spew_push_int(PROC_DEFAULT_TICKS);
		spew_push_int(PSTACK_DEFAULT_SIZE);
		spew_end_start_process(main_sym);
		spew_pop();
		spew_op(Phaltnotify);
	    } else {
		spew_begin_call(init_sym, temp);
		spew_end_call(init_sym, temp);
		spew_pop();
		spew_op(Phaltnotify);
	    }
	}

	define_begin_block();
	define_procedure(type_Void(), init_sym);
	define_end_block();
        
        spew_op(Pinitint);
        
	/* save the old SP */

	spew_sprel(0);
	spew_op(Ploadreg); spew_byte(0);
	
	spew_op(Ppush2);   spew_word(GLOBALS_ORIGIN);
	spew_op(Psetsp);
    }
}

void spew_end_pass(void)
{
    if (spew_main) {
	spew_op(Pfetchreg); spew_byte(0);
	spew_op(Psetsp);
	spew_op(Pmret0);
	spew_word(0);
    }
    if (spew__out_file) fclose(spew__out_file);
}

void spew_load_word(unsigned char *dest, Int source)
{
    if (spew_msb_first) { 
        *dest++= (char) (source >> 8);
        *dest=   (char) (source >> 0);
    }
    else {
        *dest++= (char) (source >> 0);
        *dest= (char) (source >> 8);
    }
}

void check_address(long address)
{
    if (address < code_origin ||
	address >= code_origin + spew__code_size) {
	die(("Illegal address in spew\n"));
    }
}
    
void spew_write_word(long address, Int word)
{
    check_address(address);
    spew_load_word(code + (address - code_origin), word);
}

void spew_write_byte(long address, Int byte)
{
    check_address(address);
    code[address - code_origin]= (char) byte;
}

void spew__byte(Int x)
{
/*    printf("\n<stack size %d>", current_stack_size);    */
    if (spew__out_file && !spew_bitbucket_code) {
	fprintf(spew__out_file, "%04lX: %02lX", code_current, x);
    }
    if (spew_bitbucket_code) {
	if (spew_debug)
	  debug_out(("<disabled>"));
    } else {
	if (spew_debug)
	  debug_out(("<%ld>", x));
	if (spew_codegen) {
	    if (code_current - code_origin >= spew__code_size) {
		spew__code_size += CODE_SIZE_GROW;
		code= realloc(code, (size_t) spew__code_size);
		if (!code) die(("out of memory in spew_byte\n"));
	    }
	    code[code_current - code_origin]= (char) x;
	    code_current++;
	}
    }
}

void spew_byte(Int x)
{
    spew__byte(x);
    if (spew__out_file && !spew_bitbucket_code) fprintf(spew__out_file, "\n");
}

void spew__word(Int x, char *message)
{
    if (spew_debug)
      debug_out(("[word %ld ", x));
    if (spew_msb_first) spew__byte(255 & (x>>8)); else spew__byte(255 & (x>>0));
    if (spew__out_file && !spew_bitbucket_code) fprintf(spew__out_file, "   %s\n", message);
    if (spew_msb_first) spew_byte(255 & (x>>0)); else spew__byte(255 & (x>>8));
    if (spew_debug)
      debug_out(("]"));
}

void spew_word(Int x)
{
    char message[40];
    if (spew__out_file)
      sprintf(message, "   (word %04lX %ld)", x, (x & 32768) ? x - 65536 : x);
    spew__word(x, message);
}

void spew__long(long l, char *message)
{
    if (spew_msb_first)
      spew__word((Int) (0xffff & (l >> 16)), message);
    else
      spew__word((Int) (0xffff & (l >>  0)), message);

    if (spew_msb_first)
      spew__word((Int) (0xffff & (l >>  0)), message);
    else
      spew__word((Int) (0xffff & (l >> 16)), message);
}

void spew_long(long l)
{
    char message[40];
    if (spew__out_file)
      sprintf(message, "   (long %08lX %ld)", l, l);
    spew__long(l, message);
}

#ifdef NEEDS_LOGB
double logb(double x) /* noproto */
{
    int exp;
    frexp(x, &exp);
    return (double) exp-1;
}
#endif

#ifdef NEEDS_SCALB
double scalbn(double x, Int y) /* noproto */
{
    return ldexp(x, y);
}
#endif

double spew_read_float_native(long board_float)
{
    return (double) * (float*) & board_float;
}
			      
double spew_read_float_6811(long board_float)
{
    long mantissa_int= (0x007fffff & board_float) + 0x00800000;
    long exponent_int= 0xff & (board_float >> 23);

    double mantissa= (double) mantissa_int;
    double num= scalbn(mantissa, (Int) exponent_int - 127 - 23);
    
    double ret= (0x80000000L & board_float) ? - num : num;

	if (debug_spew_float) {
		printf("read float %lx -> %f\n", board_float, ret);
	}

    return ret;
}

double spew_read_float(long board_float)
{
    switch (spew_float_type) {
      case float_native:  return spew_read_float_native(board_float);
      case float_6811:    return spew_read_float_6811(board_float);
      default:
	die(("illegal float type %ld\n", spew_float_type));
    }
    /* can never reach here */
}

void spew_float_native(double d, char *message)
{
    long l;
    float f;
    f= (float) d;
    l= * (long*) & f;
    spew__long(l, message);
}


long spew_float_6811_convert(double d)
{
    double num, exponent, mantissa;
    long   exponent_int, mantissa_int, translated;

    if (d == 0.0) {
	translated= 0;
    } else {
	num= fabs(d);
	exponent= logb(num);
	mantissa= scalbn(num, 23 - (Int) exponent);

	exponent_int= (long) exponent + 127;
	mantissa_int= (long) floor(.5 + mantissa);
	translated =
	  (d < 0.0 ? 0x80000000 : 0x00000000) |
	    ((exponent_int & 0xff) << 23) |
	      (mantissa_int & 0x007fffff) ;
	
	if (debug_spew_float) {
	    printf("spew float %f -> %lx\n", d, translated);
	    if (translated != 0)
	      printf("(exponent %f, mantissa %f)\n", exponent, mantissa);
	}
    }
    return(translated);
}

void spew_float_6811(double d, char *message)
{
    spew__long(spew_float_6811_convert(d), message);
}



void spew_float(double d)
{
    char buf[40];
    if (spew__out_file) sprintf(buf, "[float %f]", d);
    switch (spew_float_type) {
      case float_native: spew_float_native(d, buf); break;
      case float_6811:   spew_float_6811(d, buf);   break;
      default:
	die(("illegal float type %ld\n", spew_float_type));
    }
}

void spew_garbage(Type *t)
{
    Int garbage;

    if (type_id(t) == pointer_id) {
	if (type_id(type_Pointer_deref(t)) == array_id) {
	    garbage= THE_ZERO_ARRAY;
	} else {
	    garbage= 0;
	}
    }
    else {
	garbage= 0;
    }

    if (spew_debug)
      debug_out(("{garbage "));
    switch(type_sizeof(t)) {
      case 0:	break;
      case 1:   spew_byte(garbage);	     break;
      case 2:	spew_word(garbage);	     break;
      case 4:	spew_float((float) garbage); break;
      default:
	die(("Illegal size in spew_garbage"));
    }
    if (spew_debug)
      debug_out(("}"));
}

void spew_binary_subroutine_def(Symbol *sym, long addr)
{
    define_binary_subroutine(sym);
    define_set_symbol_addr(sym, addr + spew__module_start - 0x872B);
}

void spew_binary_variable_def(Symbol *sym, long addr)
{
    define_binary_variable(sym);
    define_set_symbol_addr(sym, addr + spew__module_start - 0x872B);
}

void spew_binary_module(Module *m)
{
    Int zp_offset= 0x20;
    long main_offset;
    Symbol *temp= define_temp_symbol();

    spew_jump(temp);
    main_offset= code_current;
    spew__module_start= code_current;
    
    if (pass == 1) {
	/* Spew out the bytes */
	long i, a1, a2;

	if (spew_debug) printf("begin spewing binary module\n");
	    
	for (i= 0; i< m->binrec1->len; i++) {
	    switch (0xff & (m->binrec2->data[i] - m->binrec1->data[i])) {
	      case 0:
		spew_byte(m->binrec1->data[i]);
		break;
	      case 3: /* zero page address */
		spew_byte(m->binrec1->data[i]- 0x10 + zp_offset);
		break;
	      default: /* hope it's a main address */
		a1= (m->binrec1->data[i] << 8) +
		     (unsigned char) m->binrec1->data[i+1];
		a2= (m->binrec2->data[i] << 8) +
		     (unsigned char) m->binrec2->data[i+1];
		if (a2 - a1 == 0x070B) {
		    spew_word((Int) (a1 - 0x8020 + main_offset));
		}
		else {
		    printf("delta was %lx\n", a2-a1);
		    user_error(("Bad address in .icb file\n"));
		}
		i+=1;
	    }
	}

	if (spew_debug) printf("end spewing binary module\n");
    }
    spew_label_definition(temp);
}		

void spew_binop(char op)
{
    Type *t1= define_pop();
    Type *t2= define_pop();
    Int opcode;

    if (pass == 1 && !type_compatible(t1, t2))
      user_error(("Different type arguments given to %c:  %s %c %s\n",
		  (int)op, type_name(t2), (int)op, type_name(t1)));

    switch(type_id(t1)) {
      case undef_id:
      case int_id:    opcode= spew__int_binops[op];   break;
      case long_id:   opcode= spew__long_binops[op];   break;
      case float_id:  opcode= spew__float_binops[op]; break;
      default:        opcode= Pundefined;             break;
    }

    if (pass == 1 && opcode == Pundefined) {
	user_error(("Illegal type given to %c: %s %c %s\n",
		    (int)op, type_name(t2), (int)op, type_name(t1)));
	opcode= Padd2;
    }

    spew_op(opcode);
    if (strchr("<>=", op))    /* conditionals return Int */
      define_push(type_Int());
    else 
      define_push(t1);
}

void spew_unop(char op)
{
    Type *t1= define_pop();
    Int opcode;

    switch(type_id(t1)) {
      case undef_id:
      case int_id:    opcode= spew__int_unops[op];   break;
      case long_id:   opcode= spew__long_unops[op];  break;
      case float_id:  opcode= spew__float_unops[op]; break;
      default:        opcode= Pundefined;            break;
    }

    if (pass == 1 && opcode == Pundefined) {
	user_error(("Illegal type given to %c:  %c %s\n",
		    (int)op, (int)op, type_name(t1)));
	opcode= Pneg2;
    }

    spew_op(opcode);
    define_push(t1);
}

void spew_push_garbage(Type *t)
{
    if (type_id(t) == pointer_id &&
	type_id(type_Pointer_deref(t)) == array_id) {
	spew_begin_array(t,1);
	spew_end_array(1);
    }
    else {
	Int size= type_sizeof(t);

	if (size) {
	    spew_push_op(size);
	    spew_garbage(t);
	}
	define_push(t);
    }
}

void spew_skip_garbage(Type *t)
{
    Int size= type_sizeof(t);

    if (type_id(t) == pointer_id &&
	type_id(type_Pointer_deref(t)) == array_id) {
	spew_begin_array(t,0);
	spew_end_array(0);
    }
    else {
	if (size) {
	    spew_op(Paddsp);
	    spew_word(-size);
	}
	define_push(t);
    }
}

void spew_push2(Int i, Type *t)
{
    spew_op(Ppush2);
    spew_word(i);
    define_push(t);
}

void spew_push_int(Int i)
{
    spew_push2(i, type_Int());
}

void spew_push_long(long l)
{
    spew_op(Ppush4);
    spew_long(l);
    define_push(type_Long());
}

void spew_push_float(double d)
{
    spew_op(Ppush4);
    spew_float(d);
    define_push(type_Float());
}

void spew_push_one(void)
{
    switch (type_id(define_stack_type(0))) {
      case undef_id:
      case int_id:
	spew_push_int(1);
	break;
      case long_id:
	spew_push_long(1L);
	break;
      case float_id:
	spew_push_float(1.0);
	break;
      default:
	if (pass == 1)
	  user_error(("Attempt to increment or decrement type %s\n",
		      type_name(define_stack_type(0))));
	break;
    }
}

void spew_sprel(long offset)
{
    spew__last_sprel_offset= offset;
    spew_op(Psprel);
    spew_word((Int) offset);
}


typedef union {
    Int    i;
    long   l;
    double d;
} Spew_types;

#define MAX_SPEW_ARRAY_ITEMS 1000

Int spew_array_item;
Type *spew_expected_element_type;
Type *spew_array_type;
Spew_types spew_array_init[MAX_SPEW_ARRAY_ITEMS];

void spew_begin_array(Type *array_type, Int write)
{
    if (type_id(array_type) != pointer_id ||
	type_id(type_Pointer_deref(array_type)) != array_id) {
	user_error(("Attempt to initialize non-array with multiple elements"));
	spew_array_type= type_Array(type_Undef(), -1);
	spew_expected_element_type= type_Undef();
    } else {
	spew_array_type= type_Pointer_deref(array_type);
	spew_expected_element_type=
	  type_Array_elemtype(type_Pointer_deref(array_type));
    }
    spew_array_item= 0;
}

void spew_continue_array_int(Int i)
{
    if (spew_array_item > MAX_SPEW_ARRAY_ITEMS) {
	user_error(("Array initialization too long"));
	return;
    }
    
    switch (type_id(spew_expected_element_type)) {
      case pointer_id:
	if (i != 0) goto error;
      case undef_id:
      case char_id:
      case int_id:
	spew_array_init[spew_array_item].i= i;
	break;
      default:
      error:
	user_error(("Incompatible type in array initialization (expected %s, given %s)",
		    type_name(spew_expected_element_type),
		    type_name(type_Int())));
	break;
    }
    spew_array_item++;
}
    
void spew_continue_array_long(long l)
{
    if (spew_array_item > MAX_SPEW_ARRAY_ITEMS) {
	user_error(("Array initialization too long"));
	return;
    }
    
    switch (type_id(spew_expected_element_type)) {
      case pointer_id:
	if (l != 0) goto error;
      case undef_id:
      case char_id:
      case int_id:
      case long_id:
	spew_array_init[spew_array_item].l= l;
	break;
      default:
      error:
	user_error(("Incompatible type in array initialization (expected %s, given %s)",
		    type_name(spew_expected_element_type),
		    type_name(type_Long())));
	break;
    }
    spew_array_item++;
}
    
void spew_continue_array_float(double d)
{
    if (spew_array_item > MAX_SPEW_ARRAY_ITEMS) {
	user_error(("Array initialization too long"));
	return;
    }
    
    if (!type_compatible(type_Float(), spew_expected_element_type)) {
	user_error(("Incompatible type in array initialization (expected %s, given %s)",
		    type_name(spew_expected_element_type),
		    type_name(type_Float())));
    } else {
	spew_array_init[spew_array_item].d= d;
    }
    spew_array_item++;
}
    
void spew_end_array(Int write)
{
    Type *array_type;
    Int i;
    Int specified_size= type_Array_size(spew_array_type);

    if (specified_size == -1) specified_size= spew_array_item;
    if (spew_array_item > specified_size) {
	user_error(("Specified array size too small to hold all initializers\n"));
    }

    array_type= type_Array(spew_expected_element_type, specified_size);
    if (type_id(spew_expected_element_type) == char_id &&
	(specified_size % 2) == 1) {
	specified_size++;
    }

    if (write) {
	for (i= specified_size-1; i > -1; i--) {
	    switch(type_id(spew_expected_element_type)) {
	      case float_id:
		spew_op(Ppush4);
		spew_float(i < spew_array_item ? spew_array_init[i].d : 0.0);
		break;
	      case long_id:
		spew_op(Ppush4);
		spew_long(i < spew_array_item ? spew_array_init[i].l : 0L);
		break;
	      case int_id:
	      case pointer_id:
		spew_op(Ppush2);
		spew_word(i < spew_array_item ? spew_array_init[i].i : 0);
		break;
	      case char_id:
		spew_op(Ppush2);
		spew_word(((i < spew_array_item ? spew_array_init[i-1].i : 0)
			   << 8) |
			  ((i+1 < spew_array_item ? spew_array_init[i].i : 0)
			   & 0xff));
		i--;
		break;
	      default:
		die(("Don't know how to make %s", type_name(array_type)));
	    }
	}
    }
    else {
	spew_op(Paddsp);
	spew_word(-specified_size * type_sizeof(spew_expected_element_type));
    }
    spew_op(Ppush2);
    spew_word(specified_size);
    define_push(array_type);
    spew_sprel(0);
    define_push(type_Pointer(array_type));
}

void spew_return(void)
{
    Type *actual_return_type, *proc_type, *declared_return_type;

    if (pass == 0) return;

    proc_type= symtab_get(symbol_create("current procedure"));

    if (!proc_type) {
	user_error((
		    "'return' must be used inside a procedure body.\n"));
	define_pop();
	return;
    }

    declared_return_type= type_Func_return_type(proc_type);

    actual_return_type= define_stack_type(0);

    if (pass == 1 &&
	!type_compatible(declared_return_type, actual_return_type)) {
	user_error((
	 "Attempt to return type %s from function declared as returning type %s\n",
		    type_name(actual_return_type),
		    type_name(declared_return_type)));
    }

    switch(type_sizeof(actual_return_type)) {
      case 0:
	spew_op(Pmret0);
	spew_word((Int) current_stack_size);
	break;
      case 2:
	spew_op(Pmret2);
	spew_word((Int) current_stack_size);
	break;
      case 4:
	spew_op(Pmret4);
	spew_word((Int) (current_stack_size - 2));
	break;
      default:
	die(("Illegal type size in spew_return"));
    }
    define_pop();
    spew_bitbucket_code= 1;
}

Type *spew_check_funcall(Symbol *func)
{
    Int i, nargs, not_a_func= 0;
    Value *func_val= symtab_get(func);
    Type  *proc_type;

    for (nargs= 0; type_id(define_stack_type(nargs)) != block_id; nargs++);

    if (pass == 0) {
	not_a_func= 1;
    }
    else if (!func_val) {
	if (pass == 1)
	  user_error(("Function %s undefined", symbol_name(func)));
	not_a_func= 1;
    }
    else if (type_id(proc_type= value_type(func_val)) != func_id) {
	if (pass == 1)
	  user_error(("%s is not a function", symbol_name(func)));
	not_a_func= 1;
    }
    else if (nargs > type_Func_number_args(proc_type)) {
	if (pass == 1)
	  user_error(("Too many arguments given to %s", symbol_name(func)));
    }
    else if (nargs < type_Func_number_args(proc_type)) {
	if (pass == 1)
	  user_error(("Too few arguments given to %s", symbol_name(func)));
    }
    else {
	for (i= 0; i< nargs; i++) {
	    Type *given=    define_stack_type(nargs - i - 1);
	    Type *expected= type_Func_arg_type(proc_type, i);

	    if (!type_compatible(given, expected)) {
		if (pass == 1)
		  user_error(("Argument %ld incompatible (expected %s, given %s)",
			      i+1, type_name(expected), type_name(given)));
	    }
	}
	for (i= 0; i< nargs; i++) {
	    define_pop(); /* pop arg */
	}
    }
    return not_a_func ? type_Undef() : type_Func_return_type(proc_type);
}

void spew_begin_printf(void)
{
    define_begin_block();
}

void spew_end_printf(void)
{
    Int nargs;

    for (nargs= 0; type_id(define_stack_type(nargs)) != block_id; nargs++);

    spew_push_int((Int) (define_stack_offset(nargs) - 2));
    spew_op(Pprintf);
    define_end_block_no_pop();
    define_push(type_Int());
}
    
void spew_begin_call(Symbol *func, Symbol *temp)
{
    Value *func_val= symtab_get(func);
    
    if (func_val &&
	type_id(value_type(func_val)) == func_id) {

	switch (type_Func_calling_convention(value_type(func_val))) {
	  case pcode_calling_convention:
	    spew_op(Ppush2);
	    spew_label_reference(temp);
	    break;
	  case ml_calling_convention:
	    spew_op(Ppush2);
	    spew_label_reference(func);
	    break;
	  default:
	    die(("Illegal calling convention in begin_call"));
	}
    }

    define_push(type_Label());
    define_begin_block();
}


void spew_end_call(Symbol *func, Symbol *temp)
{
    Type  *proc_type;
    Value *func_val= symtab_get(func);

    proc_type= spew_check_funcall(func);
    
    if (func_val &&
	type_id(value_type(func_val)) == func_id) {

	switch (type_Func_calling_convention(value_type(func_val))) {
	  case pcode_calling_convention:
	    spew_jump(func);
	    break;
	  case ml_calling_convention:
	    spew_op(Pcallml);
	    break;
	}
    }

    define_end_block_no_pop();
    define_pop(); /* pop call address or return address */
    
    define_push(proc_type);
    
    spew_label_definition(temp);
}

void spew_kill_process(void)
{
    spew_op(Pkillprocess);
}

void spew_begin_procedure(Symbol *proc)
{
  if (!native_flag)
    {
      spew_op(Pcheckstack);
      spew__check_stack_pc= code_current;
      spew_word(0);

      if (!strcmp(symbol_name(proc), "main")) {
	Symbol *temp= define_temp_symbol();
	Symbol *init_sym= symbol_create("_init_globals");
	spew_begin_call(init_sym, temp);
	spew_end_call(init_sym, temp);
	spew_pop();
      }
    }
  else
    {
/* *** DO THE NATIVE JUMP THANG HERE */
      spew_op(Pnative);		/* check stack and start executing */
    }
  define_set_low_water_mark();
}

void spew_end_procedure(Symbol *proc)
{
    long stack_needed= define_get_high_water_mark() + 2;
    UNUSED(proc);

    if (!native_flag)
      spew_write_word(spew__check_stack_pc, (Int) stack_needed);
}
    
void spew_begin_start_process(Symbol *func)
{
    UNUSED(func);
    spew_push_int(0);
    define_begin_block();
}

void spew_continue_start_process(Symbol *func)
{
    Int nargs;
    long nbytes;

    for (nargs= 0; type_id(define_stack_type(nargs)) != block_id; nargs++);

    nbytes= define_stack_offset(nargs) + 2;

    spew_check_funcall(func);

    spew_push_int((Int) nbytes);
    
    spew_op(Ppush2);
    spew_label_reference(func);
}

void spew_end_start_process(Symbol *func)
{
    Type *stack_size= define_pop();
    Type *ticks= define_pop();

    UNUSED(func);
    if (type_id(stack_size) != int_id &&
	type_id(stack_size) != undef_id) {
	user_error(("Second argument to start_process (timeslice) must be an integer, but %s given\n",
		    type_name(stack_size)));
    }
    if (type_id(ticks) != int_id &&
	type_id(ticks) != undef_id) {
	user_error(("Third argument to start_process (stack size) must be an integer, but %s given\n",
		    type_name(ticks)));
    }
    
    spew_op(Pstartprocess);
    
    define_end_block_no_pop();
    define_pop(); /* pop 0 return address */
    define_push(type_Int()); /* push process id# */
}

void spew_jump(Symbol *destination)
{
    spew_op(Pjumpi);
    spew_label_reference(destination);
}

void spew_conditional_jump(Opcode op, Symbol *destination)
{
    Type *condition= define_pop();

    if (pass == 1 &&
	type_id(condition) != int_id &&
	type_id(condition) != pointer_id &&
	type_id(condition) != undef_id) {
	user_error(("Type %s cannot be used in a conditional",
		    type_name(condition)));
    }
    spew_op(op);
    spew_label_reference(destination);
}

void spew_peeki(Type *type, Int global_location)   
{
    UNUSED(type);
    UNUSED(global_location);
    DIE();
}

void spew_push_op(Int size)
{
    switch (size) {
      case 2:
	spew_op(Ppush2);
	break;
      case 4:
	spew_op(Ppush4);
	break;
      default:
	die(("Illegal size to spew_push_op"));
    }
}

void spew_peek_op(Int size)   
{
    Opcode op;
    switch (size) {
      case 1:   op= Ppeek1;  break;
      case 2:   op= Ppeek2;  break;
      case 4:   op= Ppeek4;  break;
      default:	die(("Illegal size to spew_peek_op"));
    }
    if (spew__last_op == Ppush2 && spew_optimize_read_global) {
	if (code_current -3 != spew__last_op_loc)
	  die(("Problem with read global optimization"));
	switch (size) {
	  case 1:   op= Ppeeki1;  break;
	  case 2:   op= Ppeeki2;  break;
	  case 4:   op= Ppeeki4;  break;
	  default:	die(("Illegal size to spew_peek_op (read global)"));
	}
	spew_write_byte(code_current - 3, op*2);
	spew__last_op= -1;
    }
    else if (spew__last_op == Psprel && spew_optimize_read_local &&
	spew__last_sprel_offset < 256) {
	if (code_current -3 != spew__last_op_loc)
	  die(("Problem with read local optimization"));
	switch (size) {
	  case 2:   op= Pspeek2;  break;
	  case 4:   op= Pspeek4;  break;
	  default:	die(("Illegal size to spew_peek_op (read local)"));
	}
	spew_write_byte(code_current - 3, op*2);
	spew_write_byte(code_current - 2, (Int) spew__last_sprel_offset);

	code_current--;
	spew__last_op= -1;
    }
    else {
	spew_op(op);
    }
}
	
void spew_poke_op(Int size, Int pop)   
{
    switch (size) {
      case 1:
	spew_op(pop ? Ppoke1 : Ppoke1nopop);
	break;
      case 2:
	spew_op(pop ? Ppoke2 : Ppoke2nopop);
	break;
      case 4:
	spew_op(pop ? Ppoke4 : Ppoke4nopop);
	break;
      default:
	die(("Illegal size to spew_poke_op"));
    }
}
	
void spew_pop_op(Int size)   
{
    switch (size) {
      case 0:
	break;
      case 2:
	if (spew__last_op == Ppoke2nopop) {
	    if (spew__last_op_loc != code_current - 1)
	      die(("problem with poke_nopop optimization 2"));
	    spew_write_byte(code_current - 1, Ppoke2*2);
	    spew__last_op= Ppoke2;
	} else {
	    spew_op(Ppop2);
	}
	break;
      case 4:
	if (spew__last_op == Ppoke4nopop) {
	    if (spew__last_op_loc != code_current - 1)
	      die(("problem with poke_nopop optimization 4"));
	    spew_write_byte(code_current - 1, Ppoke4*2);
	    spew__last_op= Ppoke4;
	} else {
	    spew_op(Ppop4);
	}
	break;
      default:
	spew_op(Paddsp);
	spew_word(size);
	break;
    }
}

void spew_aref_op(Int size)
{
    switch (size) {
      case 1:
	spew_op(Paref1);
	break;
      case 2:
	spew_op(Paref2);
	break;
      case 4:
	spew_op(Paref4);
	break;
      default:
	die(("Illegal size %ld to spew_aref_op", size));
    }
}
	
void spew_pop(void)
{
    Type *top_of_stack= define_pop();
    Int  size= type_sizeof(top_of_stack);

    if (size) spew_pop_op(size);
}

void spew_array_reference(void)
{
    Type *index= define_pop();
    Type *array= define_pop();
    Type *ret= 0;

    if (type_id(array) != pointer_id ||
	type_id(type_Pointer_deref(array)) != array_id) {
	if (pass == 1 && type_id(array) != undef_id) {
	    user_error(("Attempt to use type %s as an array\n",
			type_name(array)));
	}
	ret= type_Undef();
    }
    if (type_id(index) != int_id) {
	if (pass == 1 && type_id(index) != undef_id) {
	    user_error(("Array index must be an integer (type %s attempted)\n",
			type_name(index)));
	}
	ret= type_Undef();
    }
    if (!ret) {
	ret= type_Array_elemtype(type_Pointer_deref(array));
    }
    spew_aref_op(type_sizeof(ret));
    define_push(type_Pointer(ret));
}

void spew_peek(void)
{
    Type *address= define_pop();

    if (type_id(address) != pointer_id) {
	if (pass == 1 && type_id(address) != undef_id) {
	    user_error(("Attempt to dereference a non-pointer type %s\n",
			type_name(address)));
	}
	define_push(type_Undef());
    } else {
	Type *deref= type_Pointer_deref(address);
	if (type_id(deref) == func_id) {
	    if (pass == 1) {
		user_error(("Attempt to use type <function> in expression\n"));
	    }
	    define_push(type_Undef());
	} else {
	    spew_peek_op(type_sizeof(deref));
	    /* char peek yields Int */
	    if (type_id(deref) == char_id) deref= type_Int();
	    define_push(deref);
	}
    }
}

void spew_bit_set_clr(Opcode op)
{
    Type *bits= define_pop();
    Type *dest= define_pop();

    if (type_id(dest) != pointer_id && !integerp(dest) && !undefp(dest) && pass == 1) {
	user_error(("First argument to %s must be integer (%s given)",
		    op == Pbitset ? "bit_set" : "bit_clear", type_name(dest)));
    }
    if (!integerp(bits) && !undefp(bits) && pass == 1) {
	user_error(("Second argument to %s must be integer (%s given)",
		    op == Pbitset ? "bit_set" : "bit_clear", type_name(bits)));
    }
    spew_op(op);
}
    

void spew_poke(Int pop)
{
    Type *source=    define_pop();
    Type *dest_addr= define_pop();
    Type *dest;
    Int  dest_size;

    if (type_id(dest_addr) != pointer_id ||
	type_id(type_Pointer_deref(dest_addr)) == func_id) {
	if (pass == 1 && type_id(dest_addr) != undef_id)
	  user_error(("Cannot assign type %s\n", type_name(dest_addr)));
	dest= type_Undef();
	dest_size= 2;
    } else {
	dest= type_Pointer_deref(dest_addr);
	dest_size= type_sizeof(dest);
	/* char poke takes Int */
	if (type_id(dest) == char_id) dest= type_Int();
	if (pass == 1 && !type_compatible(source, dest))
	  {
	    if (type_id(source) == int_id &&
		type_id(dest) == pointer_id) 
	      {
		if (!ic_wizard) 
		  user_error(("Attempt to coerce type %s to type %s (use -wizard)\n",
			      type_name(source), type_name(dest))); 
	      }
	    else
	      user_error(("Types not compatible in assignment: %s = %s\n",
			  type_name(dest), type_name(source)));
	  }
      }
    spew_poke_op(dest_size, pop);
    if (!pop) define_push(source);
}


void spew_speek(Int stack_index)   
{
    Type *type= define_stack_type(stack_index);
    long  stack_offset= define_stack_offset(stack_index);
    
    ASSERT(0L <= stack_offset);
    ASSERT(stack_offset <= 255);
    switch (type_sizeof(type)) {
      case 2:
	spew_op(Pspeek2);
	break;
      case 4:
	spew_op(Pspeek4);
	break;
      default:
	die(("Illegal type size in spew_speek"));
	break;
    }
    spew_byte((Int) stack_offset);
    define_push(type);
}

void spew_dup(void)
{
    spew_speek(0);
}

void spew_set_module(Int module)
{
    spew_module= module;
}

void spew_label_reference(Symbol *sym)
{
    if (pass == 1 && !spew_bitbucket_code)
      ref_add(sym, code_current, two_byte_absolute);
    spew_word(0);
}

void spew_label_reference_offset(Symbol *sym, Int offset)
{
    if (pass == 1 && !spew_bitbucket_code)
      ref_add(sym, code_current, two_byte_absolute);
    spew_word(offset);
}

void spew_label_definition(Symbol *sym)
{
    spew__last_op= -1;
    spew_bitbucket_code= 0;
    if (spew_debug) debug_out(("\n[label %s:]", symbol_name(sym)));
    define_set_symbol_addr(sym, code_current);
    
    if (pass == 1) {
    }
    if (spew__out_file && !spew_bitbucket_code) {
	Value *func= symtab_get(sym);
	if (func && type_id(value_type(func)) == func_id) {
	    fprintf(spew__out_file, "%04lX: 100 %s %ld\n",
		    code_current, symbol_name(sym),
		    type_Func_number_args(value_type(func)));
	} else {
	    fprintf(spew__out_file, "%04lX: 100 <%s>:\n", code_current, symbol_name(sym));
	}
    }
}

Opcode name_to_opcode(char *name)
{
    if (!strcmp(name, "add2")) return Padd2;
    if (!strcmp(name, "sub2")) return Psub2;
    if (!strcmp(name, "mult2")) return Pmult2;
    if (!strcmp(name, "div2")) return Pdiv2;
    if (!strcmp(name, "bitand2")) return Pbitand2;
    if (!strcmp(name, "bitor2")) return Pbitor2;
    if (!strcmp(name, "bitxor2")) return Pbitxor2;
    if (!strcmp(name, "equal2")) return Pequal2;
    if (!strcmp(name, "lt2")) return Plt2;
    if (!strcmp(name, "gt2")) return Pgt2;
    if (!strcmp(name, "lshift")) return Plshift;
    if (!strcmp(name, "lognot2")) return Plognot2;
    if (!strcmp(name, "logidn2")) return Plogidn2;
    if (!strcmp(name, "bitnot2")) return Pbitnot2;
    if (!strcmp(name, "neg2")) return Pneg2;
    if (!strcmp(name, "add4")) return Padd4;
    if (!strcmp(name, "sub4")) return Psub4;
    if (!strcmp(name, "mult4")) return Pmult4;
    if (!strcmp(name, "equal4")) return Pequal4;
    if (!strcmp(name, "lt4")) return Plt4;
    if (!strcmp(name, "gt4")) return Pgt4;
    if (!strcmp(name, "neg4")) return Pneg4;
    if (!strcmp(name, "fadd")) return Pfadd;
    if (!strcmp(name, "fsub")) return Pfsub;
    if (!strcmp(name, "fmult")) return Pfmult;
    if (!strcmp(name, "fdiv")) return Pfdiv;
    if (!strcmp(name, "fx2y")) return Pfx2y;
    if (!strcmp(name, "fequal")) return Pfequal;
    if (!strcmp(name, "flt")) return Pflt;
    if (!strcmp(name, "fgt")) return Pfgt;
    if (!strcmp(name, "fneg")) return Pfneg;
    if (!strcmp(name, "fsqrt")) return Pfsqrt;
    if (!strcmp(name, "fexp")) return Pfexp;
    if (!strcmp(name, "f10tx")) return Pf10tx;
    if (!strcmp(name, "fln")) return Pfln;
    if (!strcmp(name, "flog")) return Pflog;
    if (!strcmp(name, "fatan")) return Pfatan;
    if (!strcmp(name, "fsin")) return Pfsin;
    if (!strcmp(name, "fcos")) return Pfcos;
    if (!strcmp(name, "ftan")) return Pftan;
    if (!strcmp(name, "fl2int")) return Pfl2int;
    if (!strcmp(name, "int2fl")) return Pint2fl;
    if (!strcmp(name, "int2lng")) return Pint2lng;
    if (!strcmp(name, "fl2lng")) return Pfl2lng;
    if (!strcmp(name, "lng2fl")) return Plng2fl;
    if (!strcmp(name, "push2")) return Ppush2;
    if (!strcmp(name, "push4")) return Ppush4;
    if (!strcmp(name, "pushblock")) return Ppushblock;
    if (!strcmp(name, "pop2")) return Ppop2;
    if (!strcmp(name, "pop4")) return Ppop4;
    if (!strcmp(name, "addsp")) return Paddsp;
    if (!strcmp(name, "setsp")) return Psetsp;
    if (!strcmp(name, "peeki1")) return Ppeeki1;
    if (!strcmp(name, "peeki2")) return Ppeeki2;
    if (!strcmp(name, "peeki4")) return Ppeeki4;
    if (!strcmp(name, "peek1")) return Ppeek1;
    if (!strcmp(name, "peek2")) return Ppeek2;
    if (!strcmp(name, "peek4")) return Ppeek4;
    if (!strcmp(name, "bitset")) return Pbitset;
    if (!strcmp(name, "bitclr")) return Pbitclr;
    if (!strcmp(name, "pokei1")) return Ppokei1;
    if (!strcmp(name, "pokei2")) return Ppokei2;
    if (!strcmp(name, "pokei4")) return Ppokei4;
    if (!strcmp(name, "poke1")) return Ppoke1;
    if (!strcmp(name, "poke2")) return Ppoke2;
    if (!strcmp(name, "poke4")) return Ppoke4;
    if (!strcmp(name, "poke1nopop")) return Ppoke1nopop;
    if (!strcmp(name, "poke2nopop")) return Ppoke2nopop;
    if (!strcmp(name, "poke4nopop")) return Ppoke4nopop;
    if (!strcmp(name, "speek2")) return Pspeek2;
    if (!strcmp(name, "speek4")) return Pspeek4;
    if (!strcmp(name, "spoke2")) return Pspoke2;
    if (!strcmp(name, "spoke4")) return Pspoke4;
    if (!strcmp(name, "sprel")) return Psprel;
    if (!strcmp(name, "checkstack")) return Pcheckstack;
    if (!strcmp(name, "aref1")) return Paref1;
    if (!strcmp(name, "aref2")) return Paref2;
    if (!strcmp(name, "aref4")) return Paref4;
    if (!strcmp(name, "aref_arb")) return Paref_arb;
    if (!strcmp(name, "jump")) return Pjump;
    if (!strcmp(name, "jumpi")) return Pjumpi;
    if (!strcmp(name, "jfalse")) return Pjfalse;
    if (!strcmp(name, "jtrue")) return Pjtrue;
    if (!strcmp(name, "jpfalse")) return Pjpfalse;
    if (!strcmp(name, "jptrue")) return Pjptrue;
    if (!strcmp(name, "mret0")) return Pmret0;
    if (!strcmp(name, "mret2")) return Pmret2;
    if (!strcmp(name, "mret4")) return Pmret4;
    if (!strcmp(name, "fl2ascii")) return Pfl2ascii;
    if (!strcmp(name, "print2")) return Pprintlcd2;
    if (!strcmp(name, "printf")) return Pprintf;
    if (!strcmp(name, "printstr")) return Pprintstring;
    if (!strcmp(name, "printchar")) return Pprintchar;
    if (!strcmp(name, "startproc")) return Pstartprocess;
    if (!strcmp(name, "killproc")) return Pkillprocess;
    if (!strcmp(name, "haltnotify")) return Phaltnotify;
    if (!strcmp(name, "defer")) return Pdefer;
    if (!strcmp(name, "systime")) return Psystime;
    if (!strcmp(name, "loadreg")) return Ploadreg;
    if (!strcmp(name, "fetchreg")) return Pfetchreg;
    if (!strcmp(name, "callml")) return Pcallml;
    if (!strcmp(name, "bench")) return Pbench;
    if (!strcmp(name, "initint")) return Pinitint;
    if (!strcmp(name, "native")) return Pnative;
    die (("Couldn't find opcode name %s", name));
}

void spew_init_module(void)
{
  Int i;

  for (i= 0; opcodes[i].name; i++) {
    opcodes[i].opcode= name_to_opcode(opcodes[i].name);
  }
}

char *opcode_name(Int op)
{
    Int i;
    for (i= 0; opcodes[i].name; i++) {
	if (opcodes[i].opcode == op) return opcodes[i].name;
    }
    die(("Illegal opcode %lx (hex) in opcode_name", op*2));
}

void spew_op(Opcode op)
{
    spew__last_op= op;
    spew__last_op_loc= code_current;
    if (!code_inline)
      {
	if (spew_debug)
	  debug_out(("\n%lx:[%s ", code_current, opcode_name(op)));
	spew__byte(2 * (Int) op);
      }
    else
      {
	if (op == Pmret0 || op == Pmret2 || op == Pmret4)
	  code_mop(jmp);
	else
	  code_mop(jsr);
	if (spew_debug) debug_out((" %s ", opcode_name(op)));
	spew_word(jumptable[op]);
      }
    if (spew__out_file && !spew_bitbucket_code) fprintf(spew__out_file, "   %s\n", opcode_name(op));
    if (spew_debug)
      debug_out(("]"));
}

void spew_set_code_origin(long origin)
{
    code_origin= origin;
}

void spew_set_code_address(long address)
{
    code_current= address;
}
