/**
 ** symtab.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#define SYMTAB_MODULE

/* uncomment for stand-alone testing */
/* #define STAND_ALONE */

#include CONFIG
#include <stdio.h>

#include "symtab.h"
#include "util.h"

char symbol__temp_name[20];

void 	    symtab_stats      (void);

/*-------------------------------------------------------------------------*/
/* Private Constants                                                       */

#define BINDINGS_CHUNK 100
#define LOCALS_CHUNK 25

/*-------------------------------------------------------------------------*/
/* Private Variables                                                       */

Level *current_level=0, *global_level=0;
Int current_level_no= 0;
Binding *free_bindings;
Symbol **symtab;

long symtab_size;
Int symtab_size_number;
long symtab_total_symbols, symtab_collisions;
long symtab_total_symbols_limit, symtab_collisions_limit;
Int symbol__temp_id= 0;

long symtab_sizes[]=  {
    37, 67, 127, 251, 503,
    1009, 2003, 4001, 8009, 16001, 32003, 64007, 128021, 256019, 512009,
    1024021, 2048003, 4096013, 8192003, 16384001, 32768011, 65536043
  };

/*-------------------------------------------------------------------------*/
/* Public Variables                                                        */

Int symtab_debug= 0;

/*-------------------------------------------------------------------------*/
/* Private Functions                                                       */

/* out of memory checker:  checks if newly allocated foo is NULL */
void check(void *foo)
{
    if (!foo) {fprintf(stderr, "symtab: out of memory\n"); exit(1);}
}
    
Binding *new_binding(void *val, enum binding_type type, 
		     Int lev_no, Binding *next)
{
    Binding *ret;
    Int i;
    if (!free_bindings) {
	free_bindings= xmalloc(sizeof(Binding) * BINDINGS_CHUNK);
	for(i= 0; i< BINDINGS_CHUNK-1; i++)
	  free_bindings[i].next= &free_bindings[i+1];
	free_bindings[i].next= 0;
    }
    ret= free_bindings;
    free_bindings= free_bindings->next;
    ret->val= val;
    ret->type=type;
    ret->lev_no= lev_no;
    ret->next= next;
    return ret;
}

void free_binding(Binding *b)
{
  if(b->type==value_binding) {
    if(symtab_debug>1)
      printf("(freeing value)\n");
    free(b->val); 
  } else {
    if(symtab_debug>1)
      printf("(not freeing type)\n");
  }
  b->next= free_bindings;
  b->type= no_binding;
  b->val=NULL;
  free_bindings= b;
}

long hash_string(char *str)
{
    register long ret= 0;
    register long tmp;
    /*for (; *str; str++) ret = *str + 97*ret;*/
    while (*str) {
	tmp= ret << 5;
	tmp += tmp << 1;
	ret += tmp + *str++;
    }
    return ret;
}

Int symtab_grow(void)
{
    long   new_symtab_size = symtab_sizes[symtab_size_number++];
    Symbol **new_symtab;
    long   i;
    long loc;

    if (new_symtab_size * sizeof(Symbol*) > (long) MAX_ALLOCATED_DATA_SIZE) {
        return 0;
    }
    new_symtab= xmalloc_clear((size_t) (new_symtab_size * sizeof(Symbol*)));
    if (!new_symtab) return 0;
    if (symtab_debug) {
	fprintf(stderr, ">GROWING SYMTAB -- BEFORE\n");
	symtab_stats();
    }
    
    check(new_symtab);

    symtab_collisions= 0;
    for(i= 0; i< symtab_size; i++) {
	if (symtab[i]) {
	    loc= (long) ((unsigned long) hash_string(symtab[i]->name)) %
	                ((unsigned long) new_symtab_size);
	    while (new_symtab[loc]) {
		loc += 1;
		if (loc == new_symtab_size) loc= 0;
		symtab_collisions++;
	    }
	    new_symtab[loc]= symtab[i];
	}
    }

    if (symtab) free(symtab);
    symtab= new_symtab;
    symtab_size= new_symtab_size;
    symtab_collisions_limit = symtab_size / 16;
    symtab_total_symbols_limit = symtab_size / 8;

    if (symtab_debug) {
	fprintf(stderr, "<GROWING SYMTAB -- AFTER\n");
	symtab_stats();
    }
    return 1;

}

void symtab_destroy_current_block(void)
{
    Level   *temp;
    Binding *tmp_binding;
    Int     i;

    for (i= 0; i< current_level->nsyms; i++) {
	tmp_binding= current_level->syms[i]->binding;
	current_level->syms[i]->binding= tmp_binding->next;
	if (symtab_debug > 1) {
	  printf("freeing binding for symbol %s\n",
		 symbol_name(current_level->syms[i]));
	}
	free_binding(tmp_binding);
    }
    temp= current_level;
    current_level= current_level->parent;
    if (current_level) current_level->child= 0;
    current_level_no--;
    if (symtab_debug) printf("freeing block %lx, back to block %lx\n",
			     (long)temp, (long)current_level);
    free(temp);
}

/*-------------------------------------------------------------------------*/
/* Public Functions                                                        */

void symtab_stats(void)
{
    fprintf(stderr,"Symbol table stats:\n");
    if (!symtab) return;
    fprintf(stderr,"   Size: %ld,  # symbols: %ld,  # collisions: %ld\n",
	    symtab_size, symtab_total_symbols, symtab_collisions);
    fprintf(stderr,"   # symbols limit: %ld,  # collisions limit: %ld\n",
	    symtab_total_symbols_limit, symtab_collisions_limit);
    if (symtab_collisions) 
      fprintf(stderr,"   Collision percentage %f\n",
	      100.0 * symtab_collisions / (double) symtab_total_symbols);
}

Level *symtab__make_level(Level *parent)
{
    Level *new= xmalloc(sizeof(Level) + sizeof(Symbol *) * (LOCALS_CHUNK - 1));
    check(new);
    new->maxsyms= LOCALS_CHUNK;
    new->nsyms= 0;
    new->parent= parent;
    if (parent) parent->child= new;
    return new;
}

Symbol *symtab_get_binding(void *val, Int reset)
{
    static Int i= 0;

    if (reset) i= 0;
    if (i >= current_level->nsyms) return 0;
    *(void**)val= current_level->syms[i]->binding->val;
    return current_level->syms[i++];
}

void symtab_init(void)
{
    if (!symtab) symtab_grow();

    /* Free any old bindings */
    while (current_level)
      symtab_destroy_current_block();

    current_level= global_level= symtab__make_level(0);
    current_level_no= 0;
    if (symtab_debug) {
	printf("symtab: symtab initializing\n");
	printf("symtab: global level is %lx\n", (long)global_level);
    }

    symtab_set_temp_prefix("temp");
}

void symtab_backup_to_global_level(void)
{
    while (current_level && current_level != global_level)
      symtab_destroy_current_block();
}

char *symbol_name(Symbol *sym)
{
    return sym->name;
}

Symbol *symbol_create(char *name)
{
    long loc = (long) ((unsigned long) hash_string(name)) % (unsigned long) symtab_size;
    Int current_collisions= 0;

    if (symtab_debug > 1) fprintf(stderr, "symbol_create >%s< ... ", name);
    while (1) {
	if (!symtab[loc]) {
	    symtab[loc]= xmalloc(sizeof(Symbol) + strlen(name));
	    symtab[loc]->binding= 0;
	    strcpy(symtab[loc]->name, name);
	    symtab_total_symbols++;
	    symtab_collisions += current_collisions;
	    if (symtab_debug > 1) fprintf(stderr, "made a new one ");
	    break;
	}
	if (!strcmp(name, symtab[loc]->name)) break;
	loc += 1;
	if (loc == symtab_size) loc= 0;
	current_collisions++;
	if (symtab_collisions > symtab_collisions_limit &&
	    symtab_total_symbols > symtab_total_symbols_limit) {
            if (symtab_grow()) return symbol_create(name);
	    else symtab_collisions_limit *= 2;
	}
    }
    if (symtab_debug > 1) fprintf(stderr, "at loc %ld\n", loc);
    return symtab[loc];
}

void symbol_reset_temp(void)
{
    symbol__temp_id= 0;
}

void symtab_set_temp_prefix(char *prefix)
{
    strncpy(symbol__temp_name, prefix, 20);
    symbol__temp_name[19]= 0;
}

Symbol *symbol_temp(void)
{
    char buf[15];
    symbol__temp_id++;
    sprintf(buf, ".%s%ld", symbol__temp_name, symbol__temp_id);
    return symbol_create(buf);
}
    
void *symtab_get(Symbol *sym)  /* NULL if no binding */
{
    return sym->binding ? sym->binding->val : 0;
}

void symtab_begin_block(void)
{
    current_level= symtab__make_level(current_level);
    current_level_no++;
    if (symtab_debug)
      printf("symtab: Begin block: level now %lx\n", (long)current_level);
}

void symtab_end_block(void)
{
    if (current_level == global_level)
      die(("symtab:  too many end blocks\n"));
    symtab_destroy_current_block();
    if (symtab_debug)
      printf("symtab: End block: level now %lx\n", (long)current_level);
}

Int symtab_current_level(void)
{
    return current_level_no;
}

/* 0 if duplicate, 1 if successful */
Int symtab__define(Symbol *sym, void *val,enum binding_type type)
{
    Binding *current_binding= sym->binding;
    if (current_level == global_level) return symtab__define_global(sym, val,type);
    if (symtab_debug)
      printf("symtab: Locally defining '%s' to %lx\n", symbol_name(sym), (long)val);
    if (current_binding && current_binding->lev_no == current_level_no) return 0;
    if (current_level->nsyms >= current_level->maxsyms) {
	current_level->maxsyms *= 2;
	current_level= xrealloc(current_level, sizeof(Level) +
			       sizeof(Binding) * (current_level->maxsyms-1));
	check(current_level);
	if (current_level->child) current_level->child->parent= current_level;
    }
    current_level->syms[current_level->nsyms++]= sym;
    sym->binding= new_binding(val, type,
			      current_level_no, current_binding);
    return 1;
}

Int symtab_define_value(Symbol *sym, void *val)
{
  return(symtab__define(sym, val,value_binding));
}

Int symtab_define_type(Symbol *sym, void *val)
{
  return(symtab__define(sym, val,type_binding));
}

Int symtab_define_symbol(Symbol *sym, void *val)
{
  return(symtab__define(sym, val,symbol_binding));
}

Int symtab__define_global(Symbol *sym, void *val,enum binding_type type)
      /* 0 if duplicate, 1 if successful */
{
    Binding *current_binding= sym->binding;
    if (symtab_debug)
      printf("symtab: Globally defining '%s' to %lx\n", symbol_name(sym), (long)val);
    if (current_binding) return 0;
    if (global_level->nsyms >= global_level->maxsyms) {
	global_level->maxsyms *= 2;
	global_level= xrealloc(global_level, sizeof(Level) +
			       sizeof(Binding) * (global_level->maxsyms-1));
	check(global_level);
	if (current_level_no == 0) current_level= global_level;
	if (global_level->child) global_level->child->parent= global_level;
    }
    global_level->syms[global_level->nsyms++]= sym;
    sym->binding= new_binding(val, type, 0, current_binding);
    return 1;
}

Int symtab_define_global_value(Symbol *sym, void *val)
{
  return(symtab__define_global(sym, val,value_binding));
}
Int symtab_define_global_type(Symbol *sym, void *val)
{
  return(symtab__define_global(sym, val,type_binding));
}
Int symtab_define_global_symbol(Symbol *sym, void *val)
{
  return(symtab__define_global(sym, val,symbol_binding));
}

/*-------------------------------------------------------------------------*/
/* Stand-alone test code                                                   */

#ifdef STAND_ALONE

#include <ctype.h>

void main(Int argc, char **argv)
{
    char buf[200];
    Int i= 0, c;

    symtab_init();
    while ((c= getchar()) != EOF) {
	if (!(isalnum(c) || c == '_')) {
	    buf[i]= 0;
	    if (i) symbol_create(buf);
	    i=0;
	} else {
	    buf[i++]= c;
	}
    }
    symtab_stats();
}

#endif /* STAND_ALONE */


    
	    
