/**
 ** type.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#define TYPE_MODULE

#include "core.h"

#define INT_SIZE 2
#define LONG_SIZE 4
#define CHAR_SIZE 1
#define FLOAT_SIZE 4
#define ARRAY_HEADER_SIZE 2
#define POINTER_SIZE 2
#define VOID_SIZE 0

char type__buf1[50], type__buf2[50];
char *type__name1= type__buf1, *type__name2= type__buf2;

Int delete_duplicate_types= 1;

/*-------------------------------------------------------------------------*/
/* Private Functions                                                       */

char *type__name(char *buf, Type *type)
{
    switch (TYPE_ID(type)) {
      case void_id:
	strcpy(buf, "<void>");
	break;
      case char_id:
	strcpy(buf, "<char>");
	break;
      case int_id:
	strcpy(buf, "<int>");
	break;
      case long_id:
	strcpy(buf, "<long>");
	break;
      case float_id:
	strcpy(buf, "<float>");
	break;
      case pointer_id:
	strcpy(buf, "<pointer to ");
	buf += strlen(buf);
	buf= type__name(buf, type->Pointer.type);
	strcpy(buf, " >");
	break;
      case array_id:
	strcpy(buf, "<array of ");
	buf += strlen(buf);
	buf= type__name(buf, type->Array.elemtype);
	strcpy(buf, " >");
	break;
      case func_id:
	strcpy(buf, "<function>");
	break;
      case undef_id:
	strcpy(buf, "<undefined>");
	break;
      case block_id:
	strcpy(buf, "<block>");
	break;
      case label_id:
	strcpy(buf, "<label>");
	break;
      default:
	die(("Illegal type %d in type__name", TYPE_ID(type)));
	break;
    }
    buf += strlen(buf);
    return buf;
}


/*-------------------------------------------------------------------------*/
/* Public Functions                                                        */

/**************************************************************/
/* type_name                                                  */
/* Returns a readable representation of the type              */
/* Only the last two return values from this function         */
/* are valid;  after that, the buffers returned get recycled. */
/**************************************************************/

char *type_name(Type *type)
{
    char *temp;
    temp= type__name1;
    type__name1= type__name2;
    type__name2= temp;
    type__name(type__name1, type);
    return type__name1;
}

/*******************************************************************/
/* type_compatible                                                 */
/* Tells if a value of type a can be assigned to a value of type b */
/*******************************************************************/

Int type_compatible(Type *a, Type *b)
{
    if (TYPE_ID(a) == undef_id || TYPE_ID(b) == undef_id) return 1;
    if (TYPE_ID(a) == pointer_id && TYPE_ID(b) == int_id) return 1;
    if (TYPE_ID(a) != TYPE_ID(b)) return 0;
    switch(TYPE_ID(a)) {
      case char_id:
      case int_id:
      case long_id:
      case float_id:
      case void_id:
	return 1;
      case pointer_id:
	return type_compatible(a->Pointer.type, b->Pointer.type);
      case array_id:
	return type_compatible(a->Array.elemtype, b->Array.elemtype);
      case func_id:
	{
	    Int nargs= type_Func_number_args(a);
	    if (nargs != type_Func_number_args(b)) return 0;
	    while (nargs--) {
		if (!type_compatible(type_Func_arg_type(a, nargs),
				     type_Func_arg_type(b, nargs))) return 0;
	    }
	    if (type_Func_calling_convention(a) !=
		type_Func_calling_convention(b)) return 0;
	    return type_compatible(a->Func.return_type, b->Func.return_type);
	}
      default:
	die(("Illegal type id %d to type_compatible", TYPE_ID(a)));
    }
    /* Will not reach here */
    return 0;
}

/******************************************************/
/* type_sizeof                                        */
/* Returns the size in bytes needed to store the type */
/******************************************************/

Int type_sizeof(Type *type)
{
    switch (TYPE_ID(type)) {
      case void_id:
	return VOID_SIZE;
      case char_id:
	return CHAR_SIZE;
      case int_id:
	return INT_SIZE;
      case long_id:
	return LONG_SIZE;
      case float_id:
	return FLOAT_SIZE;
      case pointer_id:
	return POINTER_SIZE;
      case array_id:
	return ARRAY_HEADER_SIZE +
	       type_sizeof(type->Array.elemtype) * type->Array.size;
      case func_id:
	return 0;
      case block_id:
	return 0;
      case undef_id:
	return 2;
      case label_id:
	return 2;
      default:
	die(("type_size called with type %d\n", TYPE_ID(type)));
    }
    /* Will not reach here */
    return 0;
}

Type_id type_id(Type *type)
{
    return TYPE_ID(type);
}

/****************************************************************************/
/*** TYPE CONSTRUCTOR FUNCTIONS                                           ***/
/*** These functions are used for constructing types                      ***/
/*** Primitive types are only created once and returned multiple times.   ***/
/*** Compound types are newly created each time.                          ***/
/*** TODO:                                                                ***/
/*** make compound types use the same type structure if they are the same ***/
/****************************************************************************/

Type *type_Void(void)
{
    static Void_type ret = {void_id};
    return (Type*) &ret;
}

Type *type_Char(void)
{
    static Char_type ret = {char_id};
    return (Type*) &ret;
}

Type *type_Int(void)
{
    static Int_type ret = {int_id};
    return (Type*) &ret;
}

Type *type_Long(void)
{
    static Long_type ret = {long_id};
    return (Type*) &ret;
}

Type *type_Float(void)
{
    static Float_type ret = {float_id};
    return (Type*) &ret;
}

Type *type_Simple(Type_id id)	/* return Type of Type_id */
{
  switch(id)
    {
    case char_id: return(type_Char());
    case int_id: return(type_Int());
    case long_id: return(type_Long());
    case float_id: return(type_Float());
    case void_id: return(type_Void());
    }
  return(NULL);
}

Type *type_Array(Type *elemtype, Int size)
{
    Type *ret= (Type*) malloc(sizeof(Array_type));

    ret->Array.type_id=  array_id;
    ret->Array.elemtype= elemtype;
    ret->Array.size=     size;

    return find_original_type(ret);
}

Type *type_Array_elemtype(Type *type)
{
    ASSERT(type->Array.type_id == array_id);

    return type->Array.elemtype;
}

Int type_Array_size(Type *type)
{
    ASSERT(type->Array.type_id == array_id);

    return type->Array.size;
}

Type *type_Pointer(Type *type)
{
    Type *ret= (Type*) malloc(sizeof(Pointer_type));

    ret->Pointer.type_id= pointer_id;
    ret->Pointer.type=    type;

    return find_original_type(ret);
}

Type *type_Abstract_declarator(Symbol *symbol)
{
    Type *ret= (Type*) malloc(sizeof(Abstract_declarator_type));

    ret->Abstract_declarator.type_id= abstract_declarator_id;
    ret->Abstract_declarator.symbol= symbol;

    return find_original_type(ret);
}

Type *type_parse_Abstract_declarator(void **type, Type *abstract_declarator)
{
    Type *ret;
    switch (type_id(abstract_declarator)) {
      case abstract_declarator_id:
	ret= *type;
	*type= abstract_declarator->Abstract_declarator.symbol;
	break;
      case pointer_id:
	ret= type_Pointer(type_parse_Abstract_declarator(type,
					type_Pointer_deref(abstract_declarator)));
	break;
      case array_id:
	ret= type_Array(type_parse_Abstract_declarator(type,
				        type_Array_elemtype(abstract_declarator)),
			type_Array_size(abstract_declarator));
	break;
      default:
	die(("Illegal abstract declarator id %d\n", type_id(abstract_declarator)));
    }
    return ret;
}

Type *type_Pointer_deref(Type *type)
{
    ASSERT(type->Pointer.type_id == pointer_id);

    return type->Pointer.type;
}

Type *type_Label(void)
{
    static Label_type ret = {label_id};
    return (Type*) &ret;
}
    
Type *type_Func(Type *return_type, Calling_convention conv, Int number_args)
{
    Type *ret= (Type*) malloc(sizeof(Func_type) + sizeof(Func_arg) * (number_args-1));
    Int  i;

    ret->Func.type_id=     func_id;
    ret->Func.return_type= return_type;
    ret->Func.number_args= number_args;
    ret->Func.calling_convention= conv;

    for (i= 0; i< number_args; i++) {
	ret->Func.argument[i].type= 0;
	ret->Func.argument[i].name= 0;
    }
    return ret;
}

Calling_convention type_Func_calling_convention(Type *type)
{
    ASSERT(type->Func.type_id == func_id);

    return type->Func.calling_convention;
}

Type *type_Func_return_type(Type *type)
{
    ASSERT(type->Func.type_id == func_id);

    return type->Func.return_type;
}
    
Int type_Func_number_args(Type *type)
{
    ASSERT(type->Func.type_id == func_id);

    return type->Func.number_args;
}
    
Type *type_Func_arg_type(Type *type, Int arg_no)
{
    Type *arg_type;
    ASSERT(type->Func.type_id == func_id);
    ASSERT(0 <= arg_no && arg_no < type->Func.number_args);

    arg_type= type->Func.argument[arg_no].type;
    ASSERT(arg_type);
    
    return arg_type;
}

Symbol *type_Func_arg_name(Type *type, Int arg_no)
{
    Symbol *name;
    ASSERT(type->Func.type_id == func_id);
    ASSERT(0 <= arg_no && arg_no < type->Func.number_args);

    name= type->Func.argument[arg_no].name;
    ASSERT(name);
    
    return name;
}

void type_Func_set_arg_name(Type *type, Int arg_no, Symbol *arg_name)
{
    ASSERT(type->Func.type_id == func_id);
    ASSERT(0 <= arg_no && arg_no < type->Func.number_args);

    type->Func.argument[arg_no].name= arg_name;
}

void type_Func_set_arg_type(Type *type, Int arg_no, Type *arg_type)
{
    ASSERT(type->Func.type_id == func_id);
    ASSERT(0 <= arg_no && arg_no < type->Func.number_args);

    type->Func.argument[arg_no].type= arg_type;
}

#if 0
Type *type_Char_constant(Int val)
{
    Type *ret= (Type*) malloc(sizeof(Char_constant_type));

    ret->Char_constant.type_id= int_constant_id;
    ret->Char_constant.value=   val;

    return find_original_type(ret);
}

Type *type_Int_constant(Int val)
{
    Type *ret= (Type*) malloc(sizeof(Int_constant_type));

    ret->Int_constant.type_id= int_constant_id;
    ret->Int_constant.value=   val;

    return find_original_type(ret);
}

Type *type_Float_constant(double val)
{
    Type *ret= (Type*) malloc(sizeof(Float_constant_type));

    ret->Float_constant.type_id= float_constant_id;
    ret->Float_constant.value=   val;

    return find_original_type(ret);
}
#endif

Type *type_Block(void)
{
    static Block_type ret = {block_id};
    return (Type*) &ret;
}

Type *type_Undef(void)
{
    static Undef_type ret = {undef_id};
    return (Type*) &ret;
}
/******** Remove duplicate types **********/

Int type_host_storage_size(Type *a)
{
    switch(TYPE_ID(a)) {
      case char_id: return (sizeof(Char_type));
      case int_id:  return (sizeof(Int_type));
      case long_id: return (sizeof(Long_type));
      case float_id:return (sizeof(Float_type));
      case void_id: return (sizeof(Void_type));
      case undef_id:return (sizeof(Undef_type));
      case pointer_id: return (sizeof(Pointer_type));
      case array_id:   return (sizeof(Array_type));
      case abstract_declarator_id:  return(sizeof(Abstract_declarator_type));
      case func_id:
	{
	    Int nargs= type_Func_number_args(a);

	    return sizeof(Func_type) + sizeof(Func_arg) * (nargs - 1);
	}
      default:
	die(("Illegal type id %d to type_host_storage_size", TYPE_ID(a)));
    }
    /* Will not reach here */
    return 0;
}
    
Int type_equal(Type *a, Type *b)
{
    return !memcmp(a, b, type_host_storage_size(a));
}

Growbuf all_types;

void type_init_module(void)
{
    growbuf_init(&all_types);
}

Type *find_original_type(Type *t)
{
    Int i;
    Int ntypes= GROWBUF_LENGTH(&all_types) / sizeof(Type*);
    Type **types= (Type**) GROWBUF_DATA(&all_types);

    if (!delete_duplicate_types) return t;

    for (i= 0; i< ntypes; i++) {
	if (t == types[i]) return t;
	if (type_equal(types[i], t)) {
	    free(t);
	    return types[i];
	}
    }
    growbuf_add_ptr(&all_types, t);
    return t;
}



