/**
 ** filelib.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include CONFIG
#include <common.h>
#include <ctype.h>

#include "filelib.h"

#ifndef MAC
#include <sys/types.h>
#include <sys/stat.h>
#endif

#if defined(UNIX) && !defined(SOLARIS)
#include <sys/dir.h>
#elif defined(SOLARIS)
#include <stdio.h>
#include <dirent.h>
#endif

#ifdef PC
#include <dos.h>
Int __cdecl chsize(Int, long); /* from <io.h> microsoft c 7.00 */
#endif

#ifdef UNIX
char *filenm_simplify(char *filename)
{
    char *ret= filenm_simplify_unix(filename);
    if (!strcmp(ret, "")) ret= string_copy("/");
    if (strcmp(ret, "/")) {
	if (ret[strlen(ret)-1] == '/') {
	    ret[strlen(ret)-1]= 0;
	}
    }
    return ret;
}
#endif

#ifdef PC
char *filenm_simplify(char *filename)
{
    char *ret;
    char *drive= NULL;
    filename= free_soon(string_replace_char(filename, '\\', '/'));

    /* Take off drive letter if any */
    if (isalpha(filename[0]) && filename[1] == ':') {
	drive= string_left(filename, 2);
	filename= free_soon(string_mid(filename, 2, -1));
    }
	
    ret= filenm_simplify_unix(filename);
    
    if (!strcmp(ret, "")) {
	if (drive) free(drive);
	return ret;
    }
    
    if (strcmp(ret, "/")) {
	if (ret[strlen(ret)-1] == '/') {
	    ret[strlen(ret)-1]= 0;
	}
    }

    if (drive) {
	ret= string_append(free_soon(drive), free_soon(ret));
    }

    return ret;
}
#endif

char *filenm_simplify_unix(char *filename)
{
    char *ret= string_append(filename, "/");
    char *target, *targetstart;
    
    while ((target= string_find_substring(ret, "/./"))) {
	ret= string_delete(free_soon(ret), target-ret, 2);
    }

    while ((target= string_find_substring(ret, "/../"))) {
	for (targetstart= target-1; 1; targetstart--) {
	    if (targetstart < ret) {
		targetstart= ret-1;
		break;
	    }
	    if (*targetstart == '/') break;
	}
	ret= string_delete(free_soon(ret), targetstart - ret + 1, target - targetstart + 3);
    }
    /*ret[strlen(ret)-1]= 0;*/ /* delete trailing "/" */

    while ((target= string_find_substring(ret, "//"))) {
	ret= string_mid(free_soon(ret), (target-ret)+1, -1);
    }
    return ret;
}

#ifndef MAC
long file_length(FILE *f)
{
    struct stat file_stat;
    fstat(fileno(f), &file_stat);
    return file_stat.st_size;
}
#endif

#ifndef MAC
Int filenm_is_directory(char *filename)
{
    struct stat file_stat;

#ifdef PC
    if (isalpha(filename[0]) &&
	(!strcmp(filename + 1, ":/") ||
	 !strcmp(filename + 1, ":")))     return 1;
#endif    
    
    stat(filename, &file_stat);
    return !!(file_stat.st_mode & S_IFDIR);
}
#endif


Int is_file_directory_delimiter(Int ch)
{
    return strchr(FILE_DIRECTORY_DELIMITERS, ch) ? 1 : 0;
}

char *filenm_create(char *dir, char *filename)
{
    char *ret;

    if (is_file_directory_delimiter(string_lastchar(dir)) ||
	dir[0] == 0) {
	ret= string_printf("%s%s", dir, filename);
    }
    else {
	ret= string_printf("%s%c%s", dir, FILE_DIRECTORY_DELIMITER, filename);
    }

    if (is_file_directory_delimiter(string_lastchar(ret))) ret[strlen(ret)-1]= 0;
    
    return ret;
}

Int file_copy(FILE *dest, FILE *src)
{
    Int c;
    while (1) {
	c= getc(src);
	if (c == EOF) break;
	putc(c, dest);
    }
    return 0;
}

Int filenm_copy(char *dest, char *src)
{
    FILE *d, *s;
    Int num;
    char buf[1024];
    
/*    printf("open %s %s\n", src, READ_BINARY);*/
    s= fopen(src, READ_BINARY);
    if (!s) {
	printf("Could not open %s for reading\n", src);
	return 1;
    }
/*    printf("open %s %s\n", dest, WRITE_BINARY);*/
    d= fopen(dest, WRITE_BINARY);
    if (!d) {
	printf("Could not open %s for writing\n", dest);
	fclose(s);
	return 1;
    }

    while (1) {
	num= fread(buf, 1, 1024, s);
	if (num==0) break;
	fwrite(buf, 1, num, d);
    }

    fclose(d);
    fclose(s);

#ifdef UNIX
    {
	struct stat s;
	
	stat(src, &s);
	chmod(dest, (Int) s.st_mode);
    }
#endif	

    return 0;
}

/* 0 if success */
Int filenm_rename(char *dest, char *src)
{
    if (filenm_copy(dest, src)) return 1;
    if (filenm_delete(src)) return 1;
    return 0;
}

#ifdef UNIX
#define HAS_UNLINK
#endif

#ifdef PC
#define HAS_UNLINK
#endif

#ifdef HAS_UNLINK
Int filenm_delete(char *filename)
{
    return unlink(filename);
}
#endif

#ifdef MAC
Int filenm_delete(char *filename)
{
	printf("filenm_delete for mac not implemented!\n");
}
#endif

#if defined(UNIX) && !defined(MISSING_DIR)
char **dir_get_filenames(char *dirname)
{
    DIR *d;
    struct dirent *ent;
    Growbuf g;
    char **ret;

    growbuf_init(&g);

    d= opendir(dirname);
    
    if (d) {
	while (ent= readdir(d)) {
	    growbuf_add_ptr(&g, string_copy(ent->d_name));
	}
    }
    
    ret= (char**) growbuf_data(&g);
    qsort((void*)ret,
	  (growbuf_length(&g)/sizeof(char*)), sizeof(char*), (void*) table_strcmp);
    growbuf_add_ptr(&g, 0);
    ret= (char**) growbuf_data(&g);
    return ret;
}
#endif /* #ifdef UNIX */




#ifdef PC
char **dir_get_filenames(char *dirname)
{
    struct find_t find;
    char *path;
    Growbuf g;
    char **ret;
    Int  test;

    growbuf_init(&g);

    if (!strcmp(dirname, "")) {
	Int i;
	for (i= 'A'; i <= 'Z'; i++) {
	    growbuf_add_ptr(&g, string_printf("%c:/", i));
	}
    }
    else {

	path= filenm_create(dirname, "*.*");

	/*printf("<%s>\n", path);*/
	
	for (test= _dos_findfirst(path, -1, &find); !test;
	     test= _dos_findnext(&find)) {
	    
	    growbuf_add_ptr(&g, string_copy(find.name));
	    /*printf("[%s]\n", find.name);*/
	}
	/*printf("----------\n");*/
	free(path);
    }
    
    
    ret= (char**) growbuf_data(&g);
    qsort((void*)ret,
	  (growbuf_length(&g)/sizeof(char*)), sizeof(char*), table_strcmp);
    growbuf_add_ptr(&g, 0);
    ret= (char**) growbuf_data(&g);
    return ret;
}
#endif /* #ifdef PC */

#ifdef PC
Int file_truncate(FILE *f, long len)
{
    fflush(f);
    chsize(fileno(f), len);
    return 0;
}
#endif

#ifdef UNIX
Int file_truncate(FILE *f, long len)
{
    fflush(f);
    ftruncate(fileno(f), len);
    return 0;
}
#endif


FILE *fopen_here_or_there_with_format(char *directory, char *filename, char *mode, char *format)
{
    FILE *ret;

    ret= fopen(filename, mode);

    if (ret) {
	printf(format, filename);
    } else {
	char *newfilename= malloc(strlen(directory) + strlen(filename) + 2);
	strcpy(newfilename, directory);
	if (newfilename[0] &&
	    !is_file_directory_delimiter(newfilename[strlen(newfilename)-1])) {
	    newfilename[strlen(newfilename)+1]=0;
	    newfilename[strlen(newfilename)]=FILE_DIRECTORY_DELIMITER;
	}
	strcat(newfilename, filename);
	ret= fopen(newfilename, mode);

	if (ret) {
	    printf(format, newfilename);
	} else {
	    printf("Cannot find file %s or file %s\n",
		   filename, newfilename);
	}
	free(newfilename);
    }
    return ret;
}

FILE *fopen_here_or_there(char *directory, char *filename, char *mode)
{
    return
      fopen_here_or_there_with_format(directory, filename, mode, "Loading %s\n");
}

/* anarch start */

char *filename_here_or_there(char *directory, char *filename,char *dest)
{
    FILE *ret;

    ret= fopen(filename, "r");

    if (ret) 
      {
	fclose(ret);
	return(filename);
      } 
    else 
      {
	strcpy(dest, directory);
	if (dest[0] &&
	    !is_file_directory_delimiter(dest[strlen(dest)-1])) {
	    dest[strlen(dest)+1]=0;
	    dest[strlen(dest)]=FILE_DIRECTORY_DELIMITER;
	}
	strcat(dest, filename);
	ret= fopen(dest, "r");

	if (!ret) {
	    printf("Cannot find file %s or file %s\n",
		   filename, dest);
	    return((char *)NULL);
	}
	fclose(ret);
	return(dest);
      }
}

char *filename_here_or_there_with_format(char *directory, char *filename,char *dest,char *format)
{
    FILE *ret;

    ret= fopen(filename, "r");

    if (ret) 
      {
	fclose(ret);
	printf(format, filename);
	return(filename);
      } 
    else 
      {
	strcpy(dest, directory);
	if (dest[0] &&
	    !is_file_directory_delimiter(dest[strlen(dest)-1])) {
	    dest[strlen(dest)+1]=0;
	    dest[strlen(dest)]=FILE_DIRECTORY_DELIMITER;
	}
	strcat(dest, filename);
	ret= fopen(dest,"r");

	if (!ret) {
	    printf("Cannot find file %s or file %s\n",
		   filename, dest);
	    return((char *)NULL);
	}
	printf(format, dest);
	return(dest);
      }
}

#ifdef UNIX
char *create_temp_filename(char *tempdir,char *prefix)
{
  char temp[50];
  char *tname;

/*  if(*temp)
    strcpy(temp,tempdir);
  else */
    strcpy(temp,"/tmp/");

  strcat(temp,prefix);
  strcat(temp,"XXXXXX");
  tname=string_copy(mktemp(temp));
/*  printf("Writing output to file %s\n",tname);*/
  return(tname);
}
#endif

#ifdef not
char *create_temp_filename(char *tempdir,char *prefix)
{
  FILE *nfile;
  char command[100]="mktemp ";
  char temp[50];
  char *sptr;

  if(*tempdir) 
    strcpy(temp,"-d");
  else
    *temp=0;
  strcat(command,strcat(temp,tempdir));

  if(*prefix) 
    strcpy(temp,"-p");
  else
    *temp=0;
  strcat(command,strcat(temp,prefix));

  nfile=popen(command,"r");

  fgets(temp, 50, nfile);
  pclose(nfile);

  sptr=strchr(temp,'\n');
  if(sptr)
    *sptr=0;
  printf("Writing output to file %s\n",temp);
  return(string_copy(temp));
}
#endif

