/**
 ** io_pc.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#define IO_MODULE

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <asiports.h>
#include <ibmkeys.h>
#include <bios.h>

#include "util.h"


typedef enum {
    pc_not_a_stream,
    pc_serial_stream,
    pc_keyboard_stream
  } PC_stream_type;

typedef struct iostream_struct {
    PC_stream_type type;
    Int            data;
} IOStream;

#include "io.h"

Int io_delay_factor= 1;
Int io_hex_on_serial= 0;
Int io_disable_xy;

char *io_default_serial_name = "com1";
Int io_debug= 0;

Int io_carrier_detect(IOStream *s)
{
    return 1;  /* TODO: implement me */
}

void io_set_delay_factor(Int f)
{
    if (f < 1) f= 1;
    if (f > 100) f= 100;
    io_delay_factor= f;
}

void io_set_hex_on_serial(Int hex)
{
    io_hex_on_serial= hex;
}

#define MSEC_PER_TICK 58

long clock(void)
{
    long ret;
    _bios_timeofday(_TIME_GETCLOCK, &ret);
    return ret;
}

void io_make_stdin_unbuffered(void)
{
}

void io_sigint_interrupt(void)
{
}

/*******************/
/* SERIAL ROUTINES */
/*******************/

IOStream *io_open_serial(char *name)
{
    Int port, status;
    IOStream *ret= malloc(sizeof(IOStream));
    ret->type= pc_serial_stream;
    if (!strnicmp(name, "com1", 4)) {
        port= COM1;
    } else if (!strnicmp(name, "com2", 4)) {
        port= COM2;
    } else {
        printf("Unknown com port.  Try 'com1' or 'com2'\n");
        return 0;
    }
    ret->data= port;

    status= asiopen(port, ASINOUT | BINARY | NORMALRX, 1024, 1024, 9600L,
                    P_NONE, 1, 8, 0, 0);
    if (status != ASSUCCESS) {
        printf("Cannot open com port %s: error %d\n", name, status);
        return 0;
    }
    return ret;
}

IOStream *io_open_stdin(void)
{
    IOStream *ret= malloc(sizeof(IOStream));
    ret->type= pc_keyboard_stream;
    return ret;
}

void io_serial_init(long baud, IOStream *s)
{
    asiinit(s->data, baud, P_NONE, 1, 8);
    io_discard_input(s);
}

Int io_getchar(IOStream *s, Int timeout)
{
    IOStream *streams[2];
    Int which;
    if (io_debug) printf("enter io_getchar\n");
    streams[0]= s;
    streams[1]= 0;
    return io_getchar_multiple_inputs(streams, &which, timeout);
}

void io_putchar(Int c, IOStream *s)
{
    switch (s->type) {
      case pc_serial_stream:
        if (io_debug) printf("putchar %02X >%c<\n", c, c);
        asiputc(s->data, c);
        break;
      default:
        printf("Illegal IOStream in io_putchar\n");
        exit(1);
    }
}

Int io_getchar_nowait(IOStream *s)
{
    Int c;
    switch (s->type) {
      case pc_keyboard_stream:
        if (gfkbhit()) {
          c= getkey();
          if (c == 3) exit(0);
          return c;
        }
        else return EOF;
      case pc_serial_stream:
        c= asigetc( s->data );
        if (c >= 0 && io_debug) printf("getchar %02X >%c<\n", c, c);
        return (c >= 0 ? c : EOF);
      default:
        printf("Illegal IOStream\n");
        exit(1);
    }
}


Int io_getchar_multiple_inputs(IOStream **inputs, Int *which, Int timeout)
{
    Int j= 0;
    Int c;
    long first_tick= clock();
    long last_tick= first_tick + 1 +
      (io_delay_factor*timeout)/ MSEC_PER_TICK;
    
    if (io_debug) printf("get serial until %ld\n", last_tick);

    do {
        for (j= 0; inputs[j]; j++) {
            c= io_getchar_nowait(inputs[j]);
            if (c != EOF) {
                *which= j;
                return c;
            }
        }
        if (io_debug) printf("getchar %ld\n", clock());
    } while (timeout && (timeout < 0 || clock() <= last_tick));
    if (io_debug) printf("TIMEOUT %d msec\n", timeout);
    return EOF;
}

void io_discard_input(IOStream *s)
{
    while (io_getchar(s, 0) != EOF);
}

void msleep(Int msec)
{
    long first_tick_count= clock();
    long last_tick_count= first_tick_count + 1 +
        (io_delay_factor * msec) / MSEC_PER_TICK;
    if (io_debug) printf("msleep till %ld\n", last_tick_count);
    while (clock() <= last_tick_count) 
      if (io_debug) printf("msleep %ld\n", clock());
}

#ifdef STAND_ALONE
NOPROTO main()
{
    IOStream *kbd= io_open_stdin();
    IOStream *ser= io_open_serial(io_default_serial_name);
    IOStream *streams[3];
    Int i;
    
    streams[0]= kbd;
    streams[1]= ser;
    streams[2]= 0;
    
    for (i= 0; i< 3; i++) {
        Int c, which;
        
        c= io_getchar_multiple_inputs(streams, &which, 5000);
        printf("Got a char %d >%c< from %d\n", c, c, which);
    }
    
}
#endif
