/**
 ** iob_pc.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


/* Serial and console library routines for UNIX */

#include <stdio.h>
#include <string.h>     
#include <bios.h>

#include "util.h"
#include "byteq.h"
#include "stringlb.h"
#include "filelib.h"

#include <sys/timeb.h>
#include <conio.h>

#include "iob.h"

Int io_carrier_detect(IOStream *s)
{
   return 1;
}
void io_sigint_interrupt(void)
{
    printf("Bye!\n");
}

void io_make_stdin_unbuffered(void)
{
}

void io_make_raw(IOStream *s)
{
    (void) s;
}

/*******************/
/* SERIAL ROUTINES */
/*******************/

IOStream *io_open_serial(char *name)
{
    IOStream *ret= malloc(sizeof(IOStream));
    strncpy(ret->name, name, IOSTREAM_NAME_MAX);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
    
    ret->port= serial_port_by_name(name);
    serial_init(ret->port);

    ret->istream= 0;
    ret->ostream= 0;
    
    bytequeue_init(&ret->inbuf, 1020);
    bytequeue_init(&ret->outbuf, 1021);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;
    return ret;
}

IOStream *io_open_filename(char *name)
{
   FILE *io= fopen(name, "r");
   if (!io) return 0;
   return io_open_file(name, io, io);
}

IOStream *io_open_file(char *name, FILE *in, FILE *out)
{
    IOStream *ret= malloc(sizeof(IOStream));
    strncpy(ret->name, name, IOSTREAM_NAME_MAX);
    ret->name[IOSTREAM_NAME_MAX-1]= 0;
    
    ret->istream= in;
    ret->ostream= out;
    ret->port= 0;
    bytequeue_init(&ret->inbuf,1024);
    bytequeue_init(&ret->outbuf, 1024);
    ret->extended_charset= 0;
    ret->prefix_detected= 0;
    return ret;
}

IOStream *io_open_stdin(void)
{
   return io_open_file("console", stdin, stdout);
}

void io_highlight_video(IOStream *s, Int on)
{
    /* TODO: implement me */
}

void io_inverse_video(IOStream *s, Int on)
{
    /* TODO: implement me */
}

void io_serial_init(long baud, IOStream *s)
{
   serial_init(s->port);
   serial_set_baud(s->port, baud);
   while (1) {
      if (io_getchar(s, 0) == EOF) break;
   }
}

char *io_fn_prefix="\033[";


Int io_getchar_multiple_inputs_n(IOStream **inputs, Int *which, Int timeout, Int n)
{
    IOStream *input;
    Int c;
    while (1) {
	c= io_getchar_multiple_inputs_n_internal(inputs, which, timeout, n);
	/*printf("[%d]", c);*/
	if (c == EOF) return c;
	input= inputs[*which];
	if (!input->extended_charset) {
	    return c;
	}
	/*printf("<%d, prefix=%d>", c, input->prefix_detected);*/
	if (input->prefix_detected == 2) {
	    input->prefix_detected= 0;
	    switch (c) {
	      case 'A': return IO_UPARROW;
	      case 'B': return IO_DOWNARROW;
	      case 'C': return IO_RIGHTARROW;
	      case 'D': return IO_LEFTARROW;
	      default:	return 27; /* not quite the right thing to do */
	    }
	}
	if (io_fn_prefix[input->prefix_detected] != c) {
	    input->prefix_detected= 0;
	    return c;
	}
	input->prefix_detected++;
    }
}

#define MSEC_PER_TICK 58

Int io_write(IOStream *s, char *buf, long len)
{
   long pos= 0;
   /* Loop until all written out */
   if (s->port) {
      serial_write(s->port, buf, len);
      return len;
   } else {
      fwrite(buf, len, 1, s->ostream);
      return len;
   }
}

Int io_getchar_multiple_inputs_n_internal(IOStream **inputs, Int *which, Int timeout, Int n)
{
   Int c;
   Int i;
   long here_till= ticks() + timeout/MSEC_PER_TICK + 2;

   for (i= 0; i< n; i++) {
      if (!BYTEQUEUE_EMPTY(&inputs[i]->inbuf)) {
	 *which= i;
	 return bytequeue_remove_char(&inputs[i]->inbuf);
      }
   }

   /* TODO: don't poll when we're just doing a single getchar
      from a FILE* */

   if (n == 1 && timeout < 0) {
      c= io__getc(inputs[0]);
      *which = ((c == EOF) ? -1 : 0);
      return c;
   }
      
   while (1) {
      long delta= ticks() - here_till;
      for (i= 0; i< n; i++) {
	 if (io__peekc(inputs[i])) {
	    c= io__getc(inputs[i]);
	    *which = ((c == EOF) ? -1 : i);
	    return c;
	 }
      }
      if (timeout == 0 ||
	  (timeout >= 0 && delta >= 0)) {
	 *which= -1;
	 return EOF;
      }
   }
}

Int io__peekc(IOStream *i)
{
   Int ret;
   if (i->port) {
      ret= serial_receive_full(i->port);
      /*if (ret) printf("+"); else printf("-");*/
      /*fflush(stdout);*/
      return ret;
   }
   if (i->istream == stdin) {
      ret= kbhit();
      /*printf("keyboard peek is %ld\n", ret);*/
      return ret;
   }
   /*printf("can't peekc stream\n"); fflush(stdout);*/
   return 1;
}

Int io__getc(IOStream *i)
{
   if (i->port) return serial_getchar(i->port);
   if (i->istream == stdin)
      {
	 int c=getch();
	 if(c==0 && kbhit()) {
	    switch(getch()) {
	       case 'H':  return('P'-'@');
	       case 'K':  return('B'-'@');
	       case 'M':  return('F'-'@');
	       case 'P':  return('N'-'@');
	       default:   return(c);
	    }
	 }
	 else {
	    return c;
	 }
      }
   /*printf("bad getchar\n"); fflush(stdout);*/
   return getc(i->istream);
}

void msleep(Int mseconds)
{
   long sleep_till= ticks() + mseconds/MSEC_PER_TICK + 2;

   while (1) {
      long delta= ticks() - sleep_till;
      if (delta >= 0) return;
   }
}

long ticks(void)
{
    long ret;
    _bios_timeofday(_TIME_GETCLOCK, &ret);
    return ret;
}

long io_time_msec(void)
{
   struct timeb t;
   ftime(&t);
   return t.millitm + t.time * 1000L;
}

long mtime(void)
{
   return io_time_msec();
}

	
/* pc direct hardware serial routines follow */

#define TRANSMIT(port) ((port)+0)
#define RECEIVE(port) ((port)+0)
#define DIVISOR_LOW(port) ((port)+0)
#define DIVISOR_HIGH(port) ((port)+1)
#define INTERRUPT_ENABLE(port) ((port)+1)
#define INTERRUPT_ID(port) ((port)+2)
#define LINE_CONTROL(port) ((port)+3)
#define MODEM_CONTROL(port) ((port)+4)
#define LINE_STATUS(port) ((port)+5)
#define MODEM_STATUS(port) ((port)+6)
#define SCRATCH(port) ((port)+7)

#define DLAB 0x80

#define TRANSMIT_READY 0x20
#define RECEIVE_READY 0x01

void serial_set_dlab(Int port, Int set)
{
    if (set) 
        outp(LINE_CONTROL(port), inp(LINE_CONTROL(port)) | DLAB);
    else
        outp(LINE_CONTROL(port), inp(LINE_CONTROL(port)) & ~DLAB);
}
        
Int serial_transmit_empty(Int port)
{
    return (inp(LINE_STATUS(port)) & TRANSMIT_READY);
}

void serial_putchar(Int port, Int c)
{
    while (!serial_transmit_empty(port)) ;
    outp(TRANSMIT(port), c);
}

Int serial_write(Int port, char *buf, long len)
{
   long i;
   for (i= 0; i< len; i++) {
      serial_putchar(port, buf[i]);
   }
   return len;
}

Int serial_receive_full(Int port)
{
    return (inp(LINE_STATUS(port)) & RECEIVE_READY);
}
    
Int serial_getchar(Int port)
{
    while (!serial_receive_full(port)) ;
    return inp(RECEIVE(port));
}

long serial_get_baud_divisor(Int port)
{
   long ret;
   serial_set_dlab(port, 1);
   ret= (((unsigned char) inp(DIVISOR_LOW(port))) >> 0) + 
	(((unsigned char) inp(DIVISOR_HIGH(port))) << 8);
   serial_set_dlab(port, 0);
   return ret;
}

void serial_set_baud_divisor(Int port, long divisor)
{
   serial_set_dlab(port, 1);
   outp(DIVISOR_LOW(port), (Int) ((divisor>>0) & 0xff));
   outp(DIVISOR_HIGH(port), (Int) ((divisor>>8) & 0xff));
   serial_set_dlab(port, 0);
}

#define MAXBAUD 115200

long serial_get_baud(Int port)
{
    return MAXBAUD / serial_get_baud_divisor(port);
}

void serial_set_baud(Int port, long baud)
{
    serial_set_baud_divisor(port, MAXBAUD / baud);
}

void serial_init(Int port)
{
    serial_set_dlab(port, 0);
    outp(INTERRUPT_ENABLE(port), 0); /* turn off INTERRupts */
    /* 8 bits, 1 stop, no parity */
    outp(LINE_CONTROL(port), 0x03);
}

Int serial_port_by_name(char *name)
{
   int n;
   int port=0;
   if (!strnicmp(name, "com", 3)) name += 3;
   n= name[0]-'0';
   switch (n) {
      case 1: port= 0x3f8; break;
      case 2: port= 0x2f8; break;
      case 3: port= 0x3e8; break;
      case 4: port= 0x2e8; break;
      default:
	 fprintf(stderr, "Illegal serial port: %s.  Try com1, com2, com3, or com4\n",
		 name);
	 exit(-1);
	 return 0;
   }
   return port;
}



    

