/**
 ** s19.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include CONFIG
#include <util.h>
#include <stream.h>

#include <ctype.h>

#include "binrec.h"
#include "s19.h"

Int s19_debug= 0;

Int s19_get_hex1(Stream *s)
{
    Int ch= STREAM_GETC(s);
    if ('0' <= ch && ch <= '9') return ch - '0';
    if ('a' <= ch && ch <= 'f') return ch - 'a' + 10;
    if ('A' <= ch && ch <= 'F') return ch - 'A' + 10;
    die(("Illegal hex digit in .s19 file >%c<\n", (int)ch));
    return -1;
}

Int s19_get_hex2(Stream *s)
{
    Int a= s19_get_hex1(s);
    Int b= s19_get_hex1(s);

    return (a<<4) + b;
}
    
Int s19_get_hex4(Stream *s)
{
    Int a= s19_get_hex2(s);
    Int b= s19_get_hex2(s);

    return (a<<8) + b;
}

Binrec *read_contiguous_s19_file(FILE *in)
{
    Binrec *ret;
    Stream s;
    stream_init_from_file(&s, in);
    ret=read_contiguous_s19_stream(&s);
    stream_term(&s);
    return ret;
}

Binrec *read_contiguous_s19_string(char *str)
{
    Binrec *ret;
    Stream s;
    stream_init_from_string(&s, str);
    ret=read_contiguous_s19_stream(&s);
    stream_term(&s);
    return ret;
}

Binrec *read_contiguous_s19_stream(Stream *s)
{
    Binrec *ret= binrec_create(0,0);
    if (s19_debug) printf("Starting read contiguous\n");
    while (1) {
	Binrec *new= read_s19_record_from_stream(s);
	Binrec *newret;
	if (!new) return ret;
	if (s19_debug) printf("Got record\n");
	if (!binrec_can_concat(ret, new))
	  die((".s19 file was not contiguous\n"));
	newret= binrec_concat(ret, new);
	binrec_destroy(ret);
	binrec_destroy(new);
	ret= newret;
    }
}

char *unparse_s19_record(Binrec *rec)
{
    char *buf;
    long i;
    Int checksum= 0;
    buf= malloc((size_t) (rec->len * 2 + 30));
    buf[0]= 0;
    sprintf(buf + strlen(buf), "S1%02lX%04lX", rec->len + 3, rec->addr);
    for (i= 0; i< rec->len; i++)
      sprintf(buf + strlen(buf), "%02X", rec->data[i]);
    sprintf(buf + strlen(buf), "%02lX", checksum); /* sorry, this is bogus */
    sprintf(buf + strlen(buf), "\n");
    return buf;
}

void nuke_cr(Stream *s)
{
    Int c;
    c= STREAM_GETC(s);
    if (c != '\n' && c != '\r') STREAM_UNGETC(s, c);
}

Binrec *read_s19_record_from_stream(Stream *s)
{
    Int c;
    /* Skip whitespace */
    while (1) {
	c= STREAM_GETC(s);
	if (!isspace(c)) break;
	if (s19_debug) {printf("%ld,",c);}
	      
    }
    if (c == EOF) {
       if (s19_debug) {printf("EOF in read_s19 record\n");}
       return NULL;
    }
    if (c != 'S') die(("Illegal line in .s19 file"));
    
    c= STREAM_GETC(s);
    switch (c) {
      case '9':
	/* end of file (or something like that) */
	/* read the rest of the line */
	while (1) {
	    c= STREAM_GETC(s);
	    if (c == '\n' || c == '\r' || c == EOF) {
                nuke_cr(s);
		if (s19_debug) printf("s9 record: done\n");
                return 0;
            }
	}
      case '1':
	/* data record */
	{
	    Binrec *ret;
	    Int i, nbytes, addr;
	    
	    nbytes= s19_get_hex2(s)-3;
	    addr=   s19_get_hex4(s);
	    ret= binrec_create(addr, nbytes);

	    for (i= 0; i< nbytes; i++) {
		ret->data[i]= (char) s19_get_hex2(s);
	    }
	    
	    s19_get_hex2(s);
	    while (1) {
		c= STREAM_GETC(s);
		if (c == '\n' || c == '\r') {
                    nuke_cr(s);
                    break;
                }
		if (!isspace(c))
		  die(("Extra bytes at end of .s19 record"));
	    }

	    if (s19_debug) printf("got rec: addr:%04lX len:%ld\n", addr, nbytes);
	    return ret;
	}
      default:
	die(("Unrecognized .s19 record: S%c", (int)c));
	break;
    }
    return 0;
}
	    
