/**
 ** util.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include CONFIG

#include "util.h"
#include "stringlb.h"

void *util_unused;

#ifdef NEED_STRICMP
NOPROTO int stricmp(char *a, char *b)
{
	while (1) {
		Int c1= *a;
		Int c2= *b;
		if ('A' <= c1 && c1 <= 'Z') c1 += 32;
		if ('A' <= c2 && c2 <= 'Z') c2 += 32;
		if (c1 < c2) return -1;
		if (c1 > c2) return 1;
		if (!c1) return 0;
		a++; b++;
	}
}
#endif

#ifdef HAS_STRCASECMP
NOPROTO int stricmp(char *a, char *b)
{
    return strcasecmp(a, b);
}
#endif

#ifdef NEED_STRNICMP
NOPROTO int strnicmp(char *a, char *b, int n)
{
	while (n>0) {
		Int c1= *a;
		Int c2= *b;
		if ('A' <= c1 && c1 <= 'Z') c1 += 32;
		if ('A' <= c2 && c2 <= 'Z') c2 += 32;
		if (c1 < c2) return -1;
		if (c1 > c2) return 1;
		if (!c1) return 0;
		a++; b++; n--;
	}
	return 0;
}
#endif

#ifdef HAS_STRNCASECMP
NOPROTO int strnicmp(char *a, char *b, int n)
{
    return strncasecmp(a, b, n);
}
#endif

void *xmalloc(size_t size)
{
  void *ret= malloc(size);
  if (!ret) {
    printf("Out of memory (tried to allocate size %ld)\n", (Int)size);
    exit(1);
  }
  return ret;
}

void *xmalloc_clear(size_t size)
{
  void *ret= xmalloc(size);
  if (ret) memset(ret, 0, size);
  return ret;
}

void *xrealloc(void *old, size_t size)
{
  void *ret= realloc(old, size);
  if (!ret) {
    printf("Out of memory (tried to reallocate size %ld)\n", (Int)size);
    exit(1);
  }
  return ret;
}

void xfree(void *pointer)
{
  free(pointer);
}

void out_of_memory(void)
{
    die(("Out of memory\n"));
}

void debugf(char *format, ...)
{
    va_list l;
    va_start(l, format);
    vfprintf(stderr, format, l);
    va_end(l);
}


void errorf(char *format, ...)
{
    va_list l;
    va_start(l, format);
    vfprintf(stderr, format, l);
    va_end(l);
    die(("fatal error\n"));
}


Int intlog2(unsigned long num)
{
    Int ret= -1;
    while (num > 128) {
	num >>= 8;
	ret += 8;
    }
    while (num > 8) {
	num >>= 4;
	ret += 4;
    }
    while (num > 0) {
	num >>= 1;
	ret ++;
    }
    return ret;
}

#ifdef HACK_VSPRINTF
/*NOPROTO*/ Int vsprintf(char *dest, char *src, void * _NOPROTO)
{
    long *args= (long*) _NOPROTO;
    return sprintf(dest, src, args[0], args[1], args[2], args[3], args[4], args[5],
  		              args[6], args[7], args[8], args[9]);
}
#endif

#ifdef MALLOC_DEBUG

#undef malloc
#undef free
#undef realloc

Malloc_ptrs malloc_tab[MAX_MALLOCS];
Int mtablen;

Int dmalloc_debug;
Int debug_malloc_lognum=0;

void debug_malloc_print(FILE *f)
{
  Int i,index;
  Int logsum[debug_malloc_lognum+1];
  Int sum;

  for(i=0;i<debug_malloc_lognum+1;i++)
    logsum[i]=0;
  
  for(index=0;index<mtablen;index++)
    {
      if(malloc_tab[index].mptr!=NULL) /* found real slot */
	{
	  fprintf(f,"%d %d: %d bytes at %x in %s:%d\n",
		  malloc_tab[index].logtime,malloc_tab[index].time,
		  malloc_tab[index].size,malloc_tab[index].mptr,
		  malloc_tab[index].file,
		  malloc_tab[index].line,
		  malloc_tab[index].logtime);
	  logsum[malloc_tab[index].logtime]+=malloc_tab[index].size;
	}
    }

  sum=0;
  for(i=0;i<debug_malloc_lognum+1;i++)
    {
      fprintf(f,"%d Total: %d\n",i,logsum[i]);
      sum+=logsum[i];
    }
  debug_malloc_lognum++;
  fprintf(f,"%d Initial allocation %d\n",debug_malloc_lognum,sum);
  fflush(f);
}
		  
void *debug_malloc(size_t size,const char *file,Int line)
{
  void *ret;
  Int index;

  ret=malloc(size);
 
  for(index=0;index<mtablen;index++)
    {
      if(malloc_tab[index].mptr==NULL) /* found empty slot */
	{
	  __debug_malloc_insert(index,ret,size,file,line);
	  return(ret);
	}
    }
  /* no empty slot, add to end */
  if(mtablen>=MAX_MALLOCS)
    {
      printf("Error: Debug Malloc out of space in %s:%d\n",file,line);
      return(ret);
    }
  
  __debug_malloc_insert(mtablen,ret,size,file,line);
  mtablen++;
  return(ret);
}

void debug_free(void *ptr,const char *file,Int line)
{
  Int index;

  if(ptr==NULL)
    {
      printf("Error: Free called on NULL in %s:%d\n",file,line);
      return;
    }
  
  for(index=0;index<mtablen;index++)
    {
      if(malloc_tab[index].mptr==ptr)
	{
	  if(dmalloc_debug)
	    printf("%d Free: %d bytes at %x from %s:%d in %s:%d\n",
		   malloc_tab[index].time,
		   malloc_tab[index].size,malloc_tab[index].mptr,
		   malloc_tab[index].file,
		   malloc_tab[index].line,file,line);
	  free(ptr);
	  malloc_tab[index].mptr=NULL;
	  return;
	}
    }
  printf("Error: Freeing illegal address %x in %s:%d\n",ptr,file,line);
  return;
}

void *debug_realloc(void *ptr, size_t size, const char *file, Int line)
{
  Int index;
  void *ret;
  
  if(ptr==NULL)
    {
      printf("Error: Realloc called on NULL in %s:%d\n",file,line);
      return(NULL);
    }
  
  for(index=0;index<mtablen;index++)
    {
      if(malloc_tab[index].mptr==ptr)
	{
	  ret=realloc(ptr,size);
	  if(dmalloc_debug)
	    printf("%d Realloc: %d->%d bytes %x->%x in %s:%d\n",
		   malloc_tab[index].time,
		   malloc_tab[index].size,size,
		   malloc_tab[index].mptr,ret,
		   file,line);
	  malloc_tab[index].mptr=ret;
	  malloc_tab[index].size=size;
	  malloc_tab[index].file=file;
	  malloc_tab[index].line=line;
/*	  malloc_tab[index].logtime=debug_malloc_lognum;*/
	  return(ret);
	}
    }
  printf("Error: Reallocing illegal address %x in %s:%d\n",ptr,file,line);
  return(NULL);
}
  
void __debug_malloc_insert(Int index,void *ptr,size_t size,
			   const char *file,Int line)
{
  static Int insert_time=0;
  
  if(dmalloc_debug)
    printf("%d Malloc: %d bytes at %x in %s:%d\n",insert_time,
	   size,ptr,file,line);

  malloc_tab[index].mptr=ptr;
  malloc_tab[index].size=size;
  malloc_tab[index].file=file;
  malloc_tab[index].line=line;
  malloc_tab[index].time=insert_time++;
  malloc_tab[index].logtime=debug_malloc_lognum;
}
#endif
