/* qcam.c -- routines for accessing the Connectix QuickCam */

/* Version 0.3, January 2, 1996 */


/******************************************************************

Copyright (C) 1996 by Scott Laird

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL SCOTT LAIRD BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <stdio.h>
#include <stdlib.h>

#include <unistd.h>

#include "qcam.h"

void usage(void)
{
  fprintf(stderr,"Usage:\n");
  fprintf(stderr,"  qcam [options]\n");
  fprintf(stderr,"    Options:\n");
  fprintf(stderr,"      -x width   Set width\n");
  fprintf(stderr,"      -y height  Set height\n");
  fprintf(stderr,"      -p port    Set port\n");
  fprintf(stderr,"      -B bpp     Set bits per pixel\n");

  fprintf(stderr,"      -c val     Set contrast\n");
  fprintf(stderr,"      -w val     Set white balance\n");
  fprintf(stderr,"      -b val     Set brightness\n");
  fprintf(stderr,"      -V         Show version information\n");

}


void modegripe(struct qcam *q)
{
  fprintf(stderr,"Unsupported resolution/bit depth!\n");
  fprintf(stderr,"This program only supports 320x240, 160x120, and 80x60.\n");
  fprintf(stderr,"You specified %d x %d.  Try again.\n",q->width,q->height);
  exit(1);
}

int main(int argc, char **argv)
{
  scanbuf *scan;
  int arg;
  extern char *optarg;
  struct qcam *q;
  int width,height;

  if(geteuid()) {
    fprintf(stderr,"%s: Not installed SUID or run as root.  Exiting.\n",
	    argv[0]);
    exit(1);
  }

  q=qc_init();
  qc_getresolution(q,&width,&height);

  /* Read command line */

  while((arg=getopt(argc,argv,"hx:y:p:b:B:c:w:V"))>0) { 
    switch (arg) {
    case 'x':
      width=atoi(optarg);
      break;
    case 'y':
      height=atoi(optarg);
      break;
    case 'p':
      if (!getuid())
	q->port=strtol(optarg,NULL,0);
      break;
    case 'B':
      qc_setbitdepth(q,atoi(optarg));
      break;
    case 'b':
      qc_setbrightness(q,atoi(optarg));
      break;
    case 'c':
      qc_setcontrast(q,atoi(optarg));
      break;
    case 'w':
      qc_setwhitebal(q,atoi(optarg));
      break;
    case 'V':
      fprintf(stderr,"%s: Version 0.3\n",argv[0]);
      exit(0);
      break;
    case 'h':
      usage();
      exit(0);
      break;
    default:
      fprintf(stderr,"%s: Unknown option or error in option\n",argv[0]);
      usage();
      exit(1);
      break;
    }
  }

  if(qc_setresolution(q,width,height)) {
    fprintf(stderr,"Invalid resolution.  Exiting.\n");
    exit(1);
  }

  /* Attempt to get permission to access IO ports.  Must be root */

  if (qc_open(q)) {
    fprintf(stderr,"Cannot open QuickCam; exiting.\n");
    exit(1);
  }

  setuid(getuid());

  /* Program the QuickCam */

  qc_set(q);

  /* Scan one image */

  scan=qc_scan(q);

  /* Output the image to stdout */

  qc_writepgm(q,stdout,scan);
  free(scan);

  /* Release IO privileges */
  qc_close(q);

  return 0;
}

