/***************************************
* XFqcam
* by Paul Chinn <loomer@svpal.org>
*
* QuickCam code by Thomas Davis
* Additional QuickCam code Scott Laird <scott@laird.com>
* 
* PWC: some X/shared mem code was taken from 
* VGB "Virtual GameBoy" emulator by Marat Fayzullin and
* Elan Feingold.  It's mostly all gone now, but still
* an acknowledgment is due.
* History
* -------
* 1.02 - autofind camera, detect bimode, image stability, load flashes etc.
* 1.01 - fixed getstatus for unidirectional, Russ Nelson's shared image patches
* 1.00 - complete overhaul: total camera control, bidirectional support, multiple x depths, etc.
* 0.2  - initial forms based interface to set brightness,contrast,whitebal and depth
* <.2  - xqcam: no gui, just qcam output in an xwindow
******************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <unistd.h>
#include <asm/io.h>

#ifdef XSHMEM
/** MIT Shared Memory Extension for X ************************/
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
XShmSegmentInfo SHMInfo;
int use_shm = 1, xerror;
#endif

/** XForms include files */
#include <forms.h>
#include "controlpanel.h"


/** This entire program is driven with global variables...I know, I know...
* but it started as a hack, and what with the XForms call backs, and X
* itself sort of pushing the lazy programmer to use globals...well, sorry...
*******************************/

#define QC_BRIGHTNESS	0x0b
#define QC_CONTRAST	0x19
#define QC_WHITEBALANCE	0x1f
#define QC_XFERMODE	0x07
#define QC_XSIZE	0x13
#define QC_YSIZE	0x11
#define QC_UNK1		0x0d
#define QC_UNK2		0x0f
#define MAX_AUTO_ADJUST 15

int quickcam_port = 0;
int contrast = 65;
int brightness = 161;
int whitebalance = 133;
int bpp6 = 1;
int biMode = 0;
int canDoBi = 0;
int zoom = 0;
int xfermode;
void (*GetImage)(char *);
void (*RawToX)(char *, char *);

int xsize = 160;
int ysize = 120;
int unk1 = 0x01;
int unk2 = 0x07;



int status;
int debug = 0;


Screen *screen;
Display *disp;
Window root,win;
XGCValues values;
Colormap cmap;
int xDepth;
XImage *ximage;
char *ximagemem;
GC gc;
int xstarted;
int screen_num;
int quit=0;
int secz=0;
int monz=0;
int hgz=0;
double thresh=1.0;
int autoz=0;

FD_QuickCam *mainwin;
FD_AboutBox *aboutBox;
FD_HGram *hg;

char tbuf[115200];
char rawbuf[320*240];

int colortable[64];
int bpp4tobpp6[16]={0,63,59,55,51,47,43,39,35,32,28,24,20,14,9,5};

typedef struct stag{
	int w,h;
	char label[10];
	}SCREENMENU;

SCREENMENU choiceScreen[]={
	{320,240,"320x240"},
	{240,180,"240x180"},
	{180,160,"180x160"},
	{160,120,"160x120"},
	{80,60,"80x60"},
	{60,45,"60x45"},
	{40,30,"40x30"},
	{0,0,""}};

char *choiceQuality[]={"6 Bpp","4 Bpp",NULL};

/*VLINE is the index of the line that contains the version string so we can
* print itlater on*/

#define VLINE 2
char *aboutText[]={
"",
"@c@lXFQcam",
"@c@iversion 1.02 1/25/96",
"@c@_by Paul Chinn(loomer@1000klub.com)",
"",
"@cQuickCam routines originally by ",
"@cThomas Davis and Scott Laird (scott@laird.com)",
"@cwith additions and modifications by Paul Chinn",
"@cThis program is free for any use.",
"@c@iThanks also to members of the QuickCam reverse engineering project",
"@c@iCheck out www.crynwr.com/qcpc for more info",
"",
"@cThe latest XFQcam can be found at www.1000klub.com/~loomer",
"",
"@cSecuity, Graph, Monitor, and AutoBrightness ",
"@cModifications by Reza Naima (reza@ucsd.edu) 3/30/96",
NULL};

/* Function Prototypes*******************************************************/
void DisableButton(FL_OBJECT *);
void EnableButton(FL_OBJECT *);
void CalcXferMode(void);
int NewImage(int, int);
void ConstrainZoom(void);

/* QuickCam Functions ***************************************************/
static void reset_quickcam(int port)
{
	outb(0x20, port+2);
	outb(0x75, port);
	status = inb(port);
	outb(0x0b, port+2);
	usleep(250);
	outb(0x0e, port+2);
	usleep(250);
}

int qc_find(void)
{
	int port,ports[]={0x278,0x378,0x3bc};
	int i,reg,lastreg,count,p;
	
	for(p=0;p<3;p++)
 	{
		port=ports[p];
		if(ioperm(port,3,1))
		   	 return -1;
		count=0;
		lastreg=reg=inb(port+1)&0xf0;
		for(i=0;i<30;i++)
		{
			reg=inb(port+1)&0xf0;
			if(reg!=lastreg) count++;
			lastreg=reg;
			usleep(10000);
		}
		if(count > 3)
		   	 return port;
		else
		   	 ioperm(port,3,0);
	}
	return 0;
}
	
#if 0
static int waitfor_qc_uni(int port)
{
	unsigned char status1, status2;

	if (debug) 
		fprintf(stderr, "waiting for camera to say \"Go!\"\n");

	outb(0x06, port+2);

	do {
		status1 = inb(port+1);
	} while (!(status1 & 0x08));

	outb(0x0e, port+2);

	do {
		status2 = inb(port+1);
	} while (status2 & 0x08) ;

	if (debug) {
		fprintf(stderr, "status1 = 0x%02x ", status1);
		fprintf(stderr, "status2 = 0x%02x ", status2);
		fprintf(stderr, "status = 0x%02x\n",((status1 & 0xF0) | ( status2 >> 4 )));
	}

	return ((status1 & 0xF0 ) | ( status2 >> 4 ));

}
#endif

static int waitfor_qc_bi(int port)
{
	unsigned char status1 = 0, status2 = 1;

	if (debug) 
		fprintf(stderr, "waiting for camera to say \"Go!\"\n");
	outb(0x26, port+2);

	while ((status1 & 0x0f) != 0x08) {
		status1 = inb(port+1);
	}

	outb(0x2f, port+2);

	while ((status2 & 0x0f) != 0) {
		status2 = inb(port+1);
	}

	return ((status1 & 0xF0 ) | ( status2 >> 4 ));

}

static int getstatus(int port)
{
	unsigned char status1 , status2 ;
	
	outb(0x06, port+2);
	
	do{
		status1 = inb(port + 1);
	} while (!(status1 & 0x08));

	outb(0x0e, port+2);
	do{
		status2 = inb(port+1);
	} while ((status2 & 0x08));
/*	usleep(100000);*/
	return ((status1 & 0xF0 ) | ( status2 >> 4 ));
}

static int sendbyte( int port, int value)
{
	outb(value, port);
	outb(value, port);
	outb(value, port);
	return (getstatus(port));
}

int AverageBrightness(unsigned char *buf)
{
	int i;
	int avg=0;
	
	for(i=0; i<xsize*ysize; i++)
	  avg+=*buf++;
	avg/= xsize*ysize;
	return avg;
}

static int sendcmd( int cmd, int value)
{

/*	gettimeofday(&start,NULL);*/
	if (debug)
		fprintf(stderr, "cmd = 0x%x, value = 0x%x\n", cmd, value);
	if (sendbyte(quickcam_port, cmd) != cmd )
		return(1);

/*	usleep(25);*/

	if (sendbyte(quickcam_port, value) != value )
		return(1);
/*	gettimeofday(&stop,NULL);
	printf("sendcmd %ld-",((stop.tv_sec - start.tv_sec)*1000000)+(1000000 - start.tv_usec) + stop.tv_usec);
*/	
/*	usleep(25);*/

	return 0;
}

void set_brightness()
{
	sendcmd(QC_BRIGHTNESS, brightness);
	sendcmd(QC_BRIGHTNESS, 0x01);
	sendcmd(QC_BRIGHTNESS, 0x01);
	sendcmd(QC_BRIGHTNESS, brightness);
	sendcmd(QC_BRIGHTNESS, brightness);
	sendcmd(QC_BRIGHTNESS, brightness);
}


void set_size()
{
	if(bpp6)
	  sendcmd(QC_XSIZE, xsize/4);
	else
	  sendcmd(QC_XSIZE, xsize/2);
	
	sendcmd(QC_YSIZE, ysize);
}

void set_contrast()
{
	sendcmd(QC_CONTRAST,contrast);
}

 void set_whitebalance()
{
	sendcmd(QC_WHITEBALANCE,whitebalance);
}

void set_all()
{
	set_brightness();
	set_contrast();
	set_whitebalance();
	set_size();
	sendcmd(QC_UNK1, unk1);
	sendcmd(QC_UNK2, unk2);
}

void Get4bppImageUni(char *buffer)
{
	unsigned int b1,b2;
    unsigned int i;
	sendcmd(QC_XFERMODE, xfermode);
	for(i=0; i< ((xsize*ysize)/2); i++)
	{
	   	outb(0x06, quickcam_port+2);
		do {
			b1 = inb(quickcam_port+1);
		} while (!(b1 & 0x08));
		outb(0x0e, quickcam_port+2);
		do {
			b2 = inb(quickcam_port+1);
		} while (b2 & 0x08);
		
		*buffer++ = bpp4tobpp6[b1>>4];
		*buffer++ = bpp4tobpp6[b2>>4];
	}
		outb(0x0e, quickcam_port+2);
		do {
			b2 = inb(quickcam_port+1);
		} while (b2 & 0x08);
		outb(0x06, quickcam_port+2);
	do {
			b1 = inb(quickcam_port+1);
		} while ((b1 & 0x08));
	outb(0x0f, quickcam_port+2);
}

void Get6bppImageUni(char *buffer)
{
	unsigned char b1,b2,b3,*tbp=tbuf;
    unsigned int i;
	
	sendcmd(QC_XFERMODE, xfermode);
	for(i=0; i< ((xsize*ysize*3)/4); i++)
	{
		outb(0x06, quickcam_port+2);
		do {
			b1 = inb(quickcam_port+1);
		} while (!(b1 & 0x08));
		outb(0x0e, quickcam_port+2);
		do {
			b2 = inb(quickcam_port+1);
		} while (b2 & 0x08);
		
		*tbp++ = b1;
		*tbp++ = b2;
	}
		outb(0x0e, quickcam_port+2);
		do {
			b2 = inb(quickcam_port+1);
		} while (b2 & 0x08);

	outb(0x06, quickcam_port+2);
	do {
			b1 = inb(quickcam_port+1);
		} while ((b1 & 0x08));
	outb(0x0f, quickcam_port+2);
	tbp = tbuf;
	for(i=0; i<xsize*ysize; i+=2)
	  {
		  b1 = *tbp++;
		  b2 = *tbp++;
		  b3 = *tbp++;
		  *buffer++ = 63 - (((b1 & 0xF0) >> 2) | (b2 >> 6));
		  *buffer++ = 63 - ((b2 & 0x30) | (b3 >> 4));
		}
}

void Get6bppImageBi(char *buffer)
{
	int i,word1;
	sendcmd(QC_XFERMODE,xfermode);
	waitfor_qc_bi(quickcam_port);
	for(i=0; i< xsize*ysize; i+=4)
	{
		outb(0x26, quickcam_port+2);
		word1 = inw(quickcam_port);
		while( (word1 & 0x1) != 0x01)
			word1 = inw(quickcam_port);

		word1 = ((word1 & 0xff00) >> 3) | (word1 & 0x00ff);
		word1 = word1 << 1;
		word1 = ((word1 & 0x00ff) >> 2) | (word1 & 0xff00);
		*buffer++ = 63 - (word1 & 0x00ff);
		*buffer++ = 63 - ((word1 >> 8) & 0x00ff);
		
		outb(0x2f, quickcam_port+2);
		word1 = inw(quickcam_port);
		while ((word1 & 0x01) != 0x0) 
			word1 = inw(quickcam_port);

		word1 = ((word1 & 0xff00) >> 3) | (word1 & 0x00ff);
		word1 = word1 << 1;
		word1 = ((word1 & 0x00ff) >> 2) | (word1 & 0xff00);
		*buffer++ = 63 -(word1 & 0x00ff);
		*buffer++ = 63 - ((word1 >> 8) & 0x00ff);
		}
	outb(0x26, quickcam_port+2);
	word1 = inw(quickcam_port);
	while( (word1 & 0x1) != 0x01)
		word1 = inw(quickcam_port);
	outb(0x0f, quickcam_port+2);
	return;

}

 
/*signal handler so we can dispose of shared memory if
we get killed*/

void quitprogram(int foo)
{
  quit=1;
}

/*Callback routines to handle Xforms interaction*****************************/

void About(FL_OBJECT *obj, long val)
{
	fl_show_form(aboutBox->AboutBox, FL_PLACE_MOUSE, FL_TRANSIENT, "About XFQcam");
}

void AboutOk(FL_OBJECT *obj, long val)
{
     fl_hide_form(aboutBox->AboutBox);
}

void Slider(FL_OBJECT *obj, long val)
{
	switch(val){
	case 0:	/*brightness*/
	  brightness = fl_get_slider_value(obj);
	  set_brightness();
	  break;
	case 1: /*white balance*/
	  whitebalance = fl_get_slider_value(obj);
	  set_whitebalance();
	  break;
	case 2:
	  contrast = fl_get_slider_value(obj);
	  set_contrast();
	  break;
	}			
      }

/* handle interaction with BPP menu*/
void Quality(FL_OBJECT *obj, long val)
{
  if( fl_get_choice(obj) == 1)
    {bpp6=1;
     if(canDoBi)
       EnableButton(mainwin->BiMode);
   }
  else
    {
      bpp6=0;
      biMode=0;  /*disable 4bbp/bi for now- till we figure out encoding*/
      DisableButton(mainwin->BiMode);
    }
  set_size();	/*reset dimentions*/
  ConstrainZoom();
  CalcXferMode();
}

/*handle image size menu*/
void ScreenSize(FL_OBJECT *obj, long val)
{
  int choice;
  int nw,nh;
  choice = fl_get_choice(obj) - 1;
  nw = choiceScreen[choice].w;
  nh = choiceScreen[choice].h;
  if(NewImage(nw,nh))
    {
      xsize = nw;
      ysize = nh;
      set_size();
      fl_redraw_object(mainwin->thebox);
    }		
  ConstrainZoom();
  CalcXferMode();
}

void Zoom(FL_OBJECT *obj, long val)
{
zoom = val;
CalcXferMode();
}

void Histogram(FL_OBJECT *obj, long val)
{
}


void automode(FL_OBJECT *obj, long val)
{
  autoz=fl_get_button(obj);
 /* if (autoz)
    {
      hgz=1;
      fl_set_button(mainwin->Graph,1);
      fl_deactivate_object(mainwin->Graph);
      fl_set_object_lcol(mainwin->Graph, FL_MCOL);
      Graph(mainwin->Graph,0);
    }
  if (!autoz)
    {
      fl_activate_object(mainwin->Graph);
      fl_set_object_lcol(mainwin->Graph, FL_BLACK);
      Graph(mainwin->Graph,0);
    }     */ 
}


/*bimode button handler*/

void BiMode(FL_OBJECT *obj, long val)
{
biMode = !biMode;
CalcXferMode();
}

void Graph(FL_OBJECT *obj, long val)
{
  hgz=fl_get_button(obj);
  if (hgz) 
    {
      fl_show_form(hg->HGram, FL_PLACE_FREE, FL_TRANSIENT, "HistoGraph");
    }
  if (!hgz)  
    {
      fl_hide_form(hg->HGram);
    }
}
void st_cb(FL_OBJECT *obj, long val)
{
  thresh=fl_get_counter_value(obj);
}




void Security(FL_OBJECT *obj, long val)
{
  secz=fl_get_button(obj);
  if (secz) 
    {
      monz=1;
      fl_set_button(mainwin->Monitor,1);
      fl_deactivate_object(mainwin->Monitor);
      fl_set_object_lcol(mainwin->Monitor, FL_MCOL);

    }
  if (!secz) 
    {
      fl_activate_object(mainwin->Monitor);
      fl_set_object_lcol(mainwin->Monitor, FL_BLACK);
    }
}
void Monitor(FL_OBJECT *obj, long val)
{
  monz=fl_get_button(obj);
}

void TakePic(FL_OBJECT *obj, long val)
{
	const char *fname="snapshot.pgm";
	int i;
	char *buf;
	FILE *f;

	buf = malloc(xsize * ysize);
	GetImage(buf);
	
	fname = fl_show_fselector("Choose PGM file:",NULL,"*.pgm",fname);
	if(fname)
	{
		f = fopen(fname,"w");
		if(!f)
		{
			perror("Can't create snapshot:");
		}
		else
		{
  			fprintf(f,"P5\n");
  			fprintf(f,"%d %d\n",xsize,ysize);
  			fprintf(f,"63\n");

  			for(i=0;i<xsize * ysize;i++) 
    			fputc(buf[i],f);
			fclose(f);
			fname = fl_get_filename(); /*save filename for next time*/
		}
	}
	free(buf);
}





/*Quit button handler*/
void Quit(FL_OBJECT *obj, long val)
{
	quit = 1;
}

void RawTo8BitX(char *raw, char *xbuf)
{
	int i;
	
	for(i=0; i<xsize*ysize; i++)
	{
		*xbuf++ = colortable[(int)*raw++];
	}
}
void RawTo16BitX(char *raw, short *xbuf)
{
	int i;
	
	for(i=0; i<xsize*ysize; i++)
	{
		*xbuf++ = colortable[(int)*raw++];
	}
}
void RawTo32BitX(char *raw, int *xbuf)
{
	int i;
	
	for(i=0; i<xsize*ysize; i++)
	{
		*xbuf++ = colortable[(int)*raw++];
	}
}

/** Initialize xwindows, and prepare a shared memory buffer for
 the image.  Returns pointer to shared memory buffer. */
 
char *GetSharedMem(void)
{
#ifdef XSHMEM
  int majver, minver;
  Bool sharedpix;

  if (use_shm && XShmQueryVersion(disp, &majver, &minver, &sharedpix)) {

    SHMInfo.shmid=shmget(IPC_PRIVATE, 320* 240*(xDepth / 8), IPC_CREAT|0777);

    if(SHMInfo.shmid < 0) {
      perror("shmget failed:");
      return (NULL);
    }
 
    SHMInfo.shmaddr = shmat(SHMInfo.shmid, 0, 0);
    XShmAttach(disp, &SHMInfo);
    signal(SIGHUP, quitprogram); 
    signal(SIGINT, quitprogram);
    signal(SIGQUIT, quitprogram); 
    signal(SIGTERM, quitprogram);
    xstarted=1;
    return(SHMInfo.shmaddr);
  } else
	{
	    use_shm = 0; /*ifShmQuery fails above we no longer try and use shared memeory*/
	}	
#endif
	return malloc(320* 240*(xDepth / 8));
}

/*clean up shared memory*/
void ExitXWindows(void)
{
#ifdef XSHMEM
  if(xstarted) {
    XShmDetach(disp, &SHMInfo);
    if(SHMInfo.shmaddr)
      shmdt(SHMInfo.shmaddr);
    if(SHMInfo.shmid > 0)
      shmctl(SHMInfo.shmid, IPC_RMID, 0);
  }
#endif
}


void xqc_createpalette(Colormap cmap)
{
  int i;
	XColor col;

  
  for(i=0; i<64; i++) {
    col.red =col.green = col.blue = i * 1024;
    XAllocColor(disp, cmap, &col);
    colortable[i] = col.pixel;
  }
return;
}

/*create a new image structure and destroy the previous one*/
int NewImage(int w, int h)
{
	if(!ximage || w != ximage->width || h != ximage->height)
	{
		if(ximage) XDestroyImage(ximage);	
#ifdef XSHMEM
		if (use_shm) {
			ximage = XShmCreateImage(disp, DefaultVisual(disp, screen_num),
						 xDepth, ZPixmap, NULL, &SHMInfo, w, h);
			ximage->data = SHMInfo.shmaddr;
		} else
#endif
		ximage = XCreateImage(disp,
			  DefaultVisual(disp,screen_num), xDepth,
			  ZPixmap, 0, ximagemem, w, h,
			  8, 0);
		return 1;
	}
return 0;
}

/*compute the transfermode to use based on bpp, bimode and zoom*/
void CalcXferMode()
{
/*	printf("%d %d zoom=%d\n",xsize,ysize,zoom);*/
	if(xsize > 160 || ysize > 120)
	{
		xfermode = 0;
	}
	else
	if(xsize > 80 || ysize > 60)
	{
		if(zoom)
		  xfermode = 0;
		else
		  xfermode = 4;
	}
	else
	{
		if(zoom == 2)
		  xfermode = 0;
		else
		if(zoom == 1)
		  xfermode = 4;
		else
		  xfermode = 8;
	 }
	if(bpp6)
	  xfermode += 2;
	xfermode += biMode;
	if(xfermode == 2 || xfermode ==3 || xfermode == 6 || xfermode == 7 ||
	   xfermode == 10 || xfermode == 11)
	  if(xfermode & 0x1)
	  	GetImage = Get6bppImageBi;
		else
	  	GetImage = Get6bppImageUni;
	else
	  GetImage = Get4bppImageUni;
	
	/*printf("xfermode = %d\n",xfermode);*/
}
	

void DisableButton(FL_OBJECT *obj)
{
	fl_set_button(obj, 0);
	fl_deactivate_object(obj);
	fl_set_object_lcol(obj, FL_MCOL);
}

void EnableButton(FL_OBJECT *obj)
{
	fl_activate_object(obj);
	fl_set_object_lcol(obj, FL_BLACK);
}

/*make sure only zooms that are viable for given image size
* are available*/
void ConstrainZoom()
{
	if(xsize >160 || ysize > 120) /* no zoom available...*/
	{
		zoom = 0;
		DisableButton(mainwin->zoom2);
		DisableButton(mainwin->zoom1);
		fl_set_button(mainwin->zoom0,1);
	}
	else
	if(xsize > 80 || ysize > 60) /*1.5 zoom available)*/
	{
		DisableButton(mainwin->zoom2);
		EnableButton(mainwin->zoom1);
		if(zoom == 2)
		{	
			fl_set_button(mainwin->zoom1,1);
			zoom = 1;
		}
	}
	else
	{
		EnableButton(mainwin->zoom2);
		EnableButton(mainwin->zoom1);
	}
}

void usage(void)
{
  fprintf(stderr,"Usage:\n");
  fprintf(stderr,"  xfqcam [options]\n");
  fprintf(stderr,"    Options:\n");
  fprintf(stderr,"      -x width   Set width\n");
  fprintf(stderr,"      -y height  Set height\n");
  fprintf(stderr,"      -p port    Set port\n");
  fprintf(stderr,"      -B bpp     Set bits per pixel\n");

  fprintf(stderr,"      -c val     Set contrast\n");
  fprintf(stderr,"      -w val     Set white balance\n");
  fprintf(stderr,"      -b val     Set brightness\n");
  fprintf(stderr,"      -z val     Set Zoom(0,1,2)\n");
  fprintf(stderr,"      -V         Show version information\n");
  fprintf(stderr,"      -o         Override port bi-directional test\n");  
#ifdef XSHMEM
  fprintf(stderr,"      -S         Don't use shared memory\n");
#endif
}


void takepic(int picnum,char *buf)
{
        char fname[256];
        int i;
        FILE *f;

	sprintf(fname,"Image%d",picnum);


	
	f = fopen(fname,"w");
	if(!f)
	  {
	    perror("Can't create snapshot:");
	  }
	else
	  {
	    fprintf(f,"P5\n");
	    fprintf(f,"%d %d\n",xsize,ysize);
	    fprintf(f,"63\n");
	    
	    for(i=0;i<xsize * ysize;i++) 
	      fputc(buf[i],f);
	    fclose(f);
	  }
}

 
int main(int argc, char **argv)
{
  char fpstext[80] = "";
  char changetext[80]= "";
  int arg,index;
  extern char *optarg;
  int fakeargc=1;
  int biOverRide = 0;	
  int verbose=0;
  int privatecmap=0;
  Colormap cmap;
  struct timeval tv1, tv2;
  double framerate=0,fr;
  int first=0;
  float chg=0;
  char *bufz;
  int iz;
  char *old;
  int picnum = 0;
  static float histz[256];
  static float xz[256];
  float mean;
  double tempz=0;
  int tempint=0;

  /* Read command line */
	while((arg=getopt(argc,argv,"SohCvx:y:f:z:p:b:B:c:w:V"))>0) { 
    switch (arg) {
    case 'x':
      xsize=atoi(optarg);
      break;
    case 'y':
      ysize=atoi(optarg);
      break;
	 case 'o':
		biOverRide = 1;
	break;
	 case 'z':
	 	zoom=atoi(optarg);
	 break;
	 case 'p':
      if (!getuid())
		quickcam_port=strtol(optarg,NULL,0);
      break;
    case 'B':
      bpp6=(atoi(optarg)==6);
      break;
    case 'b':
      brightness=atoi(optarg);
      break;
    case 'c':
      contrast=atoi(optarg);
      break;
    case 'w':
      whitebalance=atoi(optarg);
      break;
    case 'V':
      fprintf(stderr,"%s:%s\n",argv[0],&aboutText[VLINE][4]);
      exit(0);
      break;
    case 'h':
      usage();
      exit(0);
      break;
	 case 'f':
		xfermode=atoi(optarg);
		break;
    case 'C':
      privatecmap=1;
      break;
#ifdef XSHMEM
	 case 'S':
      use_shm=0;
      break;
#endif
	 case 'v':
      verbose=1;
      break;
    default:
      fprintf(stderr,"%s: Unknown option or error in option\n",argv[0]);
      usage();
      exit(1);
      break;
    }
  }
 if(geteuid()) {
    fprintf(stderr,"%s: Must be installed SUID or run as root.  Exiting.\n",
	    argv[0]);
    exit(1);
  }
	
	if(quickcam_port)	/*override autodetect*/
	{
		if(quickcam_port != 0x278 && quickcam_port != 0x378 && quickcam_port != 0x3bc)
		  {
			  fprintf(stderr,"Invalid port address\n");
			  exit(1);
		}
		ioperm(quickcam_port,3,1);
	}
	else
	  {
		  quickcam_port=qc_find();
		  if(quickcam_port < 0)
		  	{
				fprintf(stderr,"Can't get IO perms\n");
				exit(1);
			}
		  else
		  	if(!quickcam_port)
		  	{
				fprintf(stderr,"Can't find a camera\n");
				exit(1);
			}
		}
	reset_quickcam(quickcam_port);
/*	printf("camera at %x status=%x\n",quickcam_port,status);*/
	if(status == 0x75)
	  canDoBi = 0;
	else
	  {canDoBi = 1;
		biMode=1;
	}
	if(biOverRide)
	  canDoBi = 1;
	/*intialize xforms and get the display */
  disp = fl_initialize(argv[0], "XFqcam", 0,0, &fakeargc, argv);

 
	if(zoom > 2 || zoom < 0)
	  {
		  fprintf(stderr,"Zoom values are: 0(no zoom), 1(1.5x), 2(2x)\n");
		  exit(1);
		  }

	/*fprintf(stderr,"Scanning from QuickCam at 0x%x at %dx%d, xfermode=%d\n",
	  quickcam_port,xsize,ysize,xfermode);
	*/
  /*open the xform control panel*/
  mainwin = create_form_QuickCam();
  aboutBox = create_form_AboutBox();
  hg = create_form_HGram();

  /*intialize various control setting*/
  fl_set_slider_bounds(mainwin->Brightness, 0, 255);
  fl_set_slider_step(mainwin->Brightness, 1);
  fl_set_slider_precision(mainwin->Brightness, 0);
  fl_set_slider_bounds(mainwin->Contrast, 0, 255);
  fl_set_slider_step(mainwin->Contrast, 1);
  fl_set_slider_precision(mainwin->Contrast, 0);
  fl_set_slider_bounds(mainwin->WhiteBal, 0, 255);
  fl_set_slider_step(mainwin->WhiteBal, 1);
  fl_set_slider_precision(mainwin->WhiteBal, 0);
  mainwin->fps->label = fpstext;
  mainwin->rez->label = changetext;
  fl_set_counter_value(mainwin->st,1.0);
  fl_set_counter_bounds(mainwin->st,0.0,20.0);
  
  for(index=0; choiceScreen[index].w; index++)
    fl_addto_choice(mainwin->ScreenSize, choiceScreen[index].label);

  for(index=0; choiceQuality[index]; index++)
    fl_addto_choice(mainwin->Quality, choiceQuality[index]);

  for(index=0; aboutText[index]; index++)
    fl_add_browser_line(aboutBox->AboutText, aboutText[index]);

  /*set the sliders to qcam values*/
  fl_set_slider_value(mainwin->Brightness, brightness);
  fl_set_slider_value(mainwin->Contrast, contrast);
  fl_set_slider_value(mainwin->WhiteBal, whitebalance);

  /*set bpp menu*/
  if(bpp6)
	fl_set_choice(mainwin->Quality, 1);
  else
    fl_set_choice(mainwin->Quality, 2);
	if(zoom == 2)
	  fl_set_button(mainwin->zoom2, 1);
	else
	if (zoom == 1)
	  fl_set_button(mainwin->zoom1,1);
	else
	  fl_set_button(mainwin->zoom0,1);
	
	/*disable bi mode if necessary*/
	if(!canDoBi)
	  DisableButton(mainwin->BiMode);
	else
	  fl_set_button(mainwin->BiMode,biMode);
	
	ConstrainZoom();
	CalcXferMode();
	
  /*show the form*/
  fl_show_form(mainwin->QuickCam, FL_PLACE_SIZE, FL_FULLBORDER, "Control Panel");

/*initialize the Xwindows stuff*/
	xDepth = fl_state[fl_get_vclass()].depth;
	if(xDepth == 8)
	  RawToX = RawTo8BitX;
	else
	if(xDepth == 16)
	  RawToX = RawTo16BitX;
	else
	if(xDepth == 32)
	  RawToX = RawTo32BitX;
	else
	  {
		  fprintf(stderr,"XFqcam only works in 8,16 & 32 bit X depths\n");
		  exit(1);
	}
  if ((ximagemem=GetSharedMem())==NULL) exit(1);

	screen_num=DefaultScreen(disp);
   gc = XCreateGC(disp, FL_ObjWin(mainwin->thebox),0, &values);

  if (privatecmap) {
    cmap = XCreateColormap(disp, win, DefaultVisual(disp, screen_num), 
			   AllocNone);
    XSetWindowColormap(disp, win, cmap);
  } else {
    cmap=DefaultColormap(disp, screen_num);
  }
  xqc_createpalette(cmap);
 /*create an image*/ 
	ximage = NULL;
	NewImage(xsize, ysize);
	
  /* Attempt to get permission to access IO ports.  Must be root */

  setuid(getuid());

	reset_quickcam(quickcam_port);
	set_all();
       
  /* Scan one image */
  if(verbose) {
    gettimeofday(&tv1,NULL);
  }



  while(!quit) {
    fl_check_forms(); /* Do checks Here */

    bufz = malloc(xsize * ysize); /*and allocate up some space*/    
    GetImage(rawbuf);
    for(iz=0;iz<xsize * ysize;iz++) bufz[iz]=rawbuf[iz];

      
    

    RawToX(rawbuf, ximagemem);
#ifdef XSHMEM
    if (use_shm) XShmPutImage(disp, FL_ObjWin(mainwin->thebox), gc, ximage, 0,0,
	       mainwin->thebox->x + 4 + ((320 - xsize)/2), 
	       mainwin->thebox->y + 4 + ((240 - ysize)/2),ximage->width, ximage->height, False);
    else
#endif
      XPutImage(disp, FL_ObjWin(mainwin->thebox), gc, ximage, 0,0,
		mainwin->thebox->x + 4 + ((320 - xsize)/2), 
		mainwin->thebox->y + 4 + ((240 - ysize)/2),ximage->width, ximage->height);
    XFlush(disp);
    
    
    
    /* Calculate frame rate */
    gettimeofday(&tv2,NULL);
    
    /* The frame rate is calculated using the TCP RTT algorithm */
    
    fr=(1.0/(tv2.tv_sec-tv1.tv_sec+(tv2.tv_usec-tv1.tv_usec)/1000000.0));
    if(framerate!=0)
      framerate=0.9*framerate+0.1*fr;
    else
      framerate=fr;
    
    sprintf(fpstext,"Rate: %.2f fps",framerate);
    fl_redraw_object(mainwin->fps);
    tv1.tv_sec=tv2.tv_sec;
    tv1.tv_usec=tv2.tv_usec;
    
    
    /* Ok, Caclulate change */
    

    if (first &&monz ) 
      {
	for(iz=0;iz<xsize * ysize;iz++) 
	  {
	    chg += ((bufz[iz]-old[iz])*(bufz[iz]-old[iz]));
	  }
	chg /= xsize * ysize;
	sprintf(changetext,"Change: %.2f",chg);
	fl_redraw_object(mainwin->rez);
	
	
      }
    if (first) free(old);
    old=bufz;
    first=1;
    
    if (hgz) {                                      
      for(iz=0;iz<65;iz++) 
	{
	  histz[iz]=0;
	  xz[iz]=iz;
	}
      for(iz=0;iz<xsize * ysize;iz++) histz[old[iz]]++;       
      fl_set_xyplot_data(hg->Histogram,xz,histz,65,"Histograph",  
 			 "Intensity","Pixles");  
    }


    if (autoz) /*Auto Brightness Stuff*/
      {
	if (!hgz) {
	  for(iz=0;iz<65;iz++) 
	    {
	      histz[iz]=0;
	      xz[iz]=iz;
	    }
	  for(iz=0;iz<xsize * ysize;iz++) histz[old[iz]]++;       
	}
	
	mean=0;
	for(iz=0;iz<65;iz++) mean+=histz[iz]*iz;
	mean /=xsize*ysize;
	if (mean>33) tempint=-MAX_AUTO_ADJUST*((mean-33)/31);
	if (mean<31) tempint=MAX_AUTO_ADJUST*(1-(mean/31));
	if (mean>33 || mean<31) 
	  {
	    tempz=fl_get_slider_value(mainwin->Brightness);
	    tempz+=tempint;
	    if (tempz>=255) tempz=254;  
	    if (tempz<=0) tempz=1;
	    fl_set_slider_value(mainwin->Brightness,tempz);
	    Slider(mainwin->Brightness,0);
	  }
      }

    if (chg>thresh && monz && secz) /*Security Stuff*/
      {
	picnum++;
	takepic(picnum,old);
      }
    
    chg=0;
    
  }

  reset_quickcam(quickcam_port);  
  ExitXWindows();
  return 0;
  
}

