/* qcam-lib.c -- Library for programming with the Connectix QuickCam.
 * See the included documentation for usage instructions and details
 * of the protocol involved. */


/* Version 0.3, January 6, 1996 */


/******************************************************************

Copyright (C) 1996 by Scott Laird

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL SCOTT LAIRD BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <unistd.h>

#include "qcam.h"
#include "qcam-os.h"

/* Prototypes for static functions.  Externally visible functions
 * should be prototyped in qcam.h */

static int qc_waithand(struct qcam *q, int val);
static void qc_command(struct qcam *q, int command);
static int qc_initfile(struct qcam *q);
static int qc_setscanmode(struct qcam *q);
static unsigned char qc_readbyte(struct qcam *q);


/* The next several functions are used for controlling the qcam
 * structure.  They aren't used inside this library, but they should
 * provide a clean interface for external programs.*/

/* Gets/sets the brightness. */

int qc_getbrightness(struct qcam *q)
{
  return q->brightness;
}

int qc_setbrightness(struct qcam *q, int val)
{
  q->brightness=val;
  return 0;
}


/* Gets/sets the contrast */

int qc_getcontrast(struct qcam *q)
{
  return q->contrast;
}

int qc_setcontrast(struct qcam *q, int val)
{
  q->contrast=val;
  return 0;
}


/* Gets/sets the white balance */

int qc_getwhitebal(struct qcam *q)
{
  return q->whitebal;
}

int qc_setwhitebal(struct qcam *q, int val)
{
  q->whitebal=val;
  return 0;
}


/* Gets/sets the resolution */

void qc_getresolution(struct qcam *q, int *x, int *y)
{
  *x=q->width;
  *y=q->height;
}

int qc_setresolution(struct qcam *q, int x, int y)
{
  q->width=x;
  q->height=y;
  return qc_setscanmode(q);
}

int qc_getheight(struct qcam *q)
{
  return q->height;
}

int qc_setheight(struct qcam *q, int y)
{
  q->height=y;
  return 0;
}

int qc_getwidth(struct qcam *q)
{
  return q->width;
}

int qc_setwidth(struct qcam *q, int x)
{
  q->width=x;
  return 0;
}

/* Gets/sets the bit depth */

int qc_getbitdepth(struct qcam *q)
{
  return q->bpp;
}

int qc_setbitdepth(struct qcam *q, int val)
{
  q->bpp=val;
  return qc_setscanmode(q);
}


/* Read from the QuickCam init file.  By default this is
 * /usr/local/etc/qcam.conf.  This can be changed in the Makefile.
 * This routine sets the 'qcam' structure to the defaults in the
 * config file. */

static int qc_initfile(struct qcam *q)
{
  FILE *f;
  char buf[256],*s;

  f=fopen(CONFIG_FILE,"r");

  if (f==NULL&&errno!=ENOENT) {
    perror("Cannot open configuration file!");
    return 1;
  }

  /* I'm not real happy with this code, but it seems to work.  The
   * main problem is that it'll accept things strings containing any
   * of these words as an attempt to set these options.  For example,
   * it'll take the word 'report' on a line by itself as an attempt to
   * set the port to 0.  It goes without saying that it won't detect
   * errors in the config file. */

  while (fgets(buf,255,f)) {
    if (buf[0]=='#') ; /* comment */
    else if ((s=strstr(buf,"port"))) q->port=strtol(s+4,NULL,0);
    else if ((s=strstr(buf,"height"))) q->height=atoi(s+6);
    else if ((s=strstr(buf,"width"))) q->width=atoi(s+5);
    else if ((s=strstr(buf,"bpp"))) q->bpp=atoi(s+3);
    else if ((s=strstr(buf,"brightness"))) q->brightness=atoi(s+10);
    else if ((s=strstr(buf,"whitebal"))) q->whitebal=atoi(s+8);
    else if ((s=strstr(buf,"contrast"))) q->contrast=atoi(s+8);
  }

  fclose(f);
  return 0;
}

/* Initialize the QuickCam driver control structure.  This is where
 * defaults are set for people who don't have a config file.*/

struct qcam *qc_init(void)
{
  struct qcam *q;

  q=malloc(sizeof(struct qcam));

  q->port=0; /* Port 0 == Autoprobe */
  q->port_mode=QC_UNIDIR;
  q->width=160;
  q->height=120;
  q->bpp=4;
  q->contrast=200;
  q->brightness=135;
  q->whitebal=128;

  qc_initfile(q);
  return q;
}


/* qc_open enables access to the port specified in q->port.  It takes
 * care of locking and enabling I/O port access by calling the
 * appropriate routines.
 *
 * Returns 0 for success, 1 for opening error, 2 for locking error,
 * and 3 for qcam not found */

int qc_open(struct qcam *q)
{
  if(q->port==0)
    if(qc_probe(q)) {
      fprintf(stderr,"Qcam not found\n");
      return 3;
    }

  if(qc_lock(q)) {
    fprintf(stderr,"Cannot lock qcam.\n");
    return 2;
  }

  if(enable_ports(q)) {
    fprintf(stderr,"Cannot open QuickCam -- permission denied.");
    return 1;
  } else {
    return 0;
  }
}


/* qc_close closes and unlocks the driver.  You *need* to call this,
 * or lockfiles will be left behind and everything will be screwed. */

int qc_close(struct qcam *q)
{
  qc_unlock(q);

  disable_ports(q);
  return 0;
}


/* qc_command is probably a bit of a misnomer -- it's used to send
 * bytes *to* the camera.  Generally, these bytes are either commands
 * or arguments to commands, so the name fits, but it still bugs me a
 * bit.  See the documentation for a list of commands. */

static void qc_command(struct qcam *q, int command)
{
  int status;
  write_lpdata(q,command);
  /* These pauses seem to be needed for some reason. */
  qc_wait(1);
  write_lpdata(q,command);
  qc_wait(1);
  write_lpdata(q,command);
  write_lpcontrol(q,6);

  qc_waithand(q,1);

  status=read_lpstatus(q);
  write_lpcontrol(q,0xe);

  qc_waithand(q,0);

  status=read_lpstatus(q);
}


/* qc_waithand busy-waits for a handshake signal from the QuickCam.
 * Almost all communication with the camera requires handshaking. */

inline static int qc_waithand(struct qcam *q, int val)
{
  if(val) 
    while(!(read_lpstatus(q)&8));
  else
    while((read_lpstatus(q)&8));

  return 0;
}


/* Try to detect a QuickCam.  It appears to flash the upper 4 bits of
   the status register at 5-10 Hz.  This is only used in the autoprobe
   code.  Be aware that this isn't the way Connectix detects the
   camera (they send a reset and try to handshake), but this should be
   almost completely safe, while their method screws up my printer if
   I plug it in before the camera. */
#if 0
int qc_detect(struct qcam *q)
{
  int reg,lastreg;
  int count=0;
  int i;

  lastreg=reg=read_lpstatus(q)&0xf0;

  for(i=0;i<30;i++) {
    reg=read_lpstatus(q)&0xf0;
    if(reg!=lastreg) count++;
    lastreg=reg;
    usleep(10000);
  }

  /* Be liberal in what you accept...  */

  if(count>3&&count<15)
    return 1; /* found */
  else 
    return 0; /* not found */
}
#endif

int qc_detect(struct qcam *q)
{
  int reg,lastreg;
  int count=0;
  int i;

  lastreg=reg=read_lpstatus(q)&0xf0;

  for(i=0;i<30;i++) {
    reg=read_lpstatus(q)&0xf0;
    if(reg!=lastreg) count++;
    lastreg=reg;
    usleep(10000);
  }

  return 1;
}


/* Reset the QuickCam.  This uses the same sequence the Windows
 * QuickPic program uses.  Someone with a bi-directional port should
 * check that bi-directional mode is detected right, and then
 * implement bi-directional mode in qc_readbyte(). */

void qc_reset(struct qcam *q)
{
  int bidir;

  write_lpcontrol(q,0x20);
  write_lpdata(q,0x75);

  bidir=read_lpdata(q)!=0x75;

  if(bidir) {
    fprintf(stderr,"Bidirectional port found.  Hack qc_readbyte to use.\n");
    q->port_mode=QC_BIDIR;
  }

  /* usleep(250);*/
  write_lpcontrol(q,0xb);
  /*  usleep(250); */
  write_lpcontrol(q,0xe);
}


/* Decide which scan mode to use.  There's no real requirement that
 * the scanmode match the resolution in q->height and q-> width -- the
 * camera takes the picture at the resolution specified in the
 * "scanmode" and then returns the image at the resolution specified
 * with the resolution commands.  If the scan is bigger than the
 * requested resolution, the upper-left hand corner of the scan is
 * returned.  If the scan is smaller, then the rest of the image
 * returned contains garbage. */

static int qc_setscanmode(struct qcam *q)
{
  switch (q->width) {
  case 80:  if(q->height==60)  q->mode=8; break;
  case 160: if(q->height==120) q->mode=4; break;
  case 320: if(q->height==240) q->mode=0; break;
  }

  if (q->mode<0) { 
    fprintf(stderr,"Error: unsupported resolution (%d x %d)!\n",
    q->width,q->height); 
    return 1;
  }

  switch (q->bpp) {
  case 4: break;
  case 6: q->mode+=2; break;
  default:
    fprintf(stderr,"Error: Unsupported bit depth\n");
    return 1;
  }
  return 0;
}


/* Reset the QuickCam and program for brightness, contrast,
 * white-balance, and resolution. */

void qc_set(struct qcam *q)
{
  qc_setscanmode(q);
  qc_reset(q);

  /* Set the brightness.  Yes, this is repetitive, but it works.
   * Shorter versions seem to fail subtly.  Feel free to try :-). */
  qc_command(q,0xb);
  qc_command(q,q->brightness);
  qc_command(q,0xb);
  qc_command(q,1);
  qc_command(q,0xb);
  qc_command(q,1);
  qc_command(q,0xb);
  qc_command(q,q->brightness);
  qc_command(q,0xb);
  qc_command(q,q->brightness);
  qc_command(q,0xb);
  qc_command(q,q->brightness);

  qc_command(q,0x11);
  qc_command(q,q->height);
  qc_command(q,0x13);
  if(q->bpp==4)
    qc_command(q,q->width/2);
  else
    qc_command(q,q->width/4);

  qc_command(q,0x11);
  qc_command(q,q->height);
  qc_command(q,0x13);
  if(q->bpp==4)
    qc_command(q,q->width/2);
  else
    qc_command(q,q->width/4);

  /* I still don't know what these do! */
  qc_command(q,0xd);
  qc_command(q,0x1);
  qc_command(q,0xf);
  qc_command(q,0x7);

  qc_command(q,0x19);
  qc_command(q,q->contrast);
  qc_command(q,0x1f);
  qc_command(q,q->whitebal);

}


/* Read a byte from the QC.  This is only used from qc_scan.  Notice
   that there are places for three modes of operation: unidirectional,
   bidirectional, and serial.  Right now only unidirectional is
   implemented.  Bidirectional uses the unidirectional code.  Serial
   code is just a case: stub, just in case someone wants to plug a Mac
   QuickCam into a 230.4k serial port and see what happens.  Wishful
   thinking. */

static unsigned char qc_readbyte(struct qcam *q)
{
  int ret;
  int hi,lo;

  switch (q->port_mode) {
  case QC_UNIDIR:  /* Unidirectional Port */
  case QC_BIDIR:  /* Bi-directional Port same as uni-dir for now. */
    write_lpcontrol(q,6);
    qc_waithand(q,1);
    hi=read_lpstatus(q)&0xf0;
    write_lpcontrol(q,0xe);
    qc_waithand(q,0);
    lo=(read_lpstatus(q)&0xf0)>>4;
    ret=hi|lo;
    break;
  case QC_SERIAL: /* Serial Interface.  Just in case.*/
  default:
    fprintf(stderr,"Mode %d not supported\n",q->port_mode);
    ret=0;
    break;
  }
  return ret;
}

/* Read a scan from the QC.  This takes the qcam structure and
 * requests a scan from the camera.  It sends the correct instructions
 * to the camera and then reads back the correct number of bytes.  In
 * previous versions of this routine the return structure contained
 * the raw output from the camera, and there was a 'qc_convertscan'
 * function that converted that to a useful format.  In version 0.3 I
 * rolled qc_convertscan into qc_scan and now I only return the
 * converted scan.  The format is just an one-dimensional array of
 * characters, one for each pixel, with 0=black up to n=white, where
 * n=2^(bit depth)-1.  Ask me for more details if you don't understand
 * this. */

scanbuf *qc_scan(struct qcam *q)
{
  char *ret;
  int i=0;
  unsigned int a=0,b=0;
  int len;

  len=q->height*q->width;

  ret=malloc(len);

  qc_command(q,0x7);
  qc_command(q,q->mode);

  while(i<len) {
    switch (q->bpp) {
    case 4:
      /* In 4bpp mode the high nibble contains the first pixel and the
       * low nibble contains the second pixel.*/
      a=qc_readbyte(q);
      ret[i++]=(16-(a>>4))&0xf;
      ret[i++]=(16-a)&0xf;
      break;
    case 6:
      /* In 6bpp mode the first pixel is in the high 6 bits of the
       * first byte, the second pixel is the two remaining bits from
       * that byte plus the 4 high bits from the second byte.  The
       * third pixel is the 4 remaining bits from the second byte plus
       * the two high bits of the third byte, and the 4th pixel is the
       * remaining six bits from the third byte.  Repeat until desired
       * effect is achieved. */
      a=qc_readbyte(q);
      ret[i++]=63-(a>>2);
      b=a; a=qc_readbyte(q);
      ret[i++]=63-((b&3)<<4|(a>>4));
      b=a; a=qc_readbyte(q);
      ret[i++]=63-((b&0xf)<<2|(a>>6));
      ret[i++]=63-((a&0x3f));
      break;
    default:
      fprintf(stderr,"Unknown bit depth in qc_scan.  Exiting.\n");
      exit(1);
    }
  }

  qc_reset(q);

  return ret;
}


/* qc_convertscan has been removed, and its code has been incorporated
 * into qc_scan.  See the release notes for version 0.3. */


/* This is a bit of a hack to write pgm files.  I should probably use
 * libpgm, but I don't have the documentation in front of me, and the
 * format is pretty trivial, so I hacked this together with help from
 * a few people. */

void qc_writepgm(struct qcam *q, FILE *f ,scanbuf *scan)
{
  int i;

  fprintf(f,"P5\n");
  fprintf(f,"%d %d\n",q->width,q->height);
  fprintf(f,"%d\n",(q->bpp==4)?15:63);

  for(i=0;i<q->height*q->width;i++) {
    fputc(scan[i],f);
  }
}

