/***************************************
XWindow Support for QuickCam
by Paul Chinn <loomer@svpal.org>
Modified by Scott Laird <scott@laird.com>
 
I took a bunch of this code from the source for VGB
"Virtual GameBoy" emulator by Marat Fayzullin and
Elan Feingold
*****************/
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <unistd.h>

/** MIT Shared Memory Extension for X ************************/
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
XShmSegmentInfo SHMInfo;


/** QuickCam include files */
#include "qcam.h"
#include "qcam-os.h"
 
/** Various X-related variables ******************************/

Screen *screen;
Display *disp;
Window root,win;
XColor col;
Colormap cmap;
XImage *ximage;
XEvent ev;
GC gc;
int screen_num;
unsigned long white,black;
int xstarted=0; 
int quit=0;

/* Set a flag to exit the loop at the end */

void quitprogram(int foo)
{
  quit=1;
}
 
/** Initialize xwindows, and prepare a shared memory buffer for
 the image.  Returns pointer to shared memory buffer. */
 
char *InitXWindows(struct qcam *q)
{
  XGCValues values;
  XSizeHints hints;
  XWMHints wmhints;
  int width, height;
  char *sbuf;
  char *window_name="QuickCam";
  char *icon_name="QuickCam";
  XTextProperty windowName, iconName;

  width=q->width;
  height=q->height;

  disp=XOpenDisplay(NULL);
  if(!disp) {printf("open display failed\n"); return NULL;}
  
  screen=DefaultScreenOfDisplay(disp);
  screen_num=DefaultScreen(disp);
  white=XWhitePixel(disp,screen_num);
  black=XBlackPixel(disp,screen_num);
  
  root=DefaultRootWindow(disp);
  
  win=XCreateSimpleWindow(disp,root,0,0,width,height,0,white,black);
  if(!win) {  
    printf("create window failed\n");
    return(NULL); 
  }
  
  /* tell window manager about our window */
  hints.flags=PSize|PMaxSize|PMinSize;
  hints.min_width=hints.max_width=hints.width=width;
  hints.min_height=hints.max_height=hints.height=height;
  wmhints.input=True;
  wmhints.flags=InputHint;

  XStringListToTextProperty(&window_name, 1 ,&windowName);
  XStringListToTextProperty(&icon_name, 1 ,&iconName);


  XSetWMProperties(disp, win, 
		   &windowName, &iconName,
		   NULL, 0,
		   &hints, &wmhints, NULL);
  
  /*  XStoreName(disp, win, "QuickCam"); */
  XSelectInput(disp, win, ExposureMask);
  XMapRaised(disp, win);
  XNextEvent(disp, &ev);
  
  gc = XCreateGC(disp, win, 0, &values);
  
  ximage = XShmCreateImage(disp, DefaultVisual(disp, screen_num), 
			   8, ZPixmap, NULL, &SHMInfo, width, height);
  if(!ximage) {
    printf("CreateImage Failed\n");
    return(NULL);
  }
 
  SHMInfo.shmid=shmget(IPC_PRIVATE, 
		       ximage->bytes_per_line*ximage->height,
		       IPC_CREAT|0777);

  if(SHMInfo.shmid < 0) {
    perror("shmget failed:");
    return (NULL);
  }
 
  sbuf = ximage->data = SHMInfo.shmaddr = shmat(SHMInfo.shmid, 0, 0);
  XShmAttach(disp, &SHMInfo);
  signal(SIGHUP, quitprogram); 
  signal(SIGINT, quitprogram);
  signal(SIGQUIT, quitprogram); 
  signal(SIGTERM, quitprogram);
  xstarted=1;
  return(sbuf);
}


void ExitXWindows(void)
{
  if(xstarted) {
    XShmDetach(disp, &SHMInfo);
    if(SHMInfo.shmaddr)
      shmdt(SHMInfo.shmaddr);
    if(SHMInfo.shmid > 0)
      shmctl(SHMInfo.shmid, IPC_RMID, 0);
  }
}


int *xqc_createpalette(Colormap cmap)
{
  int *pal;
  int i;

  pal=malloc(sizeof(int[64]));
  
  for(i=0; i<64; i++) {
    col.red =col.green = col.blue = i * 1024;
    if (!XAllocColor(disp, cmap, &col)) {
      fprintf(stderr,"XAllocColor failed on %d\n",i);
    }
    pal[i] = col.pixel;
  }
  return pal;
}


void xqc_sync(struct qcam *q, int *colortable, char *sbuf)
{
  int i;
  scanbuf *scan;

  qc_set(q);

  scan=qc_scan(q);

  for(i=0;i<q->height*q->width;i++)
    switch(q->bpp) {
    case 4: 
      sbuf[i]=colortable[(scan[i]<<2)+1];
      break;
    case 6:
      sbuf[i]=colortable[scan[i]];
      break;
    default:
      fprintf(stderr,"Unsupported depth!\n");
      exit(1);
    }	
  
  free(scan);
  
  XShmPutImage(disp, win, gc, ximage, 0,0,0,0,q->width, q->height, False);
  XFlush(disp);
}

void usage(void)
{
  fprintf(stderr,"Usage:\n");
  fprintf(stderr,"  qcam [options]\n");
  fprintf(stderr,"    Options:\n");
  fprintf(stderr,"      -x width   Set width\n");
  fprintf(stderr,"      -y height  Set height\n");
  fprintf(stderr,"      -p port    Set port\n");
  fprintf(stderr,"      -B bpp     Set bits per pixel\n");

  fprintf(stderr,"      -c val     Set contrast\n");
  fprintf(stderr,"      -w val     Set white balance\n");
  fprintf(stderr,"      -b val     Set brightness\n");
  fprintf(stderr,"      -V         Show version information\n");
  fprintf(stderr,"      -v         Verbose output\n");
  fprintf(stderr,"      -C         Use private colormap\n");
  

}

void modegripe(struct qcam *q)
{
  fprintf(stderr,"Unsupported resolution/bit depth!\n");
  fprintf(stderr,"This program only supports 320x240, 160x120, and 80x60.\n");
  fprintf(stderr,"You specified %d x %d.  Try again.\n",q->width,q->height);
  exit(1);
}

int main(int argc, char **argv)
{
  int arg;
  extern char *optarg;
  struct qcam *q;
  int *colortable;
  char *sbuf;
  int verbose=0;
  int privatecmap=0;
  Colormap cmap;
  struct timeval tv1, tv2;
  double framerate=0,fr;

  colortable=malloc(sizeof(int[64]));

  if(geteuid()) {
    fprintf(stderr,"%s: Must be installed SUID or run as root.  Exiting.\n",
	    argv[0]);
    exit(1);
  }

  q=qc_init();

  /* Read command line */

  while((arg=getopt(argc,argv,"hCvx:y:p:b:B:c:w:V"))>0) { 
    switch (arg) {
    case 'x':
      q->width=atoi(optarg);
      break;
    case 'y':
      q->height=atoi(optarg);
      break;
    case 'p':
      if (!getuid())
	q->port=strtol(optarg,NULL,0);
      break;
    case 'B':
      q->bpp=atoi(optarg);
      break;
    case 'b':
      q->brightness=atoi(optarg);
      break;
    case 'c':
      q->contrast=atoi(optarg);
      break;
    case 'w':
      q->whitebal=atoi(optarg);
      break;
    case 'V':
      fprintf(stderr,"%s: Version 0.3\n",argv[0]);
      exit(0);
      break;
    case 'h':
      usage();
      exit(0);
      break;
    case 'C':
      privatecmap=1;
      break;
    case 'v':
      verbose=1;
      break;
    default:
      fprintf(stderr,"%s: Unknown option or error in option\n",argv[0]);
      usage();
      exit(1);
      break;
    }
  }

  switch (q->width) {
  case 80:  if(q->height!=60)  modegripe(q); break;
  case 160: if(q->height!=120) modegripe(q); break;
  case 320: if(q->height!=240) modegripe(q); break;
  default:
    modegripe(q);
  }

  switch (q->bpp) {
  case 4: 
  case 6: break;
  default:
    fprintf(stderr,"%s: Error: Unsupported bit depth\n",argv[0]);
    exit(1);
  }


  /* Attempt to get permission to access IO ports.  Must be root */

  if (qc_open(q)) {
    fprintf(stderr,"Cannot open QuickCam; exiting.\n");
    exit(1);
  }

  setuid(getuid());

  fprintf(stderr,"Scanning from QuickCam at 0x%x at %dx%d @ %dbpp\n",
	  q->port,q->width,q->height,q->bpp);

  /* Start X Display */

  if ((sbuf=InitXWindows(q))==NULL) {
    fprintf(stderr,"InitXWindows failed, exiting\n");
    exit(1);
  }

  if (privatecmap) {
    cmap = XCreateColormap(disp, win, DefaultVisual(disp, screen_num), 
			   AllocNone);
    XSetWindowColormap(disp, win, cmap);
  } else {
    cmap=DefaultColormap(disp, screen_num);
  }
  colortable=xqc_createpalette(cmap);


  /* Scan one image */
  if(verbose) {
    gettimeofday(&tv1,NULL);
  }
  
  while(!quit) {
    xqc_sync(q,colortable,sbuf);

    /* Calculate frame rate */
    if(verbose) {
      gettimeofday(&tv2,NULL);

      /* The frame rate is calculated using the TCP RTT algorithm */

      fr=(1.0/(tv2.tv_sec-tv1.tv_sec+(tv2.tv_usec-tv1.tv_usec)/1000000.0));
      if(framerate!=0)
	framerate=0.9*framerate+0.1*fr;
      else
	framerate=fr;

      fprintf(stderr,"Frame rate: %f fps       \r",framerate);
      tv1.tv_sec=tv2.tv_sec;
      tv1.tv_usec=tv2.tv_usec;
    }

  }

  ExitXWindows();

  /* Release IO privileges */
  qc_close(q);

  return 0;
}

