/* console.c - Basic console I/O routines. */

#include "duart.h"

static char     digits[16] = {
	'0', '1', '2', '3', '4', '5', '6', '7',
	'8', '9', 'A', 'B', 'C', 'D', 'E', 'F'
};

/*
 * puts - Writes a string to a port, exactly as the string is defined.
 */
int
puts(s, port)
	char           *s, *port;
{
	if ((port != (char *) DUART_A) && (port != (char *) DUART_B))
		return (-1);

	/* Dump the string out the port. */
	while (*s != '\0')
		putc(*s++, port);
}

/*
 * gets - Reads a string from a port, terminated by the specified character.
 */
int
gets(s, port, term)
	char           *s, *port, term;
{
	int             i;
	char            c;

	if ((port != (char *) DUART_A) && (port != (char *) DUART_B))
		return (-1);

	i = 0;
	c = '\0';
	while (c != term)
		s[i++] = c = getc(port);

	s[--i] = '\0';

	return (i);
}

/*
 * readln - Reads a string from the console port, with echo.
 */
int
readln(s)
	char           *s;
{
	int             i, n;
	char            c;

	/* Get the string from the console port. */
	i = 0;
	c = '\0';

	while ((c != '\n') && (c != '\r')) {
		/* Get a character. */
		s[i++] = c = getc(DUART_A);

		/* Deal with the echo. */
		switch (c) {
		case '\n':
		case '\r':
			putc('\r', DUART_A);
			putc('\n', DUART_A);
			break;
		case '':
			if (--i > 0) {
				puts(" ", DUART_A);
				i--;
			}
			break;
		default:
			putc(c, DUART_A);
			break;
		}
	}

	/* Strip out the last character and replace it with 'null'. */
	i--;
	s[i] = '\0';

	return (i);
}

int
ch_atoi(c)
	char            c;
{
	int             value;

	value = 0;

	while ((value < 16) && (c != digits[value]))
		value++;

	if (c != digits[value])
		return (-1);
	else
		return (value);
}


int
atoi(str)
	char           *str;
{
	int             sum, val;

	sum = 0;

	while (*str != '\0') {
		if ((val = ch_atoi(*str++)) == -1)
			return (-1);
		else
			sum = sum * 10 + val;
	}

	return (sum);
}

void
itohex(s, i, bits)
	char           *s;
	int             i, bits;
{
	int             d, r, c;

	/* Initialize the end of the string. */
	c = (bits >> 2);
	s[c] = '\0';

	/* Convert the digits, four bits at a time. */
	while (c > 0) {
		s[--c] = digits[i & 0xf];
		i = i >> 4;
	}
}

int
hextoi(s)
	char           *s;
{
	int             sum, val;

	sum = 0;
	while (*s != '\0') {
		if ((val = ch_atoi(*s++)) == -1)
			return (-1);
		else
			sum = sum * 16 + val;
	}

	return (sum);
}

char           *
strncpy(s1, s2, n)
	char           *s1, *s2;
	int             n;
{
	int             i;
	char           *sp;

	sp = s1;
	i = 0;
	for (i = 0; (i < n) && *s2; i++)
		*s1++ = *s2++;

	*s1 = '\0';

	return (sp);
}

char           *
strcpy(s1, s2)
	char           *s1, *s2;
{
	int             i;
	char           *sp;

	sp = s1;
	i = 0;
	for (i = 0; *s2; i++)
		*s1++ = *s2++;

	*s1 = '\0';

	return (sp);
}

putchar(c)
	char            c;
{
	if (c == '\n')
		putc('\r', DUART_A);
	putc(c, DUART_A);
}

/*
 * N.B. support of "%d" via decprint() requires more code (for ldiv, etc.) --
 * dont use it if at all possible.
 */

printf(string)
	char           *string;
{
	unsigned       *argp = (unsigned *) &string;
	char           *s = string;
	int             c, width = 0;
	char            pad = ' ';

	argp++;
	while (*s) {
		if (*s == '%') {
			s++;
			if (*s == '-') {
				pad = '0';
				s++;
			} else {
				pad = ' ';
			}
			if ('0' <= *s && *s <= '9') {
				width = *s - '0';
				s++;
			} else {
				width = 0;
			}
			switch (*s++) {
			case 's':
				puts((char *) (*argp++), DUART_A);
				break;
			case 'x':
				hexprint(*argp++, width);
				break;
			case 'X':
				hexprint(*argp++, 8);
				break;
			case 'd':
				decprint((int) (*argp++), width, pad);
				break;
			case 'c':
				c = *argp++;
				putchar(c);
				break;
			default:
				if (width != 0)
					putchar(*(s - 3));
				putchar(*(s - 2));
				putchar(*(s - 1));
			}
		} else {
			putchar(*s++);
		}
	}
}

/*
 * unsigned hexadecimal print
 */
hexprint(n, width)
	unsigned        n;
	int             width;
{
	unsigned        i, digit, pflag = 0;
	if (width <= 0)
		width = 1;
	for (i = 0; i < 8; i++) {
		digit = (n >> 28);
		if (digit != 0 || i == 8 - width)
			pflag = 1;
		if (pflag) {
			if (digit > 9)
				digit += 'a' - 10;
			else
				digit += '0';
			putchar((char) digit);
		}
		n <<= 4;
	}
}

/*
 * Signed decimal print.  We build the character string backwards.
 */
decprint(n, width, pad)
	int             n, width, pad;
{
	unsigned        digit, nflag = 0;
	static char     decbuf[12];
	char           *p = decbuf + 11;

	*p-- = 0;
	if (n < 0) {
		n = -n;
		nflag++;
	}
	if (width == 0)
		width = 1;
	if (n == 0) {
		*p-- = '0';
	}
	while (n != 0) {
		digit = n % 10;
		n = n / 10;
		*p-- = digit + '0';
	}
	while (p > decbuf + 10 - width)
		*p-- = pad;
	if (nflag)
		*p-- = '-';
	p++;
	puts(p, DUART_A);
}
