/* exception.c - Handles system exceptions. */

#include "globals.h"
#include "exception.h"
#include "kern.h"

extern void     unass_trap();

/*
 * exc_dispatch - Calls appropriate exception handler.
 */
void
exc_dispatch(pstat, exc_frame)
	struct pstat   *pstat;
	struct exc_frame *exc_frame;
{
	/*
	 * If interrupt, call handler, else print suitable message.
	 */
	if ((exc_frame->vec >= EXC_USR_INT0) && 
		(exc_frame->vec <= EXC_USR_INT7))
		call_user_ifunc(exc_frame->vec);
	else if (exc_frame->vec == EXC_BUSERR)
		printf("\nBus error.\n");
	else if (exc_frame->vec == EXC_ADDRERR)
		printf("\nAddress error.\n");
	else if (exc_frame->vec == EXC_ILLINST)
		printf("\nIllegal instruction.\n");
	else if (exc_frame->vec == EXC_ZDIV)
		printf("\nDivide by zero.\n");
	else if (exc_frame->vec == EXC_CHK)
		printf("\nCHK instruction.\n");
	else if (exc_frame->vec == EXC_TRAPV)
		printf("\nTRAPV instruction.\n");
	else if (exc_frame->vec == EXC_TRACE)
		printf("\nTrace exception.\n");
	else if ((exc_frame->vec >= EXC_AUTO1) && 
		(exc_frame->vec <= EXC_AUTO7))
		printf("\nAutovector level %x\n", 
			exc_frame->vec - EXC_AUTO1 + 1);
	else if ((exc_frame->vec >= EXC_TRAP0) && 
		(exc_frame->vec <= EXC_TRAP15))
		printf("\nTrap #%x\n", exc_frame->vec - EXC_TRAP0);
	else if (exc_frame->vec == EXC_UNINIT)
		printf("\nException: Uninitialized Interrupt.\n");
	else if (exc_frame->vec == EXC_PRIV)
		printf("\nException: Previlege violation.\n");
	else
		printf("\nException, vec = #%x\n", exc_frame->vec);

	/*
	 * If this is a system exception, print out stack frame information.
	 */
	if (exc_frame->vec < N_SVEC) {
		printf("exc_dispatch: show exception frame\n");
		show_exc_frame(exc_frame);
	}

	/*
	 * For group one and two exceptions, restore saved context.
	 */
	if (exc_frame->group != EXC_GROUP0)
		restore(pstat);
	else
		panic("Fatal exception");
}

/*
 * compute_vec - Compute the vector number and group from the vector address.
 */
compute_vec(vec_addr, vec, exc_type, vec_num)
	long            vec_addr;
	int            *vec;
	int            *exc_type;
	int             vec_num;
{
	short           offset;
	char            vec_str[5];
	void            trap2();
	int             retval = 0;

	/* 
	 * Vector number is determined from the trap
	 * handler address. 
	 */
	offset = (vec_addr - 1) - (long) trap2;
	*vec = (offset / 6) + 2;

	/*
	 * Figure out the exception group number.
	 */
	if ((*vec == EXC_BUSERR) || (*vec == EXC_ADDRERR))
		*exc_type = EXC_GROUP0;
	else if ((*vec == EXC_TRACE) || (*vec == EXC_ILLINST) ||
		 (*vec == EXC_PRIV) ||
		 ((*vec >= EXC_SPUR) && (*vec <= EXC_AUTO7)) ||
		 (*vec == EXC_UNINIT) ||
		 ((*vec >= EXC_USR_INT0) && (*vec <= EXC_USR_INT7)))
		*exc_type = EXC_GROUP1;
	else if ((*vec == EXC_CHK) || (*vec == EXC_ZDIV) ||
		 (*vec == EXC_TRAPV) ||
		 ((*vec >= EXC_TRAP0) && (*vec <= EXC_TRAP15)))
		*exc_type = EXC_GROUP2;
	else
		*exc_type = EXC_INVALID;

	return retval;
}

/*
 * get_context - Decodes stack frame information following exception.
 */
void
get_context(regs, fp, vec_addr, vec_num)
	struct m68k_regs *regs;
	long            fp, vec_addr;
	int             vec_num;
{
	struct pstat    pstat;
	struct exc_frame exc_frame;
	long            pc, get_usp();
	int             vec, exc_type, frame_size;
	short           cpu_mode, sr;
	void            trap2(), unass_trap();
	int             retval;

	/* Compute the vector number and exception group number. */
	retval = compute_vec(vec_addr, &vec, &exc_type, vec_num);

	pstat.flags = 0;

	/*
	 * Get the exception frame, depending on type.
	 */
	if (exc_type == EXC_GROUP0) {
		exc_frame.frame.frame0 = *((struct exc_frame0 *) fp);
		pc = exc_frame.frame.frame0.pc;
		sr = exc_frame.frame.frame0.sr;
		cpu_mode = get_cpu_mode(sr);
		frame_size = sizeof(struct exc_frame0);
	} else if ((exc_type == EXC_GROUP1) || (exc_type == EXC_GROUP2)) {
		exc_frame.frame.frame12 = *((struct exc_frame12 *) fp);
		pc = exc_frame.frame.frame12.pc;
		sr = exc_frame.frame.frame12.sr;
		cpu_mode = get_cpu_mode(sr);
		frame_size = sizeof(struct exc_frame12);
	} else if (exc_type == EXC_INVALID) {
		printf("\nget_context: unassigned exception, vec = %x\n", vec);
		panic("get_context: Unknown exception");
	}

	exc_frame.vec = vec;
	exc_frame.group = exc_type;

	/* Change 'regs->a7' to point to pre-exception stack location. */
	regs->a7 = fp + frame_size;

	/* Put together the processor status structure. */
	pstat.regs = *regs;
	pstat.usp = get_usp();
	pstat.sr = sr;
	pstat.pc = pc;

	/* Go process the exception. */
	exc_dispatch(&pstat, &exc_frame);

	/* Panic if 'get_context' tries to return. */
	panic("get_context: illegal attempt to return to 'get_context'.");
}
