/* duart.c - Boot code for the 68681 serial communications chip. */

#include "exception.h"
#include "duart.h"

void            duart_tx_ready(), duart_rx_ready();

/*
 * init_duart - Sets up the DUART for two-channel operation.
 */
void
init_duart(a_baud, b_baud)
	int             a_baud, b_baud;
{
	volatile char  *io1_base, *duart_a, *duart_b;
	unsigned char   ch_baud_a, ch_baud_b, baud_char();
	char            dummy;

	ch_baud_a = baud_char(a_baud);
	if (ch_baud_a == 0xff) {
		printf("Incorrect baud rate for channel A: %x\n", a_baud);
		return;
	}
	ch_baud_b = baud_char(b_baud);
	if (ch_baud_b == 0xff) {
		printf("Incorrect baud rate for channel B: %x\n", b_baud);
		return;
	}
	if ((a_baud == 19200 && b_baud > 19200) ||
	    (a_baud > 19200 && b_baud == 19200)) {
		printf("Can't mix %x baud and %x baud\n", a_baud, b_baud);
		return;
	}
	/* Get register base addresses. */
	io1_base = (char *) IO1_BASE;
	duart_a = (char *) DUART_A;
	duart_b = (char *) DUART_B;

	/* Initialize the timer countdown registers. */
	*(io1_base + CTUR) = CTUR_DEFAULT;
	*(io1_base + CTLR) = CTLR_DEFAULT;

	/* Enable the RTS outputs. */
	*(io1_base + OPCR) = OPCR_DEFAULT;
	*(io1_base + OPR_SET) = RTS_A | RTS_B;

	if (b_baud <= 19200 && a_baud <= 19200)
		*(io1_base + ACR) = (ACR_DEFAULT | 0x80);	/* BRG set 1 */
	else			/* 38K baud requested */
		*(io1_base + ACR) = ACR_DEFAULT;	/* BRG set 0 */

	*(duart_a + CSR) = ch_baud_a;
	*(duart_b + CSR) = ch_baud_b;

	/* Set up the serial port control registers. */
	*(duart_a + MR1) = MR1_DEFAULT;
	*(duart_a + MR2) = MR2_DEFAULT;
	*(duart_a + CR) = CR_DEFAULT;
	*(duart_b + MR1) = MR1_DEFAULT;
	*(duart_b + MR2) = MR2_DEFAULT;
	*(duart_b + CR) = CR_DEFAULT;

	/* Set the interrupt vector register and enable timer interrupts. */
	*(io1_base + IVR) = DUART_INT_VEC;
	*(io1_base + IMR) = IMR_DEFAULT;
}

unsigned char
baud_char(num)
	int             num;
{
	unsigned char   ch_baud;

	switch (num) {
	case 50:
		ch_baud = BAUD50;
		break;
	case 110:
		ch_baud = BAUD110;
		break;
	case 135:
		ch_baud = BAUD135;
		break;
	case 200:
		ch_baud = BAUD200;
		break;
	case 300:
		ch_baud = BAUD300;
		break;
	case 600:
		ch_baud = BAUD600;
		break;
	case 1200:
		ch_baud = BAUD1200;
		break;
	case 1050:
		ch_baud = BAUD1050;
		break;
	case 2400:
		ch_baud = BAUD2400;
		break;
	case 4800:
		ch_baud = BAUD4800;
		break;
	case 9600:
		ch_baud = BAUD9600;
		break;
	case 19200:
	case 38400:
		ch_baud = BAUD38400;
		break;
	default:
		ch_baud = 0xff;
		break;
	}

	return (ch_baud);
}

/*
 * duart_reset_isr - Resets the timer interrupt status register after an
 * interrupt.
 */
void
duart_reset_isr()
{
	volatile char  *io1_base;
	char            dummy;

	/* Get the device base address. */
	io1_base = (char *) IO1_BASE;

	/*
	 * Clear the interrupt status register by reading the stop counter
	 * command address.
	 */
	dummy = *(io1_base + C_STOP);
}

/* putc - Stuffs a character out one of the DUART channels. */
void
putc(c, port)
	volatile char   c, *port;
{
	volatile char   status;

	/* Loop until output buffer is ready. */
	status = 0;

	while (!(status & TX_RDY))
		status = *(port + SR);

	/* Send character to transmit buffer. */
	*(port + TB) = c;
}

/* getc - Gets a character from one of the DUART channels. */
char
getc(port)
	volatile char  *port;
{
	volatile char   status;

	/* Loop until input buffer is ready. */
	status = 0;

	while (!(status & RX_RDY))
		status = *(port + SR);

	/* Get character from receive buffer. */
	return (*(port + RB));
}

/*
 * DUART interrupt handler
 */
void
duart_ihandler()
{
	volatile char   status;

	status = *((char *) DUART_A + ISR);

	if (status & 0x1)	/* ttya tx ready */
		duart_tx_ready(0);
	if (status & 0x10)	/* ttyb tx ready */
		duart_tx_ready(1);
	if (status & 0x2)	/* ttya rx ready */
		duart_rx_ready(0);
	if (status & 0x20)	/* ttyb rx ready */
		duart_rx_ready(1);
}

void
duart_tx_ready(tty)
	int             tty;
{
	printf("DUART%c TX READY\n", tty ? 'B' : 'A');
}

void
duart_rx_ready(tty)
	int             tty;
{
	printf("DUART%c RX READY\n", tty ? 'B' : 'A');
}

duart_imr_set(set, clear)
	char            set, clear;
{
	volatile static int imr;

	imr = (imr | set) & (~clear);
	*((char *) DUART_A + IMR) = imr;
}
