#include "kern.h"

#define TCB_PRIQ_SIZE	(MAX_PRI + 1)
struct tcbq tcb_priq[TCB_PRIQ_SIZE];

struct tcb *cur_thread;

struct tcb *idle_thread;
struct tcb idle_tcb;
char idle_stack[128];

int idle_fun() { for (;;) ; }

void
init_kern()
{
	int ix;

	for (ix = 0; ix < TCB_PRIQ_SIZE; ix++) {
		tcb_priq[ix].next = 0;
		tcb_priq[ix].prev = 0;
	}

	idle_thread = spawn(&idle_tcb, "idle", idle_stack, idle_fun, 
				sizeof(idle_stack), MAX_PRI);
	cur_thread = idle_thread;
}

struct tcb *
spawn(new_tcb, name, stack, code, stksiz, pri)
	struct tcb *new_tcb;
	char *name;
	char *stack;
{

	if (!new_tcb)
		return 0;

	if (!stack)
		return 0;

	new_tcb->pstat.pc = code;
	new_tcb->pstat.sr = 0x2000;	/* XXX */
	new_tcb->pstat.flags = 0;

	new_tcb->pstat.regs.d0 = 0;
	new_tcb->pstat.regs.d1 = 0;
	new_tcb->pstat.regs.d2 = 0;
	new_tcb->pstat.regs.d3 = 0;
	new_tcb->pstat.regs.d4 = 0;
	new_tcb->pstat.regs.d5 = 0;
	new_tcb->pstat.regs.d6 = 0;
	new_tcb->pstat.regs.d7 = 0;
	new_tcb->pstat.regs.a0 = 0;
	new_tcb->pstat.regs.a1 = 0;
	new_tcb->pstat.regs.a2 = 0;
	new_tcb->pstat.regs.a3 = 0;
	new_tcb->pstat.regs.a4 = 0;
	new_tcb->pstat.regs.a5 = 0;
	new_tcb->pstat.regs.a6 = 0;

	new_tcb->pstat.regs.a7 = new_tcb->pstat.usp = new_tcb->stkbeg = 
			(int)stack + stksiz - STACK_FENCE_SIZE;
	new_tcb->stkend = (int)stack + STACK_FENCE_SIZE;
	new_tcb->stksiz = stksiz - STACK_FENCE_SIZE;

	if (pri > MAX_PRI)
		pri = MAX_PRI;

	new_tcb->pri = pri;
	new_tcb->stat = 0;
	new_tcb->name = name;

	insque(new_tcb, &tcb_priq[new_tcb->pri]);

	return new_tcb;
}

void 
ready(thread)
	struct tcb *thread;
{
	if (!thread)
		thread = cur_thread;

	thread->stat = TASK_READY;
}

void 
suspend(thread)
	struct tcb *thread;
{
	if (!thread)
		thread = cur_thread;

	thread->stat = TASK_SUSPEND;
}

struct tcb *
pick()
{
	int ix;
	struct tcb *thread;
	struct tcbq *que;

	for (ix = 0; ix < TCB_PRIQ_SIZE; ix++) {
		que = &tcb_priq[ix];

		for (thread = que->next; thread; thread = thread->next) {
			if (thread->stat == TASK_READY)
				return thread;
		}
	}

	return 0;
}

void 
reschedule()
{
	struct tcb *thread;
	struct tcb *old_thread;

	thread = pick();
	if (!thread) {
		panic("no ready threads to run\n");
		return;
	}

	if (thread == cur_thread)
		return;

	old_thread = cur_thread;
	cur_thread = thread;

	cswitch(&old_thread->pstat, &cur_thread->pstat);
}

void
yield()
{
	suspend(cur_thread);
	reschedule();
}

void
run(thread)
	struct tcb *thread;
{
	ready(thread);
	reschedule();
}

void
kill(thread)
	struct tcb *thread;
{
	if (!thread)
		thread = cur_thread;

	remque(thread);
	reschedule();
}

void
exit(code)
{
	printf("thread %s(%X) exiting with code = %d\n", 
		cur_thread->name, cur_thread, code);

	kill(0);
}

void
sem_init(sem)
	struct sem *sem;
{
	sem->count = 0;
	sem->waitq.next = 0;
	sem->waitq.prev = 0;
}

void
sem_p(sem)
	struct sem *sem;
{
	if (--sem->count < 0) {
		insque(cur_thread, &sem->waitq);
		yield();
	}
}

void 
sem_v(sem)
	struct sem *sem;
{
	struct tcb *thread;

	if (sem->count++ < 0) {
		thread = sem->waitq.next;

		if (!thread)
			panic("sem_v: no thread waiting on ready sem\n");

		if (thread->stat != TASK_SUSPEND)
			panic("sem_v: thread not suspended on ready sem\n");

		if (thread->pri < cur_thread->pri)
			run(thread);
		else
			ready(thread);
	}
}
