/*
 * srec.c - Client program for S-record server. 
 */

#include "globals.h"
#include "exception.h"
#include "duart.h"
#include "filserv.h"
#include "kern.h"

struct tcb shell_tcb;
struct tcb *shell;
char shell_stack[2048];

#define SHELL_PRI	10

int shell_fun();

extern	void            timer_proc();
extern  duart_ihandler();

void
send_message(code, data)
	char            code, *data;
{
	/* Send the message. */
	putc('', DUART_B);
	putc(code, DUART_B);
	puts(data, DUART_B);
	puts("\n", DUART_B);
}

void
show_time()
{
	unsigned char   hrs, mins, secs, ticks;

	/* Get the time from the system clock. */
	get_time(&hrs, &mins, &secs, &ticks);

	/* Display it, after a fashion. */
	printf("The time is: %x hours %x mins %x secs %x ticks\n", 
		hrs, mins, secs, ticks);
}

int
get_message(code, msg)
	char           *code, *msg;
{
	char            line[LINE_SIZE], *lp, *mp;

	/* Get a line of input data from the comm port. */
	gets(line, DUART_B, '\n');

	/* Make sure the first character is the escape character. */
	if (line[0] != ESC_CHAR) 
		return (FALSE);
	else
		*code = line[1];

	/* Copy the rest of the message to the 'msg' string. */
	lp = &line[2];
	mp = msg;

	while (*lp != '\0')
		*msg++ = *lp++;

	*msg = '\0';

	/* If this is a quit message, print the reason. */
	if (*code == 'Q')
		printf(">>> %s\n", mp);

	return (TRUE);
}

int
unpack_record(msg, type, count, addr, data, checksum)
	char           *msg, *type, *count, *addr, *data, *checksum;
{
	int             byte_count, addr_count, data_count;
	char           *mp;

	/* Blow off if first character isn't 'S'. */
	mp = msg;
	if (*mp != 'S')
		return (FALSE);

	/* Get the type and byte count, which are fixed length fields. */
	strncpy(type, mp, 2);
	mp += 2;

	strncpy(count, mp, 2);
	byte_count = 2 * hextoi(count);
	mp += 2;

	/* Determine the address and data field character counts. */
	switch (type[1]) {
	case '0':
	case '1':
	case '5':
	case '9':
		addr_count = 4;
		break;
	case '2':
	case '8':
		addr_count = 6;
		break;
	case '3':
	case '7':
		addr_count = 8;
		break;
	default:
		return (FALSE);
		break;
	}

	data_count = byte_count - addr_count - 2;

	/* Read out the rest of the fields. */
	strncpy(addr, mp, addr_count);
	mp += addr_count;
	strncpy(data, mp, data_count);
	mp += data_count;
	strncpy(checksum, mp, 2);

	return (TRUE);
}

int
test_checksum(count, addr, data, checksum)
	char           *count, *addr, *data, *checksum;
{
	unsigned int    sum;
	char            byte[3];

	sum = 0;

	while (*count != '\0') {
		sum += hextoi(strncpy(byte, count, 2));
		count += 2;
	}

	while (*addr != '\0') {
		sum += hextoi(strncpy(byte, addr, 2));
		addr += 2;
	}

	while (*data != '\0') {
		sum += hextoi(strncpy(byte, data, 2));
		data += 2;
	}

	return ((~sum & 0xff) == hextoi(checksum));
}

void
dump_data(addr, data)
	char           *addr, *data;
{
	char           *ap, byte[3];

	/* Convert the address string to a pointer value. */
	ap = (char *) hextoi(addr);

	/* Load the data into memory, one byte at a time. */
	while (*data != '\0') {
		/* Get the next character pair. */
		strncpy(byte, data, 2);

		data += 2;

		/* Convert into an eight-bit value and dump to memory. */
		*ap++ = (unsigned char) (hextoi(byte) & 0xff);
	}
}

int
process_record(msg, start_addr)
	char           *msg, **start_addr;
{
	char            type[3], count[3], addr[9], data[65], checksum[3];

	/* Break the record down into its component parts. */
	if (!unpack_record(msg, type, count, addr, data, checksum)) {
		printf("Error unpacking record.\n");
		return (FALSE);
	}

	/* Test the checksum. */
	if (!test_checksum(count, addr, data, checksum)) {
		printf("Bad checksum.\n");
		return (FALSE);
	}

	switch (type[1]) {
	case '0':		/* Ignore header records. */
	case '5':
		break;
	case '1':		/* Dump data records. */
	case '2':
	case '3':
		dump_data(addr, data);
		break;
	case '7':		/* Get start address. */
	case '8':
	case '9':
		*start_addr = (char *) hextoi(addr);
		break;
	}

	return (TRUE);
}

#define ACK_MSG "Acknowledge receipt of S-record."
#define ERR_MSG "Error processing S-record."
#define AGAIN_MSG "Please send S-record again"

int
get_file(line, start_addr)
	char           *line, **start_addr;
{
	int             valid;
	char            code, msg[LINE_SIZE];

	/* Send a file request message to the server. */
	send_message('F', line);

	/* Receive, decode and load all of the S-records, a line at a time. */
	*start_addr = 0;

	for (;;) {
		valid = FALSE;

		while (!valid) {
			if (get_message(&code, msg)) {
				if (code == 'L') {
					valid = process_record(msg, start_addr);
					if (valid == TRUE)
						send_message('A', ACK_MSG);
					else
						send_message('E', ERR_MSG);
				} else if (code == 'D') {
					printf("Download complete.\n");
					return (TRUE);
				} else if (code == 'Q') {
					printf("Download failed.\n");
					return (FALSE);
				}
			}

			if (!valid) {
				printf("retry...\n");
				send_message('E', AGAIN_MSG);
			}
		}
	}
}

void
run_code(code_valid, start_addr)
	int             code_valid;
	char           *start_addr;
{
	char            vec[9];

	if (!code_valid) {
		printf("You need to download a program first!\n");
		return;
	}

	/* Print out the address of the program entry point. */
	itohex(vec, (int) start_addr, sizeof(int));

	printf("Jumping to address %X\n", vec);

	/* Call this address. */
	call(start_addr);

	/* Return to loader. */
	printf("\nReturning to loader.");
}

void
print_help()
{
	printf("\nCommands:\n");
	printf("\ts - Send a text string message to server\n");
	printf("\tt - Print out the time in hex\n");
	printf("\tf - Request file transfer\n");
	printf("\tg - Execute downloaded code\n");
	printf("\tk - Kill the server\n");
	printf("\tb - Set baud rates\n");
	printf("\tT - issue a trap instruction\n");
	printf("\tB - try generating a bus error\n");
	printf("\tr - run test\n");
	printf("\t? - Print this message\n");
}

void
main()
{
	/*
	 * Install system and user exception vectors 
	 */
	init_vectors();
	install_ifunc(EXC_USR_INT0, timer_proc);

	/* 
	 * Initialize serial channels:
	 * ttya - 9600 baud - console
	 * ttyb - 38400 baud - download
	 */
	init_duart(9600, 38400);

	init_kern();	/* init threading kernel */

	shell = spawn(&shell_tcb, "shell", shell_stack,
			shell_fun, sizeof(shell_stack), 10);
	run(shell);
}

shell_fun()
{
	int             code_valid, dummy;
	char            command, *lp, *ep, *start_addr, line[LINE_SIZE];
	int             baud_a, baud_b;
	int		trap_num;
	int		ix;
	char		*cp;

	/* Clear the real-time clock. */
	set_time(0, 0, 0);

	/* Greet the user and go. */
	printf("\nS-record loader is up and running.  Type '?' for help.\n");

	code_valid = FALSE;

	for (;;) {
		/* Get a command line from the interface port. */
		show_time();
		printf("%s> ", CPU_DESC);

		readln(line);

		/* Skip over whitespace in command. */
		lp = line;
		skip_spaces(&lp);

		/* Execute the command, if it's not null. */
		if ((command = *lp++) != '\0') {
			/* Skip whitespace up to argument. */
			skip_spaces(&lp);

			/* Choose the correct response. */
			switch (command) {
			case 's':	/* send a message to filserv */
				send_message('M', lp);
				break;
			case 't':	/* show current ticks */
				show_time();
				break;
			case 'f':	/* download a file  */
				code_valid = get_file(lp, &start_addr);
				break;
			case 'g':	/* goto a location and run the code
					 * there */
				run_code(code_valid, start_addr);
				break;
			case 'k':	/* kill the server */
				send_message('Q', "Please shut down server.");
				break;
			case 'b':	/* set baud rates */
				baud_a = read_num(&lp);
				baud_b = atoi(lp);
				init_duart(baud_a, baud_b);
				break;
			case '?':	/* help */
				print_help();
				break;
			case 'T':
				trap_num = read_num(&lp);
				printf("issuing trap %x\n", trap_num);
				trapit(trap_num);
				break;
			case 'B':
				/* try a bus error. */
				puts("\nTrying a bus error...\n");
				cp = (char *) 0x0d00000;
				ix = *cp;
				break;
			case 'r':
				/* thread/sem test */
				test0();
				break;
			case '\0':
				break;
			default:
				printf("ERR: Unknown command:\n");
				printf(line);
				break;
			}
		}
	}
}
