/// @file linktypes.hh
/// @author Joshua Petitt <petitj01@tartarus.uwa.edu.au>
/// @author Stefan Schmitt <sschmitt@ee.uwa.edu.au>
/// @version 1.0
/// @date 2003-07
///  
/// Declares all non-generic SignalLink types.

#ifndef LINKTYPES_HH
#define LINKTYPES_HH

#include "behavior/behavior.hh"
#include <cstring>

namespace EyeMind {

/// Carries a speed.
class SpeedLink : public SignalLink
{
private:
	SpeedType speed; ///< Current speed.
	SpeedType next_speed; ///< Next speed.

public:
	/// Default constructor; nulls data members.
	SpeedLink();

	/// @name Data feeding functions
	//@{
	virtual void Speed(const SpeedType& s); ///< Set new speed.
	SpeedType Speed(){ return speed; } ///< Returns current speed.
	//@}

	/// Propagates all values.
	virtual bool Update();
};



/// Contains all distances to the left, front and right.
struct DistancesType {
	unsigned left, front, right, back;
};


/// Carries distances.
class DistanceLink : public SignalLink
{
private:
	DistancesType dist; ///< Current value.
	DistancesType next_dist; ///< Next value.

public:
	/// Default constructor; nulls data members.
	DistanceLink();
	
	/// @name Data feeding functions
	//@{
	virtual void Distances(const DistancesType& d);
	DistancesType Distances(){ return dist; }
	//@}
	
	/// Propagates all values.
	virtual bool Update();
};


/// Carries an Image.
/// To avoid memory leakage, the image is \b never buffered.
class ImageLink : public SignalLink
{
private:
	colimage* image;

public:
	/// Default constructor; nulls data members.
	ImageLink();

	/// @name Data feeding functions
	//@{
	virtual void Image(colimage *i);
	colimage* Image(){ return image; }
	//@}
};

#if 0

/// Carries a position.
class PositionLink : public SignalLink
{
private:
	PositionType pos; ///< Current position.
	PositionType next_pos; ///< Next position.
public:
	/// Default constructor; nulls data members.
	PositionLink();

	/// @name Data feeding functions
	//@{
	virtual void Position(const PositionType& p); ///< Set new position.
	PositionType Position(){ return pos; } ///< Returns current position.
	//@}

	/// Propagates all values.
	virtual bool Update();
};
#endif

#if 0
/// Contains polar coordinates.
struct PolarType
{
	float r;   ///< Distance in meters.
	float phi; ///< Angle in radians.
};



/// Carries polar coordinates.
class PolarLink : public SignalLink
{
private:
	PolarType coord; ///< Current coordinates.
	PolarType next_coord; ///< Next coordinates.

public:
	/// Default constructor; nulls data members.
	PolarLink();

	/// @name Data feeding functions
	//@{
	virtual void Polar(const PolarType& p); ///< Set new coordinates.
	PolarType Polar(){ return coord; } ///< Returns current coordinates.
	//@}

	/// Propagates all values.
	virtual bool Update();
};
#endif

/// Contains polar coordinates.
struct BallType
{
	float r;   ///< Distance in meters.
	float phi; ///< Angle in radians.
	bool  visible; ///< ball is visible
	bool  close;  ///< ball is close
};

/// Carries polar coordinates.
class BallLink : public SignalLink
{
private:
	BallType ball; ///< Current coordinates.
	BallType next_ball; ///< Next coordinates.

public:
	/// Default constructor; nulls data members.
	BallLink();

	/// @name Data feeding functions
	//@{
	virtual void Ball(const BallType& b); ///< Set new coordinates.
	BallType Ball(){ return ball; } ///< Returns current coordinates.
	//@}

	/// Propagates all values.
	virtual bool Update();
};


}; // namespace EyeMind

#endif // LINKTYPES_HH

