/**************************************************************************\
 *
 *  This file is part of the Coin 3D visualization library.
 *  Copyright (C) 1998-2002 by Systems in Motion. All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public License
 *  version 2.1 as published by the Free Software Foundation. See the
 *  file LICENSE.LGPL at the root directory of the distribution for
 *  more details.
 *
 *  If you want to use Coin for applications not compatible with the
 *  LGPL, please contact SIM to acquire a Professional Edition license.
 *
 *  Systems in Motion, Prof Brochs gate 6, 7030 Trondheim, NORWAY
 *  http://www.sim.no support@sim.no Voice: +47 22114160 Fax: +47 22207097
 *
\**************************************************************************/

/*!
  \class SoWWWAnchor SoWWWAnchor.h Inventor/nodes/SoWWWAnchor.h
  \brief The SoWWWAnchor class adds URL callbacks to the highlighted geometry.
  \ingroup nodes

  In addition to highlighting geometry under the cursor, the application
  programmer can set callbacks. It is possible to set one callback for
  picking, the fetch callback, and one callback for highlighting.

  \since Inventor 2.1
*/


#include <Inventor/nodes/SoWWWAnchor.h>
#include <Inventor/nodes/SoSubNodeP.h>
#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/actions/SoHandleEventAction.h>
#include <Inventor/SoPickedPoint.h>
#include <coindefs.h>
#include <../tidbits.h> // coin_atexit()


/*!
  \enum SoWWWAnchor::Mapping
  Enum that says how a picked node's position should be mapped to the URL.
*/
/*!
  \var SoWWWAnchor::Mapping SoWWWAnchor::NONE
  The position of the picked node is not mapped to the URL.
*/
/*!
  \var SoWWWAnchor::Mapping SoWWWAnchor::POINT
  The position of the picked node is mapped to the URL as object space
  coordinates, adding a parameter string to the end of the URL. To
  assure that the URL works with all browsers, the coordinates are
  divided by commas sent as the hex representation.

  If a model by the name of sim.wrl resided at www.coin3d.org and the
  picked point had the coordinates [1.5, 10, 6.77], the resulting URL
  would be "http://www.coin3d.org/sim.wrl?1.5%2c10%2c6.77".
*/


/*!
  \var SoSFString SoWWWAnchor::name
  The name of the URL which the anchor points to.
*/
/*!
  \var SoSFString SoWWWAnchor::description
  The description of the URL.
*/
/*!
  \var SoSFEnum SoWWWAnchor::map
  Enum describing how a node's position should be mapped to the URL.
*/

// *************************************************************************

#ifndef DOXYGEN_SKIP_THIS

class SoWWWAnchorP {
 public:
  SoWWWAnchorP(SoWWWAnchor * owner) {
    this->owner = owner;
    this->fullname = "";
  }
  SoWWWAnchor * owner;
  SbString fullname;

  static SoWWWAnchorCB * fetchfunc;
  static void * fetchdata;
  static SoWWWAnchorCB * highlightfunc;
  static void * highlightdata;
};

// static members
SoWWWAnchorCB * SoWWWAnchorP::fetchfunc;
void * SoWWWAnchorP::fetchdata;
SoWWWAnchorCB * SoWWWAnchorP::highlightfunc;
void * SoWWWAnchorP::highlightdata;

// We forgot about this class before locking the Coin 1.0 ABI, so we
// have to use an SbDict to store per-instance data members for this
// class. In Coin 2.0 it's ok to break ABI-compatibility again, so we
// don't use this little hack there, but rather the usual Cheshire Cat
// pattern.

static SbDict * wwwanchor_private_data_dict = NULL;

static void
wwwanchor_private_data_cleanup(void)
{
  delete wwwanchor_private_data_dict;
  wwwanchor_private_data_dict = NULL;
}

static SoWWWAnchorP *
wwwanchor_get_private_data(const SoWWWAnchor * thisp)
{
  if (wwwanchor_private_data_dict == NULL) {
    wwwanchor_private_data_dict = new SbDict;
    coin_atexit((coin_atexit_f *)wwwanchor_private_data_cleanup);
  }
  void * pimpl;
  if (!wwwanchor_private_data_dict->find((unsigned long) thisp, pimpl)) {
    pimpl = (void*) new SoWWWAnchorP((SoWWWAnchor*) thisp);
    (void) wwwanchor_private_data_dict->enter((unsigned long) thisp, pimpl);
  }
  return (SoWWWAnchorP*) pimpl;
}

#undef PRIVATE
#define PRIVATE(p) (wwwanchor_get_private_data(this))

#endif // DOXYGEN_SKIP_THIS

// *************************************************************************


SO_NODE_SOURCE(SoWWWAnchor);


/*!
  Constructor.
*/
SoWWWAnchor::SoWWWAnchor()
{
  SO_NODE_INTERNAL_CONSTRUCTOR(SoWWWAnchor);

  SO_NODE_ADD_FIELD(name, ("<Undefined URL>"));
  SO_NODE_ADD_FIELD(description, (""));
  SO_NODE_ADD_FIELD(map, (NONE));

  SO_NODE_DEFINE_ENUM_VALUE(Map, NONE);
  SO_NODE_DEFINE_ENUM_VALUE(Map, POINT);
  SO_NODE_SET_SF_ENUM_TYPE(map, Map);
}

/*!
  Destructor.
*/
SoWWWAnchor::~SoWWWAnchor()
{
  wwwanchor_private_data_dict->remove((unsigned long)this);
}

// doc in super
void
SoWWWAnchor::initClass(void)
{
  SO_NODE_INTERNAL_INIT_CLASS(SoWWWAnchor);
}


/*!
  Sets the full URL to \a url. If this is set, this URL will be used in
  callbacks instead of the URL set in SoWWWAnchor::name.

  \sa SoWWWAnchor::getFullURLName()
 */
void
SoWWWAnchor::setFullURLName(const SbString & url)
{
  PRIVATE(this)->fullname = url;
}

/*!
  Returns the full URL if it's set by
  SoWWWAnchor::setFullURLName(). Otherwise the contents of
  SoWWWAnchor::name is returned.

  \sa SoWWWAnchor::setFullURLName()
 */
const SbString &
SoWWWAnchor::getFullURLName(void)
{
  if (PRIVATE(this)->fullname.getLength() > 0) {
    return PRIVATE(this)->fullname;
  }

  return this->name.getValue();
}

// documented in superclass
void
SoWWWAnchor::handleEvent(SoHandleEventAction * action)
{
  const SoEvent * event = action->getEvent();
  if (event->isOfType(SoMouseButtonEvent::getClassTypeId()) &&
      SoWWWAnchorP::fetchfunc) {
    const SoMouseButtonEvent * mbevent = (SoMouseButtonEvent*)event;
    if (SoMouseButtonEvent::isButtonPressEvent(mbevent,
                                               SoMouseButtonEvent::BUTTON1)) {
      SbString s = this->getFullURLName();
      if (this->map.getValue() == POINT) {
        const SoPickedPoint * pp = action->getPickedPoint();
        const SbVec3f point = pp->getObjectPoint(NULL);
        SbString temp;
        temp.sprintf("?%g%%2c%g%%2c%g", point[0], point[1], point[2]);
        s.operator+=(temp);
      }

      SoWWWAnchorP::fetchfunc(s, SoWWWAnchorP::fetchdata, this);
    }
  }
  inherited::handleEvent(action);
}

/*!
  Sets the callback function \a f that is called when a SoWWWAnchor node is
  clicked on. This callback can among other things be used to provide a
  browser with the URL of this node.

  The callback will be called with the URL, \a userData and a pointer to
  this node as arguments.
 */
void
SoWWWAnchor::setFetchURLCallBack(SoWWWAnchorCB * f, void * userData)
{
  SoWWWAnchorP::fetchfunc = f;
  SoWWWAnchorP::fetchdata = userData;
}

/*!
  Sets the callback function \a f that is called when a SoWWWAnchor node
  is highlighted. This callback can among other things be used to provide
  the user with a visual clue on which URL the node points to, for example
  by showing the URL as a string.

  The callback will be called with the URL, \a userData and a pointer to
  this node as arguments.
 */
void
SoWWWAnchor::setHighlightURLCallBack(SoWWWAnchorCB * f, void * userData)
{
  SoWWWAnchorP::highlightfunc = f;
  SoWWWAnchorP::highlightdata = userData;
}

/*!
  Calls the highlight callback set up with
  SoWWWAnchor::setHighlightURLCallBack().
*/
void
SoWWWAnchor::redrawHighlighted(SoAction * act, SbBool isNowHighlighting)
{
  inherited::redrawHighlighted(act, isNowHighlighting);

  if (SoWWWAnchorP::highlightfunc) {
    SbString s = this->getFullURLName();
    SoWWWAnchorP::highlightfunc(s, SoWWWAnchorP::highlightdata, this);
  }
}
