/**************************************************************************\
 *
 *  This file is part of the Coin family of 3D visualization libraries.
 *  Copyright (C) 1998-2002 by Systems in Motion.  All rights reserved.
 *
 *  This library is free software; you can redistribute it and / or
 *  modify it under the terms of the GNU General Public License
 *  version 2 as published by the Free Software Foundation.  See the
 *  file LICENSE.GPL at the root directory of this source distribution
 *  for more details.
 *
 *  If you desire to use this library in software that is incompatible
 *  with the GNU GPL, and / or you would like to take advantage of the
 *  additional benefits with regard to our support services, please
 *  contact Systems in Motion about acquiring a Coin Professional
 *  Edition License.  See <URL:http://www.coin3d.org> for more
 *  information.
 *
 *  Systems in Motion, Prof Brochs gate 6, 7030 Trondheim, NORWAY
 *  <URL:http://www.sim.no>, <mailto:support@sim.no>
 *
\**************************************************************************/

// src/Inventor/Qt/viewers/SoQtViewer.cpp.  Generated from SoGuiViewer.cpp.in by configure.

// NOTE: The SoQtViewer.cpp sourcecode file is completely
// autogenerated from "templatized" source code.

// *************************************************************************

/*!
  \class SoQtViewer Inventor/Qt/viewers/SoQtViewer.h
  \brief The SoQtViewer class is the top level base viewer class.
  \ingroup components viewers

  This is an abstract class, which adds the following features to it's
  SoQtRenderArea superclass: convenient methods for camera
  handling, an automatic headlight configuration.

  As for the camera handling: when setting a new scenegraph for the
  viewer, the scenegraph will automatically be scanned for a node
  derived from SoCamera. If not found, the viewer will itself set up a
  camera for the scene. The camera can then be conveniently controlled
  by the application programmers in many aspects:

  <ul>

  <li>camera type: toggle between using an orthographic camera and a
  perspective camera with SoQtViewer::toggleCameraType()</li>

  <li>zoom out to exactly encompass all scene geometry within the view
  by using SoQtViewer::viewAll()</li>

  <li>tag a specific position and orientation for the camera as the
  "home" position with SoQtViewer::saveHomePosition(), which one
  can then return to by using
  SoQtViewer::resetToHomePosition()</li>

  <li>automatically fit the near and far clipping planes of the camera
  around the scene's geometry by using
  SoQtViewer::setAutoClipping()</li>

  <li>control stereo viewing parameters</li>

  </ul>


  The SoQtViewer class automatically adds a headlight to the scene,
  which will always point in the approximate same direction as the
  current viewer camera, thereby securing that the scene geometry is
  always lighted and visible. (If you don't want the constant
  headlight, but rather want to light the scene on your own, this
  behavior can be turned off with SoQtViewer::setHeadlight()).


  SoQtViewer-derived viewers all inherit the following keyboard
  controls from this class (but only when the viewer is in "examine
  mode", ie SoQtViewer::isViewing() returns \c TRUE):

  <ul>

  <li>"s": put the viewer in "seek mode", where the end user may click
  anywhere on scene geometry to trigger an animation which moves the
  camera towards the point clicked</li>

  <li>"Home": hit this key to move camera back to last saved "home
  position"</li>

  <li>arrow keys: moves camera slightly left, right, up or down</li>

  <li>"q": exit application</li>

  </ul>
*/

// *************************************************************************


/*!
  \enum SoQtViewer::AutoClippingStrategy
  Enum for auto clipping strategy.
  
  \sa setAutoClippingStrategy()
*/

/*!
  \var SoQtViewer::AutoClippingStrategy SoQtViewer::CONSTANT_NEAR_PLANE

  Constant near plane auto clipping strategy. Explained in detail in
  the documentation for the SoQtViewer::setAutoClippingStrategy()
  method.
*/

/*!
  \var SoQtViewer::AutoClippingStrategy SoQtViewer::VARIABLE_NEAR_PLANE

  Variable near plane auto clipping strategy. Explained in detail in
  the documentation for the SoQtViewer::setAutoClippingStrategy()
  method.
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif // HAVE_CONFIG_H

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <Inventor/Qt/common/gl.h>
#include <Inventor/Qt/SoAny.h>
#include <Inventor/Qt/common/SbGuiList.h>

#include <Inventor/SoDB.h>

#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoLocateHighlight.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoOrthographicCamera.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoSwitch.h>
#include <Inventor/nodekits/SoBaseKit.h>

#include <Inventor/actions/SoGetBoundingBoxAction.h>
#include <Inventor/actions/SoGetMatrixAction.h>
#include <Inventor/actions/SoSearchAction.h>
#include <Inventor/actions/SoRayPickAction.h>

#include <Inventor/events/SoMouseButtonEvent.h>
#include <Inventor/errors/SoDebugError.h>
#include <Inventor/misc/SoCallbackList.h>
#include <Inventor/sensors/SoTimerSensor.h>
#include <Inventor/events/SoKeyboardEvent.h>
#include <Inventor/SoSceneManager.h>
#include <Inventor/SoPickedPoint.h>
#include <Inventor/SoLists.h>
#include <Inventor/SbLinear.h>

#include <soqtdefs.h>
#include <Inventor/Qt/SoQt.h>
#include <Inventor/Qt/viewers/SoQtViewer.h>
#include <math.h>
#include <float.h> // FLT_MAX

#if HAVE_SOPOLYGONOFFSET
#include <Inventor/nodes/SoPolygonOffset.h>
#endif // HAVE_SOPOLYGONOFFSET

// (note: this *must* be a #define, not a static variable -- to avoid
// initialization race conditions with the static variables being set
// to the value of this)
#define UNINITIALIZED_ENVVAR -1 // value of envvars before tested

// environment variables
static int COIN_SHOW_FPS_COUNTER = UNINITIALIZED_ENVVAR;

// *************************************************************************

#ifndef DOXYGEN_SKIP_THIS

// The private data for the SoQtViewer.
class SoQtViewerP {
public:
  SoQtViewerP(SoQtViewer * publ);
  ~SoQtViewerP(void);

  SoSeparator * createSuperScene(void);
  SoSeparator * createFPSSuperimposition(void);

  static void convertOrtho2Perspective(const SoOrthographicCamera * in,
                                       SoPerspectiveCamera * out);
  static void convertPerspective2Ortho(const SoPerspectiveCamera * in,
                                       SoOrthographicCamera * out);

  SoCamera * camera;
  SoQtViewer::Type type;
  SbBool viewingflag;
  SoGetBoundingBoxAction * autoclipbboxaction;
  SoSeparator * sceneroot;
  SoNode * scenegraph;


  // Seek functionality
  SoTimerSensor * seeksensor;
  float seekperiod;
  SbBool inseekmode;
  SbBool seektopoint;
  SbVec3f camerastartposition, cameraendposition;
  SbRotation camerastartorient, cameraendorient;
  float seekdistance;
  SbBool seekdistanceabs;

  // Camera handling
  SbBool deletecamera;

  // Home position storage.
  SoOrthographicCamera * storedortho;
  SoPerspectiveCamera * storedperspective;

  SoDirectionalLight * headlight;

  // Drawstyles
  SoQtViewer::DrawStyle drawstyles[2];
  SoSwitch * drawstyleroot, * hiddenlineroot, * polygonoffsetparent;
  SoBaseColor * sobasecolor;
  SoComplexity * socomplexity;
  SoDrawStyle * sodrawstyle;
  SoLightModel * solightmodel;
  SoMaterialBinding * somaterialbinding;
  SoSeparator * usersceneroot;
  SoSwitch * superimpositionroot;
#if HAVE_SOPOLYGONOFFSET
  SoPolygonOffset * sopolygonoffset;
#endif // HAVE_SOPOLYGONOFFSET
  // Automatic setting of clipping planes
  SbBool adjustclipplanes;

  SoSeparator * fpsRoot;

  // Keep track of the frames-per-second counter.
  // Const value trick for old compilers.
  enum Constants { FRAMESARRAY_SIZE = 100 };
  SbVec2f frames[FRAMESARRAY_SIZE];
  float totalcoin, totaldraw;
  double lastgettimeofday;
  int framecount;

  // Misc
  SoType cameratype;
  SbBool cursoron, localsetbuffertype;
  SoCallbackList * interactionstartCallbacks, * interactionendCallbacks;
  int interactionnesting;
  SoQtViewer::BufferType buffertype;
  SbBool stereoviewing;
  float stereooffset;
  SbColor wireframeoverlaycolor;

  void reallyRedraw(const SbBool clearcol, const SbBool clearz = TRUE);

  // Seek functionality
  static void seeksensorCB(void * data, SoSensor *);

  // Drawstyles
  void changeDrawStyle(SoQtViewer::DrawStyle style);
  SbBool drawInteractiveAsStill(void) const;
  SbBool drawAsHiddenLine(void) const;
  SbBool drawAsWireframeOverlay(void) const;
  SoQtViewer::DrawStyle currentDrawStyle(void) const;

  // Automatic setting of clipping planes
  void setClippingPlanes(void);

  // Methods to keep track of frames-per-second value.
  void resetFrameCounter(void);
  SbVec2f addFrametime(const double ft);
  void recordFPS(const double rendertime);

  // Misc
  static void interactivestartCB(void *, SoQtViewer * thisp);
  static void interactiveendCB(void *, SoQtViewer * thisp);
  void moveCameraScreen(const SbVec2f & screenpos);
  void getCameraCoordinateSystem(SoCamera * camera, SoNode * root,
                                 SbMatrix & matrix, SbMatrix & inverse);

  SoSearchAction * searchaction;
  SoGetMatrixAction * matrixaction;
  SbPList * superimpositions;
  SbGuiList<SbBool> superimpositionsenabled;
  SoQtViewer * pub;

  // auto clipping parameters
  SoQtViewer::AutoClippingStrategy autoclipstrategy;
  float autoclipvalue;
  SoQtAutoClippingCB * autoclipcb;
  void * autoclipuserdata;
};

#define PRIVATE(ptr) (ptr->pimpl)
#define PUBLIC(ptr) (ptr->pub)


SoQtViewerP::SoQtViewerP(SoQtViewer * publ)
{
  PUBLIC(this) = publ;
  this->searchaction = new SoSearchAction;
  this->matrixaction = new SoGetMatrixAction(SbViewportRegion(100,100));
  this->superimpositions = NULL;

  this->storedortho = new SoOrthographicCamera;
  this->storedortho->ref();
  this->storedperspective = new SoPerspectiveCamera;
  this->storedperspective->ref();

  // initialize auto clipping parameters
  this->autoclipstrategy = SoQtViewer::VARIABLE_NEAR_PLANE;
  this->autoclipvalue = 0.6f;
  this->autoclipcb = NULL;
}

SoQtViewerP::~SoQtViewerP()
{
  if (this->superimpositions) {
    delete this->superimpositions;
    this->superimpositions = NULL;
  }
  delete this->searchaction;
  delete this->matrixaction;

  this->storedortho->unref();
  this->storedperspective->unref();
}

SoSeparator *
SoQtViewerP::createSuperScene(void)
{
  static const char * superSceneGraph[] =
  {
    "#Inventor V2.1 ascii",
    "",
    "Separator {",
    "  renderCaching OFF",
    "  renderCulling OFF",
    "  pickCulling OFF",
    "  boundingBoxCaching OFF",
    
    // Headlight. By inserting this before any scenegraph camera, the
    // light will always be pointing in the correct direction.
    "  DEF soqt->headlight DirectionalLight {",
    "    direction 1 -1 -10",
    "  }",

    "  DEF soqt->drawstyleroot Switch {",
    "    whichChild -1",
    "    DEF soqt->lightmodel LightModel {",
    "      model BASE_COLOR",
    "    }",
    "    DEF soqt->drawstyle DrawStyle {",
    "      pointSize ~",
    "      lineWidth ~",
    "      linePattern ~",
    "    }",
    "    DEF soqt->complexity Complexity {",
    "      textureQuality 0.0",
    "      value 0.1",
    "    }",
    "  }",
    "  DEF soqt->hiddenlineroot Switch {",
    "    whichChild -1",
    "    DEF soqt->basecolor BaseColor { }",
    "    DEF soqt->materialbinding MaterialBinding {",
    "      value OVERALL",
    "    }",
    "    DEF soqt->polygonoffsetparent Switch {",
    "      whichChild -1",
#if HAVE_SOPOLYGONOFFSET
    "      DEF soqt->polygonoffset PolygonOffset { }",
#endif // HAVE_SOPOLYGONOFFSET
    "    }",
    "  }",
    "  DEF soqt->userscenegraphroot Separator {",
    "  }",
    "}",
    NULL
  };

  int i, bufsize;
  for (i = bufsize = 0; superSceneGraph[i]; i++)
    bufsize += strlen(superSceneGraph[i]) + 1;
  char * buf = new char [bufsize + 1];
  for (i = bufsize = 0; superSceneGraph[i]; i++) {
    strcpy(buf + bufsize, superSceneGraph[i]);
    bufsize += strlen(superSceneGraph[i]);
    buf[bufsize] = '\n';
    bufsize++;
  }
  SoInput * input = new SoInput;
  input->setBuffer(buf, bufsize);
  SoNode * root = NULL;
  SbBool ok = SoDB::read(input, root);
  delete input;
  delete [] buf;
  if (!ok) {
    SoDebugError::post("SoQtViewer::SoQtViewer",
		       "couldn't create viewer superscene");
    return NULL;
  }
  assert(root->isOfType(SoSeparator::getClassTypeId()));
  root->ref();

  this->searchaction->reset();
  this->searchaction->setSearchingAll(TRUE);
  this->searchaction->setInterest(SoSearchAction::FIRST);

#define LOCATE_NODE(member, type, name) \
  do { \
    member = NULL; \
    this->searchaction->setName(SbName(name)); \
    this->searchaction->apply(root); \
    if (this->searchaction->isFound()) { \
      SoNode * node = this->searchaction->getPath()->getTail(); \
      assert(node != NULL); \
      if (node->isOfType(type::getClassTypeId())) \
        member = (type *) node; \
    } else { \
      SoDebugError::post("SoQtViewerP::createSuperScene", \
                         "didn't locate node \"%s\"", name); \
    } \
  } while (FALSE)

  LOCATE_NODE(this->headlight, SoDirectionalLight, "soqt->headlight");
  LOCATE_NODE(this->drawstyleroot, SoSwitch, "soqt->drawstyleroot");
  LOCATE_NODE(this->hiddenlineroot, SoSwitch, "soqt->hiddenlineroot");
  LOCATE_NODE(this->polygonoffsetparent, SoSwitch,
	      "soqt->polygonoffsetparent");
  LOCATE_NODE(this->usersceneroot, SoSeparator, "soqt->userscenegraphroot");

  LOCATE_NODE(this->sobasecolor, SoBaseColor, "soqt->basecolor");
  LOCATE_NODE(this->socomplexity, SoComplexity, "soqt->complexity");
  LOCATE_NODE(this->sodrawstyle, SoDrawStyle, "soqt->drawstyle");
  LOCATE_NODE(this->solightmodel, SoLightModel, "soqt->lightmodel");
  LOCATE_NODE(this->somaterialbinding, SoMaterialBinding, "soqt->materialbinding");
  if (this->sobasecolor) this->sobasecolor->setOverride(TRUE);
  if (this->socomplexity) this->socomplexity->setOverride(TRUE);
  if (this->sodrawstyle) this->sodrawstyle->setOverride(TRUE);
  if (this->solightmodel) this->solightmodel->setOverride(TRUE);
  if (this->somaterialbinding) this->somaterialbinding->setOverride(TRUE);
#ifdef HAVE_SOPOLYGONOFFSET
  LOCATE_NODE(this->sopolygonoffset, SoPolygonOffset, "soqt->polygonoffset");
  if (this->sopolygonoffset) this->sopolygonoffset->setOverride(TRUE);
#endif // HAVE_SOPOLYGONOFFSET

#undef LOCATE_NODE
  this->searchaction->reset();

  root->unrefNoDelete();
  return (SoSeparator *) root;
}

// FIXME: this method is unnecessary robust, and generally
// weird. 20020522 mortene.
SoSeparator *
SoQtViewerP::createFPSSuperimposition(void)
{
  static const char * fpsSceneGraph[] =
  {
    "#Inventor V2.1 ascii",
    "",
    "Separator {",
    "  renderCaching OFF",
    "  renderCulling OFF",
    "  pickCulling OFF",
    "  boundingBoxCaching OFF",
    "}",
    NULL
  };

  int i, bufsize;
  for (i = bufsize = 0; fpsSceneGraph[i]; i++)
    bufsize += strlen(fpsSceneGraph[i]) + 1;
  char * buf = new char [bufsize + 1];
  for (i = bufsize = 0; fpsSceneGraph[i]; i++) {
    strcpy(buf + bufsize, fpsSceneGraph[i]);
    bufsize += strlen(fpsSceneGraph[i]);
    buf[bufsize] = '\n';
    bufsize++;
  }
  SoInput * input = new SoInput;
  input->setBuffer(buf, bufsize);
  SoNode * root = NULL;
  SbBool ok = SoDB::read(input, root);
  delete input;
  delete [] buf;
  if (!ok) {
    SoDebugError::post("SoQtViewer::SoQtViewer",
		       "couldn't create viewer superscene");
    return NULL;
  }
  assert(root->isOfType(SoSeparator::getClassTypeId()));
  root->ref();

  root->unrefNoDelete();
  return NULL;
  return (SoSeparator *) root;
}

// Returns the coordinate system the current camera is located in. If
// there are transformations before the camera in the scene graph,
// this must be considered before doing certain operations. \a matrix
// and \a inverse will not contain the transformations caused by the
// camera fields, only the transformations traversed before the camera
// in the scene graph.
void
SoQtViewerP::getCameraCoordinateSystem(SoCamera * camera,
                                          SoNode * root,
                                          SbMatrix & matrix,
                                          SbMatrix & inverse)
{
  this->searchaction->reset();
  this->searchaction->setSearchingAll(TRUE);
  this->searchaction->setInterest(SoSearchAction::FIRST);
  this->searchaction->setNode(camera);
  this->searchaction->apply(root);

  matrix = inverse = SbMatrix::identity();
  if (this->searchaction->getPath()) {
    this->matrixaction->apply(this->searchaction->getPath());
    matrix = this->matrixaction->getMatrix();
    inverse = this->matrixaction->getInverse();
  }
  this->searchaction->reset();
}

// FIXME: this code is really not applying the correct solution,
// trying to convert between SoOrthographic::height and
// SoPerspectiveCamera::heightAngle. What we should really try to do
// is:
//
//       * when going from orthocam -> perspectivecam: set the
//       heightAngle field to it's default value (45), and move
//       camera to a position where the scene/model would fill about
//       the same screenspace as it did in the orthocam
//
//       * when going from perspectivecam -> orthocam: keep the
//       current position, but tune the view-volume height so the
//       scene/model takes up about the same screenspace
//
// 20020522 mortene.

void
SoQtViewerP::convertOrtho2Perspective(const SoOrthographicCamera * in,
                                         SoPerspectiveCamera * out)
{
  out->aspectRatio.setValue(in->aspectRatio.getValue());
  out->focalDistance.setValue(in->focalDistance.getValue());
  out->orientation.setValue(in->orientation.getValue());
  out->position.setValue(in->position.getValue());
  out->viewportMapping.setValue(in->viewportMapping.getValue());
  
  float focaldist = in->focalDistance.getValue();

  // focalDistance==0.0f happens for empty scenes.
  if (focaldist != 0.0f) {
    out->heightAngle = 2.0f * atan(in->height.getValue() / 2.0f / focaldist);
  }
  else {
    // 45 is the default value of this field in SoPerspectiveCamera.
    out->heightAngle = M_PI / 4.0f;
  }
}

void
SoQtViewerP::convertPerspective2Ortho(const SoPerspectiveCamera * in,
                                         SoOrthographicCamera * out)
{
  out->aspectRatio.setValue(in->aspectRatio.getValue());
  out->focalDistance.setValue(in->focalDistance.getValue());
  out->orientation.setValue(in->orientation.getValue());
  out->position.setValue(in->position.getValue());
  out->viewportMapping.setValue(in->viewportMapping.getValue());

  float focaldist = in->focalDistance.getValue();

  out->height = 2 * focaldist * tan(in->heightAngle.getValue() / 2.0f);
}

void
SoQtViewerP::reallyRedraw(const SbBool clearcol, const SbBool clearz)
{
  // Recalculate near/far planes. Must be done in reallyRedraw() --
  // not actualRedraw() -- so the clipping planes are correct even
  // when rendering multiple times with different camera settings.
  if (PUBLIC(this)->isAutoClipping()) { this->setClippingPlanes(); }

  if (this->drawAsHiddenLine()) {

    // First pass: render as filled, but with the background color.

    this->solightmodel->model.setIgnored(FALSE); // override as SoLightModel::BASE
    this->sodrawstyle->style.setIgnored(TRUE); // draw as-is filled/lines/points
    this->socomplexity->type.setIgnored(TRUE); // as-is rendering space
    this->socomplexity->value.setIgnored(TRUE); // as-is complexity on non-simple shapes
    // textureQuality field of socomplexity node is always 0.0

    this->sobasecolor->rgb.setValue(PUBLIC(this)->getBackgroundColor());
    this->sobasecolor->rgb.setIgnored(FALSE);
    this->somaterialbinding->value.setIgnored(FALSE); // override with OVERALL
    this->polygonoffsetparent->whichChild = SO_SWITCH_ALL;

    PUBLIC(this)->getSceneManager()->render(clearcol, clearz);

    // Second pass, render wireframe on top.

    this->sodrawstyle->style = SoDrawStyle::LINES;
    this->sodrawstyle->style.setIgnored(FALSE); // force lines
    this->sobasecolor->rgb.setIgnored(TRUE); // use as-is line colors
    this->somaterialbinding->value.setIgnored(TRUE); // as-is
    this->polygonoffsetparent->whichChild = SO_SWITCH_NONE;

    PUBLIC(this)->getSceneManager()->render(FALSE, FALSE);

    return;
  }
  if (this->drawAsWireframeOverlay()) {
    // First pass: render as-is, with polygon offset

    this->solightmodel->model.setIgnored(TRUE);
    this->somaterialbinding->value.setIgnored(TRUE);
    this->sobasecolor->rgb.setIgnored(TRUE);
    this->sodrawstyle->style.setIgnored(TRUE); // draw as-is filled/lines/points
    this->socomplexity->type.setIgnored(TRUE); // as-is rendering space
    this->socomplexity->value.setIgnored(TRUE); // as-is complexity on non-simple shapes
    this->socomplexity->textureQuality.setIgnored(TRUE);

    this->somaterialbinding->value.setIgnored(TRUE); // override with OVERALL
    this->polygonoffsetparent->whichChild = SO_SWITCH_ALL;

    PUBLIC(this)->getSceneManager()->render(clearcol, clearz);

    // Second pass, render wireframe on top.
    this->sobasecolor->rgb.setValue(this->wireframeoverlaycolor);
    this->sobasecolor->rgb.setIgnored(FALSE);
    this->somaterialbinding->value.setIgnored(FALSE); // override with OVERALL

    this->solightmodel->model.setIgnored(FALSE); // override as SoLightModel::BASE
    this->sodrawstyle->style = SoDrawStyle::LINES;
    this->sodrawstyle->style.setIgnored(FALSE); // force lines
    this->polygonoffsetparent->whichChild = SO_SWITCH_NONE;
    this->socomplexity->textureQuality.setIgnored(FALSE);

    PUBLIC(this)->getSceneManager()->render(FALSE, FALSE);

    SbBool oldnot;

    // disable override nodes
    (void) this->sobasecolor->rgb.enableNotify(FALSE);
    this->sobasecolor->rgb.setIgnored(TRUE);
    (void) this->sobasecolor->rgb.enableNotify(TRUE);

    (void) this->somaterialbinding->value.enableNotify(FALSE);
    this->somaterialbinding->value.setIgnored(TRUE);
    (void) this->somaterialbinding->value.enableNotify(TRUE);

    (void) this->solightmodel->model.enableNotify(FALSE);
    this->solightmodel->model.setIgnored(TRUE);
    (void) this->solightmodel->model.enableNotify(TRUE);

    (void) this->socomplexity->textureQuality.enableNotify(FALSE);
    this->socomplexity->textureQuality.setIgnored(TRUE);
    (void) this->socomplexity->textureQuality.enableNotify(TRUE);

    (void) this->sodrawstyle->style.enableNotify(FALSE);
    this->sodrawstyle->style.setIgnored(TRUE);
    (void) this->sodrawstyle->style.enableNotify(TRUE);
    return;
  }

  SbBool clearzbuffer = TRUE;
  SoQtViewer::DrawStyle style = this->currentDrawStyle();
  switch (style) {
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_LOW_RES_POINT:
  case SoQtViewer::VIEW_BBOX:
    clearzbuffer = FALSE;
  default:
    break; // Include "default:" case to avoid compiler warning.
  }

  PUBLIC(this)->getSceneManager()->render(clearcol, clearzbuffer && clearz);
}


// *************************************************************************

// Returns a boolean to indicate if the dynamic drawstyle equals
// the static drawstyle.

SbBool
SoQtViewerP::drawInteractiveAsStill(void) const
{
  SbBool moveasstill = this->drawstyles[SoQtViewer::INTERACTIVE] == SoQtViewer::VIEW_SAME_AS_STILL;
  if (! moveasstill)
    moveasstill = this->drawstyles[SoQtViewer::INTERACTIVE] == this->drawstyles[SoQtViewer::STILL];
  if (! moveasstill)
    moveasstill =
      this->drawstyles[SoQtViewer::INTERACTIVE] == SoQtViewer::VIEW_NO_TEXTURE &&
      this->drawstyles[SoQtViewer::STILL] != SoQtViewer::VIEW_AS_IS;
  return moveasstill;
}

// Returns the current drawing style.
SoQtViewer::DrawStyle
SoQtViewerP::currentDrawStyle(void) const
{
  SbBool interactivemode = PUBLIC(this)->getInteractiveCount() > 0 ? TRUE : FALSE;

  if (!interactivemode || this->drawInteractiveAsStill())
    return this->drawstyles[SoQtViewer::STILL];
  else
    return this->drawstyles[SoQtViewer::INTERACTIVE];
}

// Returns a boolean to indicate if the current drawstyle settings implies
// hidden line rendering.
SbBool
SoQtViewerP::drawAsHiddenLine(void) const
{
  return ((this->currentDrawStyle() == SoQtViewer::VIEW_HIDDEN_LINE) ? TRUE : FALSE);
}

// Returns a boolean to indicate if the current drawstyle settings
// implies wirefram overlay rendering.
SbBool
SoQtViewerP::drawAsWireframeOverlay(void) const
{
  return ((this->currentDrawStyle() == SoQtViewer::VIEW_WIREFRAME_OVERLAY) ? TRUE : FALSE);
}

// Use the given style setting to set the correct states in the
// rendering control nodes. This will affect the way the scene is
// currently rendered.
void
SoQtViewerP::changeDrawStyle(SoQtViewer::DrawStyle style)
{
  // Turn on/off Z-buffering based on the style setting.
  switch (style) {
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_LOW_RES_POINT:
  case SoQtViewer::VIEW_BBOX:
    PUBLIC(this)->glLockNormal();
    // FIXME: shouldn't this be done "lazy", i.e. before we do any
    // actual rendering? 20001126 mortene.
    glDisable(GL_DEPTH_TEST);
    PUBLIC(this)->glUnlockNormal();
    break;

  default:
    PUBLIC(this)->glLockNormal();
    // FIXME: shouldn't this be done "lazy", i.e. before we do any
    // actual rendering? 20001126 mortene.
    glEnable(GL_DEPTH_TEST);
    PUBLIC(this)->glUnlockNormal();
    break;
  }

  // Render everything as its supposed to be done, don't override
  // any of the settings in the ``real'' graph.
  if (style == SoQtViewer::VIEW_AS_IS) {
    this->drawstyleroot->whichChild = SO_SWITCH_NONE;
    return;
  }

  this->drawstyleroot->whichChild = SO_SWITCH_ALL;
  if ((style == SoQtViewer::VIEW_HIDDEN_LINE) ||
      (style == SoQtViewer::VIEW_WIREFRAME_OVERLAY)) {
    this->hiddenlineroot->whichChild = SO_SWITCH_ALL;
    return;
  } else {
    this->hiddenlineroot->whichChild = SO_SWITCH_NONE;
  }

  // Set or unset lightmodel override.
  switch (style) {
  case SoQtViewer::VIEW_NO_TEXTURE:
  case SoQtViewer::VIEW_LOW_COMPLEXITY:
    this->solightmodel->model.setIgnored(TRUE); // as-is BASE or PHONG
    break;

  case SoQtViewer::VIEW_LINE:
  case SoQtViewer::VIEW_POINT:
  case SoQtViewer::VIEW_BBOX:
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_LOW_RES_POINT:
    this->solightmodel->model.setIgnored(FALSE); // force BASE lighting
    break;

  default:
    assert(FALSE); break;
  }


  // Set or unset drawstyle override.
  switch (style) {
  case SoQtViewer::VIEW_NO_TEXTURE:
  case SoQtViewer::VIEW_LOW_COMPLEXITY:
    this->sodrawstyle->style.setIgnored(TRUE); // as-is drawing style filled/lines/points
    break;

  case SoQtViewer::VIEW_LINE:
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_BBOX:
    this->sodrawstyle->style = SoDrawStyle::LINES;
    this->sodrawstyle->style.setIgnored(FALSE); // force line rendering
    break;

  case SoQtViewer::VIEW_POINT:
  case SoQtViewer::VIEW_LOW_RES_POINT:
    this->sodrawstyle->style = SoDrawStyle::POINTS;
    this->sodrawstyle->style.setIgnored(FALSE); // force point rendering
    break;

  default:
    assert(FALSE); break;
  }

  // Set or unset complexity value override.
  switch (style) {
  case SoQtViewer::VIEW_NO_TEXTURE:
  case SoQtViewer::VIEW_LINE:
  case SoQtViewer::VIEW_POINT:
  case SoQtViewer::VIEW_BBOX:
    this->socomplexity->value.setIgnored(TRUE); // as-is complexity
    break;

  case SoQtViewer::VIEW_LOW_COMPLEXITY:
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_LOW_RES_POINT:
    this->socomplexity->value.setIgnored(FALSE); // force complexity setting of 0.1
    break;

  default:
    assert(FALSE); break;
  }

  // Set or unset complexity textureQuality override (the value of the
  // override-field is always 0.0, ie signalling "textures off").
  switch (style) {
  case SoQtViewer::VIEW_HIDDEN_LINE:
  case SoQtViewer::VIEW_NO_TEXTURE:
  case SoQtViewer::VIEW_LINE:
  case SoQtViewer::VIEW_POINT:
  case SoQtViewer::VIEW_BBOX:
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_LOW_RES_POINT:
    this->socomplexity->textureQuality.setIgnored(FALSE); // textures off
    break;

  default:
    this->socomplexity->textureQuality.setIgnored(TRUE); // don't override
    break;
  }

  // Set or unset complexity type override.
  switch (style) {
  case SoQtViewer::VIEW_NO_TEXTURE:
  case SoQtViewer::VIEW_LOW_COMPLEXITY:
  case SoQtViewer::VIEW_LINE:
  case SoQtViewer::VIEW_POINT:
  case SoQtViewer::VIEW_LOW_RES_LINE:
  case SoQtViewer::VIEW_LOW_RES_POINT:
    this->socomplexity->type.setIgnored(TRUE); // as-is
    break;

  case SoQtViewer::VIEW_BBOX:
    this->socomplexity->type = SoComplexity::BOUNDING_BOX;
    this->socomplexity->type.setIgnored(FALSE); // force bounding box rendering
    break;

  default:
    assert(FALSE); break;
  }

#if 0 // debug
  SoDebugError::postInfo("SoQtViewer::changeDrawStyle",
                         "\n"
                         "\tdrawstyle style: 0x%02x (isIgnored() == %s)\n"
                         "\tlightmodel model: 0x%02x, (isIgnored() == %s)\n"
                         "\tcomplexity type: 0x%02x, (isIgnored() == %s)\n"
                         "\tcomplexity value: %f, (isIgnored() == %s)\n"
                         "",
                         this->sodrawstyle->style.getValue(),
                         this->sodrawstyle->style.isIgnored() ? "T" : "F",
                         this->solightmodel->model.getValue(),
                         this->solightmodel->model.isIgnored() ? "T" : "F",
                         this->socomplexity->type.getValue(),
                         this->socomplexity->type.isIgnored() ? "T" : "F",
                         this->socomplexity->value.getValue(),
                         this->socomplexity->value.isIgnored() ? "T" : "F");
#endif // debug
}

// Position the near and far clipping planes just in front of and
// behind the scene's bounding box. This will give us the optimal
// utilization of the z buffer resolution by shrinking it to its
// minimum depth.
//
// Near and far clipping planes are specified in the camera fields
// nearDistance and farDistance.
void
SoQtViewerP::setClippingPlanes(void)
{
  // This is necessary to avoid a crash in case there is no scene
  // graph specified by the user.
  if (this->camera == NULL) return;

  if (this->autoclipbboxaction == NULL)
    this->autoclipbboxaction =
      new SoGetBoundingBoxAction(PUBLIC(this)->getViewportRegion());
  else
    this->autoclipbboxaction->setViewportRegion(PUBLIC(this)->getViewportRegion());

  this->autoclipbboxaction->apply(this->sceneroot);

  SbXfBox3f xbox = this->autoclipbboxaction->getXfBoundingBox();

  SbMatrix cammat;
  SbMatrix inverse;
  this->getCameraCoordinateSystem(this->camera, this->sceneroot, cammat, inverse);
  xbox.transform(inverse);

  SbMatrix mat;
  mat.setTranslate(- this->camera->position.getValue());
  xbox.transform(mat);
  mat = this->camera->orientation.getValue().inverse();
  xbox.transform(mat);
  SbBox3f box = xbox.project();

  // Bounding box was calculated in camera space, so we need to "flip"
  // the box (because camera is pointing in the (0,0,-1) direction
  // from origo.
  float nearval = -box.getMax()[2];
  float farval = -box.getMin()[2];

  // FIXME: what if we have a weird scale transform in the scenegraph?
  // Could we end up with nearval > farval then? Investigate, then
  // either use an assert() (if it can't happen) or an SoQtSwap()
  // (to handle it). 20020116 mortene.

  // Check if scene is completely behind us.
  if (farval <= 0.0f) { return; }

  if (this->camera->isOfType(SoPerspectiveCamera::getClassTypeId())) {
    // Disallow negative and small near clipping plane distance.
    
    float nearlimit; // the smallest value allowed for nearval
    if (this->autoclipstrategy == SoQtViewer::CONSTANT_NEAR_PLANE) {
      nearlimit = this->autoclipvalue;
    }
    else {
      assert(this->autoclipstrategy == SoQtViewer::VARIABLE_NEAR_PLANE);
      // From glFrustum() documentation: Depth-buffer precision is
      // affected by the values specified for znear and zfar. The
      // greater the ratio of zfar to znear is, the less effective the
      // depth buffer will be at distinguishing between surfaces that
      // are near each other. If r = far/near, roughly log (2) r bits
      // of depth buffer precision are lost. Because r approaches
      // infinity as znear approaches zero, you should never set znear
      // to zero.

      GLint depthbits[1];
      glGetIntegerv(GL_DEPTH_BITS, depthbits);
      
      int use_bits = (int) (float(depthbits[0]) * (1.0f-this->autoclipvalue)); 
      float r = (float) pow(2.0, (double) use_bits);
      nearlimit = farval / r;
    }
    
    if (nearlimit >= farval) {
      // (The "5000" magic constant was found by fiddling around a bit
      // on an OpenGL implementation with a 16-bit depth-buffer
      // resolution, adjusting to find something that would work well
      // with both a very "stretched" / deep scene and a more compact
      // single-model one.)
      nearlimit = farval / 5000.0f;
    }
    
    // adjust the near plane if the the value is too small.
    if (nearval < nearlimit) {
      nearval = nearlimit;
    }
    
    if (this->autoclipcb) {
      SbVec2f nearfar;
      nearfar[0] = nearval;
      nearfar[1] = farval;
      
      nearfar = this->autoclipcb(this->autoclipuserdata, nearfar);
      
      nearval = nearfar[0];
      farval = nearfar[1]; 
    }
  }

  // Some slack around the bounding box, in case the scene fits
  // exactly inside it. This is done to minimize the chance of
  // artifacts caused by the limitation of the z-buffer
  // resolution. One common artifact if this is not done is that the
  // near clipping plane cuts into the corners of the model as it's
  // rotated.
  const float SLACK = 0.001f;

  this->camera->nearDistance = nearval * (1.0f - SLACK);
  this->camera->farDistance = farval * (1.0f + SLACK);


  // FIXME: there's a possible optimization to take advantage of here,
  // since we are able to sometimes know for sure that all geometry is
  // completely inside the view volume. I quote from the "OpenGL FAQ
  // and Troubleshooting Guide":
  //
  //  "10.050 I know my geometry is inside the view volume. How can I
  //  turn off OpenGL's view-volume clipping to maximize performance?
  //
  //   Standard OpenGL doesn't provide a mechanism to disable the
  //   view-volume clipping test; thus, it will occur for every
  //   primitive you send.
  //
  //   Some implementations of OpenGL support the
  //   GL_EXT_clip_volume_hint extension. If the extension is
  //   available, a call to
  //   glHint(GL_CLIP_VOLUME_CLIPPING_HINT_EXT,GL_FASTEST) will inform
  //   OpenGL that the geometry is entirely within the view volume and
  //   that view-volume clipping is unnecessary. Normal clipping can
  //   be resumed by setting this hint to GL_DONT_CARE. When clipping
  //   is disabled with this hint, results are undefined if geometry
  //   actually falls outside the view volume."
  //
  // 20020117 mortene.


  if (SOQT_DEBUG && 0) { // debug
    SoDebugError::postInfo("SoQtViewer::setClippingPlanes",
                           "near, far: %f (%f), %f (%f)",
                           nearval, this->camera->nearDistance.getValue(),
                           farval, this->camera->farDistance.getValue());
  }
}

// Translate camera a distance equal to the difference in projected,
// normalized screen coordinates given by the argument.
void
SoQtViewerP::moveCameraScreen(const SbVec2f & screenpos)
{
  SoCamera * cam = PUBLIC(this)->getCamera();
  assert(cam);

  if (SOQT_DEBUG && 0) { // debug
    SoDebugError::postInfo("SoQtViewer::moveCameraScreen",
                           "screenpos: <%f, %f>, campos: <%f, %f, %f>",
                           screenpos[0], screenpos[1],
                           cam->position.getValue()[0],
                           cam->position.getValue()[1],
                           cam->position.getValue()[2]);
  }

  SbViewVolume vv = cam->getViewVolume(PUBLIC(this)->getGLAspectRatio());
  SbPlane panplane = vv.getPlane(cam->focalDistance.getValue());

  SbLine line;
  vv.projectPointToLine(screenpos + SbVec2f(0.5, 0.5f), line);
  SbVec3f current_planept;
  panplane.intersect(line, current_planept);
  vv.projectPointToLine(SbVec2f(0.5f, 0.5f), line);
  SbVec3f old_planept;
  panplane.intersect(line, old_planept);

  // Reposition camera according to the vector difference between the
  // projected points.
  cam->position = cam->position.getValue() - (current_planept - old_planept);

  if (SOQT_DEBUG && 0) { // debug
    SoDebugError::postInfo("SoQtViewer::moveCameraScreen",
                           "newcampos: <%f, %f, %f>",
                           cam->position.getValue()[0],
                           cam->position.getValue()[1],
                           cam->position.getValue()[2]);
  }
}

// Called when viewer enters interactive mode (animation, drag, ...).
void
SoQtViewerP::interactivestartCB(void *, SoQtViewer * thisp)
{
  // In interactive buffer mode, doublebuffering is used during interaction.
  if (PRIVATE(thisp)->buffertype == SoQtViewer::BUFFER_INTERACTIVE) {
    PRIVATE(thisp)->localsetbuffertype = TRUE;
    thisp->SoQtRenderArea::setDoubleBuffer(TRUE);
    PRIVATE(thisp)->localsetbuffertype = FALSE;
  }

  // Use the dynamic drawstyle.
  if (!PRIVATE(thisp)->drawInteractiveAsStill())
    PRIVATE(thisp)->changeDrawStyle(PRIVATE(thisp)->drawstyles[SoQtViewer::INTERACTIVE]);
}

// Called when viewer goes out of interactive mode and into "frozen"
// mode.
void
SoQtViewerP::interactiveendCB(void *, SoQtViewer * thisp)
{
  // In interactive buffer mode, doublebuffering is used during
  // interaction, singelbuffering while the camera is static.
  if (PRIVATE(thisp)->buffertype == SoQtViewer::BUFFER_INTERACTIVE) {
    PRIVATE(thisp)->localsetbuffertype = TRUE;
    thisp->SoQtRenderArea::setDoubleBuffer(FALSE);
    PRIVATE(thisp)->localsetbuffertype = FALSE;
  }

  // Back to static drawstyle.
  if (!PRIVATE(thisp)->drawInteractiveAsStill())
    PRIVATE(thisp)->changeDrawStyle(PRIVATE(thisp)->drawstyles[SoQtViewer::STILL]);
}

// Called repeatedly during the seek animation.
void
SoQtViewerP::seeksensorCB(void * data, SoSensor * s)
{
  SbTime currenttime = SbTime::getTimeOfDay();

  SoQtViewer * thisp = (SoQtViewer *)data;
  SoTimerSensor * sensor = (SoTimerSensor *)s;

  float t =
    (currenttime - sensor->getBaseTime()).getValue() / PRIVATE(thisp)->seekperiod;
  if ((t > 1.0f) || (t + sensor->getInterval().getValue() > 1.0f)) t = 1.0f;
  SbBool end = (t == 1.0f);
  
  t = (float) ((1.0 - cos(M_PI*t)) * 0.5);
  
  PRIVATE(thisp)->camera->position = PRIVATE(thisp)->camerastartposition +
    (PRIVATE(thisp)->cameraendposition - PRIVATE(thisp)->camerastartposition) * t;
  PRIVATE(thisp)->camera->orientation = 
    SbRotation::slerp(PRIVATE(thisp)->camerastartorient,
                      PRIVATE(thisp)->cameraendorient, 
                      t);

  if (end) thisp->setSeekMode(FALSE);
}

// Reset the frames-per-second counter upon window resize events,
// abnormal delays, etc.
//
// The methods for recording FPS values are Coin extensions, not
// available in the original Open Inventor API.
//
// \sa addFrametime(), recordFPS()
void
SoQtViewerP::resetFrameCounter(void)
{
  this->framecount = 0;
  for (int i = 0; i < SoQtViewerP::FRAMESARRAY_SIZE; i++)
    this->frames[i] = SbVec2f(0.0f, 0.0f);
  this->totalcoin = 0.0f;
  this->totaldraw = 0.0f;
  this->lastgettimeofday = SbTime::getTimeOfDay().getValue();
}

// Adds the time spent drawing the last frame to the array of past
// frame times. Returns the current averaged fps-value.
//
// The methods for recording FPS values are Coin extensions, not
// available in the original Open Inventor API.
//
// \sa resetFrameCounter(), recordFPS()
SbVec2f
SoQtViewerP::addFrametime(const double ft)
{
  this->framecount++;

  int arrayptr = (this->framecount - 1) % FRAMESARRAY_SIZE;

  this->totalcoin += (ft - this->frames[arrayptr][0]);
  float coinfps =
    this->totalcoin / SoQtMin(this->framecount, (int) FRAMESARRAY_SIZE);

  double timeofday = SbTime::getTimeOfDay().getValue();
  double ct = timeofday - this->lastgettimeofday;
  this->totaldraw += (ct - this->frames[arrayptr][1]);
  float drawfps =
    this->totaldraw / SoQtMin(this->framecount, (int) FRAMESARRAY_SIZE);

  this->frames[arrayptr] = SbVec2f(ft, ct);
  this->lastgettimeofday = timeofday;

  return SbVec2f(1.0f / coinfps, 1.0f / drawfps);
}

static unsigned char fps2dfont[][12] = {
  {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 }, //
  {  0,  0, 12, 12,  0,  8, 12, 12, 12, 12, 12,  0 }, // !
  {  0,  0,  0,  0,  0,  0,  0,  0,  0, 20, 20, 20 }, // \"
  {  0,  0, 18, 18, 18, 63, 18, 18, 63, 18, 18,  0 }, // #
  {  0,  8, 28, 42, 10, 10, 12, 24, 40, 42, 28,  8 }, // $
  {  0,  0,  6, 73, 41, 22,  8, 52, 74, 73, 48,  0 }, // %
  {  0, 12, 18, 18, 12, 25, 37, 34, 34, 29,  0,  0 }, // &
  { 12, 12, 24,  0,  0,  0,  0,  0,  0,  0,  0,  0 }, // '
  {  0,  6,  8,  8, 16, 16, 16, 16, 16,  8,  8,  6 }, // (
  {  0, 48,  8,  8,  4,  4,  4,  4,  4,  8,  8, 48 }, //)
  {  0,  0,  0,  0,  0,  0,  8, 42, 20, 42,  8,  0 }, // *
  {  0,  0,  0,  8,  8,  8,127,  8,  8,  8,  0,  0 }, // +
  {  0, 24, 12, 12,  0,  0,  0,  0,  0,  0,  0,  0 }, // ,
  {  0,  0,  0,  0,  0,  0,127,  0,  0,  0,  0,  0 }, // -
  {  0,  0, 24, 24,  0,  0,  0,  0,  0,  0,  0,  0 }, // .
  {  0, 32, 32, 16, 16,  8,  8,  8,  4,  4,  2,  2 }, // /
  {  0,  0, 28, 34, 34, 34, 34, 34, 34, 34, 28,  0 }, // 0
  {  0,  0,  8,  8,  8,  8,  8,  8, 40, 24,  8,  0 }, // 1
  {  0,  0, 62, 32, 16,  8,  4,  2,  2, 34, 28,  0 }, // 2
  {  0,  0, 28, 34,  2,  2, 12,  2,  2, 34, 28,  0 }, // 3
  {  0,  0,  4,  4,  4,126, 68, 36, 20, 12,  4,  0 }, // 4
  {  0,  0, 28, 34,  2,  2,  2, 60, 32, 32, 62,  0 }, // 5
  {  0,  0, 28, 34, 34, 34, 60, 32, 32, 34, 28,  0 }, // 6
  {  0,  0, 16, 16, 16,  8,  8,  4,  2,  2, 62,  0 }, // 7
  {  0,  0, 28, 34, 34, 34, 28, 34, 34, 34, 28,  0 }, // 8
  {  0,  0, 28, 34,  2,  2, 30, 34, 34, 34, 28,  0 }, // 9
  {  0,  0, 24, 24,  0,  0,  0, 24, 24,  0,  0,  0 }, // :
  {  0, 48, 24, 24,  0,  0,  0, 24, 24,  0,  0,  0 }, // ;
  {  0,  0,  0,  2,  4,  8, 16,  8,  4,  2,  0,  0 }, // <
  {  0,  0,  0,  0,  0,127,  0,127,  0,  0,  0,  0 }, // =
  {  0,  0,  0, 16,  8,  4,  2,  4,  8, 16,  0,  0 }, // >
  {  0,  0, 16, 16,  0, 16, 28,  2,  2,  2, 60,  0 }, // ?
  {  0,  0, 28, 32, 73, 86, 82, 82, 78, 34, 28,  0 }, // @
  {  0,  0, 33, 33, 33, 63, 18, 18, 18, 12, 12,  0 }, // A
  {  0,  0, 60, 34, 34, 34, 60, 34, 34, 34, 60,  0 }, // B
  {  0,  0, 14, 16, 32, 32, 32, 32, 32, 18, 14,  0 }, // C
  {  0,  0, 56, 36, 34, 34, 34, 34, 34, 36, 56,  0 }, // D
  {  0,  0, 62, 32, 32, 32, 60, 32, 32, 32, 62,  0 }, // E
  {  0,  0, 16, 16, 16, 16, 30, 16, 16, 16, 30,  0 }, // F
  {  0,  0, 14, 18, 34, 34, 32, 32, 32, 18, 14,  0 }, // G
  {  0,  0, 34, 34, 34, 34, 62, 34, 34, 34, 34,  0 }, // H
  {  0,  0, 62,  8,  8,  8,  8,  8,  8,  8, 62,  0 }, // I
  {  0,  0,112,  8,  8,  8,  8,  8,  8,  8, 62,  0 }, // J
  {  0,  0, 33, 33, 34, 36, 56, 40, 36, 34, 33,  0 }, // K
  {  0,  0, 30, 16, 16, 16, 16, 16, 16, 16, 16,  0 }, // L
  {  0,  0, 33, 33, 33, 45, 45, 45, 51, 51, 33,  0 }, // M
  {  0,  0, 34, 34, 38, 38, 42, 42, 50, 50, 34,  0 }, // N
  {  0,  0, 12, 18, 33, 33, 33, 33, 33, 18, 12,  0 }, // O
  {  0,  0, 32, 32, 32, 60, 34, 34, 34, 34, 60,  0 }, // P
  {  3,  6, 12, 18, 33, 33, 33, 33, 33, 18, 12,  0 }, // Q
  {  0,  0, 34, 34, 34, 36, 60, 34, 34, 34, 60,  0 }, // R
  {  0,  0, 60,  2,  2,  6, 28, 48, 32, 32, 30,  0 }, // S
  {  0,  0,  8,  8,  8,  8,  8,  8,  8,  8,127,  0 }, // T
  {  0,  0, 28, 34, 34, 34, 34, 34, 34, 34, 34,  0 }, // U
  {  0,  0, 12, 12, 18, 18, 18, 33, 33, 33, 33,  0 }, // V
  {  0,  0, 34, 34, 34, 54, 85, 73, 73, 73, 65,  0 }, // W
  {  0,  0, 34, 34, 20, 20,  8, 20, 20, 34, 34,  0 }, // X
  {  0,  0,  8,  8,  8,  8, 20, 20, 34, 34, 34,  0 }, // Y
  {  0,  0, 62, 32, 16, 16,  8,  4,  4,  2, 62,  0 }, // Z
  {  0, 14,  8,  8,  8,  8,  8,  8,  8,  8,  8, 14 }, // [
  {  0,  2,  2,  4,  4,  8,  8,  8, 16, 16, 32, 32 }, // [backslash]
  {  0, 56,  8,  8,  8,  8,  8,  8,  8,  8,  8, 56 }, // ]
  {  0,  0,  0,  0,  0, 34, 34, 20, 20,  8,  8,  0 }, // ^
  {  0,127,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 }, // _
  {  0,  0,  0,  0,  0,  0,  0,  0,  0, 24, 24, 12 }, // `
  {  0,  0, 29, 34, 34, 30,  2, 34, 28,  0,  0,  0 }, // a
  {  0,  0, 60, 34, 34, 34, 34, 50, 44, 32, 32, 32 }, // b
  {  0,  0, 14, 16, 32, 32, 32, 16, 14,  0,  0,  0 }, // c
  {  0,  0, 26, 38, 34, 34, 34, 34, 30,  2,  2,  2 }, // d
  {  0,  0, 28, 34, 32, 62, 34, 34, 28,  0,  0,  0 }, // e
  {  0,  0, 16, 16, 16, 16, 16, 16, 62, 16, 16, 14 }, // f
  { 28,  2,  2, 26, 38, 34, 34, 34, 30,  0,  0,  0 }, // g
  {  0,  0, 34, 34, 34, 34, 34, 50, 44, 32, 32, 32 }, // h
  {  0,  0,  8,  8,  8,  8,  8,  8, 56,  0,  8,  8 }, // i
  { 56,  4,  4,  4,  4,  4,  4,  4, 60,  0,  4,  4 }, // j
  {  0,  0, 33, 34, 36, 56, 40, 36, 34, 32, 32, 32 }, // k
  {  0,  0,  8,  8,  8,  8,  8,  8,  8,  8,  8, 56 }, // l
  {  0,  0, 73, 73, 73, 73, 73,109, 82,  0,  0,  0 }, // m
  {  0,  0, 34, 34, 34, 34, 34, 50, 44,  0,  0,  0 }, // n
  {  0,  0, 28, 34, 34, 34, 34, 34, 28,  0,  0,  0 }, // o
  { 32, 32, 60, 34, 34, 34, 34, 50, 44,  0,  0,  0 }, // p
  {  2,  2, 26, 38, 34, 34, 34, 34, 30,  0,  0,  0 }, // q
  {  0,  0, 16, 16, 16, 16, 16, 24, 22,  0,  0,  0 }, // r
  {  0,  0, 60,  2,  2, 28, 32, 32, 30,  0,  0,  0 }, // s
  {  0,  0, 14, 16, 16, 16, 16, 16, 62, 16, 16,  0 }, // t
  {  0,  0, 26, 38, 34, 34, 34, 34, 34,  0,  0,  0 }, // u
  {  0,  0,  8,  8, 20, 20, 34, 34, 34,  0,  0,  0 }, // v
  {  0,  0, 34, 34, 34, 85, 73, 73, 65,  0,  0,  0 }, // w
  {  0,  0, 34, 34, 20,  8, 20, 34, 34,  0,  0,  0 }, // x
  { 48, 16,  8,  8, 20, 20, 34, 34, 34,  0,  0,  0 }, // y
  {  0,  0, 62, 32, 16,  8,  4,  2, 62,  0,  0,  0 }, // z
  {  0,  6,  8,  8,  8,  4, 24,  4,  8,  8,  8,  6 }, // {
  {  0,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8 }, // |
  {  0, 48,  8,  8,  8, 16, 12, 16,  8,  8,  8, 48 }, // }
  {  0,  0,  0,  0,  0,  0, 78, 57,  0,  0,  0,  0 }  // ~
};

static void
printString(const char * s)
{
  int i,n;
  n = strlen(s);
  for (i = 0; i < n; i++)
    glBitmap(8, 12, 0.0, 2.0, 10.0, 0.0, fps2dfont[s[i] - 32]);
}

static void
Draw2DString(const char * str, SbVec2s glsize, SbVec2f position)
{
  // Store GL state.
  glPushAttrib(GL_ALL_ATTRIB_BITS);

  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_TEXTURE_2D);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  glOrtho(0.0, glsize[0], 0.0, glsize[1], -1, 1);

  glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

  glColor3f(0.0, 0.0, 0.0);
  glRasterPos2f(position[0] + 1, position[1]);
  printString(str);
  glRasterPos2f(position[0] - 1, position[1]);
  printString(str);
  glRasterPos2f(position[0], position[1] + 1);
  printString(str);
  glRasterPos2f(position[0], position[1] - 1);
  printString(str);

  glColor3f(1.0, 1.0, 0.0);
  glRasterPos2f(position[0], position[1]);
  printString(str);

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();

  glPopAttrib();
}


// FIXME: the following is just a temporary hack to enable the FPS
// counter. We should really write a proper interface against it, so
// applications can set up feedback loops to control scene complexity
// and get a nice and steady maximum framerate, for instance.
//
// For anyone who want to execute that task, check what TGS has done
// first. If their API is fine, use the same approach.
//
// 20001124 mortene.

// Draw a text string showing the current frame-per-seconds value in
// the lower left corner of the OpenGL canvas (after recording
// information needed to calculate the fps).
//
// The methods for recording FPS values are Coin extensions, not
// available in the original Open Inventor API.
//
// \sa resetFrameCounter(), addFrametime()
void
SoQtViewerP::recordFPS(const double rendertime)
{
  if (COIN_SHOW_FPS_COUNTER == UNINITIALIZED_ENVVAR) {
    const char * env = SoAny::si()->getenv("COIN_SHOW_FPS_COUNTER");
    COIN_SHOW_FPS_COUNTER = env ? atoi(env) : 0;
  }

  if (COIN_SHOW_FPS_COUNTER) {
    SbVec2f fps = this->addFrametime(rendertime);

    char buffer[64];
    int nr = sprintf(buffer, "%.1f/%.1f fps", fps[0], fps[1]);
    assert(nr < 64);
    Draw2DString(buffer, PUBLIC(this)->getGLSize(), SbVec2f(10, 10));
  }
}

#endif // DOXYGEN_SKIP_THIS

// *************************************************************************

SOQT_OBJECT_ABSTRACT_SOURCE(SoQtViewer);

// *************************************************************************

/*!
  \enum SoQtViewer::Type

  Hints about what context the viewer will be used in.  Usually not
  very interesting for the application programmer, it doesn't matter
  much which value is used for the viewer type.  This "feature" of the
  viewer is included just to be compatible with the old SGI Inventor
  API.
*/
/*!
  \var SoQtViewer::Type SoQtViewer::BROWSER

  If a user-supplied scenegraph passed into the setSceneGraph()
  function does not contain a camera, setting the viewer type to
  BROWSER will make the viewer in that case automatically set up a
  camera outside the scene, as part of the viewer's private and hidden
  "supergraph".
*/
/*!
  \var SoQtViewer::Type SoQtViewer::EDITOR

  If a user-supplied scenegraph passed into the setSceneGraph)
  function does not contain a camera, setting the viewer type to
  EDITOR will make the viewer in that case automatically set up a
  camera \e in the user-supplied scene.

  So if you want to avoid having the SoQtViewer class muck about
  with your supplied scenegraph, set the type-flag to
  SoQtViewer::BROWSER instead.
*/

/*!
  \enum SoQtViewer::DrawType

  Contains valid values for the first argument to the
  SoQtViewer::setDrawStyle() call. Decides the effect of the second
  argument.

  \sa SoQtViewer::setDrawStyle(), SoQtViewer::DrawStyle
*/
/*!
  \var SoQtViewer::DrawType SoQtViewer::STILL

  If this value is passed as the first argument of
  SoQtViewer::setDrawStyle(), the second argument decides which
  draw style to use when the viewer camera is standing still in the
  same position with the same orientation -- i.e. when the end user is
  \e not interacting with the scene camera.
*/
/*!
  \var SoQtViewer::DrawType SoQtViewer::INTERACTIVE

  If this value is passed as the first argument of
  SoQtViewer::setDrawStyle(), the second argument decides which
  draw style to use when the end user is interacting with the scene
  camera, causing continuous animation redraws.
*/

/*!
  \enum SoQtViewer::DrawStyle

  Decides drawstyle for a scene with either a still camera or an
  animating camera.

  \sa SoQtViewer::setDrawStyle(), SoQtViewer::DrawType
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_AS_IS

  Normal rendering, draws all scene geometry in it's original style.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_HIDDEN_LINE

  Draw scene in "hidden line" mode: that is, as wireframe with no
  "see-through".

  Note that this is actually an expensive way to render, as the scene
  must be rendered twice to achieve the effect of hiding lines behind
  the invisible geometry.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_WIREFRAME_OVERLAY

  Render the scene as normal, but overlay a set of lines showing the
  contours of all polygons.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_NO_TEXTURE

  Render scene without textures.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_LOW_COMPLEXITY

  Render all "complex" shape types with low complexity to improve
  rendering performance.

  "Complex shapes" in this context includes spheres, cones, cylinder,
  NURBS surfaces, and others which are tesselated to polygons before
  being rendered.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_LINE

  View all polygon geometry in wireframe mode.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_POINT

  Render only the vertex positions of the geometry.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_BBOX

  View the scene's bounding boxes, instead of rendering the full
  geometry.

  A very efficient way of optimizing rendering performance for scenes
  with high primitive counts while moving the camera about is to set
  this mode for the SoQtViewer::INTERACTIVE DrawType.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_LOW_RES_LINE

  Render as wireframe and don't bother with getting them rendered
  correctly in depth.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_LOW_RES_POINT

  Render as vertex points and don't bother with getting them rendered
  correctly in depth.
*/
/*!
  \var SoQtViewer::DrawStyle SoQtViewer::VIEW_SAME_AS_STILL

  Always render a scene with an animating camera (ie
  SoQtViewer::INTERACTIVE DrawType) in the same manner as scene
  with a still camera.
*/

/*!
  \enum SoQtViewer::BufferType
  Set of valid values for SoQtViewer::setBufferingType().
*/
/*!
  \var SoQtViewer::BufferType SoQtViewer::BUFFER_SINGLE
  Change underlying OpenGL canvas to be single-buffered.
*/
/*!
  \var SoQtViewer::BufferType SoQtViewer::BUFFER_DOUBLE
  Change underlying OpenGL canvas to be double-buffered.
*/
/*!
  \var SoQtViewer::BufferType SoQtViewer::BUFFER_INTERACTIVE

  Set up so animation rendering is done in a double-buffered OpenGL
  canvas, but ordinary rendering happens directly in the front-buffer.

  This mode can be useful with absurdly large scenes, as the rendering
  will \e visibly progress, and one will avoid having the end user
  wonder why nothing is happening while the scene is rendered to the
  back buffer in the default SoQtViewer::BUFFER_DOUBLE mode.
*/

// *************************************************************************

/*!
  \internal

  Return the parent node in the scene graph of the given \a node.
  NB: this is just a quick'n'dirty thing for often executed code,
  and doesn't cover cases where nodes have multiple parents.
*/

static SoGroup *
get_parent_of_node(SoQtViewerP * pimpl, SoNode * root, SoNode * node)
{
  SbBool oldsearch = SoBaseKit::isSearchingChildren();
  SoBaseKit::setSearchingChildren(TRUE);

  assert(node && root && "get_parent_of_node() called with null argument");
  if (pimpl == NULL) {
    SoSearchAction search;
    search.setSearchingAll(TRUE);
    search.setNode(node);
    search.apply(root);
    assert(search.getPath() && "node not found in scenegraph");
    SoNode * parent = ((SoFullPath *)search.getPath())->getNodeFromTail(1);
    assert(parent && "couldn't find parent");
    SoBaseKit::setSearchingChildren(oldsearch);
    return (SoGroup *)parent;
  } else {
    pimpl->searchaction->reset();
    pimpl->searchaction->setSearchingAll(TRUE);
    pimpl->searchaction->setNode(node);
    pimpl->searchaction->apply(root);
    assert(pimpl->searchaction->getPath() && "node not found in scenegraph");
    SoNode * parent =
      ((SoFullPath *) pimpl->searchaction->getPath())->getNodeFromTail(1);
    assert(parent && "couldn't find parent");
    pimpl->searchaction->reset();
    SoBaseKit::setSearchingChildren(oldsearch);
    return (SoGroup *)parent;
  }
}

// *************************************************************************

/*!
  Constructor. \a parent, \a name and \a embed are passed on to
  SoQtRenderArea, so see the documentation for our parent
  constructor for for more information on those.

  The \a t type setting hints about what context the viewer will be
  used in.  Usually not very interesting for the application
  programmer, but if you want to make sure the SoQtViewer class
  doesn't muck about with your supplied scenegraph, set the type-flag
  to SoQtViewer::BROWSER.  (This "feature" of the viewer is
  included just to be compatible with the old SGI Inventor API.)

  The \a build flag decides whether or not to delay building the
  widgets / window which is going to make up the components of the
  viewer.
*/
SoQtViewer::SoQtViewer(QWidget * parent,
                             const char * name,
                             SbBool embed,
                             SoQtViewer::Type t,
                             SbBool build)
  : inherited(parent, name, embed, TRUE, TRUE, FALSE)
{
  PRIVATE(this) = new SoQtViewerP(this);

  // initialization of protected data
  PRIVATE(this)->type = t;
  PRIVATE(this)->viewingflag = TRUE;
  PRIVATE(this)->camera = NULL;
  PRIVATE(this)->scenegraph = NULL;

  // initialization of internal data
  PRIVATE(this)->cursoron = TRUE;
  PRIVATE(this)->localsetbuffertype = FALSE;

  PRIVATE(this)->cameratype = SoPerspectiveCamera::getClassTypeId();
  PRIVATE(this)->deletecamera = FALSE;
  PRIVATE(this)->buffertype = this->isDoubleBuffer() ? BUFFER_DOUBLE : BUFFER_SINGLE;

  PRIVATE(this)->interactionstartCallbacks = new SoCallbackList;
  PRIVATE(this)->interactionendCallbacks = new SoCallbackList;
  PRIVATE(this)->interactionnesting = 0;

  PRIVATE(this)->seekdistance = 50.0f;
  PRIVATE(this)->seekdistanceabs = FALSE;
  PRIVATE(this)->seektopoint = TRUE;
  PRIVATE(this)->seekperiod = 2.0f;
  PRIVATE(this)->inseekmode = FALSE;
  PRIVATE(this)->seeksensor = new SoTimerSensor(SoQtViewerP::seeksensorCB, this);

  PRIVATE(this)->sceneroot = PRIVATE(this)->createSuperScene();
  PRIVATE(this)->sceneroot->ref();

  PRIVATE(this)->drawstyles[STILL] = VIEW_AS_IS;
  PRIVATE(this)->drawstyles[INTERACTIVE] = VIEW_SAME_AS_STILL;

  this->addStartCallback(SoQtViewerP::interactivestartCB);
  this->addFinishCallback(SoQtViewerP::interactiveendCB);

  PRIVATE(this)->adjustclipplanes = TRUE;
  PRIVATE(this)->autoclipbboxaction = NULL;

  PRIVATE(this)->stereoviewing = FALSE;
  PRIVATE(this)->stereooffset = 0.1f;

  PRIVATE(this)->wireframeoverlaycolor = SbColor(1.0f, 0.0f, 0.0f);

  if (build) {
    this->setClassName("SoQtViewer");
    QWidget * widget = this->buildWidget(this->getParentWidget());
    this->setBaseWidget(widget);
  }

  PRIVATE(this)->fpsRoot = PRIVATE(this)->createFPSSuperimposition();
  if (PRIVATE(this)->fpsRoot) {
    this->addSuperimposition(PRIVATE(this)->fpsRoot);
    this->setSuperimpositionEnabled(PRIVATE(this)->fpsRoot, FALSE);
  }
  PRIVATE(this)->resetFrameCounter();
}

// *************************************************************************

/*!
  Destructor.
*/

SoQtViewer::~SoQtViewer()
{
  delete PRIVATE(this)->autoclipbboxaction;

  delete PRIVATE(this)->interactionstartCallbacks;
  delete PRIVATE(this)->interactionendCallbacks;

  delete PRIVATE(this)->seeksensor;

  if (PRIVATE(this)->scenegraph) this->setSceneGraph(NULL);
  PRIVATE(this)->sceneroot->unref();
  if (PRIVATE(this)->superimpositions != NULL) {
    for (int i = PRIVATE(this)->superimpositions->getLength() - 1; i >= 0; i--) {
      SoNode * node = (SoNode *) (*PRIVATE(this)->superimpositions)[i];
      node->unref();
    }
  }
  delete PRIVATE(this);
}

// *************************************************************************

// Note: the following function documentation block will also be used
// for all the miscellaneous viewer subclasses, so keep it general.
/*!
  Set the camera we want to view the scene with.

  The camera passed in as an argument to this method \e must be part
  of the viewer's scenegraph.

  If the application code doesn't explicitly set up a camera through
  this method, the viewer will automatically scan through the
  scenegraph to find a camera to use. If no camera is available in the
  scenegraph at all, it will set up it's own camera.

  \sa getCamera()
*/
void
SoQtViewer::setCamera(SoCamera * cam)
{
  if (PRIVATE(this)->camera) {
    // If we made the camera, detach it. Otherwise just leave it in
    // the graph.
    if (PRIVATE(this)->deletecamera) {
      SoGroup * cameraparent =
	get_parent_of_node(PRIVATE(this), PRIVATE(this)->sceneroot, PRIVATE(this)->camera);
      cameraparent->removeChild(PRIVATE(this)->camera);
      PRIVATE(this)->deletecamera = FALSE;
    }

    PRIVATE(this)->camera->unref();
  }

  PRIVATE(this)->camera = cam;

  if (PRIVATE(this)->camera) {
    PRIVATE(this)->camera->ref();
    this->saveHomePosition();
    PRIVATE(this)->cameratype = PRIVATE(this)->camera->getTypeId();
  }
}

// *************************************************************************

/*!
  Returns the camera currently used by the viewer for the user's main
  viewpoint.

  It \e is possible that this function returns \c NULL, for instance
  if there's no scenegraph present in the viewer.  (This is mostly
  meant as a note for developers extending the SoQt library, as
  application programmers usually controls if and when a viewer
  contains a scenegraph, and therefore know in advance if this method
  will return a valid camera pointer.)

  \sa setCamera()
*/

SoCamera *
SoQtViewer::getCamera(void) const
{
  return PRIVATE(this)->camera;
}

// *************************************************************************

/*!
  When the viewer has to make its own camera as a result of the graph
  passed to setSceneGraph() not containing any camera nodes, this call
  can be made in advance to decide which type the camera will be of.

  Default is to use an SoPerspectiveCamera.

  If this method is called when there is a scene graph and a camera
  already set up, it will delete the old camera and set up a camera
  with the new type if the \a t type is different from that of the
  current camera.

  \sa getCameraType()
*/

void
SoQtViewer::setCameraType(SoType t)
{
  SoType perspectivetype = SoPerspectiveCamera::getClassTypeId();
  SoType orthotype = SoOrthographicCamera::getClassTypeId();
  SbBool oldisperspective = PRIVATE(this)->cameratype.isDerivedFrom(perspectivetype);
  SbBool newisperspective = t.isDerivedFrom(perspectivetype);

  if ((oldisperspective && newisperspective) ||
      (!oldisperspective && !newisperspective)) // Same old, same old..
    return;

  if (SOQT_DEBUG) {
    SbBool valid = TRUE;
    if (t == SoType::badType()) valid = FALSE;
    if (valid) {
      valid = FALSE;
      if (newisperspective) valid = TRUE;
      if (t.isDerivedFrom(orthotype)) valid = TRUE;
    }

    if (!valid) {
      SoDebugError::post("SoQtViewer::setCameraType",
                         "not a valid camera type: '%s'",
                         t == SoType::badType() ?
                         "badType" : t.getName().getString());
      return;
    }
  }

  if (PRIVATE(this)->camera != NULL) {
    SoCamera * newcamera = (SoCamera *)t.createInstance();

    // Transfer and convert values from one camera type to the other.
    if (newisperspective) {
      SoQtViewerP::convertOrtho2Perspective((SoOrthographicCamera *)PRIVATE(this)->camera,
                                               (SoPerspectiveCamera *)newcamera);
    }
    else {
      SoQtViewerP::convertPerspective2Ortho((SoPerspectiveCamera *)PRIVATE(this)->camera,
                                               (SoOrthographicCamera *)newcamera);
    }

    SoGroup * cameraparent =
      get_parent_of_node(PRIVATE(this), PRIVATE(this)->sceneroot, PRIVATE(this)->camera);
    cameraparent->insertChild(newcamera,
                              cameraparent->findChild(PRIVATE(this)->camera));
    SoCamera * oldcamera = !PRIVATE(this)->deletecamera ? PRIVATE(this)->camera : NULL;

    // Store the current home position, as it will be implicitly reset
    // by setCamera().
    SoOrthographicCamera * homeo = new SoOrthographicCamera;
    SoPerspectiveCamera * homep = new SoPerspectiveCamera;
    homeo->ref();
    homep->ref();
    homeo->copyContents(PRIVATE(this)->storedortho, FALSE);
    homep->copyContents(PRIVATE(this)->storedperspective, FALSE);

    this->setCamera(newcamera); // This will set PRIVATE(this)->cameratype.

    // Restore home position.
    PRIVATE(this)->storedortho->copyContents(homeo, FALSE);
    PRIVATE(this)->storedperspective->copyContents(homep, FALSE);
    homeo->unref();
    homep->unref();

    PRIVATE(this)->deletecamera = TRUE;
    if (oldcamera) { cameraparent->removeChild(oldcamera); }
  }
  else { // A camera has not been instantiated for the scene.
    PRIVATE(this)->cameratype = t; // No call to setCamera(), so set type explicitly.
  }
}

// *************************************************************************

/*!
  Returns camera type which will be used when the viewer has to make its
  own camera.

  Note that this call does \e not return the current cameratype, as one
  might expect. Use getCamera() and SoType::getTypeId() for that inquiry.

  \sa setCameraType()
*/

SoType
SoQtViewer::getCameraType(void) const
{
  return PRIVATE(this)->cameratype;
}

// *************************************************************************

/*!
  Reposition the current camera so we can see the complete scene.
*/

void
SoQtViewer::viewAll(void)
{
  if (PRIVATE(this)->camera && PRIVATE(this)->scenegraph)
    PRIVATE(this)->camera->viewAll(PRIVATE(this)->scenegraph, this->getViewportRegion());
}

// *************************************************************************

/*!
  Store the current camera settings for later retrieval with
  resetToHomePosition().

  \sa resetToHomePosition()
*/

void
SoQtViewer::saveHomePosition(void)
{
  if (! PRIVATE(this)->camera) return; // probably a scene-less viewer

  // FIXME: the old code here used SoType::isDerivedFrom() to test for
  // camera type. This caused a crash in SoFieldData::overlay(), line
  // 235 since the types of the field containers were not equal when
  // calling copyContents() (I use a camera which is derived from
  // SoPerspectiveCamera in my application).
  //
  // We should probably not use copyContents() to copy the camera
  // field values, but copy field by field. Also, we should probably
  // use SoType::createInstance() to store a copy of the camera, not
  // just assume it's either a perspective or an orthographic camera.
  //   pederb, 2002-05-27

  SoType t = PRIVATE(this)->camera->getTypeId();
  if (t == SoOrthographicCamera::getClassTypeId()) {
    PRIVATE(this)->storedortho->copyContents(PRIVATE(this)->camera, FALSE);
    SoQtViewerP::convertOrtho2Perspective((SoOrthographicCamera *)PRIVATE(this)->camera,
                                             PRIVATE(this)->storedperspective);
  }
  else if (t == SoPerspectiveCamera::getClassTypeId()) {
    PRIVATE(this)->storedperspective->copyContents(PRIVATE(this)->camera, FALSE);
    SoQtViewerP::convertPerspective2Ortho((SoPerspectiveCamera *)PRIVATE(this)->camera,
                                             PRIVATE(this)->storedortho);
  }
  else {
#if SOQT_DEBUG
    SoDebugError::postWarning("SoQtViewer::saveHomePosition",
                              "Only SoPerspectiveCamera and SoOrthographicCamera is supported.");
#endif // SO@GUI_DEBUG
  }
}

// *************************************************************************

/*!
  Restore the saved camera settings.

  \sa saveHomePosition()
*/

void
SoQtViewer::resetToHomePosition(void)
{
  if (! PRIVATE(this)->camera) return; // probably a scene-less viewer
  
  SoType t = PRIVATE(this)->camera->getTypeId();
  if (t == SoOrthographicCamera::getClassTypeId()) {
    PRIVATE(this)->camera->copyContents(PRIVATE(this)->storedortho, FALSE);
  }                                                                 
  else if (t == SoPerspectiveCamera::getClassTypeId()) {
    PRIVATE(this)->camera->copyContents(PRIVATE(this)->storedperspective, FALSE);
  }
}

// *************************************************************************

/*!
  Turn the camera headlight on or off.

  Default is to have a headlight turned on.

  \sa isHeadlight(), getHeadlight()
*/

void
SoQtViewer::setHeadlight(SbBool on)
{
  PRIVATE(this)->headlight->on = on;
}

// *************************************************************************

/*!
  Returns status of the viewer headlight, whether it is on or off.

  \sa setHeadlight(), getHeadlight()
*/

SbBool
SoQtViewer::isHeadlight(void) const
{
  return PRIVATE(this)->headlight->on.getValue();
}

// *************************************************************************

/*!
  Returns the a pointer to the directional light node which is the
  viewer headlight.

  The fields of the node is available for user editing.

  \sa isHeadlight(), setHeadlight()
*/

SoDirectionalLight *
SoQtViewer::getHeadlight(void) const
{
  return PRIVATE(this)->headlight;
}

// *************************************************************************

/*!
  Set up a drawing style. The \a type argument specifies if the given
  \a style should be interpreted as the drawstyle during animation or
  when the camera is static.

  Default values for the drawing style is to render the scene "as is"
  in both still mode and while the camera is moving.

  See the documentation for the \a DrawType and \a DrawStyle for more
  information.

  \sa getDrawStyle()
*/
void
SoQtViewer::setDrawStyle(SoQtViewer::DrawType type,
                            SoQtViewer::DrawStyle style)
{
  if (SOQT_DEBUG) {
    if ((type != STILL) && (type != INTERACTIVE)) {
      SoDebugError::postWarning("SoQtViewer::setDrawStyle",
                                "unknown drawstyle type setting 0x%x", type);
      return;
    }
  }

  if (style == this->getDrawStyle(type)) {
    if (SOQT_DEBUG && 0) { // debug
      SoDebugError::postWarning("SoQtViewer::setDrawStyle",
                                "drawstyle for type 0x%02x already 0x%02x",
                                type, style);
    }
    return;
  }

  PRIVATE(this)->drawstyles[type] = style;
  PRIVATE(this)->changeDrawStyle(PRIVATE(this)->currentDrawStyle());
}

// *************************************************************************

/*!
  Return current drawstyles for the given type (\a STILL or
  \a INTERACTIVE).

  \sa setDrawStyle()
*/

SoQtViewer::DrawStyle
SoQtViewer::getDrawStyle(const SoQtViewer::DrawType type) const
{
  if (SOQT_DEBUG) {
    if ((type != STILL) && (type != INTERACTIVE)) {
      SoDebugError::postWarning("SoQtViewer::setDrawStyle",
                                "unknown drawstyle type setting 0x%x", type);
      return PRIVATE(this)->drawstyles[STILL];
    }
  }

  return PRIVATE(this)->drawstyles[type];
}

// *************************************************************************

/*!
  Set the viewer's buffer type. Available types are \c
  SoQtViewer::BUFFER_SINGLE, \c SoQtViewer::BUFFER_DOUBLE and \c
  SoQtViewer::BUFFER_INTERACTIVE.

  (With a buffer type of \c SoQtViewer::BUFFER_INTERACTIVE, the
  viewer will render with doublebuffering during user interaction and
  with single buffering otherwise.)

  Default is \c SoQtViewer::BUFFER_DOUBLE.

  \sa getBufferingType()
*/

void
SoQtViewer::setBufferingType(SoQtViewer::BufferType type)
{
  if (type == PRIVATE(this)->buffertype) return;

  if (type != BUFFER_SINGLE &&
      type != BUFFER_DOUBLE &&
      type != BUFFER_INTERACTIVE) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::setBufferingType",
                                "unknown buffer type 0x%x", type);
    }
    return;
  }

  PRIVATE(this)->buffertype = type;

  PRIVATE(this)->localsetbuffertype = TRUE;
  inherited::setDoubleBuffer(type == BUFFER_DOUBLE);
  PRIVATE(this)->localsetbuffertype = FALSE;
}

// *************************************************************************

/*!
  Return the viewer's buffer type.

  \sa setBufferingType()
*/

SoQtViewer::BufferType
SoQtViewer::getBufferingType(void) const
{
  return PRIVATE(this)->buffertype;
}

// *************************************************************************

// Note: this documentation for setViewing() will also be used for all
// the miscellaneous viewer subclasses, so keep it general.
/*!
  Set view mode.

  If the view mode is on, user events will be caught and used to
  influence the camera position / orientation. If view mode is off,
  all events in the viewer canvas (like for instance keypresses or
  mouseclicks and -movements) will be passed along to the scene graph.

  Default is to have the view mode active.

  \sa isViewing()
*/
void
SoQtViewer::setViewing(SbBool enable)
{
  if (PRIVATE(this)->viewingflag == enable) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::setViewing",
                                "unnecessary called");
    }
    return;
  }

  PRIVATE(this)->viewingflag = enable;

  // Turn off the selection indicators when we go back from picking
  // mode into viewing mode.
  if (PRIVATE(this)->viewingflag) {
    SoGLRenderAction * action = this->getGLRenderAction();
    if (action != NULL)
      SoLocateHighlight::turnOffCurrentHighlight(action);
  }
}

// *************************************************************************

/*!
  Return state of view mode.

  \c TRUE means that the mode of the viewer is set such that user
  interaction with the mouse is used to modify the position and
  orientation of the camera.

  \sa setViewing()
*/
SbBool
SoQtViewer::isViewing(void) const
{
  return PRIVATE(this)->viewingflag;
}

// *************************************************************************

/*!
  Set whether or not the mouse cursor representation should be visible
  in the viewer canvas.

  Default value is on.

  \sa isCursorEnabled()
*/

void
SoQtViewer::setCursorEnabled(SbBool on)
{
  PRIVATE(this)->cursoron = on;
}

// *************************************************************************

/*!
  Returns visibility status of mouse cursor.

  \sa setCursorEnabled()
*/

SbBool
SoQtViewer::isCursorEnabled(void) const
{
  return PRIVATE(this)->cursoron;
}

// *************************************************************************

/*!
  Turn on or off continuous automatic adjustments of the near and far
  clipping planes.

  If on, the distance from the camera position to the near and far
  planes will be calculated to be a "best fit" around the geometry in
  the scene, to maximize the "stretch" of values for the visible
  geometry in the z-buffer. This is important, as z-buffer resolution
  is usually limited enough that one will quickly see flickering in
  the rasterization of close polygons upon lousy utilization of the
  z-buffer.

  Automatic clipping is on as default.

  For better control over what happens in boundary conditions (for
  instance when the distance between near and far planes get very far,
  or if geometry gets very close to the camera position), it is
  possible to use the SoQtViewer::setAutoClippingStrategy() method
  to fine-tune the near/far clipping plane settings.

  \sa getAutoClipping()
*/

void
SoQtViewer::setAutoClipping(SbBool enable)
{
  if (SOQT_DEBUG) {
    if (PRIVATE(this)->adjustclipplanes == enable) {
      SoDebugError::postWarning("SoQtViewer::setAutoClipping",
                                "unnecessary called");
      return;
    }
  }

  PRIVATE(this)->adjustclipplanes = enable;
  if (enable) { this->scheduleRedraw(); }
}

/*!
  Set the auto clipping strategy used for auto clipping.

  When auto clipping is enabled, the near plane distance is calculated
  so that it is just in front of the scene bounding box. If this near
  plane is behind or very close to the projection point, one of the
  following strategies will be used to calculate the new clipping
  plane.

  The VARIABLE_NEAR_PLANE strategy considers the number of z buffer
  bits available for the current OpenGL context, and uses \a value to
  calculate the number of bits that is lost because of the far/near
  ratio. \a value should be in the range [0.0, 1.0]. A higher \a value
  will increase the z-buffer precision, but also push the near plane
  further away from the projection point.
  
  The CONSTANT_NEAR_PLANE strategy simply sets the near plane to 
  \a value. If \a value at some point approaches the far clipping
  plane distance, the near plane distance will be set to far plane 
  distance divided by 5000.0.

  The default strategy is VARIABLE_NEAR_PLANE.


  It is also possible to register a callback method \a cb, which will
  then be invoked after the near and far clipping planes has been
  calculated by the SoQtViewer code. The callback can then adjust
  the values for the distance to the near and far planes to exactly
  match the needs of the application (for instance at specific parts
  in the scene), to limit the distance to either plane, or whatever
  else needs to be controlled.

  The signature of the SoQtAutoClippingCB callback must match:
  \code
  SbVec2f myfunc(void * data, const SbVec2f & nearfar);
  \endcode

  The first argument is the \a cbuserdata passed in along with the
  callback function pointer itself (ie the callback function's
  closure). The second argument is the near and far clipping plane
  distances from the camera position, as calculated internally by the
  viewer.

  The function callback can then modify the near and far clipping
  plane distances to what will \e actually be used by the viewer.

  This is a good way of dynamically modifying the near and far
  distances such that they at all times exactly matches the specific
  layout of the application scene, for instance with regard to the
  trade-off between z-buffer resolution and how early geometry is
  clipped at the near plane (or at the far plane).

  Note that the internal near/far calculations should be good enough
  for the vast majority of scenes. Application programmers should only
  need to set up their own adjustments upon "unusual" scenes, like for
  instance scenes with a large world space, but where one would still
  like to be able to get up extremely close on details in some parts
  of the scene.


  \sa setAutoClipping() 
*/
void 
SoQtViewer::setAutoClippingStrategy(const AutoClippingStrategy strategy,
                                       const float value,
                                       SoQtAutoClippingCB * cb,
                                       void * cbuserdata)
{
  PRIVATE(this)->autoclipstrategy = strategy;
  PRIVATE(this)->autoclipvalue = value;
  PRIVATE(this)->autoclipcb = cb;
  PRIVATE(this)->autoclipuserdata = cbuserdata;

  if (PRIVATE(this)->autoclipstrategy == VARIABLE_NEAR_PLANE) {
    // normalize the value so that the near plane isn't too near or
    // too far from the projection point.  FIXME: calibrate this
    // normalization, pederb, 2002-04-25
    float v = SoQtClamp(value, 0.0f, 1.0f); // just in case
    v *= 0.8f;
    v += 0.1f; // v will be in range [0.1, 0.9]

    PRIVATE(this)->autoclipvalue = v;
  }
  if (PRIVATE(this)->adjustclipplanes) {
    this->scheduleRedraw();
  }
}

// *************************************************************************

/*!
  Return value of the automatic near/far clipplane adjustment indicator.

  \sa setAutoClipping()
*/

SbBool
SoQtViewer::isAutoClipping(
  void) const
{
  return PRIVATE(this)->adjustclipplanes;
}

// *************************************************************************

/*!
  Turn stereo viewing on or off.

  \sa isStereoViewing()
*/

void
SoQtViewer::setStereoViewing(// virtual
  SbBool enable)
{
  PRIVATE(this)->stereoviewing = enable;
  this->scheduleRedraw();
}

// *************************************************************************

/*!
  Returns a boolean indicating whether or not we're in stereo viewing
  mode.

  NOTE: in the original InventorXt API, this method was virtual.  It is not
  virtual here.

  \sa setStereoViewing()
*/

SbBool
SoQtViewer::isStereoViewing(
  void) const
{
  return PRIVATE(this)->stereoviewing;
}

// *************************************************************************

/*!
  Set the offset between the two viewpoints when in stereo mode.

  NOTE: In the original InventorXt API, this method was not virtual.

  \sa getStereoOffset()
*/

void
SoQtViewer::setStereoOffset(// virtual
  const float dist)
{
  PRIVATE(this)->stereooffset = dist;
  this->scheduleRedraw();
}

/*!
  Return the offset distance between the two viewpoints when in stereo mode.

  \sa setStereoOffset()
*/

float
SoQtViewer::getStereoOffset(
  void) const
{
  return PRIVATE(this)->stereooffset;
}

// *************************************************************************

/*!
  Toggle between seeking to a point or seeking to an object.

  Default is to seek to a point.

  \sa isDetailSeek()
*/

void
SoQtViewer::setDetailSeek(const SbBool on)
{
  if (SOQT_DEBUG) {
    if (PRIVATE(this)->seektopoint == on) {
      SoDebugError::postWarning("SoQtViewer::setDetailSeek",
                                "unnecessary called");
      return;
    }
  }

  PRIVATE(this)->seektopoint = on;
}

// *************************************************************************

/*!
  Returns a value indicating whether or not seeks will be performed
  to the exact point of picking or just towards the picked object.

  \sa setDetailSeek()
*/

SbBool
SoQtViewer::isDetailSeek(void) const
{
  return PRIVATE(this)->seektopoint;
}

// *************************************************************************

/*!
  Set the duration of animating the camera repositioning
  after a successful seek. Call with \a seconds equal to \a 0.0 to make
  the camera jump immediately to the correct spot.

  Default value is 2 seconds.

  \sa getSeekTime()
*/

void
SoQtViewer::setSeekTime(const float seconds)
{
  if (seconds < 0.0f) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::setSeekTime",
                                "an attempt was made to set a negative seek "
                                "time duration");
    }
    return;
  }
  PRIVATE(this)->seekperiod = seconds;
}

// *************************************************************************

/*!
  Returns the camera repositioning duration following a seek action.

  \sa setSeekTime()
*/

float
SoQtViewer::getSeekTime(void) const
{
  return PRIVATE(this)->seekperiod;
}

// *************************************************************************

/*!
  Add a function to call when user interaction with the scene starts.

  \sa removeStartCallback(), addFinishCallback()
*/

void
SoQtViewer::addStartCallback(SoQtViewerCB * func, void * data)
{
  PRIVATE(this)->interactionstartCallbacks->addCallback((SoCallbackListCB *)func, data);
}

/*!
  Remove one of the functions which has been set up to be called when user
  interaction with the scene starts.

  \sa addStartCallback(), removeFinishCallback()
*/

void
SoQtViewer::removeStartCallback(SoQtViewerCB * func, void * data)
{
  PRIVATE(this)->interactionstartCallbacks->removeCallback((SoCallbackListCB *)func,
                                                  data);
}

// *************************************************************************

/*!
  Add a function to call when user interaction with the scene ends.

  \sa removeFinishCallback(), addStartCallback()
*/

void
SoQtViewer::addFinishCallback(SoQtViewerCB * func, void * data)
{
  PRIVATE(this)->interactionendCallbacks->addCallback((SoCallbackListCB *)func, data);
}

/*!
  Remove one of the functions which has been set up to be called when user
  interaction with the scene ends.

  \sa addFinishCallback(), removeStartCallback()
*/

void
SoQtViewer::removeFinishCallback(SoQtViewerCB * func, void * data)
{
  PRIVATE(this)->interactionendCallbacks->removeCallback((SoCallbackListCB *)func,
                                                data);
}

// *************************************************************************

/*!
  Set the color of the overlay wireframe to \a color.

  \sa getWireframeOverlayColor()
*/
void SoQtViewer::setWireframeOverlayColor(const SbColor & color)
{
  PRIVATE(this)->wireframeoverlaycolor = color;
  this->scheduleRedraw();
}

// *************************************************************************

/*!
  Returns the current color of the overlay wireframe. The default
  color is [1,0,0], ie pure red.

  \sa setWireframeOverlayColor()
*/
const SbColor &SoQtViewer::getWireframeOverlayColor(void) const
{
  return PRIVATE(this)->wireframeoverlaycolor;
}

// *************************************************************************

/*!
  Overloaded to update the local bufferingtype variable.

  \sa setBufferingType(), getBufferingType()
*/

void
SoQtViewer::setDoubleBuffer(const SbBool on)
{
  if (!PRIVATE(this)->localsetbuffertype)
    PRIVATE(this)->buffertype = on ? BUFFER_DOUBLE : BUFFER_SINGLE;

  inherited::setDoubleBuffer(on);
}

// *************************************************************************

/*!
  Give the viewer a scenegraph to render and interact with. Overloaded
  from parent class so the viewer can add it's own nodes to control
  rendering in different styles, rendering with a headlight, etc.

  The \a root node will be inserted under the \e viewer's root node,
  which also covers the nodes necessary to implement the different
  preferences drawing style settings.

  If no camera is part of the scene graph under \a root, one will be
  added automatically.

  \sa getSceneGraph(), setCameraType()
*/

void
SoQtViewer::setSceneGraph(SoNode * root)
{
  if (root == PRIVATE(this)->scenegraph) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::setSceneGraph",
                                "called with the same root as already set");
    }
    return;
  }

  // If the SoQtRenderArea hasn't yet set up its pointer to the
  // SoQtViewer "viewer root" (i.e. the viewer-generated root above
  // the user-supplied root), do that first.
  if (!inherited::getSceneGraph())
    inherited::setSceneGraph(PRIVATE(this)->sceneroot);

  if (PRIVATE(this)->scenegraph) {
    if (this->getCamera())
      this->setCamera(NULL);
    // Release the old user-supplied graph.
    PRIVATE(this)->usersceneroot->removeChild(PRIVATE(this)->scenegraph);
    // old: PRIVATE(this)->sceneroot->removeChild(PRIVATE(this)->scenegraph);
  }

  PRIVATE(this)->scenegraph = root;
  if (!root) return;

  PRIVATE(this)->usersceneroot->addChild(PRIVATE(this)->scenegraph);
  // old: PRIVATE(this)->sceneroot->addChild(PRIVATE(this)->scenegraph);

  // Search for a camera in the user-supplied scenegraph.
  // SoSearchAction search;
  // search.setType(SoCamera::getClassTypeId());
  // search.apply(PRIVATE(this)->scenegraph);
  SbBool oldsearch = SoBaseKit::isSearchingChildren();
  SoBaseKit::setSearchingChildren(TRUE);

  PRIVATE(this)->searchaction->reset();
  PRIVATE(this)->searchaction->setType(SoCamera::getClassTypeId());
  PRIVATE(this)->searchaction->apply(PRIVATE(this)->scenegraph);

  SoBaseKit::setSearchingChildren(oldsearch);

  SoCamera * scenecamera = NULL;
  if (PRIVATE(this)->searchaction->isFound()) {
    SoFullPath * fullpath =
      (SoFullPath *) PRIVATE(this)->searchaction->getPath();
    scenecamera = (SoCamera *)fullpath->getTail();
  }

#if 0 // debug
  SoDebugError::postInfo("SoQtViewer::setSceneGraph",
                         "camera %sfound in graph",
                         scenecamera ? "" : "not ");
#endif // debug

  // Make our own camera if none was available.
  if (!scenecamera) {
    scenecamera = (SoCamera *) PRIVATE(this)->cameratype.createInstance();
    PRIVATE(this)->deletecamera = TRUE;

    // If type==BROWSER, camera should be inserted in the private
    // viewer "supergraph", if it's equal to EDITOR it should be
    // inserted in the user-supplied scenegraph.
    if (PRIVATE(this)->type == SoQtViewer::BROWSER) {
      PRIVATE(this)->sceneroot->insertChild(scenecamera, 1);
    }
    else { // PRIVATE(this)->type == SoQtViewer::EDITOR
      if (PRIVATE(this)->scenegraph->isOfType(SoGroup::getClassTypeId())) {
        // At the uppermost leftmost position in the user-supplied
        // scenegraph.
        ((SoGroup *)PRIVATE(this)->scenegraph)->insertChild(scenecamera, 0);
      }
      else {
        // Make an extra depth level to fit the camera node into the
        // user-scenegraph.
        SoGroup * g = new SoGroup;
        g->addChild(scenecamera);
        g->addChild(PRIVATE(this)->scenegraph);
        PRIVATE(this)->usersceneroot->removeChild(PRIVATE(this)->scenegraph);
        PRIVATE(this)->usersceneroot->addChild(g);
        PRIVATE(this)->scenegraph = g;
      }
    }

    scenecamera->viewAll(PRIVATE(this)->scenegraph, this->getViewportRegion());
  }

  this->setCamera(scenecamera);
}

// *************************************************************************

// doc in super
SoNode *
SoQtViewer::getSceneGraph(void)
{
  // Overloaded from parent class to return the root of the scene
  // graph set by the user, without the extras added by the viewer to
  // control rendering.
  return PRIVATE(this)->scenegraph;
}

// *************************************************************************

// Note: the following function documentation block will also be used
// for all the miscellaneous viewer subclasses, so keep it general.
/*!
  Put the viewer in or out of "waiting-to-seek" mode.

  If the user performs a mouse button click when the viewer is in
  "waiting-to-seek" mode, the camera will be repositioned so the
  camera focal point lies on the point of the geometry under the mouse
  cursor.

  \sa isSeekMode(), setDetailSeek()
*/
void
SoQtViewer::setSeekMode(SbBool enable)
{
  if (SOQT_DEBUG) {
    // User might have switched mode during seek, so if enable==FALSE,
    // isViewing() is irrelevant.
    if (enable) { assert(this->isViewing()); }
  }

  if (!enable && PRIVATE(this)->seeksensor->isScheduled()) {
    PRIVATE(this)->seeksensor->unschedule();
    this->interactiveCountDec();
  }

  PRIVATE(this)->inseekmode = enable;
}

// *************************************************************************

/*!
  Return a flag which indicates whether or not the viewer is in
  "waiting-to-seek" mode.

  (The actual animated translation will not occur until the end user
  really \e starts the seek operation, typically by clicking with the
  left mousebutton.)

  \sa setSeekMode()
*/
SbBool
SoQtViewer::isSeekMode(void) const
{
  return PRIVATE(this)->inseekmode;
}

// *************************************************************************

/*!
  Call this method to initiate a seek action towards the 3D
  intersection of the scene and the ray from the screen coordinate's
  point and in the same direction as the camera is pointing.

  Returns \c TRUE if the ray from the \a screenpos position intersect
  with any parts of the onscreen geometry, otherwise \c FALSE.
*/
SbBool
SoQtViewer::seekToPoint(const SbVec2s screenpos)
{
  if (! PRIVATE(this)->camera)
    return FALSE;

  SoRayPickAction rpaction(this->getViewportRegion());
  rpaction.setPoint(screenpos);
  rpaction.setRadius(2);
  rpaction.apply(PRIVATE(this)->sceneroot);

  SoPickedPoint * picked = rpaction.getPickedPoint();
  if (!picked) {
    // FIXME: this inc seems bogus, but is needed now due to buggy
    // code in for instance the examinerviewer
    // processSoEvent(). 20020510 mortene.
#if 1
    this->interactiveCountInc(); // decremented in setSeekMode(FALSE)
#endif // FIXME
    this->setSeekMode(FALSE);
    return FALSE;
  }

  SbVec3f hitpoint;
  if (PRIVATE(this)->seektopoint) {
    hitpoint = picked->getPoint();
  } 
  else {
    SoGetBoundingBoxAction bbaction(this->getViewportRegion());
    bbaction.apply(picked->getPath());
    SbBox3f bbox = bbaction.getBoundingBox();
    hitpoint = bbox.getCenter();
  }

  PRIVATE(this)->camerastartposition = PRIVATE(this)->camera->position.getValue();
  PRIVATE(this)->camerastartorient = PRIVATE(this)->camera->orientation.getValue();

  // move point to the camera coordinate system, consider
  // transformations before camera in the scene graph
  SbMatrix cameramatrix, camerainverse;
  PRIVATE(this)->getCameraCoordinateSystem(PRIVATE(this)->camera,
                                           PRIVATE(this)->sceneroot,
                                           cameramatrix,
                                           camerainverse);
  camerainverse.multVecMatrix(hitpoint, hitpoint);

  float fd = PRIVATE(this)->seekdistance;
  if (!PRIVATE(this)->seekdistanceabs)
    fd *= (hitpoint - PRIVATE(this)->camera->position.getValue()).length()/100.0f;
  PRIVATE(this)->camera->focalDistance = fd;

  SbVec3f dir = hitpoint - PRIVATE(this)->camerastartposition;
  dir.normalize();

  // find a rotation that rotates current camera direction into new
  // camera direction.
  SbVec3f olddir;
  PRIVATE(this)->camera->orientation.getValue().multVec(SbVec3f(0, 0, -1), olddir);
  SbRotation diffrot(olddir, dir);
  PRIVATE(this)->cameraendposition = hitpoint - fd * dir;
  PRIVATE(this)->cameraendorient = PRIVATE(this)->camera->orientation.getValue() * diffrot;

  if (PRIVATE(this)->seeksensor->isScheduled()) {
    PRIVATE(this)->seeksensor->unschedule();
    this->interactiveCountDec();
  }

  PRIVATE(this)->seeksensor->setBaseTime(SbTime::getTimeOfDay());
  PRIVATE(this)->seeksensor->schedule();
  this->interactiveCountInc();

  return TRUE;
}

// Documented in superclass. Overridden from parent class to be able
// to do the necessary two-pass rendering e.g. if the drawing style is
// hidden line.
void
SoQtViewer::actualRedraw(void)
{
  SbTime redrawtime = SbTime::getTimeOfDay();

  SbBool clearcol = this->isClearBeforeRender();

  if (this->isStereoViewing()) {
    SbColor bgcol = this->getSceneManager()->getBackgroundColor();

    SbBool quadstereo = this->isQuadBufferStereo();
#ifndef HAVE_SOCAMERA_SETSTEREOMODE
    SoCamera * camera = this->getCamera();
    SbVec3f camerapos = camera->position.getValue();
    SbVec3f cameradir(0.0f, 0.0f, -1.0f);
    SbVec3f offsetvec(1.0f, 0.0f, 0.0f);
    float offset = this->getStereoOffset() * 0.5f;
    SbRotation camerarot = camera->orientation.getValue();
    camerarot.multVec(cameradir, cameradir);
    camerarot.multVec(offsetvec, offsetvec);
    SbVec3f focalpoint = camerapos + cameradir * camera->focalDistance.getValue();

    SbBool notifystore = camera->isNotifyEnabled();
    SbBool positionstore = camera->position.isNotifyEnabled();
    SbBool orientationstore = camera->orientation.isNotifyEnabled();
    // turn off notification to avoid redraws
    camera->enableNotify(FALSE);
    camera->position.enableNotify(FALSE);
    camera->orientation.enableNotify(FALSE);

    camera->position = camerapos - offsetvec * offset;
    SbVec3f dir = focalpoint - camera->position.getValue();
    SbRotation rot(cameradir, dir);
    camera->orientation = camerarot * rot;
    if (quadstereo) {
      glDrawBuffer(this->isDoubleBuffer() ? GL_BACK_LEFT : GL_FRONT_LEFT);
      PRIVATE(this)->reallyRedraw(clearcol);
    }
    else { // red/cyan
      glDrawBuffer(this->isDoubleBuffer() ? GL_BACK : GL_FRONT);
      glClearColor(bgcol[0], bgcol[1], bgcol[2], 0.0f);
      glClear(GL_DEPTH_BUFFER_BIT|GL_COLOR_BUFFER_BIT);
      glColorMask(1,0,0,1); // draw red-only
      PRIVATE(this)->reallyRedraw(FALSE, FALSE);
    }

    // right eye
    camera->position = camerapos + offsetvec * offset;
    dir = focalpoint - camera->position.getValue();
    rot.setValue(cameradir, dir);
    camera->orientation = camerarot * rot;

    if (quadstereo) {
      glDrawBuffer(this->isDoubleBuffer() ? GL_BACK_RIGHT : GL_FRONT_RIGHT);
      PRIVATE(this)->reallyRedraw(clearcol);
    }
    else {
      glColorMask(0,1,1,1); // draw green and blue
      PRIVATE(this)->reallyRedraw(FALSE, TRUE);
    }

    // restore camera values and enable notification
    camera->position = camerapos;
    camera->orientation = camerarot;
    camera->position.enableNotify(positionstore);
    camera->orientation.enableNotify(orientationstore);
    camera->enableNotify(notifystore);
    if (quadstereo) {
      glDrawBuffer(this->isDoubleBuffer() ? GL_BACK : GL_FRONT);
    }
    else {
      glColorMask(1,1,1,1); // restore GL color mask
    }
#else // HAVE_SOCAMERA_SETSTEREOMODE

    if (!quadstereo) {
      SoCamera * camera = this->getCamera();
      if (this->isDoubleBuffer())
        glDrawBuffer(GL_BACK);
      else
        glDrawBuffer(GL_FRONT);
      glClearColor(bgcol[0], bgcol[1], bgcol[2], 0.0f);
      glClear(GL_DEPTH_BUFFER_BIT|GL_COLOR_BUFFER_BIT);
      glColorMask(1,0,0,1); // draw red-only
      camera->setStereoAdjustment(this->getStereoOffset());
      camera->setStereoMode(SoCamera::LEFT_VIEW);
      PRIVATE(this)->reallyRedraw(FALSE, FALSE);

      camera->setStereoMode(SoCamera::RIGHT_VIEW);
      glColorMask(0,1,1,1); // draw green and blue
      PRIVATE(this)->reallyRedraw(FALSE, TRUE);
      camera->setStereoMode(SoCamera::MONOSCOPIC);
      glColorMask(1,1,1,1); // restore GL color mask
    }
    else {
      SoCamera * camera = this->getCamera();
      if (this->isDoubleBuffer())
        glDrawBuffer(GL_BACK_LEFT);
      else
        glDrawBuffer(GL_FRONT_LEFT);
      camera->setStereoAdjustment(this->getStereoOffset());
      camera->setStereoMode(SoCamera::LEFT_VIEW);
      PRIVATE(this)->reallyRedraw(clearcol);
      camera->setStereoMode(SoCamera::RIGHT_VIEW);
      if (this->isDoubleBuffer())
        glDrawBuffer(GL_BACK_RIGHT);
      else
        glDrawBuffer(GL_FRONT_RIGHT);
      PRIVATE(this)->reallyRedraw(clearcol);
      camera->setStereoMode(SoCamera::MONOSCOPIC);
      if (this->isDoubleBuffer())
        glDrawBuffer(GL_BACK);
      else
        glDrawBuffer(GL_FRONT);
    }
#endif // HAVE_SOCAMERA_SETSTEREOMODE
  }
  else PRIVATE(this)->reallyRedraw(clearcol);

  if (PRIVATE(this)->superimpositions != NULL) {
    SoGLRenderAction * raaction = this->getSceneManager()->getGLRenderAction();
    SbBool first = TRUE;
    SbBool zWasEnabled = FALSE;
    for (int i = 0; i < PRIVATE(this)->superimpositions->getLength(); i++) {
      if (PRIVATE(this)->superimpositionsenabled[i] != FALSE) {
	if (first) {
	  // save Z buffer state and disable
          zWasEnabled = glIsEnabled(GL_DEPTH_TEST) ? TRUE : FALSE;
	  glDisable(GL_DEPTH_TEST);
          first = FALSE;
	}
	SoNode * scene = (SoNode *) (*PRIVATE(this)->superimpositions)[i];
	raaction->apply(scene);
      }
    }
    if (!first && zWasEnabled) glEnable(GL_DEPTH_TEST);
  }

  redrawtime = SbTime::getTimeOfDay() - redrawtime;
  PRIVATE(this)->recordFPS(redrawtime.getValue());
}

// *************************************************************************

/*!
  To be able to trigger callback functions when user interaction starts
  and/or stops, we need to keep track of the viewer state (i.e. are we in
  still mode or in animation mode?).

  SoQtViewer automatically adds callbacks to switch between still and
  moving draw style, and to switch between single/double buffer when
  the buffer type is \a INTERACTIVE.

  \sa interactiveCountDec(), getInteractiveCount()
  \sa addStartCallback(), addFinishCallback()
  \sa removeStartCallback(), removeFinishCallback()
  \sa setDrawStyle(), setBufferingType()
*/

void
SoQtViewer::interactiveCountInc(void)
{
  // Catch problems with missing interactiveCountDec() calls.
  assert(PRIVATE(this)->interactionnesting < 100);

  if (++(PRIVATE(this)->interactionnesting) == 1) {
    PRIVATE(this)->interactionstartCallbacks->invokeCallbacks(this);
    PRIVATE(this)->resetFrameCounter();
  }

#if 0 // debug
  SoDebugError::postInfo("SoQtViewer::interactiveCountInc", "%d -> %d",
                         PRIVATE(this)->interactionnesting - 1,
                         PRIVATE(this)->interactionnesting);
#endif // debug
}

// *************************************************************************

/*!
  To be able to trigger callback functions when user interaction starts
  and/or stops, we need to keep track of the viewer state (i.e. are we in
  still mode or in animation mode?).

  SoQtViewer automatically adds callbacks to switch between still and
  moving draw style, and to switch between single/double buffer when
  the buffer type is \a INTERACTIVE.

  \sa interactiveCountInc(), getInteractiveCount()
  \sa addStartCallback(), addFinishCallback()
  \sa removeStartCallback(), removeFinishCallback()
  \sa setDrawStyle(), setBufferingType()
*/

void
SoQtViewer::interactiveCountDec(void)
{
  // FIXME: The UI toolkits may cause the interactionnesting to go
  // below zero by triggering press and release events in different
  // widgets. mariusbu 20010709.

  // FIXME: just to clarify; this is due to programming mistakes on
  // our behalf and should be cleaned up. We're using a simple
  // work-around / ignore strategy for now, though, as getting this
  // 100% correct is hard (there are so many possible ways of user
  // interaction with a viewer canvas) and the end-user will usually
  // not notice any problems at all. So that's why we are using a
  // warning instead of an assert(). 20010815 mortene.

  // FIXME: here's one known way to trigger the bug: hit "s" in the
  // examinerviewer in EXAMINE mode, then while seeking hit ESC to put
  // the viewer in INTERACT mode. When the seek is completed, the
  // count will become -1.  20010912 mortene.

  // FIXME: and another one (tested with SoXt): click and hold LMB in
  // the canvas while in INTERACT mode, then hit 'Esc' to switch to
  // EXAMINE mode, then release LMB. 20020325 mortene.

  if (SOQT_DEBUG) {
    if (PRIVATE(this)->interactionnesting <= 0) {
      SoDebugError::postWarning("SoQtViewer::interactiveCountDec",
                                "interaction count nesting went below zero. "
                                "This is due to an internal SoQt bug.");
    }
  }

  if (--(PRIVATE(this)->interactionnesting) <= 0) {
    PRIVATE(this)->interactionendCallbacks->invokeCallbacks(this);
    PRIVATE(this)->interactionnesting = 0;
  }
}

// *************************************************************************

/*!
  Return current interaction count nesting. If equal to zero, the viewer
  is in animation mode, otherwise the camera is still.

  \sa interactiveCountInc(), interactiveCountDec()
*/

int
SoQtViewer::getInteractiveCount(void) const
{
  return PRIVATE(this)->interactionnesting;
}

// *************************************************************************

/*!
  Set the value used for calculating how close the camera and intersection
  hit point should be made at the end of a seek operation.

  The value can be interpreted as an absolute value in the given world
  unit (which typically is meters) or as a percentage value of the
  distance between the camera starting position and the intersection
  hit point. This can be controlled through the
  setSeekValueAsPercentage() method. It is as default used as an
  absolute value.

  Default value is 50 (absolute distance or percent).

  \sa getSeekDistance(), setSeekValueAsPercentage(), setSeekTime()
*/

void
SoQtViewer::setSeekDistance(const float distance)
{
  if (distance <= 0.0f) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::setSeekDistance",
                                "invalid seek distance value: %f",
                                distance);
    }
    return;
  }
  PRIVATE(this)->seekdistance = distance;
}

// *************************************************************************

/*!
  Returns the current seek distance. Value given as an absolute scalar
  length or as a percentage value of the original distance between
  the hitpoint and the camera starting position.

  \sa setSeekDistance(), isSeekValueAsPercentage()
*/

float
SoQtViewer::getSeekDistance(void) const
{
  return PRIVATE(this)->seekdistance;
}

// *************************************************************************

/*!
  Control whether or not the seek distance value should be interpreted as
  a percentage value or as an absolute distance. See documentation on
  setSeekDistance() for more information.

  \sa setSeekDistance(), isSeekValueAsPercentage()
*/

void
SoQtViewer::setSeekValueAsPercentage(const SbBool on)
{
  if (SOQT_DEBUG) {
    if ((on && this->isSeekValuePercentage()) ||
        (!on && !this->isSeekValuePercentage())) {
      SoDebugError::postWarning("SoQtViewer::setSeekDistanceAsPercentage",
                                "unnecessary called, value already %s",
                                on ? "on" : "off");
      return;
    }
  }

  PRIVATE(this)->seekdistanceabs = on ? FALSE : TRUE;
}

// *************************************************************************

/*!
  Returns an boolean which indicates if the seek distance value from
  getSeekDistance() should be interpreted as a percentage value or
  as an absolute value.

  \sa setSeekValuePercentage(), getSeekDistance()
*/

SbBool
SoQtViewer::isSeekValuePercentage(void) const
{
  return PRIVATE(this)->seekdistanceabs ? FALSE : TRUE;
}

// ************************************************************************

/*!
  This method is obsoleted in Coin SoQt.
 */
void
SoQtViewer::computeSeekFinalOrientation(void)
{
  SOQT_OBSOLETED();
}

// *************************************************************************

/*!
  If the current camera is of perspective type, switch to
  orthographic, and vice versa.

  Automatically calls SoQtViewer::setCameraType() so the change
  will immediately take place.
*/
void
SoQtViewer::toggleCameraType(void)
{
  SoType perspectivetype = SoPerspectiveCamera::getClassTypeId();
  SoType orthotype = SoOrthographicCamera::getClassTypeId();

  this->setCameraType(PRIVATE(this)->cameratype.isDerivedFrom(perspectivetype)
                       ? orthotype : perspectivetype);
}

// ************************************************************************

/*!
  Copies the settings of \a camera into our current camera.  Cameras
  must be of the same class type.
 */
void
SoQtViewer::changeCameraValues(// virtual, protected
  SoCamera * camera)
{
  assert(camera != NULL);

  SoCamera * cam = this->getCamera();
  if (!cam) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::changeCameraValues",
                                "no current camera in the scenegraph");
    }
    return;
  }
  if (cam->getTypeId() != camera->getTypeId()) {
    if (SOQT_DEBUG) {
      SoDebugError::postWarning("SoQtViewer::changeCameraValues",
                                "tried to copy data from camera of "
                                "different type");
    }
    return;
  }

  cam->copyFieldValues(camera, FALSE);
}

// *************************************************************************

// doc in super
void
SoQtViewer::sizeChanged(const SbVec2s & size)
{
  inherited::sizeChanged(size);
}

// *************************************************************************

// Documented in superclass.
SbBool
SoQtViewer::processSoEvent(const SoEvent * const event)
{
  const SoType type(event->getTypeId());
  const SoKeyboardEvent * keyevent = NULL;

  if (type.isDerivedFrom(SoKeyboardEvent::getClassTypeId())) {
    keyevent = (SoKeyboardEvent *) event;
    // the ESC key switches between view and interact mode
    if (keyevent->getKey() == SoKeyboardEvent::ESCAPE &&
        keyevent->getState() == SoButtonEvent::DOWN) {
      this->setViewing(this->isViewing() ? FALSE : TRUE);
      return TRUE;
    }
  }

  // If not viewing, break off further handling and pass the event on
  // to the SoQtRenderArea, which will pass it on to the
  // scenegraph.
  if (!this->isViewing()) { return inherited::processSoEvent(event); }


  if (keyevent && (keyevent->getState() == SoButtonEvent::DOWN)) {
    switch (keyevent->getKey()) {
    case SoKeyboardEvent::S:
      this->setSeekMode(this->isSeekMode() ? FALSE : TRUE);
      return TRUE;
    case SoKeyboardEvent::Q:
      this->hide();
      SoQt::exitMainLoop();
      return TRUE;
    case SoKeyboardEvent::HOME:
      this->resetToHomePosition();
      return TRUE;
    case SoKeyboardEvent::LEFT_ARROW:
      PRIVATE(this)->moveCameraScreen(SbVec2f(-0.1f, 0.0f));
      return TRUE;
    case SoKeyboardEvent::UP_ARROW:
      PRIVATE(this)->moveCameraScreen(SbVec2f(0.0f, 0.1f));
      return TRUE;
    case SoKeyboardEvent::RIGHT_ARROW:
      PRIVATE(this)->moveCameraScreen(SbVec2f(0.1f, 0.0f));
      return TRUE;
    case SoKeyboardEvent::DOWN_ARROW:
      PRIVATE(this)->moveCameraScreen(SbVec2f(0.0f, -0.1f));
      return TRUE;
    default:
      break;
    }
  }

  if (this->isSeekMode()) {
    if (type.isDerivedFrom(SoMouseButtonEvent::getClassTypeId())) {
      SoMouseButtonEvent * const e = (SoMouseButtonEvent *) event;
      if (e->getButton() == SoMouseButtonEvent::BUTTON1) {
        if (e->getState() == SoButtonEvent::DOWN) {
          this->seekToPoint(e->getPosition());
        }
        else {
          // We got an LMB UP-event while in seek-mode, and we just
          // swallow the event.
        }
        return TRUE;
      }
    }
  }

  return FALSE;
}

// *************************************************************************

/*!
  This method is for setting up a superimposed scene graph on top
  of the viewer scene graph.  It will be used for adding spin-rotation
  coordinate systems, fly-viewer speed indicators and similar things.

  This method is not part of the original InventorXt API.
*/

void
SoQtViewer::addSuperimposition(SoNode * scene)
{
  if (PRIVATE(this)->superimpositions == NULL) {
    PRIVATE(this)->superimpositions = new SbPList;
  }
  PRIVATE(this)->searchaction->reset();
  PRIVATE(this)->searchaction->setType(SoCamera::getClassTypeId());
  PRIVATE(this)->searchaction->setInterest(SoSearchAction::FIRST);
  PRIVATE(this)->searchaction->apply(scene);
  if (!PRIVATE(this)->searchaction->isFound()) {
    // FIXME: set up default environment if there is no camera in the
    // superimposition scene
    if (SOQT_DEBUG) {
      SoDebugError::postInfo("SoQtViewer::addSuperimposition",
                             "cameraless superimpositions are not "
                             "supported yet");
    }
    return;
  }
  scene->ref();
  PRIVATE(this)->superimpositions->append(scene);
  PRIVATE(this)->superimpositionsenabled.append(TRUE);
}

// *************************************************************************

/*!
  This method is not part of the original InventorXt API.
*/

void
SoQtViewer::removeSuperimposition(SoNode * scene)
{
  assert(scene);
  int idx = -1;
  if (PRIVATE(this)->superimpositions == NULL) goto error;
  idx = PRIVATE(this)->superimpositions->find(scene);
  if (idx == -1) goto error;
  assert(PRIVATE(this)->superimpositions != NULL);
  PRIVATE(this)->superimpositions->remove(idx);
  PRIVATE(this)->superimpositionsenabled.remove(idx);
  scene->unref();
  return;

 error:
  if (SOQT_DEBUG) {
    SoDebugError::post("SoQtViewer::removeSuperimposition",
                       "no such superimposition");
  }
  return;
}

// *************************************************************************

/*!
  This method sets whether the superimposed scene graph should be traversed
  or not.

  This method is not part of the original InventorXt API.
*/

void
SoQtViewer::setSuperimpositionEnabled(SoNode * scene,
                                         const SbBool enable)
{
  int idx = -1;
  if (PRIVATE(this)->superimpositions == NULL) goto error;
  idx = PRIVATE(this)->superimpositions->find(scene);
  if (idx == -1) goto error;
  PRIVATE(this)->superimpositionsenabled[idx] = enable;
  return;

 error:
  if (SOQT_DEBUG) {
    SoDebugError::post("SoQtViewer::setSuperimpositionEnabled",
                       "no such superimposition");
  }
  return;
}

// *************************************************************************

/*!
  This method returns whether the superimposed scene is rendered or not.

  This method is not part of the original InventorXt API.
*/

SbBool
SoQtViewer::getSuperimpositionEnabled(SoNode * scene) const
{
  int idx = -1;
  if (PRIVATE(this)->superimpositions == NULL) goto error;
  idx = PRIVATE(this)->superimpositions->find(scene);
  if (idx == -1) goto error;
  return PRIVATE(this)->superimpositionsenabled[idx];

 error:
  if (SOQT_DEBUG) {
    SoDebugError::post("SoQtViewer::getSuperimpositionEnabled",
                       "no such superimposition");
  }
  return FALSE;
}

// *************************************************************************
