// Copyright (C) 1999-2002 Open Source Telecom Corporation.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Common C++.
//
// The exception is that, if you link the Common C++ library with other files
// to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
//
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
//
// This exception applies only to the code released under the
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.

#include <cc++/config.h>
#include <cc++/export.h>
#include <cc++/numbers.h>
#include <cstdlib>
#include <cstdio>

#ifdef	WIN32
#include "time.h"
#endif

#ifdef	CCXX_NAMESPACES
namespace ost {
using std::string;
#ifdef __BORLANDC__
using std::time_t;
using std::tm;
using std::localtime;
#endif
#endif

Date::Date()
{
	time_t now;
	struct tm *dt;
	time(&now);
	dt = localtime(&now);

	toJulian(dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
}

Date::Date(struct tm *dt)
{
	toJulian(dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
}

Date::Date(time_t tm)
{
	struct tm *dt = localtime(&tm);
	toJulian(dt->tm_year + 1900, dt->tm_mon + 1, dt->tm_mday);
}

Date::Date(char *str, size_t size)
{
	time_t now;
	struct tm *dt;
	int year;
	char *md;

	if(!size)
		size = strlen(str);

	if(size < 5)
	{
		time(&now);
		dt = localtime(&now);
		year = dt->tm_year + 1900;
		md = str;
	}
	else if(size < 7)
	{
		time(&now);
		dt = localtime(&now);
		Number nyear(str, 2);
		year = ((dt->tm_year + 1900) / 100) * 100 + nyear();
		md = str + 2;
	}
	else
	{
		Number nyear(str, 4);
		year = nyear();
		md = str + 4;
	}
	Number nmonth(md, 2);
	Number nday(md + 2, 2);
	toJulian(year, nmonth(), nday());
}

Date::Date(int year, unsigned month, unsigned day)
{
	toJulian(year, month, day);
};

bool Date::isValid(void)
{
	if(julian == 0x7fffffffl)
		return false;
	return true;
}

char *Date::getDate(char *buf)
{
	fromJulian(buf);
	return buf;
}

time_t Date::getDate(void)
{
	char buf[9];
	struct tm dt;
	memset(&dt, 0, sizeof(dt));
	fromJulian(buf);
	Number nyear(buf, 4);
	Number nmonth(buf + 4, 2);
	Number nday(buf + 6, 2);
	
	dt.tm_year = nyear() - 1900;
	dt.tm_mon = nmonth() - 1;
	dt.tm_mday = nday();
	return mktime(&dt);
}

int Date::getYear(void)
{
	char buf[9];
	fromJulian(buf);
	Number num(buf, 4);
	return num();
}

unsigned Date::getMonth(void)
{
	char buf[9];
	fromJulian(buf);
	Number num(buf + 4, 2);
	return num();
}

unsigned Date::getDay(void)
{
	char buf[9];
	fromJulian(buf);
	Number num(buf + 6, 2);
	return num();
}

unsigned Date::getDayOfWeek(void)
{
	return (unsigned)((julian + 1l) % 7l);
}

string Date::operator()()
{
	char buf[9];

	fromJulian(buf);
	string date(buf);

	return date;
}

long Date::getValue(void)
{
	char buf[9];
	fromJulian(buf);
	return atol(buf);
}

Date&	Date::operator++()
{
	++julian;
	update();
	return *this;
}

Date&	Date::operator--()
{
	--julian;
	update();
	return *this;
}

Date& Date::operator+=(long val)
{
	julian += val;
	update();
	return *this;
}

Date& Date::operator-=(long val)
{
	julian -= val;
	update();
	return *this;
}

int Date::operator==(Date &d)
{
	return julian == d.julian;
}

int Date::operator!=(Date &d)
{
	return julian != d.julian;
}

int Date::operator<(Date &d)
{
	return julian < d.julian;
}

int Date::operator<=(Date &d)
{
	return julian <= d.julian;
}

int Date::operator>(Date &d)
{
	return julian > d.julian;
}

int Date::operator>=(Date &d)
{
	return julian >= d.julian;
}

void Date::toJulian(long year, long month, long day)
{
	julian = 0x7fffffffl;
	
	if(month < 1 || month > 12)
		return;

	if(day < 1 || day > 31)		
		return;

	if(year == 0)
		return;

	if(year < 0)
		year--;

	julian = day - 32075l +
		1461l * (year + 4800l + ( month - 14l) / 12l) / 4l +
		367l * (month - 2l - (month - 14l) / 12l * 12l) / 12l -
		3l * ((year + 4900l + (month - 14l) / 12l) / 100l) / 4l;
}

void Date::fromJulian(char *buffer)
{
	ZNumber nyear(buffer, 4);
	ZNumber nmonth(buffer + 4, 2);
	ZNumber nday(buffer + 6, 2);

	long t1, t2, month, year;

	t1 = julian + 32075;
	year = 4 * t1 / 1461;
	t1 = t1 - 1461 * year / 4;
	t2 = 3 * (year + 100) / 400;
	t1 = t1 + t2;
	month = 12 * t1 / 367;
	t2 = month / 11;
	nday = t1 - 367 * month / 12;
	nmonth = month + 2 - 12 * t2;
	year = year - 4800;

	if(year <= 0)
		--year;

	nyear = year;
	buffer[8] = '\0';
}

Date operator+(Date &date, long val)
{
	Date d = date;
	d.julian += val;
	d.update();
	return d;
}

Date operator+(long val, Date &date)
{
	Date d = date;
	d.julian += val;
	d.update();
	return d;
}

Date operator-(Date &date, long val)
{
	Date d = date;
	d.julian -= val;
	d.update();
	return d;
}

Date operator-(long val, Date &date)
{
	Date d = date;
	d.julian -= val;
	d.update();
	return d;
}

DateNumber::DateNumber(char *str) :
Number(str, 10), Date(str, 10)
{};

#ifdef	CCXX_NAMESPACES
};
#endif
