#ifndef _SYNCHRONIZATION_H_
#define _SYNCHRONIZATION_H_

#include "StringifiedException.h"

DEF_EXCEPTION_WDM( ESynchronization, "Synchonization error", StringifiedException );

class CCriticalSection
{
	void* impl_;

public:
	CCriticalSection( bool enter = false );
	~CCriticalSection();

	void enter();
	void leave();
};

class CEnterCriticalSection
{
	CCriticalSection& cs_;
	bool entered_;
public:
	CEnterCriticalSection( CCriticalSection& cs ) : cs_( cs ), entered_(true)
		{ cs_.enter(); }
	~CEnterCriticalSection()
		{ if( entered_ ) cs_.leave(); }
	void leave()
		{ cs_.leave(); entered_ = false; }
	void enter()
		{ cs_.enter(); entered_ = true; }
};

class CSemaphore
{
	void* impl_;

public:
	CSemaphore( long initial_value = 0 );
	CSemaphore( const CSemaphore& );
	~CSemaphore();

	/** returns false if timeout expired, otherwise false */
	bool wait( unsigned long timeout = (unsigned long)-1 );
	void release();
	bool locked();
};

class CMultiReadExclusiveWrite
{
	size_t counter_;
	CCriticalSection resource_;
	CCriticalSection lokal_;
public:
	CMultiReadExclusiveWrite();
	void enterRead();
	void leaveRead();
	void enterWrite();
	void leaveWrite();
};

template<class T> class atomic
{
protected:
	T val_;

public:
	CCriticalSection cs;

	atomic( const T& val = T() ) : val_(val) {}

	atomic<T>& operator=( const T& val )
	{
		CEnterCriticalSection lock( cs );
		val_ = val;
		return *this;
	}

	operator T () const
	{
		CEnterCriticalSection lock( const_cast<CCriticalSection&>(cs) );
		return val_;
	}
};

template<typename T> class atomic_counter : public atomic<T>
{
public:
	atomic_counter( const T& val = T(0) ) : atomic<T>(val) {}

	T operator++(int)
	{
		CEnterCriticalSection lock( cs );
		return val_++;
	}

	T operator--(int)
	{
		CEnterCriticalSection lock( cs );
		return val_--;
	}

	T operator++()
	{
		CEnterCriticalSection lock( cs );
		return ++val_;
	}

	T operator--()
	{
		CEnterCriticalSection lock( cs );
		return --val_;
	}
};

#endif /* _SYNCHRONIZATION_H_ */
