#include "thread_adapter.h" /* import the localRobis symbol */

#include "gui.h"
#include "RobiInfos.h"
#include "RobiPos.h"
#include "Settings.h"
#include "settings.h"
#include "inventor.h"
#include "console.h"

#include <Inventor/Qt/SoQt.h>
#include <Inventor/Qt/viewers/SoQtExaminerViewer.h>
#include <Inventor/Qt/viewers/SoQtPlaneViewer.h>
#include <Inventor/actions/SoBoxHighlightRenderAction.h>

#include <qaction.h>
#include <qapplication.h>
#include <qstatusbar.h>
#include <qpixmap.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qkeycode.h>
#include <qtimer.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qmessagebox.h>
#include <qsplitter.h>
#include <qslider.h>

#include "gui.moc"
#include "fileopen.xpm"

/* these are the maximum values for the sliders in the settings dialog */
const double MAX_PSDSD = 10.0;  /* in mm */
const double MAX_VWLDSD = 10.0; /* in mm */
const double MAX_VWRDSD = 10.0; /* in  */
const double MAX_DUR = 2.0;     /* in s */
const double MAX_ST2RTR = 10.0; /* in 1 */

#define STD_EXCEPTION_HANDLING\
	catch( exception& e )\
	{ QMessageBox::critical(0, "Excpetion", e.what()); }\
	catch( const char* e )\
	{ QMessageBox::critical(0, "Excpetion", e); }

struct QListViewItemRobi : public QListViewItem, public GuiUpdateRobiSelection
{
	LocalRobi& robi;
	RobiNode* robiNode;
	/* needed for the interactive position changing */
	Position oldPos;
	QListView* listView;
	QListViewItemRobi( LocalRobi& r, QListView* v, RobiNode* rn ) :
		robi(r), robiNode(rn), listView(v),
		/* the columns are "ID", "Name", "Program" */
		QListViewItem( v, QString::number(r.id), r.name.c_str(),
					   r.program.c_str() )
		{ robiNode->guiRef = this; }
	void onUpdateSelection( bool selected )
		{ listView->setSelected( this, selected ); }
};

QSimulationTimeLabel::QSimulationTimeLabel( QWidget* p ) : QLabel( p )
{
	QTimer* t = new QTimer( this );
	connect( t, SIGNAL( timeout() ), SLOT( update() ) );
	t->start( 1000, FALSE );
}
	
extern Time globalTime;
void QSimulationTimeLabel::update()
{
	setText( tr(" Simulation Time: ") + QString::number( globalTime.seconds() ) );
}

AppController::AppController( QString simulationFile ) :
    QMainWindow( 0, "EyeSim 5.0 alpha", WDestructiveClose ),
	lrs(theLocalRobis()),
	simulationFile( simulationFile )
{
	/* initialize and setup the SoQt-viewer stuff */
	SoQt::init( this );

    QSplitter* w = new QSplitter( QSplitter::Horizontal, this , "main" );
	setCentralWidget( w );

// 	SoQtPlaneViewer viewer(top);
	viewer = new SoQtExaminerViewer( w );
	viewer->setBackgroundColor(SbColor(0.2,0.2,0.2));
//	viewer->setTitle("sim");
	viewer->setHeadlight(FALSE);
//	viewer->setDecoration(FALSE);
	viewer->setGLRenderAction( new SoBoxHighlightRenderAction );
//	viewer->viewAll();
	//viewer->setViewing(FALSE);
	//viewer->setAutoClipping(FALSE);

	RobiInfoContainer* ric = new RobiInfoContainer( w );
	robiInfoList = ric->RobiInfoList;
	connect( robiInfoList, SIGNAL(doubleClicked(QListViewItem*)),
			 this, SLOT(doubleClickedOnRobi(QListViewItem*)) );
	connect( robiInfoList, SIGNAL(selectionChanged()),
			 this, SLOT(robiSelectionChanged()) );
//	robiInfoList->setSelectionMode( QListView::Single );

	/* create a modal form */
	robiPosForm = new RobiPosForm( this, 0, true );

	/* setup the settings form */
	settingsForm = new SettingsForm( this, 0, false );
	settingsForm->psdsdSlider->setValue( int(psdStandardDeviation
		* (float)settingsForm->psdsdSlider->maxValue() / MAX_PSDSD * 1000.) );
	settingsForm->vwldsdSlider->setValue( int(vwLinearDistanceStandardDeviation
		* (float)settingsForm->vwldsdSlider->maxValue() / MAX_VWLDSD * 1000.) );
	settingsForm->vwrdsdSlider->setValue( int(vwRotationalDistanceStandardDeviation
		* (float)settingsForm->vwrdsdSlider->maxValue() / MAX_VWRDSD * 180. / M_PI) );
	settingsForm->durSlider->setValue( int(displayUpdateRate
		* (float)settingsForm->durSlider->maxValue() / MAX_DUR) );
	settingsForm->st2rtrSlider->setValue( int(simulationToRealTimeRatio
		* (float)settingsForm->st2rtrSlider->maxValue() / MAX_ST2RTR) );
	settingsChanged(0);

	/* setup the QActions */
    openAction = new QAction( "Start New", QPixmap( fileopen ),
							  "Start &New", CTRL+Key_N, this, "start_new" );
    connect( openAction, SIGNAL( activated() ), SLOT( start() ) );
    QMimeSourceFactory::defaultFactory()->setPixmap( "fileopen",
													 QPixmap( fileopen ) );

    pauseAction = new QAction( "Pause", "&Pause", CTRL+Key_P, this, "pause" );
	pauseAction->setToggleAction( true );
    connect( pauseAction, SIGNAL( toggled(bool) ), SLOT( pause(bool) ) );

    stopAction = new QAction( "Stop", "&Stop", CTRL+Key_S, this, "stop" );
    connect( stopAction, SIGNAL( activated() ), SLOT( stop() ) );

    quitAction = new QAction( "Quit", "&Quit", CTRL+Key_Q, this, "quit" );
    connect( quitAction, SIGNAL( activated() ), qApp, SLOT( closeAllWindows() ) );

	modeAction = new QAction( "Mode", "Mouse &Mode", CTRL+Key_M, this,
							  "mode" );
	modeAction->setToggleAction( true );
	connect( modeAction, SIGNAL( toggled(bool) ), SLOT( mouseMode(bool) ));

	robiPosFormAction = new QAction( "Change Pos", "&Change Pos", CTRL+Key_C,
									 this, "chpos" );
	connect( robiPosFormAction, SIGNAL( activated() ),
			 SLOT( changeRobiPosInteractively() ) );
	connect( robiPosForm->AngleSlider, SIGNAL( valueChanged(int) ),
			 SLOT( robiPosChanged(int) ) );
	connect( robiPosForm->HPosSlider, SIGNAL( valueChanged(int) ),
			 SLOT( robiPosChanged(int) ) );
	connect( robiPosForm->VPosSlider, SIGNAL( valueChanged(int) ),
			 SLOT( robiPosChanged(int) ) );

	QAction* settingsFormAction = new QAction(
		"Settings", "Se&ttings", CTRL+Key_T, this, "settins" );
	connect( settingsFormAction, SIGNAL( activated() ),
			 settingsForm, SLOT( show() ) );
	connect( settingsForm->psdsdSlider, SIGNAL( valueChanged(int) ),
			 SLOT( settingsChanged(int) ) );
	connect( settingsForm->vwldsdSlider, SIGNAL( valueChanged(int) ),
			 SLOT( settingsChanged(int) ) );
	connect( settingsForm->vwrdsdSlider, SIGNAL( valueChanged(int) ),
			 SLOT( settingsChanged(int) ) );
	connect( settingsForm->durSlider, SIGNAL( valueChanged(int) ),
			 SLOT( settingsChanged(int) ) );
	connect( settingsForm->st2rtrSlider, SIGNAL( valueChanged(int) ),
			 SLOT( st2rtrChanged(int) ) );
	
	QToolBar* simTools = new QToolBar( this, "simulation control" );
    simTools->setLabel( tr( "Simulation Controls" ) );
    openAction->addTo( simTools );
    pauseAction->addTo( simTools );
	stopAction->addTo( simTools );
//	simTools->insertSeparator();
	modeAction->addTo( simTools );
	robiPosFormAction->addTo( simTools );
	settingsFormAction->addTo( simTools );

    QPopupMenu * sim = new QPopupMenu( this );
    menuBar()->insertItem( "&Simulation Control", sim );
    openAction->addTo( sim );
    pauseAction->addTo( sim );
    stopAction->addTo( sim );
	sim->insertSeparator();
    quitAction->addTo( sim );

    QPopupMenu * set = new QPopupMenu( this );
	menuBar()->insertItem( "Extra", set );
	settingsFormAction->addTo( set );
	robiPosFormAction->addTo( set );

	statusBar()->addWidget( new QSimulationTimeLabel( statusBar() ), 0, TRUE );
    statusBar()->message( "Ready", 2000 );

	setUsesTextLabel(true);
    resize( 600, 400 );
	setCaption( name() );

	simStarted = false;
	simPaused = false;
	updateActions();
}

AppController::~AppController()
{
	if( simStarted )
		stop();
}

void AppController::updateActions()
{
	pauseAction->setEnabled( simStarted );
	stopAction->setEnabled( simStarted );
	openAction->setEnabled( !simStarted );
	robiPosFormAction->setEnabled( simStarted );
}

void AppController::showEvent( QShowEvent* e )
{
	if( !e->spontaneous() && !simulationFile.isNull() )
		start( simulationFile );
}

void AppController::start( QString fileName )
{
	try {

	Core::setupSimulation( fileName );
	viewer->setSceneGraph( theSceneGraph.root() );
	viewer->viewAll();
	/* this hack is to recenter the view, there is a problem with 
	 * the resize of the widget */
 	viewer->setDecoration( !viewer->isDecoration() );
 	viewer->setDecoration( !viewer->isDecoration() );

	robiInfoList->clear();
	robiInfoList->setColumnAlignment( 0, Qt::AlignRight );
		
	for( size_t i = 0; i < lrs.size(); ++i )
		(void) new QListViewItemRobi( *lrs[i], robiInfoList,
									  theSceneGraph.robiNode(i) );
	
	/* if we have only one robi, we automaticaly open its console window */
	if( robiInfoList->childCount() == 1 )
		doubleClickedOnRobi( robiInfoList->firstChild() );

	Core::startSimulation();
	simStarted = true;
	updateActions();

	} STD_EXCEPTION_HANDLING
}

void AppController::start()
{
    QString fn = QFileDialog::getOpenFileName(
		QString::null, "EyeSim Simulation Description - *.sim", this );

    if( !fn.isEmpty() )
		start( fn );
}

void AppController::pause( bool val )
{
	Core::pauseSimulation( val );
	simPaused = val;
}

void AppController::stop()
{
	try {

	pauseAction->setOn( false );

	Core::stopSimulation();
	robiInfoList->clear();
	viewer->setSceneGraph( 0 );
	viewer->viewAll();
	theSceneGraph.clear();

	for( list<QConsole*>::iterator i = consoles.begin();
		 i != consoles.end(); ++i )
	{
		delete (*i);
	}
	consoles.clear();

	simStarted = false;
	simPaused = false;
	updateActions();

	} STD_EXCEPTION_HANDLING
}

void AppController::doubleClickedOnRobi( QListViewItem* item )
{
	try {

	QListViewItemRobi* r = dynamic_cast<QListViewItemRobi*>(item);
	if( r == 0 )
		throw StringifiedException(
			EMsgE()	<< "invalid QListViewItem in " << __PRETTY_FUNCTION__ );

	/* first check if there is already a console for this robi */
	for( list<QConsole*>::iterator i = consoles.begin();
		 i != consoles.end(); ++i )
	{
		if( (*i)->id() == r->robi.id )
		{
			(*i)->show();
			return;
		}
	}

	/* if not create one */
	consoles.push_back( new QConsole( r->robi ) );
	consoles.back()->show();

	} STD_EXCEPTION_HANDLING
}

void AppController::robiSelectionChanged()
{
	for( QListViewItemIterator i( robiInfoList ); i.current(); ++i )
	{
		QListViewItemRobi* ritem = dynamic_cast<QListViewItemRobi*>(i.current());
		theSceneGraph.setRobiNodeSelected( ritem->robi.id, ritem->isSelected() );
	}
}

void AppController::mouseMode( bool selectMode )
{
	viewer->setViewing( ! selectMode );
}

void AppController::changeRobiPosInteractively()
{
	bool paused = false;
	if( !simPaused )
	{
		paused = true;
		pause( true );
	}

	/* if we have only one robi, we automaticaly select it */
	if( robiInfoList->childCount() == 1 )
		robiInfoList->firstChild()->setSelected(true);

	size_t number_of_selected_robis = 0;
	for( QListViewItemIterator i( robiInfoList ); i.current(); ++i )
	{
		if( ! i.current()->isSelected() )
			continue;

		QListViewItemRobi* ritem = dynamic_cast<QListViewItemRobi*>(i.current());
		ritem->oldPos = ritem->robiNode->robi->pos;

		++number_of_selected_robis;
	}

	if( number_of_selected_robis > 0 )
	{
		robiPosForm->AngleSlider->setValue( 0 );
		robiPosForm->HPosSlider->setValue( 0 );
		robiPosForm->VPosSlider->setValue( 0 );

		int res = robiPosForm->exec();

		for( QListViewItemIterator i( robiInfoList ); i.current(); ++i )
		{
			if( ! i.current()->isSelected() )
				continue;
		
			QListViewItemRobi* ritem =
				dynamic_cast<QListViewItemRobi*>(i.current());
			
			/* update the VWController information for each robi */
			if( res == QDialog::Accepted )
				Core::changeRobiPosition( ritem->robi.id,
											ritem->robiNode->robi->pos );
			else
				ritem->robiNode->robi->pos = ritem->oldPos;
		}
	}
	else
		QMessageBox::information(
			0, "Information", "you have to select at least one robi first" );
	
	if( paused )
		pause( false );
}

void AppController::robiPosChanged( int val )
{
	/* this is to prevent any action while resetting the slider values in
	 * the changeRobiPosInteractively method. */
	if( ! robiPosForm->isVisible() )
		return;

	float phi = (float)robiPosForm->AngleSlider->value() / 360.0 * 2. * M_PI;
	float x = (float)robiPosForm->HPosSlider->value() /
		(float)robiPosForm->HPosSlider->maxValue() *
		fabs( theWorld.boundingBox.p.x - theWorld.boundingBox.q.x );
	float y = (float)robiPosForm->VPosSlider->value() /
		(float)robiPosForm->VPosSlider->maxValue() *
		fabs( theWorld.boundingBox.p.y - theWorld.boundingBox.q.y );

	for( QListViewItemIterator i( robiInfoList ); i.current(); ++i )
	{
		if( ! i.current()->isSelected() )
			continue;

		QListViewItemRobi* ritem = dynamic_cast<QListViewItemRobi*>(i.current());
		GlobalRobi& robi = *ritem->robiNode->robi;

		/* calculate the new position */
		Position pos( ritem->oldPos.x + x, ritem->oldPos.y + y,
					  ritem->oldPos.phi + phi );
		
		/* update the position of the robi, it would be better to 
		 * modify the scene graph here directly, it would also make it
		 * clearer, that the position of the robi has not changed yet. */
		robi.pos = pos;
	}

	/* update the SceneGraph */
	theSceneGraph.onUpdate();
}

void AppController::settingsChanged( int )
{
	psdStandardDeviation = (float)settingsForm->psdsdSlider->value()
		/ (float)settingsForm->psdsdSlider->maxValue() * MAX_PSDSD / 1000.;
	settingsForm->psdsdLabel->setNum( psdStandardDeviation * 1000 );

	vwLinearDistanceStandardDeviation = (float)settingsForm->vwldsdSlider->value()
		/ (float)settingsForm->vwldsdSlider->maxValue() * MAX_VWLDSD / 1000.;
	settingsForm->vwldsdLabel->setNum( vwLinearDistanceStandardDeviation * 1000 );

	vwRotationalDistanceStandardDeviation =
		(float)settingsForm->vwrdsdSlider->value()
		/ (float)settingsForm->vwrdsdSlider->maxValue() * MAX_VWRDSD / 180. * M_PI;
	settingsForm->vwrdsdLabel->setNum(
		vwRotationalDistanceStandardDeviation / M_PI * 180. );

	displayUpdateRate = (float)settingsForm->durSlider->value()
		/ (float)settingsForm->durSlider->maxValue() * MAX_DUR;
	settingsForm->durLabel->setNum( displayUpdateRate );
}

void AppController::st2rtrChanged( int )
{
	float v = (float)settingsForm->st2rtrSlider->value()
		/ (float)settingsForm->st2rtrSlider->maxValue() * MAX_ST2RTR;
	settingsForm->st2rtrLabel->setNum( v );

	Core::changeSimulationToRealTimeRatio( v );
}
