#include "eyebot.h"
#include "sim.h"
#include "settings.h"

ost::ThreadKey LocalThread::thisLocalRobi;

void LocalThread::run()
{
	thisLocalRobi.setKey( localRobi );
	token.wait();
	code();
	localRobi->enter();
}

void LocalThread::Final()
{
	/* if this method gets executed, this LocalThread is the
	 * currentThread and this thread got scheduled just now, so our
	 * nyat is 0 and the virtualTime is uptodate */

	dbg_msg( "thread '" << name << "' (uid:" << uid << ") of robi '"
			 << localRobi->name << "' (id:" << localRobi->id
			 << ") terminated." << endl );

	LocalRobi::thread_iterator_t nextThread = localRobi->findNextThread(
		localRobi->currentThread );

	if( nextThread == localRobi->currentThread )
	{
		localRobi->threads.erase( localRobi->currentThread );
		localRobi->mtPermit = false;
		localRobi->mainThreadLock.release();
	}
	else
	{
		localRobi->threads.erase( localRobi->currentThread );
		/* in this case, nextThread is executable and therefor after
		 * the virtualTaskSwitch, too */
		nextThread = localRobi->doVirtualTaskSwitches( nextThread );

		/* start the next thread */
		localRobi->currentThread = nextThread;
		(*nextThread)->token.release();
	}
		
	delete this;
}

LocalRobi::thread_iterator_t
LocalRobi::findNextThread( LocalRobi::thread_iterator_t i )
{
	thread_iterator_t nextThread = i;

	/* simply find first executable */

	do {
		if( ++nextThread == threads.end() )
			nextThread = threads.begin();
	} while( nextThread != i && ! (*nextThread)->executable( virtualTime ) );

	return nextThread;
}

LocalRobi::thread_iterator_t LocalRobi::doVirtualTaskSwitches( 
	LocalRobi::thread_iterator_t nextThread )
{
	/* increment virtualTime by the not yet assigned time of the next
	 * thread, this is possibly more than fits in the current time
	 * slice. we need to do this in the ! mtPermit case, too. */
	virtualTime += (*nextThread)->nyat;

	/* preemptive mode means possibly more (virtual) task switches ... */
	if( mtMode == PREEMPT && mtPermit )
	{
		/* as long as we used already more time than the current time
		 * slice contains ... */
		while( virtualTime >= ntst )
		{
			/* ... we set nyat to the amount of time that does not fit in
			 * the current time slice */
			(*nextThread)->nyat = virtualTime - ntst;

			/* and virtualTime to the next task switch time */
			virtualTime = ntst;
			processTimers();
		
			/* and ntst gets incremented */
			ntst += LocalThread::timeSlice;

			/* then we start over, with the next thread */
			nextThread = findNextThread( nextThread );
			virtualTime += (*nextThread)->nyat;
		}

		/* the thread, that gets executed next has no nyat */
		(*nextThread)->nyat = 0.0;
	}

	return nextThread;
}

void LocalRobi::reschedule()
{
	/* is this the main thread? */
	if( LocalThread::thisLocalRobi.getKey() == 0 )
	{
		/* find one executable Thread */
		currentThread = findNextThread( threads.begin() );
		if( (**currentThread).executable( virtualTime ) )
		{
			(**currentThread).token.release();
			mainThreadLock.wait();
		}

		return;
	}

	/* just get the next thread to be scheduled */
	thread_iterator_t nextThread = 
		doVirtualTaskSwitches( findNextThread( currentThread ) );

	//BUG: check sleeping threads
	/* in case we just suspended/semLocked/sleeped the current Thread
	 * and there is no other executable thread ... */
	if( ! (*nextThread)->executable( virtualTime ) )
	{
		/* ...  return to the main thread. */
		mtPermit = false;
		mainThreadLock.release();
		(*currentThread)->token.wait();
	}

	/* only perform the task switch, if we really switch to another
	 * thread. */
	if( currentThread != nextThread )
	{
		thread_iterator_t prevThread = currentThread;
		currentThread = nextThread;
		/* task switch */
		(*nextThread)->token.release();
		(*prevThread)->token.wait();
	}
}

void LocalRobi::enter( const RoBiOSFunctionFeatures& ff )
{
	cpuTime.stop();
	virtualTime += cpuTime.elapsed() * robi2HostCPUSpeedFactor;

	if( timerIRQHandler && ! ff.allowExecutionInTimerIRQHandler )
		dbg_msg( "function '" << ff.name
				 << "' callen inside timer-irq handler." << endl );

	if( mtMode != NOTASK && ! ff.allowExecutionInMultiTaskingMode )
		dbg_msg( "function '" << ff.name
				 << "' callen in multi-tasking mode." << endl );

	/* if we are in preemptive multi-tasking mode and we passed the
	 * current next task switch time (ntst) then we reschedule. */
	//BUG: could possibly remove the first evaluation
	if( mtMode == PREEMPT && mtPermit && virtualTime >= ntst &&
		!timerIRQHandler )
	{
		(*currentThread)->nyat = virtualTime - ntst;
		virtualTime = ntst;
		processTimers();
		ntst += LocalThread::timeSlice;

		thread_iterator_t oldThread = currentThread;
		/* just get the next thread to be scheduled */
		currentThread = doVirtualTaskSwitches( findNextThread( currentThread ) );

		/* we don't need to check, if the new thread is executable,
		 * because there exists at least one executable thread (the
		 * thread that called this enter() method */
		
		if( currentThread != oldThread )
		{
			/* task switch */
			(*currentThread)->token.release();
			(*oldThread)->token.wait();
		}
	}

	if( mtMode != PREEMPT && !timerIRQHandler && virtualTime >= ntit )
		processTimers();

	if( lastSynchronization.steps() < virtualTime.steps() )
	{	
		theCoreMessenger->synchronize( virtualTime );
		lastSynchronization.steps( virtualTime.steps() );
	}
}

void LocalRobi::leave( const RoBiOSFunctionFeatures& ff )
{
	virtualTime += ff.robiCPUTime;
	cpuTime.start();
}

/** this method gets executed when ever ntst gets incremented in preemptive
 * mode and in cooperative mode whenever the virtualTime exceeds the next
 * timer irq date. Since this is the only date when a preemptive task
 * schedule can happen, we need to check for messages and resume
 * blocked threads only at these dates. */
void LocalRobi::processTimersAndRADIOMessages()
{
	processTimers();
	radio.processWaitingThreads( *this );
}

void LocalRobi::processTimers()
{
	if( timers.empty() )
		return;

	timerIRQHandler = true;

	Time correctedVT = virtualTime;
	
	while( correctedVT >= ntit )
	{
		virtualTime = ntit;

		for( map<int, Timer>::iterator i = timers.begin();
			 i != timers.end(); ++i )
		{
			Timer& t = i->second;
			if( ++t.counter == t.scale )
			{
				t.counter = 0;
				cpuTime.start();
				t.code();
				cpuTime.stop();
				virtualTime += cpuTime.elapsed() * robi2HostCPUSpeedFactor;
			}
		}

		/* this is the time that the execution of the irq handlers
		 * engrossed */
		Time t = virtualTime - ntit;
//BUG: it is unclear, what happens in the real world if this case happens
		if( t >= LocalThread::timeSlice )
			dbg_msg( "timer-irq handler consumed " << t << " s (>= 1/100s)\n" );
		correctedVT += t;
		ntit += (int(t / LocalThread::timeSlice) + 1) * LocalThread::timeSlice;
	}

	virtualTime = correctedVT;

	timerIRQHandler = false;
}

#define ENTER_ROBIOS_MT \
	ENTER_ROBIOS;

#define ENTER_ROBIOS_MT_MODE \
	ENTER_ROBIOS_MT; \
	if( robi.mtMode == robios::NOTASK ) \
		return -1; 

#define DEFINE_LOCAL_THREAD \
	LocalThread& t = thread ? **thread->thread : **robi.currentThread;

namespace robios {

struct tcb
{
	list<LocalThread*>::iterator thread;
	tcb( list<LocalThread*>::iterator thread ) : thread(thread) {}
};

int OSMTInit(BYTE mode)
{
	ENTER_ROBIOS_MT;
	
	if( mode != PREEMPT && mode != COOP || robi.mtMode != NOTASK )
		return -1;

	robi.mtMode = (robios::OSMTMode)mode;

	return 0;
}

int OSMTStatus()
{
	ENTER_ROBIOS_MT;
	return robi.mtMode;
}

struct tcb* OSSpawn(char* name, void (*code)(), int stksiz, int pri, int uid)
{
	ENTER_ROBIOS_MT;
	if( robi.mtMode == NOTASK )
		return 0;

	LocalThread* t = new LocalThread( robi, name, code, stksiz, pri, uid );
	t->start();

//BUG: memory leak if robi program does not OSKill all threads
	return new tcb(	robi.threads.insert( robi.threads.end(), t ) );
}

int OSReady(struct tcb* thread)
{
	ENTER_ROBIOS_MT_MODE;
	DEFINE_LOCAL_THREAD;
	t.suspended = false;
	return 0;
}

int OSSuspend(struct tcb* thread)
{
	ENTER_ROBIOS_MT_MODE;
	DEFINE_LOCAL_THREAD;
	t.suspended = true;
	return 0;
}

int OSReschedule()
{
	ENTER_ROBIOS_MT_MODE;
	robi.reschedule();
	return 0;
}

int OSYield()
{
	ENTER_ROBIOS_MT_MODE;
	(*robi.currentThread)->suspended = true;
	robi.reschedule();
	return 0;
}

int OSRun(struct tcb* thread)
{
	ENTER_ROBIOS_MT_MODE;
	DEFINE_LOCAL_THREAD;
	t.suspended = false;
	robi.reschedule();
	return 0;
}

int OSGetUID(struct tcb* thread)
{
	ENTER_ROBIOS_MT_MODE;
	DEFINE_LOCAL_THREAD;
	return t.uid;
}

int OSKill(struct tcb* thread)
{
	ENTER_ROBIOS_MT_MODE;

	if( thread && thread->thread != robi.currentThread )
	{
		/* if we want to kill another thread, we just have to remove
		 * the thread from the queue and clean up the memory, which 
		 * includes the destruction of the thread itself. */
		robi.threads.erase( thread->thread );
		delete *(thread->thread);
		delete thread;
		robi.reschedule();
	}
	else
	{
		if( thread )
			delete thread;
		
		(**robi.currentThread).Exit();
	}
			
	return 0;
}

int OSExit(int code)
{
	ENTER_ROBIOS_MT_MODE;
	(**robi.currentThread).Exit();
	return 0;
}

int OSPanic(char* msg)
{
	ENTER_ROBIOS_MT_MODE;
	robi.lcd.putString( msg );
//BUG: better implementation?
	ost::Thread::sleep( (unsigned long)-1 );
	return 0;
}

int OSSleep(int timeout)
{
	ENTER_ROBIOS_MT_MODE;
	(*robi.currentThread)->wakeupTime =
		robi.ntst + timeout * LocalThread::timeSlice;
	robi.reschedule();
	return 0;
}

int OSForbid()
{
	ENTER_ROBIOS_MT_MODE;
	robi.mtPermit = false;
	return 0;
}

int OSPermit()
{
	ENTER_ROBIOS_MT_MODE;
	robi.mtPermit = true;
	robi.ntst = LocalThread::timeSlice
		* int(robi.virtualTime / LocalThread::timeSlice);

	/* is this the main thread? */
	if( LocalThread::thisLocalRobi.getKey() == 0 )
	{
		/* find one executable Thread */
		robi.currentThread = robi.findNextThread( robi.threads.begin() );
		if( (**robi.currentThread).executable( robi.virtualTime ) )
		{
			(**robi.currentThread).token.release();
			robi.mainThreadLock.wait();
		}
		else
			robi.mtPermit = false;
	}

	return 0;
}

/***************** Semaphores ******************/

int OSSemInit(struct sem* s, int val)
{
	ENTER_ROBIOS_MT;
	robi.semaphores.push_back( LocalSem( val ) );
	s->impl = &robi.semaphores.back();
	return 0;
}

int OSSemP(struct sem* s)
{
	ENTER_ROBIOS_MT;

//BUG: sem initialized?
	LocalSem& semaphore = *reinterpret_cast<LocalSem*>(s->impl);

	if( semaphore.counter > 0 )
		--semaphore.counter;
	else
	{
		semaphore.waitingThreads.push( *robi.currentThread );
		(*robi.currentThread)->semLocked = true;
//		dbg_msg( "reschedule due to sem lock\n" );
		robi.reschedule();
	}

	return 0;
}

int OSSemV(struct sem* s)
{
	ENTER_ROBIOS_MT;

//BUG: sem initiealized?
	LocalSem& semaphore = *reinterpret_cast<LocalSem*>(s->impl);

	if( semaphore.waitingThreads.empty() )
		++semaphore.counter;
	else
	{
		if( semaphore.counter > 0 )
			dbg_msg( "semLocked threads while sem->counter is "
					 << semaphore.counter << endl );
		semaphore.waitingThreads.front()->semLocked = false;
		semaphore.waitingThreads.pop();
//		dbg_msg( "releasing sem locked thread\n" );
	}

	return 0;
}


/* Time related OS functions ************************************************ */

#define ENTER_ROBIOS_TIME \
	ENTER_ROBIOS;

int OSSetTime(int h, int m, int s)
{
	ENTER_ROBIOS_TIME;
	Time t = h*60*60 + m*60 + s;
	robi.systemStartTime = t - robi.virtualTime;
	return 0;
}

int OSGetTime(int* h, int* m, int* s, int* ticks)
{
	ENTER_ROBIOS_TIME;
	double t = robi.virtualTime + robi.systemStartTime;
	*ticks = int(t / LocalThread::timeSlice) % int(1.0 / LocalThread::timeSlice);
	*s = int(t) % 60;
	*m = int(t/60) % 60;
	*h = int(t/3600);
	return 0;
}

int OSShowTime()
{
	ENTER_ROBIOS_TIME;
	double t = robi.virtualTime + robi.systemStartTime;
	stringstream ss;
	ss << int(t/3600) << ':' << int(t/60) % 60 << ':' << int(t) % 60;
	robi.lcd.putString( ss.str().c_str() );
	return 0;
}

int OSGetCount()
{
	ENTER_ROBIOS_TIME;
	return int(robi.virtualTime / LocalThread::timeSlice);
}

int OSWait(int n)
{
	ENTER_ROBIOS_MT;
	if( n < 0 )
		return -1;

	robi.virtualTime += LocalThread::timeSlice * n;
	robi.ntst += LocalThread::timeSlice * n;
	robi.processTimers();

	return 0;
}

TimerHandle OSAttachTimer(int scale, TimerFnc function)
{
	ENTER_ROBIOS_MT;
	if( robi.timers.size() == robi.maxNumberOfTimers )
		return 0;

	if( robi.timers.empty() )
		robi.ntit = LocalThread::timeSlice
			* int(robi.virtualTime / LocalThread::timeSlice);

	int handle = robi.nextTimerHandle++;
	robi.timers[handle] = Timer( scale, function );
	return handle;
}

int OSDetachTimer(TimerHandle handle)
{
	ENTER_ROBIOS_MT;
	return robi.timers.erase( handle ) ? 1 : 0;
}

} /* namespace robios */
