#include "thread_adapter.h"
#include "core.h"
#include "Synchronization.h"

CCriticalSection* globalLock = 0;
Client2CoreMessagesThreadImplementation theCoreMessengerObject;
Client2CoreMessages* theCoreMessenger = &theCoreMessengerObject;
Core2ClientMessagesThreadImplementation theClientsMessengerObject;
Core2ClientMessages* theClientsMessenger = &theClientsMessengerObject;


struct ThreadAdapterInfo
{
	CSemaphore synchronizeLock;
	RobiThread *thread;
	LocalRobi localRobi;
};

/* there is a vector of pointers used because the copy constructors of
 * RobiThread and LocalRobi are not stl-container compatible */
static vector<ThreadAdapterInfo*> infos;
static ost::ThreadKey thisInfo;

static vector<LocalRobi*> localRobis;
const vector<LocalRobi*>& theLocalRobis()
{
	return localRobis;
}

inline ThreadAdapterInfo* getInfo()
{
	return reinterpret_cast<ThreadAdapterInfo*>( thisInfo.getKey() );
}

LocalRobi& LocalRobi::thisRobi()
{
	ThreadAdapterInfo* info = getInfo();
	if( info )
		return info->localRobi;
	else
		return *(reinterpret_cast<LocalRobi*>(
			LocalThread::thisLocalRobi.getKey() ));
}

void Client2CoreMessagesThreadImplementation::setVWController( const VWData& vwData )
{
	CEnterCriticalSection lock( *globalLock );

	Core::setVWController( LocalRobi::thisRobi().id, vwData );
}


void Client2CoreMessagesThreadImplementation::synchronize( const Time& time )
{
	ThreadAdapterInfo* info = getInfo();

//BUG: 
	if( info == 0 )
		info = infos[LocalRobi::thisRobi().id];

	CEnterCriticalSection lock( *globalLock );

	Core::synchronize( info->localRobi.id, time );

	lock.leave();

//	debug( "wait " << info->id << endl);
	info->synchronizeLock.wait();
//	debug( "woken " << info->id << endl);
}

void Client2CoreMessagesThreadImplementation::setKicker(
	const double_pointxy& reflectionCoefficients, double speed )
{
	CEnterCriticalSection lock( *globalLock );

	Core::setKicker( LocalRobi::thisRobi().id, reflectionCoefficients, speed );
}

void Core2ClientMessagesThreadImplementation::resumeRobi( id_t id )
{
//	debug( "release " << id << endl );
	infos[id]->synchronizeLock.release();
}

void Client2CoreMessagesThreadImplementation::sendRADIOMessage(
	const RADIOMessage& message )
{
	/* this lock might not be necessary as long as changing the number
	 * of robis during the simulation is not allowed */
	CEnterCriticalSection lock( *globalLock );

	Core::sendRADIOMessage( message );
}

void Core2ClientMessagesThreadImplementation::deliverRADIOMessage(
	id_t id, const RADIOMessage& message )
{
	infos[id]->localRobi.radio.deliver( message );
}

void Core2ClientMessagesThreadImplementation::reportCollision(
	id_t id, const Time& time )
{
	/* there is no extra synchronizaiton neccessary here, since this
	 * function get only called, when all robi threads are locked by
	 * there synchronization semaphore */
	infos[id]->localRobi.collisionTime = time;
}

const Robi& Core2ClientMessagesThreadImplementation::startRobi(
	const char* description, id_t id )
{
	if( globalLock == 0 )
		globalLock = new CCriticalSection;

	CEnterCriticalSection lock( *globalLock );

	if( infos.size() != id )
		throw Eyae( "internal error (ids mixed up somehow)" );

	try {
		ThreadAdapterInfo* info = new ThreadAdapterInfo;

		info->localRobi.id = id;
		info->localRobi.readDescription( description );
		info->localRobi.vwController.axisOffset = info->localRobi.axisOffset;
		info->thread = new RobiThread( info->localRobi.program.c_str(), id );
		info->thread->start();

		infos.push_back( info );
		localRobis.push_back( &info->localRobi );

		return info->localRobi;
	} catch( ost::DSO* dso ) {
		throw dso->getError();
	} catch( ost::Thread* ) {
		throw Eyae( "internal error (could not start robi thread)" );
	}
}

void Core2ClientMessagesThreadImplementation::cleanup()
{
	for( vector<ThreadAdapterInfo*>::iterator i = infos.begin();
		 i != infos.end(); ++i )
	{
		delete (**i).thread;
		delete *i;
	}

	infos.clear();
	localRobis.clear();

	if( globalLock )
		delete globalLock;
	globalLock = 0;
}

static void registerThread( id_t id )
{
 	CEnterCriticalSection lock( *globalLock );

	ThreadAdapterInfo* info = infos[id];
	thisInfo.setKey( info );

	dbg_msg( "registered Thread " << id << endl );

	lock.leave();
	
	info->synchronizeLock.wait();
}


void RobiThread::run()
{
	try {

		registerThread( id );
		LocalRobi::thisRobi().cpuTime.start();
		(*main)();

	} catch( const std::exception& e ) {
		cout << endl << "EXCEPTION during execution of robi '"
			 << infos[id]->localRobi.name << "' (id:"
			 << id << ") : " << e.what() << endl;
	}
}

void RobiThread::Final()
{
	try {

		cout << "robi '" << infos[id]->localRobi.name << "' (id:"
			 << id << ") terminated." << endl;

		CEnterCriticalSection lock( *globalLock );
		Core::synchronize( id, Time::never() );
		lock.leave();

	} catch( const std::exception& e ) {
		cout << endl << "EXCEPTION in " << __func__ << " of robi '"
			 << infos[id]->localRobi.name << "' (id:"
			 << id << ") : " << e.what() << endl;
	}
}


/* this is the implementation of the functions in ndrng.h */

#include "stocc.h"

static StochasticLib theRandomNumberGenerator( 0 );
static CCriticalSection theRandomNumberGeneratorLock;

double generateNormalDistributedNumber( double standardDeviation )
{
	if( standardDeviation == 0.0 )
		return 0;

	CEnterCriticalSection lock( theRandomNumberGeneratorLock );

	return theRandomNumberGenerator.Normal( 0, standardDeviation );
}
