#include <cc++/thread.h>
#include <iostream>
#include <algorithm>
#include <functional>
#include <cmath>
#include <vector>
#include <time.h>
#include "timer.h"
#include "Synchronization.h"

#include "core.h"
#include "world.h"
#include "inventor.h"
#include "settings.h"
#include "parser.h"

Time globalTime;
static Time lastGlobalTime; /* necessary for the collision detection */
static timer<realtime> realTime;
double realTimeAccumulator;
double lastUpdateOfS2RTR;

vector<Time> virtualTimes;

World theWorld;
SceneGraph theSceneGraph( theWorld );

CSemaphore *pauseRequest;
CSemaphore *pauseAcc;
static bool paused;

void Core::setVWController( id_t id, const VWData& vwData )
{
	GlobalRobi& robi = theWorld.robis[id];
	robi.vwController.update( vwData );
}

void Core::setKicker( id_t id, const double_pointxy& reflectionCoefficients, 
					  double speed )
{
	GlobalRobi& robi = theWorld.robis[id];
	robi.kickerReflectionCoefficients = reflectionCoefficients;
	robi.kickerSpeed = speed;
}

void Core::sendRADIOMessage( const RADIOMessage& message )
{
	dbg_msg( message << endl );

	if( message.receiver >= theWorld.robis.size() ||
		message.sender >= theWorld.robis.size() )
	{
		dbg_msg( "messages '" << message
				 << "' contains invalid sender or receiver." << endl );
		return;
	}

	if( message.receiver == robios::BROADCAST )
		for( size_t i = 0; i < theWorld.robis.size(); ++i )
			theClientsMessenger->deliverRADIOMessage( i, message );
	else
		theClientsMessenger->deliverRADIOMessage( message.receiver, message );
}

void Core::synchronize( id_t id, const Time& time )
{
	virtualTimes[id] = time;

	Time min = *min_element( virtualTimes.begin(), virtualTimes.end() );
	
	if( min.steps() > globalTime.steps() )
	{
		while( globalTime.steps() < min.steps() )
		{
			/* save the current globalTime before updating it */
			lastGlobalTime = globalTime;

#ifdef DONT_GIVE_A_DAMN_ON_THE_DISPLAY
			/* current global time is set to the minimum of the virtual
			 * times */
			globalTime = min.steps();
#else
			/* this line increases the current globalTime by a small
			 * step to provide a continuous update of the display,
			 * therefore the displayUpdateRate is involved. The
			 * max(1,x) makes sure, we make at least one step;
			 * otherwise the simulation would stop completely (see
			 * BUG# 46). */
			globalTime.steps( std::min( min.steps(), globalTime.steps()
				+ std::max( 1, int(displayUpdateRate / Time::delta) ) ) );
#endif

			/* calculate new (now current) world */
			theWorld.update( globalTime, lastGlobalTime );

			double s = (globalTime - lastUpdateOfS2RTR)
				- (realTime.elapsed() + realTimeAccumulator)
				* simulationToRealTimeRatio;
			/* a value of 0 means "as fast as possible */
			if( simulationToRealTimeRatio != 0.0 )
				usleep( max( 0, int( s * 1000000.)) );
		}

		/* check with a handshake mechanism if we shall pause */
		pauseAcc->release();
		pauseRequest->wait();
		pauseRequest->release();
		pauseAcc->wait();
		
		/* wake up all robis that virtual time is the current global
		 * time */
		for( id_t i=0; i < virtualTimes.size(); ++i )
		{
			if( globalTime.steps() == virtualTimes[i].steps() )
				theClientsMessenger->resumeRobi( i );
		}
	}
}

void Core::setupSimulation( const char* fileName )
{
	/* setup the supported entries of the configuration file */
	Configuration c;
	c << "world" << "maze" << "robi";
	parseConfigurationFile( fileName, c );
	
	if( ! c["world"].empty() )
		theWorld.readWorld( c["world"].c_str() );
	else if( ! c["maze"].empty() )
		theWorld.readMaze( c["maze"].c_str() );
	else
		throw EInvalidParameter(
			"No environment file specified in simulation description" );

	/* overwrite the std settings, if specified */
	readSettingsFile( fileName );

	globalTime = 0;
	realTimeAccumulator = 0;
	lastUpdateOfS2RTR = 0;

	pauseRequest = new CSemaphore( 1 );
	pauseAcc = new CSemaphore( 0 );
	paused = false;

	virtualTimes.resize( theWorld.robis.size() );

	for( int id=0; id < theWorld.robis.size(); ++id )
	{
		theWorld.robis[id] = theClientsMessenger->startRobi(	c["robi"].c_str(), id );
#ifdef DEBUG
		cerr << "FIXME: setting axis offset " << theWorld.robis[id].axisOffset << endl;
		// theWorld.robis[id].vwController.axisOffset = theWorld.robis[id].axisOffset;
#endif
	}
}

void Core::startSimulation()
{
	for( int id=0; id < theWorld.robis.size(); ++id )
		theClientsMessenger->resumeRobi( id );

	realTime.start();
}

void Core::pauseSimulation( bool pause )
{
	if( pause == paused )
		return;

	if( pause )
	{
		/* block the threads */
		pauseRequest->wait();
		/* wait for them to be blocked */
		pauseAcc->wait();
		
		realTime.stop();
		realTimeAccumulator += realTime.elapsed();
	}
	else
	{
		pauseAcc->release();
		/* release the blocked thread */
		pauseRequest->release();

		realTime.start();
	}

	paused = pause;
}

void Core::stopSimulation()
{
	pauseSimulation( true );
	theClientsMessenger->cleanup();
	virtualTimes.clear();

	delete pauseAcc;
	delete pauseRequest;
}

void Core::changeRobiPosition( id_t i, const Position& newPos )
{
	GlobalRobi& robi = theWorld.robis[i];
	
	robi.vwController.distance =
		robi.vwController.remainingDistance( virtualTimes[i] );
	
	robi.vwController.startPos = newPos;
	robi.pos = newPos;

	if( theWorld.onUpdateHandler )
		theWorld.onUpdateHandler->onUpdate();
}

void Core::changeSimulationToRealTimeRatio( double r )
{
	lastUpdateOfS2RTR = globalTime;
	realTime.stop();
	realTimeAccumulator = 0;
	simulationToRealTimeRatio = r;
	realTime.start();
}
