#include "sim.h" /* dbg_msg */
#include "inventor.h"

#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoFaceSet.h>
#include <Inventor/nodes/SoNormal.h>
#include <Inventor/nodes/SoNormalBinding.h>
#include <Inventor/nodes/SoCylinder.h>
#include <Inventor/nodes/SoCube.h>
#include <Inventor/nodes/SoSphere.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoPointLight.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoDB.h>
#include <Inventor/SoPath.h>

void made_selection( void * userdata, SoPath * path )
{
	path->truncate( 2 );
	RobiNode* rn = dynamic_cast<RobiNode*>(path->getTail());
	if( ! rn )
		dbg_msg( "internal error: selected object was no robi" )
	else
		if( rn->guiRef )
			rn->guiRef->onUpdateSelection( (bool) userdata );
	
    theSceneGraph.onUpdate(); // to redraw
}

SceneGraph::SceneGraph( World& world ) : world(world), rootSG(0)
{
	/* install the onUpdate callback referencing to this object */
 	world.onUpdateHandler = this;
}

RobiNode* SceneGraph::robiNode( size_t index )
{
	if( selection == 0 )
		return 0;

	return dynamic_cast<RobiNode*>( selection->getChild( index ) );
}

void SceneGraph::setRobiNodeSelected( size_t index, bool selected )
{
	SoNode* rn = selection->getChild( index );
	if( selected )
		selection->select( rn );
	else
		selection->deselect( rn );
}

SoSeparator* SceneGraph::setupWalls()
{
/* draw t-shaped wall segments, this is done a better gui viewing
 * quality. T-shaped means that the profile of a wall segment is not
 * like the character 'l' but like an upside down 'T'. */
#define DRAW_T_SHAPED_WALLS

#ifdef DRAW_T_SHAPED_WALLS
	size_t r = 2;
#else
	size_t r = 1;
#endif /* DRAW_WALL_ROOFS */

	size_t n = world.segments.size();
	normals.resize( n * r );
	vertices.resize( 4 * n * r );
	numvertices.resize( n * r );

	for( size_t i=0; i < n; ++i )
	{
		Segment& seg = world.segments[i];
		normals[i] = pointxy2SbVec3f( seg.n );
		vertices[4*i+0] = pointxy2SbVec3f( seg.p, .0 );
		vertices[4*i+1] = pointxy2SbVec3f( seg.p, .1 );
		vertices[4*i+2] = pointxy2SbVec3f( seg.q, .1 );
		vertices[4*i+3] = pointxy2SbVec3f( seg.q, .0 );
		numvertices[i] = 4;

#ifdef DRAW_T_SHAPED_WALLS	  
		double_pointxy sn = seg.n * .005;
		normals[i+n] = SbVec3f(1,1,1);
		vertices[4*(n+i)+0] = pointxy2SbVec3f( seg.p+sn, .0 );
		vertices[4*(n+i)+1] = pointxy2SbVec3f( seg.p-sn, .0 );
		vertices[4*(n+i)+2] = pointxy2SbVec3f( seg.q-sn, .0 );
		vertices[4*(n+i)+3] = pointxy2SbVec3f( seg.q+sn, .0 );
		numvertices[i+n] = 4;
#endif /* DRAW_T_SHAPED_WALLS */
	}

	// Define the FaceSet
	SoVertexProperty* vp = new SoVertexProperty;
#if __GNUC__ == 2
	vp->normal.setValues( 0, n * r, normals.begin() );
	vp->normalBinding = SoNormalBinding::PER_FACE;
	vp->vertex.setValues( 0, 4 * n * r, vertices.begin() );

	SoFaceSet *fs = new SoFaceSet;
	fs->numVertices.setValues( 0, n * r, numvertices.begin() );
	fs->vertexProperty.setValue( vp );

	SoMaterial* m = new SoMaterial();
	m->diffuseColor.setValue(.7, .7, .7);
//	m->ambientColor.setValue(.4, .4, .4);
	m->emissiveColor.setValue(.4, .4, .45);
	m->shininess.setValue(0.0);

	SoCube* floor = new SoCube;
	floor->width = world.boundingBox.q.x - world.boundingBox.p.x;
	floor->height = world.boundingBox.q.y - world.boundingBox.p.y;
	floor->depth = 0.;
	SoTransform* floorTrans = new SoTransform;
	floorTrans->translation.setValue(
		pointxy2SbVec3f( world.boundingBox.center() ) );
	SoMaterial* floorMat = new SoMaterial();
	floorMat->diffuseColor.setValue(.1, .6, .1);
	floorMat->emissiveColor.setValue(.2, .2, .2);
	floorMat->shininess.setValue(.0);

	SoSeparator* w = new SoSeparator;
	w->ref();
	w->addChild( m );
	w->addChild( fs );
	w->addChild( floorTrans );
	w->addChild( floorMat );
	w->addChild( floor );
	w->unrefNoDelete();

	return w;
#else
	return 0;
#endif
}

SoSeparator* SceneGraph::setupRobis()
{
	if( world.robis.empty() )
		return new SoSeparator;

	/* setup the _one_ robi model */
	SoSeparator* robiSG = new SoSeparator;
	robiSG->ref();

	SoRotationXYZ* rot = new SoRotationXYZ;

//BUG: using the iv file of the first robi for all robis
	SoInput myInput;
	if( !myInput.openFile(world.robis[0].model.c_str()) )
	{
		dbg_msg( "can not open iv-file '" << world.robis[0].model
				 << "', using dummy model.\n" );

		SoScale* scale = new SoScale;
		scale->scaleFactor.setValue( theWorld.robis[0].radius,
									 theWorld.robis[0].radius,
									 theWorld.robis[0].radius/2 );
		rot->axis.setValue( SoRotationXYZ::Z );
		rot->angle.setValue( M_PI / 2 );
		robiSG->addChild( scale );
		robiSG->addChild( rot );
		SoMaterial* m = new SoMaterial();
		m->diffuseColor.setValue(1.0, 1.0, .0);
		m->specularColor.setValue(.5, .5, .5);
		m->shininess.setValue( 0.2 );
		robiSG->addChild( m );
		robiSG->addChild( new SoCylinder );
	}
	else
	{
		rot->axis.setValue( SoRotationXYZ::X );
		rot->angle.setValue( M_PI / 2 );
		robiSG->addChild( rot );

		robiSG->addChild( SoDB::readAll(&myInput) );
	}

	robiSG->unrefNoDelete();

	/* setup the transformations and links for every robi. */
	robiTransforms.clear();
	SoSelection* robisSG = selection = new SoSelection;
	selection->addSelectionCallback( made_selection, (void *)1L );
	selection->addDeselectionCallback( made_selection, (void *)0L );
	selection->policy = SoSelection::SHIFT;
	robisSG->ref();

	for( vector<GlobalRobi>::iterator i = world.robis.begin();
		 i != world.robis.end(); ++i )
	{
		SoTransform* t = new SoTransform;
		t->rotation.setValue( SbVec3f(0,0,1), i->pos.phi );
		t->translation.setValue( pointxy2SbVec3f( i->pos ) );
		robiTransforms.push_back( t );

		RobiNode* new_robi = new RobiNode( &(*i) );
		new_robi->ref();
		new_robi->addChild( t );
		new_robi->addChild( robiSG );
		new_robi->unrefNoDelete();

		robisSG->addChild( new_robi );
	}

	robisSG->unrefNoDelete();
	return robisSG;
}

SoSeparator* SceneGraph::setupBalls()
{
	if( world.balls.empty() )
		return new SoSeparator;

	/* setup the _one_ ball model */
	SoSeparator* ballSG = new SoSeparator;
	ballSG->ref();

	SoMaterial* m = new SoMaterial();
	m->diffuseColor.setValue(1.0, .0, .0);
	m->specularColor.setValue(.5, .5, .5);
	m->emissiveColor.setValue(.7, .2, .2);
	m->shininess.setValue( 0.2 );

	ballSG->addChild( m );
	ballSG->addChild( new SoSphere );
	ballSG->unrefNoDelete();

	/* setup the transformations and links for every ball. */
	ballTransforms.clear();
	SoSelection* ballsSG = new SoSelection;
	ballsSG->ref();

	for( vector<Ball>::iterator i = world.balls.begin();
		 i != world.balls.end(); ++i )
	{
		SoTransform* t = new SoTransform;
		t->translation.setValue( pointxy2SbVec3f( i->pos, i->radius ) );
		t->scaleFactor.setValue( SbVec3f(i->radius, i->radius, i->radius) );
		ballTransforms.push_back( t );

		SoSeparator* new_ball = new SoSeparator;
		new_ball->ref();
		new_ball->addChild( t );
		new_ball->addChild( ballSG );
		new_ball->unrefNoDelete();

		ballsSG->addChild( new_ball );
	}

	ballsSG->unrefNoDelete();
	return ballsSG;
}

void SceneGraph::onUpdate()
{
#ifdef DEBUG
#define CHECK_TRANSFORMATION_PARAMETER( P ) \
	if( isnan(P) ) \
		throw EInvalidParameter( EMsgE() \
			<< "invalid transformation parameter in " << __FUNC__ << ": " << P );
#else
#define CHECK_TRANSFORMATION_PARAMETER( P )
#endif /* DEBUG */

	for( size_t i = 0; i < robiTransforms.size(); ++i )
	{
		CHECK_TRANSFORMATION_PARAMETER( world.robis[i].pos.phi )
		CHECK_TRANSFORMATION_PARAMETER( world.robis[i].pos.x )
		CHECK_TRANSFORMATION_PARAMETER( world.robis[i].pos.y )
		robiTransforms[i]->rotation.setValue(
			SbVec3f(0,0,1), world.robis[i].pos.phi );
		robiTransforms[i]->translation.setValue(
			pointxy2SbVec3f( world.robis[i].pos ) );
	}

	for( size_t i = 0; i < ballTransforms.size(); ++i )
	{
		CHECK_TRANSFORMATION_PARAMETER( world.balls[i].pos.x )
		CHECK_TRANSFORMATION_PARAMETER( world.balls[i].pos.y )
		ballTransforms[i]->translation.setValue(
			pointxy2SbVec3f( world.balls[i].pos, world.balls[i].radius ) );
	}

#undef CHECK_TRANSFOMATION_PARAMETER
}

void SceneGraph::clear()
{
	vertices.clear();
	normals.clear();
	numvertices.clear();
	robiTransforms.clear();
	ballTransforms.clear();

	rootSG->unref();
	rootSG = 0;
}

SoSeparator* SceneGraph::root()
{
	if( ! rootSG )
	{
		rootSG = new SoSeparator;
		rootSG->ref();

#define USE_PHONG_SHADING
#ifdef USE_PHONG_SHADING
		SoPointLight* pl = new SoPointLight();
		pl->location.setValue( pointxy2SbVec3f(world.boundingBox.center(), 1.5) );
		rootSG->addChild( pl );

		SoShapeHints* sh = new SoShapeHints();
		sh->vertexOrdering.setValue( SoShapeHints::COUNTERCLOCKWISE );
		sh->shapeType.setValue( SoShapeHints::UNKNOWN_SHAPE_TYPE );
		rootSG->addChild( sh );
#else
		SoLightModel* lm = new SoLightModel;
		lm->model = SoLightModel::BASE_COLOR;
		rootSG->addChild( lm );
#endif /* USE_PHONG_SHADING */

		rootSG->addChild( setupWalls() );
		rootSG->addChild( setupRobis() );
		rootSG->addChild( setupBalls() );
	}

	return rootSG;
}
