#include "eyebot.h"
#include "sim.h"
#include "ndrng.h"
#include "settings.h"
#include <stdio.h>

void setupVWControllers( LocalRobi& robi, double v, double w,
						 const voodo& distance = voodo() )
{
	// cerr << "setupVW called w/ v: " << v << " w: " << w << endl;
	if( v > robi.maxV )
		v = robi.maxV;
	if( v < -robi.maxV )
		v = -robi.maxV;

	if( w > robi.maxW )
		w = robi.maxW;
	if( w < -robi.maxW )
		w = -robi.maxW;
	
	// cerr << "setupVW clip w/ v: " << v << " w: " << w << endl;

	/* if we have an infinite command we merely add an error induced
	 * value to the angular velocity, which means the radius of the
	 * driven cirle is wrong.
	 * if have a turn command, we modify only the distance, which
	 * results in wrong angle about which the robi is turned.
	 * else we modify both w and distance. */

	double real_v = v;
	double real_w = w;
	voodo real_distance = distance;
	double internal_v = v;
	double internal_w = w;
	voodo internal_distance = distance;

	if( distance.isInfinite() )
	{
		real_w += generateNormalDistributedNumber(
				vwRotationalDistanceStandardDeviation );
	}
	else if( v == 0.0 )
	{
		real_distance = distance + generateNormalDistributedNumber(
				vwRotationalDistanceStandardDeviation );
		real_distance = distance;
		/* set the internal data so that we arive at the same time, but
		 * drive another speed */
	//	 cerr << "TURN w: " << w << " dist: " << distance << " rdist: " << real_distance
	//	 << endl;
		internal_w = w * distance / real_distance;
	}
	else
	{
		real_distance = distance + generateNormalDistributedNumber(
				vwLinearDistanceStandardDeviation );
		real_w += generateNormalDistributedNumber(
				vwRotationalDistanceStandardDeviation );
		real_distance = distance;
		real_w = w;
		/* set the internal data so that we arive at the same time, but
		 * drive another speed */
		internal_v = v * distance / real_distance;
	}

	// VWData real( robi.virtualTime, real_v, real_w, real_distance );
	// VWData internal( robi.virtualTime, internal_v, internal_w, internal_distance );
	VWController real( robi.virtualTime, real_v, real_w, real_distance );
	VWController internal( robi.virtualTime, internal_v, internal_w, internal_distance );

	theCoreMessenger->setVWController( real );
	robi.vwController.update( internal );
	robi.collisionTime = Time::never();
}

#define ENTER_ROBIOS_VW \
ENTER_ROBIOS; \
	if( h != 1 ) return -1;

namespace robios {

	VWHandle VWInit(DeviceSemantics semantics, int Timescale)
	{
		ENTER_ROBIOS;

		/* VW_DRIVE == -700 and valid timescale is greater than 0
		 * if init==1, VWInit has been executed before*/
		if(semantics==VW_DRIVE && Timescale > 0 && robi.vwController.init == 0) {
			robi.vwController.init =1;
			return 1;
		}
		else return 0;
	}

	int VWRelease(VWHandle h)
	{
		ENTER_ROBIOS_VW;
		return 0;
	}

	int VWSetSpeed(VWHandle h, meterPerSec v, radPerSec w)
	{
		ENTER_ROBIOS_VW;
		setupVWControllers( robi, v, w );
		return 0;
	}

	int VWGetSpeed(VWHandle h, SpeedType* pvw)
	{
		ENTER_ROBIOS_VW;
		//TODO: implement speed ramp
		if( robi.virtualTime >= robi.collisionTime )
		{
			pvw->v = 0;
			pvw->w = 0;
		}
		else
		{
			pvw->v = robi.vwController.v;
			pvw->w = robi.vwController.w;
		}

		return 0;
	}

	int VWSetPosition(VWHandle h, meter x, meter y, radians phi)
	{
		ENTER_ROBIOS_VW;

		/* two possibilities:
		 * 1. no command, or command completed -> remainingDistance == 0 and
		 * so setting distance to 0 and the startPos to the new pos results
		 * in evaluating calcPos to startPos.
		 * 2. running command -> remainingDistance != 0 ... (thats easy) */
		robi.vwController.distance = robi.vwController.remainingDistance(
				std::min(robi.virtualTime, robi.collisionTime) );
		robi.vwController.startPos = Position( x, y, phi );
		return 0;
	}

	int VWGetPosition(VWHandle h, PositionType* pos)
	{
		ENTER_ROBIOS_VW;
		Position cp = robi.vwController.calcPos(
				std::min(robi.virtualTime, robi.collisionTime) );
		pos->x = cp.x;
		pos->y = cp.y;
		pos->phi = cp.phi;
		return 0;
	}

	int VWStartControl(VWHandle h, float Vv, float Tv, float Vw, float Tw)
	{
		/* the PI-controller parameters can be negative. it simply means that the 
		 * pi-controler may not be optimum.*/
		ENTER_ROBIOS_VW;
		return 0;
	}

	int VWStopControl(VWHandle h)
	{
		ENTER_ROBIOS_VW;
		return 0;
	}

	int VWDriveStraight(VWHandle h, meter d, meterPerSec v)
	{
		ENTER_ROBIOS_VW;
		if( v < 0 )
			return 0;
		setupVWControllers( robi, v * signum(d), 0.0, fabs(d) );
		return 0;
	}

	int VWDriveTurn(VWHandle h, radians d, radPerSec w)
	{
		ENTER_ROBIOS_VW;
		if( w < 0 )
			return 0;
		setupVWControllers( robi, 0.0, w * signum(d), fabs(d) );
		return 0;
	}

	int VWDriveCurve(VWHandle h, meter d, radians w, meterPerSec v)
	{
		ENTER_ROBIOS_VW;
		if( v <= 0.0 || d == 0.0 )
			return 0;
		setupVWControllers( robi, v * signum(d), w / fabs(d/v), fabs(d) );
		return 0;
	}

	float VWDriveRemain(VWHandle h)
	{
		ENTER_ROBIOS_VW;
		if( robi.vwController.distance.isInfinite() ||
				robi.vwController.done( std::min(robi.virtualTime, robi.collisionTime) ) )
			return 0.0;
		else
			return robi.vwController.remainingDistance(
					std::min(robi.virtualTime, robi.collisionTime) );
	}

	int VWDriveDone(VWHandle h)
	{
		ENTER_ROBIOS_VW;
		return robi.vwController.distance.isInfinite() ||
			robi.vwController.done( robi.virtualTime ) ? 1 : 0;
	}

	int VWDriveWait(VWHandle h)
	{
		ENTER_ROBIOS_VW;
		if( robi.vwController.distance.isInfinite() ||
				robi.vwController.done( robi.virtualTime ) ) {
			return 0;
		}
		robi.virtualTime = robi.vwController.startTime +
			robi.vwController.duration();
		return 0;
	}

	int VWStalled (VWHandle h)
	{
		ENTER_ROBIOS_VW;
		return
			( robi.vwController.distance.isInfinite() ||
				! robi.vwController.done( robi.virtualTime ) ) &&
			robi.virtualTime >= robi.collisionTime ? 1 : 0;
	}

} /* namespace robios */
