#include "world.h" /* for the ball things */
#include "settings.h"
#include "parser.h"
#include "libutil/string_helpers.h"
#include <iostream>

/** The rate with which all the user interface displays should be
 * updatd. The unit is one second, i.e. it is the reciprocal of frames
 * per second. The default is 0.2, which means at least 5 fps. */
double displayUpdateRate = 0.2;

/** The ratio of simulated time to real time, e.g. a value of 1.0 means
 * the simulation runs in "real time", a value of 2.0, that the
 * simulation runns twice as fast. A value of 0.0 means "as fast as
 * possible". */
double simulationToRealTimeRatio = 1.0;

/** This value is the largest distance, that a robi has from an object
 * after it collided with it, i.e. the ideal value would be 0, BUT this 
 * is not feasable. Default value will be set to one mm, since the
 * accuracy of all distance giving sensors is just one mm. [m] */
double collisionDistanceAccuracy = 0.001;

/** these are the parameters for the error model, which simply adds a
 * random number of a gauss distribution with a standard deviation
 * like specified to the corresponding value. The default value is 0,
 * which means no error is added at all. [m]/[] */
double psdStandardDeviation = 0.0;
double vwLinearDistanceStandardDeviation = 0.0;
double vwRotationalDistanceStandardDeviation = 0.0;

/** this is the deceleration factor of the ball movement, the balls
 * velocity decreases by Ball::friction * t, where t is the simulated
 * time. [m/s] */
double Ball::friction = 0.015;

/** the ball will stop, when he reached this threshold velocity, so
 * possible values are greater or equal to 0. [m/s] */
double Ball::thresholdVelocity = 0.001;

/** what could that possibly be? the default value is the radius of a
 * golf ball in [m] */
double Ball::radius = 0.02125;

/** the default values are the usual collision coefficients for an
 * elastic impact (1, -1). the first coefficient gets multiplied with the
 * component paralell to the wall, the second with the component
 * perpendicular to the wall, i.e. the second component hast to be
 * <= 0. */
double_pointxy Ball::wallReflectionCoefficients = double_pointxy(1, -1);
double_pointxy Ball::robiReflectionCoefficients = double_pointxy(1, -1);

/** the ratio between robi cpu speed and host cpu speed, e.g. 10 means
 * the host cpu is 10 times faster than the robi cpu. This value is
 * not really of any use at the moment, since the accuracy with which
 * the host cpu time can be measured is not sufficient. */
double robi2HostCPUSpeedFactor = 10;

/** the maximum distance the psd sensors can measure, values greater
 * than maxReach are mapped to the constant PSD_OUT_OF_RANGE. */
double PSD::maxReach = 2.0;

/** the background color used for the generation of the robi camera
 * images, given as (r,g,b) in float format with values between 0 and
 * 1. */
double_triple backgroundColor = double_triple(0.3);


void readSettingsFile( const char* fileName )
{
	Configuration c;
	c << "displayUpdateRate" << "simulationToRealTimeRatio"
	  << "collisionDistanceAccuracy" << "psdStandardDeviation"
	  << "vwLinearDistanceStandardDeviation" 
	  << "vwRotationalDistanceStandardDeviation"
	  << "Ball::friction" << "Ball::thresholdVelocity" << "Ball::radius"
	  << "Ball::wallReflectionCoefficients"
	  << "Ball::robiReflectionCoefficients"
	  << "robi2HostCPUSpeedFactor" << "PSD::maxReach" << "backgroundColor";

	try {
		parseConfigurationFile( fileName, c );
	} catch( const std::exception& e ) {
		cerr << "Warning: error opening ini file: " << e.what() << endl;
	}
	
#define READ_VARIABLE( VAR, TYPE ) \
	if( ! c[#VAR].empty() ) \
		VAR = str_to<TYPE>( c[#VAR] );

	READ_VARIABLE( simulationToRealTimeRatio, double )
	READ_VARIABLE( displayUpdateRate, double )
	READ_VARIABLE( collisionDistanceAccuracy, double )
	READ_VARIABLE( psdStandardDeviation, double )
	READ_VARIABLE( vwLinearDistanceStandardDeviation, double )
	READ_VARIABLE( vwRotationalDistanceStandardDeviation, double )
	READ_VARIABLE( Ball::friction, double )
	READ_VARIABLE( Ball::thresholdVelocity, double )
	READ_VARIABLE( Ball::radius, double )
	READ_VARIABLE( Ball::wallReflectionCoefficients, double_pointxy )
	READ_VARIABLE( Ball::robiReflectionCoefficients, double_pointxy )
	READ_VARIABLE( robi2HostCPUSpeedFactor, double )
	READ_VARIABLE( PSD::maxReach, double )
	READ_VARIABLE( backgroundColor, double_triple )
}
