#ifndef _sim_h_
#define _sim_h_

/* the <cc++/thread.h> file has to be included before every stdlib
 * file because otherwise the compiler reports an error while parsing
 * the pthread_rwlock_t _lock; line in this file (I don't know why
 * :-() */
#include <cc++/thread.h>
#include <vector>
#include <list>
#include <string>
#include <queue>
#include <map>
#include <fstream>
#include "Synchronization.h"
#include "timer.h"
#include "point.h"
#include "libutil/matrix.h"
#include "libutil/triple.h"
#include "features.h"

using namespace std;

namespace robios {
#include "types.h"
};

#define dbg_msg(x) cout << x << flush;

template<typename T> inline int signum( const T& v )
{ return v == T(0) ? 0 : ( v < T(0) ? -1 : +1 ); }

inline double mod2pi( double phi )
{
	return phi < 0 ? phi + 2*M_PI : (phi >= 2*M_PI ? phi - 2*M_PI : phi);
}

struct Time
{
	static const double delta = 0.05;
	/* this value is arbitrary, one simulation must not run longer
	 * that never */
	inline static const Time never() { return Time(1e200); }
	double s;

	Time( double s = 0. ) : s(s) {}
	explicit Time( size_t steps ) : s(steps * delta) {}
	inline double seconds() const { return s; }
	/* this +.001 was necessary for the problem of comparing after assigning */
	inline size_t steps() const { return size_t(s / delta + .001); }
	inline size_t steps( size_t steps ) { s = delta * steps; return steps; }
	inline Time integral() const { return Time( steps() ); }
	inline Time fractional() const
		{ double tmp; return Time( modf( s / delta, &tmp ) * delta ); }
 	inline Time& operator=( double sec ) { s = sec; return *this; }
	inline Time& operator+=( double sec ) { s += sec; return *this; }
/* 	inline Time operator-( const Time& t ) const { return Time(s - t.s); } */
/* 	inline Time operator+( double d ) const { return Time(s+d); } */

	inline operator double() const { return s; }
};

struct Position : public pointxy<double>
{
	double phi;
	Position( double x = 0.0, double y = 0.0, double phi = 0.0 ) :
		pointxy<double>( x, y ), phi(phi) {}
	Position( robios::meter x, robios::meter y, robios::radians phi ) :
		pointxy<double>( x / 1000., y / 1000. ), phi(mod2pi(phi) ) {}
	inline Position& operator=( const Position& p )
		{ pointxy<double>::operator=(p); phi = p.phi; return *this; }
	inline Position& operator=( const pointxy<double>& p )
		{ pointxy<double>::operator=(p); return *this; }
};

inline istream& operator>>( istream& is, Position& pos )
{ return is >> pos.x >> pos.y >> pos.phi; }
inline ostream& operator<<( ostream& os, const Position& pos )
{ return os << pos.x << " \t" << pos.y << " \t" << pos.phi; }

struct voodo
{
	double d;
	bool inf;
	voodo() : d(0.), inf(true) {}
	voodo( double d ) : d(d), inf(false) {}
	inline operator double() const
		{ if( inf ) throw "inf"; return d; }
	inline const double& operator=( double new_d )
		{ inf = false; return d = new_d; }
	inline bool isInfinite() const { return inf; }
	inline void setInfinite() { inf = true; }
	inline bool isNull() const { return !inf && d == 0.0; }
};

/* we need this forward declaration for the VWData constructor which
 * in turn needs it for the robis maxV/W values. */
class LocalRobi;

/** VWData stores all data that describes one drive command (either
 * just a VWSetSpeed or a VWDrive... command). Attention, the
 * modelling of moving backwards is modeled with negative v instead of
 * negative distance like in the VW... interface. The constructor
 * modifies the passed value according to the used error model. */
class VWData
{
	public:

	double v, w;
	voodo distance;
	Time startTime;

	VWData() : v(0), w(0) {}
	VWData( const Time& t, double v, double w,
			voodo distance = voodo() ):
		startTime(t), v(v), w(w), distance(distance) {}
};

class VWController : public VWData
{
	public:
	Position startPos;
	
	/* false: havn't been initialised; true: init once. */
	bool init; 
	
	VWController( const Position& startPos = Position() ) :
		startPos(startPos), init(false) {axisOffset=0;}
		// startPos(startPos), init(false) {axisOffset=0.0f;}
	
	VWController( const Time& t, double v, double w, 
			voodo distance = voodo())
		: VWData(t,v,w,distance) { axisOffset=0.0f; }

	inline void reset( const Position& pos )
		{ v = w = distance = 0.0; startPos = pos; }

	bool done( const Time& time ) const
	{
		return (stopped() || ! distance.isInfinite() &&
				time >= startTime + duration() );
	}

	inline bool stopped() const
		{ return v == 0.0 && w == 0.0 || distance.isNull(); }

	void update( const VWData& vw );
	Position calcPos( const Time& time );
	voodo remainingDistance( const Time& time ) const;
	voodo duration() const;
	
	private:	
	double axisOffset;
};

struct Ball
{
	static double friction;
	static double thresholdVelocity;
	static double radius;
	static double_pointxy wallReflectionCoefficients;
	static double_pointxy robiReflectionCoefficients;

	double_pointxy deceleration;
	double_pointxy startVelocity;
	double_pointxy startPosition;
	Time startTime;
	Time stopTime;
	/** the current position and virtual time of the ball */
	double_pointxy pos;
	Time time;

	Ball( const double_pointxy& p ) : pos(p)
		{ setState( p, double_pointxy(0,0), Time(0.0) ); }
	double_pointxy calcPosition( const Time& t ) const;
	double_pointxy calcVelocity( const Time& t ) const;

	/** sets the complete state of the ball, so that is starts at time
	 * t and position pos moving with initial speed v. */
	void setState( const double_pointxy& pos, const double_pointxy& v,
				   const Time& t );
};
inline ostream& operator<<( ostream& os, const Ball& ball )
{ return os << ball.pos << " \t" << ball.radius; }

/** static robi description */
struct Robi
{
	id_t id;
	double radius;
	string program;
	string name;
	string model;

	/* the next two variables are a duplication of information, this is 
	 * ugly. (I just wanted to make that clear :-)) */
	double axisOffset;
	/* we need this to display the psds in the gui */
	vector<Position> psdPositions;
};

struct GlobalRobi : public Robi
{
	Position pos;
	VWController vwController;
	double_pointxy kickerReflectionCoefficients;
	double kickerSpeed;

	GlobalRobi( const Position& initPos ) :
		pos( initPos ), vwController( initPos ), kickerSpeed(0),
		kickerReflectionCoefficients(Ball::robiReflectionCoefficients) {}

	template<typename T, typename U>
		inline static bool intersecting( const T& robi1, const U& robi2 )
	{
		double r = robi1.radius + robi2.radius;
		double x = robi1.pos.x - robi2.pos.x;
		double y = robi1.pos.y - robi2.pos.y;
		return r*r >= x*x + y*y;
	}

	inline GlobalRobi& operator=( const Robi& r )
		{ Robi::operator=(r); return *this; }
};

inline ostream& operator<<( ostream& os, const GlobalRobi& robi )
{ return os << robi.pos; }

struct PSD
{
	static double maxReach;
	robios::DeviceSemantics devSem;
	Position relPos;
	bool started;
	bool init;
	int cache;

	PSD() : started(false), devSem(0), cache(-1), init(false) {}
};

struct PSDs
{
	Time timeStamp;
	bool cycleMode;
	vector<PSD> psds;

	robios::PSDHandle init( robios::DeviceSemantics s );
	int start( const Time& time, robios::PSDHandle h, bool cycle );
	void stop();
	int get( const Time& time, id_t id, robios::PSDHandle h );
	bool check( const Time& time ) const;
	bool invalidHandle( robios::PSDHandle h ) const;
	bool busy( const Time& time ) const;
};

class LCD
{
	void incCursorRow();
	void clear( size_t row, size_t col );
	void extendROI( size_t x, size_t y );

public:
	cmatrix<char> textBuf;
	cmatrix<uchar_triple> imageBuf;
	pointxy<size_t> roi;
	static const size_t menuWidth = 4;
	size_t r, c;
	bool scrolling;
	bool dirtyText, dirtyImage;

	/** we need to lock the data structures, since the delivery
	 * function gets executed within an other thread context than this
	 * robi's. */
	CCriticalSection lock;

	LCD( size_t columns = 16, size_t rows = 7,
		 size_t width = 128, size_t height = 64 ) :
		textBuf(columns+1, rows+1), imageBuf(width, height),
		scrolling(false), r(0), c(0)
		{ clear(); }

	void setCursor( int row, int col );
	void putString( const char* s );
	void clear();
	void setMenu( int pos, const char* s );

	void putImage( robios::colimage* buf );
	void putImage( robios::image* buf );
	void putImage( robios::BYTE* buf );
	void setPixel( int x, int y, int c );

	string text();
};

/* I choosed to use forward declarations instead of including the
 * .h file to speed up the compilation process. */
class SoCamera;
class SoSeparator;
class SoOffscreenRenderer;

struct Camera
{
	double pan, tilt;
	double relPan, relTilt;
	double_pointxy relPos;
	double relPosZ;
	double_pointxy maxImageSize;
	SoCamera* camera;
	SoSeparator* root;
	SoOffscreenRenderer* renderer;
	
	Camera() : pan(0), tilt(0), relPan(0), relTilt(0), relPosZ(0),
		camera(0), root(0) {}
	~Camera() { release(); }
	void init();
	void release();
	bool render( robios::colimage* buf );
	void updatePos( const Position& robiPos );
	inline bool initialized() const { return root; }
};

struct LocalRobi;

/** This class represents a simulated thread on a robi. The used model a
 * cooperative one. To implement this, I use a token model, i.e. ony one
 * thread should me executed at a time, the thread with the token. */
struct LocalThread : public ost::Thread
{
	string name;
	int uid;
	int priority;
	int stackSize;
	void (*code)();

	Time nyat; /* not yet asigned robi time */
	bool suspended;
//BUG: what is the appropriate behaviour when sleeping threads are suspended?
	Time wakeupTime;
	bool semLocked;
	inline bool executable( const Time& time ) const
		{ return !suspended && !semLocked && wakeupTime <= time; }

	/* implementation dependent stuff (commonc++ Threads) */
	CSemaphore token;
	LocalRobi* localRobi;
	static ost::ThreadKey thisLocalRobi;
	static const double timeSlice = 0.01;

	LocalThread( LocalRobi& robi, char *name, void (*code)(),
				 int stksiz, int pri, int uid ) :
		name(name), code(code), stackSize(stksiz), priority(pri),
		uid(uid), localRobi(&robi), suspended(true), semLocked(false)
		{ setCancel( cancelImmediate ); }
	virtual ~LocalThread() { terminate(); }
	
	void run();
	void final();
	inline void exit()
		{ if( isThread() ) Thread::exit(); else dbg_msg("bug in exit\n"); }
};

struct LocalSem
{
	queue< LocalThread* > waitingThreads;
	int counter;

	LocalSem( int val ) : counter(val) {}
};

struct Timer
{
	robios::TimerFnc code;
	size_t scale;
	size_t counter;

	Timer( size_t scale, robios::TimerFnc code ) :
		scale(scale), code(code), counter(0) {}
	Timer() : code(0) {}
};

struct RADIOMessage
{
	Time sendingTime;
	unsigned char* buffer;
	size_t size;
	robios::BYTE receiver, sender;

	RADIOMessage() {}
	RADIOMessage( robios::BYTE rec, robios::BYTE snd,
				  unsigned char* b, size_t s, const Time& t );

	Time receptionTime() { return sendingTime; }
};

inline ostream& operator<<( ostream& os, const RADIOMessage& m )
{
	return os << "message from: " << (int)m.sender
			  << " to: " << (int)m.receiver
			  << " with: " << m.size << " bytes";
}

struct RADIOMessageQueue
{
	queue<RADIOMessage> messages;

	/** we store the threads here, that are waiting for messages */
	queue<LocalThread*> waitingThreads;

	/** we need to lock the data structures, since the delivery
	 * function gets executed within an other thread context than this
	 * robi's. */
	CCriticalSection lock;

	/** this is for faster process, if we don't use radio
	 * communication */
	bool initialized;

	void deliver( const RADIOMessage& message );
	int check();
	void receive( LocalRobi& robi, RADIOMessage& message );

	/** this method gets called together with the processTimer method
	 * to wake up waiting threads in case that a message arived. */
	void processWaitingThreads( LocalRobi& robi );

	RADIOMessageQueue() : initialized(false) {}
};

struct Keyboard
{
	int key;

	/** we need to lock the data structures, since the delivery
	 * function gets executed within an other thread context than this
	 * robi's. */
	CCriticalSection lock;

	void keyDown( int key );
	void keyUp();
	int check();
	int wait( LocalRobi& robi );
	Keyboard() : key(0) {}
};

struct LocalRobi : public Robi
{
	double maxV;
	double maxW;

	/** general stuff */
	Time virtualTime;
	Time lastSynchronization;
	timer<cputime> cpuTime;
	Time systemStartTime;

	VWController vwController;
	Time collisionTime;
	PSDs psds;
	LCD lcd;
	Keyboard keyboard;
	vector<Camera> cameras;
	RADIOMessageQueue radio;
	std::ofstream rs232out;

	/** multi-tasking */
	robios::OSMTMode mtMode;
	bool mtPermit;
	/* ATTENTION: it is important, to use a list here, to guarantee
	 * that the iterators are still valid after inserting and removing
	 * elements */
	list<LocalThread*> threads;
	typedef list<LocalThread*>::iterator thread_iterator_t;
	thread_iterator_t currentThread;
	Time ntst; /* next task switch time */
	CSemaphore mainThreadLock;
	list<LocalSem> semaphores;

	/* timer-irq */
	map<int, Timer> timers;
	int nextTimerHandle;
	static const size_t maxNumberOfTimers = 16;
	bool timerIRQHandler;
	Time ntit; /* next timer irq time */
	void processTimers();

	void processTimersAndRADIOMessages();

	LocalRobi() : mtMode(robios::NOTASK), mtPermit(false), nextTimerHandle(1),
		timerIRQHandler(false), collisionTime(Time::never()) {}

	thread_iterator_t findNextThread( thread_iterator_t i );
	thread_iterator_t doVirtualTaskSwitches( thread_iterator_t nextThread );
	void reschedule();

	void enter( const RoBiOSFunctionFeatures& f = DummyFeatures );
	void leave( const RoBiOSFunctionFeatures& f = DummyFeatures );
	void readDescription( const char* fileName );
	static LocalRobi& thisRobi();
};

struct RobiosWatchdog
{
	LocalRobi& robi;
	const RoBiOSFunctionFeatures& features;
	RobiosWatchdog( LocalRobi& r, const RoBiOSFunctionFeatures& f ) :
		robi(r), features(f) { robi.enter( features ); }
	~RobiosWatchdog() { robi.leave( features ); }
};

#define ENTER_ROBIOS \
	LocalRobi& robi = LocalRobi::thisRobi(); \
	static const RoBiOSFunctionFeatures* _features = 0; \
	if( ! _features ) \
		_features = findFunctionFeatures( __func__ ); \
	RobiosWatchdog _rwd( robi, *_features ); 

typedef size_t id_t;

/* This is the set of messages, the local robi implementation can sent
 * to the core. They use theCoreMessenger pointer. */
struct Client2CoreMessages
{
	virtual void setVWController( const VWData& vwData ) = 0;
	virtual void synchronize( const Time& time ) = 0;
	virtual void sendRADIOMessage( const RADIOMessage& message ) = 0;
	virtual void setKicker( const double_pointxy& reflectionCoefficients,
							double speed ) = 0;
};

extern Client2CoreMessages* theCoreMessenger;

#endif
