#! /usr/bin/perl -w

# Author: Mari Wang <mariw@tartarus.uwa.edu.au>
# 
# Script converting .txt files generated by c programs timing the
# RoBIOS operating system functions on the Eyebot Robots into .html
# files displaying the information in a convenient format.  
# 
# The first argument is the the html file.

if (@ARGV < 2 ) {
    print "Usage: $0 <outfile> <infile(s)>\n";
    exit;
}

# test whether file exists, prompt for overwriting
if (-e $ARGV[0]) { 
    print "$ARGV[0] allready exists, do you want to overwrite?\n";
    $answer = <STDIN>;
    chomp($answer);
    until ($answer eq "n" || $answer eq "y") {
	print "Please answer \"y\" or \"n\".\n".
	      "Answering \"n\" will cause the program to ".
	      "exit without any damage done.\n";
	$answer = <STDIN>;
	chomp($answer);
    }
    if ($answer eq "n") {
	print "Specify correct filename and try again\n";
	exit;
    }
}

# Open html file for writing
open(OUT,">".$ARGV[0])  || die "Could not open $ARGV[0]";

write_header_to_file();

shift @ARGV; 

#process text files
foreach $file (@ARGV) {
    open(FILE,"<$file") || die "Error opening $file";
    while (<FILE>) {
	$function_type = $1 if (/^Testing: (.*)$/);
	$name          = $1 if (/^Name: (.*)$/);
	$cpu_speed     = $1 if (/^Speed: (.*)$/);
	$OS_Version    = $1 if (/^Version: (.*)$/);
	last if (/^\s*$/);
    }

    write_table_header_to_file();

    %data = ();

    while (<FILE>) {
	chomp;
	next if /^[\s\4]*$/;
	if (/^Warning:/) {
	    $Warning = $_;
	    write_warning_to_file();
	    next;
	}
	print "L: \"$_\"\n";
	($func)             = m/^(.*\))/;
	($ips)              = / (\d*\.\d*)$/;
	($ms)               = 10**6/$ips;
	$data{$func}{"ips"} = $ips;
	$data{$func}{"ms"}  = round_number($ms, 2);
    }
    foreach $func (sort keys %data) {
	write_table_row_to_file();
    }
}

write_footer_to_file();

# Takes a number and the desired number of decimals, and formats the
# number accordingly
sub round_number {
    my($number,$decimal) = @_;
    $number = sprintf("%.${decimal}f", $number);
    return $number;
}

# Write header to html file 
sub write_header_to_file{
    print OUT <<EOF;
<HTML>
<HEAD>
<TITLE>Timing of RoBIOS functions</TITLE>
</HEAD>

<BODY>
<BR><BR>
<H1 ALIGN=CENTER>Timing of RoBIOS functions</H1>
<BR><BR>

<TABLE ALIGN=CENTER CELLPADDING=2 CELLSPACING=0 BORDER=1>
EOF
}

sub write_table_header_to_file{
    print OUT <<EOF;
<TR>
<TH COLSPAN=3>$function_type</TH>
</TR>

<TR>
<TD COLSPAN=3 ALIGN=CENTER>
Tested on Robot $name Running RoBIOS version $OS_Version at $cpu_speed
</TD>
</TR>

<TR>
<TD>Function(parameters)</TD>
<TD>microseconds per call</TD>
<TD>Number of calls per second</TD>
</TR>

EOF
}

# Write table row for each function
sub write_table_row_to_file {
	print OUT <<EOF;
<TR>
<TD>$func</TD>
<TD ALIGN="RIGHT">$data{$func}{"ms"}</TD>
<TD ALIGN="RIGHT">$data{$func}{"ips"}</TD>
</TR>

EOF
}

# Write footer to file
sub write_footer_to_file{
    print OUT <<EOF;

</TABLE>
</BODY>
</HTML>
EOF
}

sub write_warning_to_file{
    print OUT <<EOF;
<TR>
<TD COLSPAN='3'>$Warning for next function</TD>
</TR>
EOF
}
