/* ******** user space EyeCam driver ***
   * Author 
   *   Stephen.Humble (steve@ucc.gu.uwa.edu.au>
   * Derived from original kernel driver by 
   *   Paul McAlpine <mcalp-pr@ee.uwa.edu.au>
   *   Petter Reinholdtsen <pere@td.org.uit.no>

   code allows for noisy data because it
   resynchronises to FST at end of every frame
   resets fifo if any unusual condition occurs 
   Checks for unusual long sync or no sync pulse 
   condition on signal lines
   version .7  
   ***************************************** */
/* updated 12 Feb 2000 */
#include "eyebot.h"
#include "eyecam_lib.h"
#include <stdlib.h>
#include <stdio.h>
#include "irq.h"
#include "logger.h"
#define LARGE (1 << 24)

/*  #define INT_OFFSET_PAR (0x74) */
#define INT_OFFSET_PAR (29) 

/* stuff particular to the FIFO buffer */
#define HALFFULL 1024
#define FIFO_HALF_SIZE 1024

/* data values particular to the CMOS sensor array */
#define EC_ROWS 120
#define EC_HIGH 60
#define EC_LINE 160
#define EC_RGBSIZE 80

#define EMPTY_SET (ParStatReg & (1<<5))
#define ERROR_SET (ParStatReg & (1<<3))

   /* clear LF bit and set it to strobe data from FIFO fixed polarity +ve pulse */
#define LF_PULSE  CLEAR_BIT(ParConReg, BIT_CON_READ); \
   SET_BIT(ParConReg, BIT_CON_READ);

/* set LF bit and clear it to strobe data from FIFO fixed polarity - is now correct short +ve pulse */
#define LF_PULSE_noirq  ParConReg=0x25; \
   ParConReg=0x27

/* set LF bit and clear it to strobe data from FIFO fixed polarity - is now correct short +ve pulse */
#define LF_PULSE_irq  ParConReg=0x35; \
   ParConReg=0x37

/* setting LF_BIT makes it goto 0 volts. */
#define LF_0V  SET_BIT(ParConReg, BIT_CON_READ);

/* clearing LF_BIT makes it go to +5 volts. */
#define LF_5V   CLEAR_BIT(ParConReg, BIT_CON_READ);

#define LF_0VPULSE SET_BIT(ParConReg, BIT_CON_READ); \
   CLEAR_BIT(ParConReg, BIT_CON_READ);

#define YY 100

/* diagnostic codes for everything */
#define EYELIBhmm 'h'
#define EYELIBrealirq 'i'
#define EYELIBemptyok '0'
#define EYELIBemptyfail 'F'
#define EYELIBheysync 'S'
#define EYELIBweird 'W'
#define EYELIBemptys 'e'
#define EYELIBhalfjam 'H'
#define EYELIBackreset 'A'
#define EYELIBfulls '9'
#define EYELIBlongfst 'L'
#define EYELIBfisync 's'
#define EYELIBfiedge 't'
#define EYELIBnofst 'N'
#define EYELIBframes 'f'
#define EYELIBresets 'R'
#define EYELIBstate 's'

int hmm=0;
int realirq=0;
int emptyok=0;
int emptyfail=0;	
int heysync=0;
int weird=0;
int emptys=0;
int halfjam=0;
int ackreset=0;
int fulls=0;
int longfst=0;
int fisync=0;
int fiedge=0;
int nofst=0;
int frames=0;
int resets=0;
int state=1;
int fullup=1;
int eyecam_irqs=0;
struct eyecam_frame *Z;
struct eyecam_frame *last_frame;

int CMOSCAMStop (void);
int CMOSCAMInit(void);

/**********************************************************/
/* print diagnostic counters to screen */
int eyecam_status()
{
   printf("m%dS%dR%d\n",hmm,state,resets);
   printf("f%dE%dL%d\n",frames,emptys,longfst);
   printf("N%de%ds%d\n",nofst,fiedge,fisync);
   printf("i%dA%dH%d\n",eyecam_irqs,ackreset,halfjam);
   printf("H%de%dF%d\n",heysync,emptyok,emptyfail);
   printf("U%dW%dr%d\n",fullup,weird,realirq);
   return(eyecam_irqs);
}

/*********************************************************/
int testme()
{
  return(0);
}

/************************************************/
void WAITx()
{
  int i;
  volatile int y = 100;
  for (i=0;i<y;i++);
}

/***********************************************************/
/* reset FIFO toggle init HIGH init LOW  init HIGH */ 
int reset_fifo()
{
  WAITx();
  CLEAR_BIT(ParConReg, BIT_CON_READ);   /* dont want read */
  SET_BIT(ParConReg, BIT_CON_INIT);     /* make sure reset is high */
  CLEAR_BIT(ParConReg, BIT_CON_INIT);   /* reset FIFO */
  SET_BIT(ParConReg, BIT_CON_INIT);     /* unreset FIFO */
  resets++;
  return(0); 
}

/******************************************************/
/* try to empty the fifo fail if i cant */
int empty_fifo()
{
  int x=0;
  for(x=0;x<10000 && EMPTY_SET;x++) {LF_PULSE; }
  if(x<10000){
    emptyok++;
    return(0);
  }else{
    emptyfail++;
    return(-1);
  }
}

/***********************************************************/
int restart_fifo()
{
  int reloop=0;
  CMOSCAMStop();
 top:
 if(reloop++ >30)return(-1);
 LF_5V;
 reset_fifo();
  if(!ACK_SET){    /* half full? */

    if(!EMPTY_SET){
      printf("EMPTY+ACK?\n");
      weird++;
      reset_fifo();
      goto top;
    }
    if(empty_fifo()<0){
      reset_fifo();
    }
    if(!ACK_SET){
      reset_fifo();
      goto top;
    }
  }
  if(ACK_SET){  /* not half full at all */
    if(!ERROR_SET){
      printf("NoAck+FULL?\n");
      weird++;
      reset_fifo();
      goto top;
    }
    if(empty_fifo()<0){
      reset_fifo();
    } 
  }
  CMOSCAMInit();
  return(0);
}

/***************************************************************/
/* returns total buffer size. */
/* returns 0 if fifo fails to work correctly */
int fifo_getsize()
{
  int now;
  int x=0;
  CMOSCAMStop();
  reset_fifo();  /* reset fifo so it's ready to work */
  CMOSCAMInit();  /* start the camera to load the fifo up */

  now=OSGetCount();
  while(ERROR_SET){
    if((now+10)>OSGetCount())return(0);  /* wait for fifo to fill, timeout failure returns 0. */
  }
 CMOSCAMStop();

/*  empty the fifo fail if i cant */
  for(x=0;x<20000 && EMPTY_SET;x++) {LF_PULSE; }
  if(x<20000){
    return(x);
  }else{
    return(0);
  }
}

/**********************************************************/
/* give user pointer to latest image data */
int eyecam_kickme()
{
 emptyok=0;
 emptyfail=0;	
 heysync=0;
 weird=0;
 emptys=0;
 halfjam=0;
 ackreset=0;
 fulls=0;
 longfst=0;
 fisync=0;
 fiedge=0;
 nofst=0;
 frames=0;
 resets=0;
 fullup=1;
  restart_fifo();
  return(eyecam_irqs);
}
/**********************************************************/
/* scale value is downsize multiplier */
/* use this function to get an initialised frame structure */
struct eyecam_frame *eyecam_frame_alloc(int scale)
{
  int size=0;
  struct eyecam_frame *Zn=malloc(sizeof(struct eyecam_frame));
  Zn->state=FR_START;
  Zn->next=0;
  Zn->v_scale=scale;
  Zn->h_scale=scale;
  Zn->v_size=EC_RGBSIZE/scale;
  Zn->h_size=EC_HIGH/scale;
  size=sizeof(short)*Zn->v_size*Zn->h_size;
  if((Zn->red=malloc(size)) ==NULL){
    printf("aloc red error");
    return(0);
  }
  if((Zn->green=malloc(size))==NULL){
    printf("aloc green error");
    return(0);
  }
  if((Zn->blue=malloc(size))==NULL ){
    printf("aloc blue error");
    return(0);
  }
  if((Zn->green2=malloc(size))==NULL){
    printf("aloc green2 error");
    return(0);
  }
  Zn->fifo_ef=0;
  Zn->fifo_ff=0;
  Zn->fifo_hf=0;
  printf("aloc%db RGG2B\n",size);
  return(Zn);
}

/**********************************************************/
/* routine reads a single video line of data
 * this would be nice to turn into assembly language and shift into TPURAM */
/* __inline */
int readline(short int **col1,short int **col2)
{
  register int x;
  short int *a1;  
  #ifdef EC_DSIZE
  for(x=EC_DSIZE;x>0;x--){
    LF_PULSE;
      }
#endif
  a1=*col1;
  for(x=EC_RGBSIZE;x>0;x--){
    *a1++= *(volatile BYTE *)ParBase;
    LF_PULSE;
   }
  *col1=a1;

  a1=*col2;
  for(x=EC_RGBSIZE;x>0;x--){
    *a1++= *(volatile BYTE *)ParBase;
    LF_PULSE;
   }
  *col2=a1;
       /*    was originally *(*col2)++= *(volatile BYTE *)ParBase; */
  return(0);
}

/**********************************************************/
/* this routine skips a single video line of data 
   this would be nice to turn into assembly language and shift into TPU RAM */
int skipline()
{
  int x;
  for(x=0;x<EC_LINE;x++)	/* read thru data bits */
    {
      LF_PULSE;
	}
  return(0);
}

/**********************************************************/
/**********************************************************/
/**********************************************************/
/*
  Half full interrupt handler.
  fills up the user provided eyecam_frame structure and then when it's full
  it uses the next pointer to start on the next frame
  video output format is 4 linear word arrays planes for red , green, green2  and blue
  see structure definition for detail
  I suggest you set the next pointer to the same structure if you wish to prevent
  the FIFO filling up.
*/
#define FI_SYNC 1
#define FI_EDGE 2
#define FI_RUN 3
#define FI_ERROR 4
short int *red;
short int *blue;
short int *green;
short int *green2;
int tics=0;


int set_pointers(struct eyecam_frame *Zn)
{
    red=Zn->red;
    green=Zn->green;
    blue=Zn->blue;
    green2=Zn->green2;
    tics=0;
    return(1);
}

/* put eycam capture into re synchronise mode */
int resync(struct eyecam_frame *Zn)
{
	state=FI_SYNC;
	log("err");
      set_pointers(Zn);
	Z->fifo_ef++;
	restart_fifo();
	return(1);
}

void eyecam_read(void)
{
  int x;
  static int row=0;
  int count=3;
  hmm=0;	/* keep track of actual number of loop itterations in in one interrupt */
  /***********************************************read in some video lines of data *****/
  eyecam_irqs++;
  do{
    hmm++;
    count--;
    if(!ACK_SET){
      ackreset++;
      count=5;    /* read 5 chunks after half full goes out */
    };

    /* check our half full eventualy goes out - otherwise reset FIFO */
    if(hmm>=50){
      resync(Z);
      halfjam++;
      count=-1;
      break;
    }
    if(!EMPTY_SET){  /* check fifo is not empty */
	emptys++;
      	continue;
    }

    if(!ERROR_SET){  /* is FIFO full ? re synchronize and reset frame buffer */
      resync(Z);
      restart_fifo();
      fullup++;
      continue;
    }
    if(state==FI_RUN)
      {
	if(!BUSY_SET){
	  heysync++;
	}
	row++;
	if(row>=(EC_ROWS-1)){		/* check for last line */
	  frames++;
	  Z->state=FR_DONE;
	  last_frame=Z;
	  if(Z->next!=0)Z=Z->next; 
	  row=0;
	  set_pointers(Z);
	  state=FI_SYNC;
	  Z->state=FR_RUNNING;
	  continue;
	}

	for(x=0;x<12;x++){   /* get rid of the data bytes- i will remove this part */
	  LF_PULSE;
	}
	if(row&1){	/* odd line or even line (red green or blue green)? */
	  readline(&red,&green);
	}else{
	  readline(&green2,&blue);
	} 
	continue;
      }

    /***********************************************wait for sync to go positive *****/
    if(state==FI_SYNC)
      {
	for(x=0;x<EC_LINE && !BUSY_SET;x++){
	  LF_PULSE;
	}
	if(x<EC_LINE){
	  fisync++;
	  state=FI_EDGE;
	  tics=0;
	  continue;
	}else{
	  tics+=180;
	}
	if(tics>=(EC_LINE*EC_ROWS)){
	  /* re synchronize and reset frame buffer */
	  resync(Z);
	  restart_fifo();
	  nofst++;
	  /* printf("No FST pulse\n"); */
	  tics=0;
	}	/* FST error */
	continue;
      }
    /***********************************************wait for sync to go low again *****/
    if(state==FI_EDGE)
      {
	for(x=0;x<EC_LINE && BUSY_SET;x++){
	  LF_PULSE;
	}
	if(x<EC_LINE){
	  fiedge++;
	  state=FI_RUN;
	  row=0;
	  continue;
	}else{
	  tics+=180;
	}
	if(tics>(300*EC_LINE)){
	  /* re synchronize and reset frame buffer */
	  resync(Z);
	  restart_fifo();
	  longfst++;
	  continue;
	}	/* FST error - i guess a FIFO reset would be good */
      }
  }while(count>0);		/* read in about 4 chunks of data */
}

  /**************************   read in some video lines of data *****/
void eyecam_irq(void)
{
 realirq++;
  printf("realirq %d\n",eyecam_irqs);
  AUBeep();
  eyecam_read();
  IS_BIT_SET(ParStatReg, BIT_STATUS_INTP); /* Reset parallell port INTP line */
  /* Evil hack.  Restore registers, stored automatically by C on the top! */
  /* __asm("  movm.l -20(%a6),#0x43c\n"
	"  unlk %a6\n"
	"  rte");
  */
	__asm("  unlk %a6\n"
	"  rte");
  return;
}

/**********************************************************/
int CMOSCAMInit (void)
{
   BYTE Setup_0_Val = 0x07;  /* 8-wire parallel mode, autobrightness on */
   BYTE Setup_1_Val = 0x64;  /* 66=164*124,  65=160x120 pixels, 8-bit ADC, 60 fps timing */
   BYTE Setup_4_Val = 0x49;  /* normal FST, fast QCK, validate sync pulse and image data */
   BYTE Clk_Div_Val = 0x02;  /* 0=30fps, 1=15fps, 2=7.5fps, 3=3.7fps (clock divisor = 2^3 = 8) */
/* dummy message for startup */
   writeMessage(0xff, 0x00);

/* write to camera registers */
   if (writeMessage(SETUP_0, Setup_0_Val) == 255)
      return 255;
   if (writeMessage(SETUP_1, Setup_1_Val) == 255)
      return 255;
   if (writeMessage(SETUP_4, Setup_4_Val) == 255)
      return 255;
   if (writeMessage(CLK_DIV, Clk_Div_Val) == 255)
      return 255;
   return(0);
}

/**********************************************************/
int CMOSCAMStop (void)
{
    BYTE Setup_4_Val = 0x41;  /* disable FST, fast QCK, validate sync pulse and image data */
   /* dummy message for startup */
   writeMessage(0xff, 0x00);
   if (writeMessage(SETUP_4, Setup_4_Val) == 255)return 255;
      return(0);
}

int
eyecam_finish(void)
{
  CMOSCAMStop();
  free_irq(INT_OFFSET_PAR, 0);

  return 0;
}

/********************************************************/
/* settup the FIFO , eyecam and the interrupt system */
int eyecam_go(struct eyecam_frame *buf)
{
  volatile int a=0;
  int retry=5;
  /* reset diagnostic counters */
  emptyok=0;
  emptyfail=0;	
  heysync=0;
  weird=0;
  emptys=0;
  halfjam=0;
  ackreset=0;
  fulls=0;
  longfst=0;
  fisync=0;
  fiedge=0;
  nofst=0;
  frames=0;
  resets=0;
  fullup=1;
  eyecam_irqs=0;

  Z=buf;
  red=Z->red;
  blue=Z->blue;
  green=Z->green;
  green2=Z->green2;

  SET_BIT(ParConReg, BIT_CON_MODE); /* Set input mode */
  ParStatReg=0xAA; /* Set input mode */
  ParDataReg=0xFF; /* Set input mode */

  SET_BIT(ParConReg, BIT_CON_INIT); /* unreset fifo */

  CLEAR_BIT(ParConReg, BIT_CON_INTR);  /* Disable parport interrupt */

  while((CMOSCAMInit()==255)&&retry-->0){ /* error condition check */
    for (a=0;a<200;a++)WAITx();
  }

  /* enable parport interrupt */
  /*
  if (0 > request_irq(INT_OFFSET_PAR, eyecam_irq, 0, "EYEcam FIFO",0))
   LCDPutString("request_irq() failed");
   SET_BIT(ParConReg, BIT_CON_INTR);

   OSEnable();
   OSAttachTimer(1, eyecam_read);
  */
   /* void(*)(void)INT_OFFSET_PAR(); */
  return 0;
}

/**********************************************************/
/* give user pointer to latest image data */
struct eyecam_frame *eyecam_get()
{
   return(last_frame);
}

/**********************************************************/
/* give user back image structure */
struct eyecam_frame *eyecam_getframe()
{
   return(last_frame);
}

/**********************************************************/
void acknowledge(void)
{
    SDA_clear();
    WAITx();
    SCL_set();
    WAITx();
    SCL_clear();
    WAITx();
}

/**********************************************************/
BYTE readByte(void)
{
    int i;
    BYTE result = 0;

    for (i=0;i<8;i++) {
        SCL_set();

        WAITx();

        result <<= 1;
        result += SDA_read();

        SCL_clear();
        WAITx();
    }

    return result;
}

/**********************************************************/
int writeMessage(BYTE index, BYTE data)
{
    BYTE address = WRITE;
	
    /* generate start condition */
    startCondition();

    /* send message */
    if (sendByte(address) == 255)
        return 255;
    if (sendByte(index) == 255)
        return 255;
    if (sendByte(data) == 255)
        return 255;

    /* generate stop condition */
    stopCondition();
    return 0;
}

/**********************************************************/
void stopCondition(void)
{
    SCL_set();
    SDA_set();
    WAITx();
}

/**********************************************************/
void startCondition(void)
{
    SDA_set();
    SCL_set();
    WAITx();
    SDA_clear();
    WAITx();
    SCL_clear();
}

/**********************************************************/

void SDA_set(void)
{
    BYTE b;
	
    b = OSReadParCTRL();

    b |= SDA;
    OSWriteParCTRL(b);
}

/**********************************************************/

int SDA_read(void)
{
    BYTE b;

    b = OSReadParCTRL() & SDA;
    return (b>0);
}

/**********************************************************/

void SDA_clear(void)
{
    BYTE b;
	
    b = OSReadParCTRL();

    b &= ~SDA;
    OSWriteParCTRL(b);
}

/**********************************************************/

void SCL_clear(void)
{
    BYTE b;
	
    b = OSReadParCTRL();

    b &= ~SCL;
    OSWriteParCTRL(b);
}

/**********************************************************/

void SCL_set(void)
{
    BYTE b;
	
    b = OSReadParCTRL();

    b |= SCL;
    OSWriteParCTRL(b);
}
 
/*****************hiya*****************************************/ 

int sendByte(BYTE value)
{
    int i;

    for (i=0;i<8;i++) {
        if ((value & 0x80) == 0) {
            SDA_clear();
        } else {
            SDA_set();
        }

        WAITx(); /* set up data */
        SCL_set(); /* clock */
        WAITx(); /* let cam read it */
        SCL_clear();
        WAITx(); 

        value <<= 1;
    }

    SCL_set();
#if OURWAY
    while (SDA_read());
#endif
    WAITx();

    if (SDA_read() == 1)
        return (255);
	
    SCL_clear();

    return 0;
}













