/** @name Bit locations of the various pins in the byte
	returned by OSReadParCTRL and OSReadParSR and of
	course OSWrite... */
/*@{*/  

#ifndef EYECAM
#define ParDataReg (*(((volatile BYTE*)ParBase)+0))
#define ParStatReg (*(((volatile BYTE*)ParBase)+1))
#define ParConReg  (*(((volatile BYTE*)ParBase)+2))

#define CLEAR_BIT(a, b)  ((a) &= ~ (1<<(b)))
#define SET_BIT(a, b)    ((a) |=   (1<<(b)))
#define IS_BIT_SET(a, b) ((a) &    (1<<(b)))
/* Parallel port control register bits */
#define BIT_CON_READ 1
#define BIT_CON_INIT 2
#define BIT_CON_INTR 4
#define BIT_CON_MODE 5

#define BIT_STATUS_INTP 6
#define BIT_STATUS_BUSY 7

#define BUSY_SET (ParStatReg & (1<<7))
#define ACK_SET (ParStatReg & (1<<6))
#define EMPTY_SET (ParStatReg & (1<<5))
#define ERROR_SET (ParStatReg & (1<<3))

#define BYTE unsigned char
/** QCK (00001000): (SR) pixel sample clock. */
#define QCK 0x08 /* look at OSReadParSR() doc before changing */
/** FST (00010000): (SR) Frame start signal.*/
#define FST 0x10
/** SIN (00000100): (CTRL) Frame timing reset. */
#define SIN 0x04
/** SCL (00000001): (CTRL) Serial bus clock. */
#define SCL 0x01
/** SDA (00001000): (CTRL) Serial bus data. */
#define SDA 0x08
/*@}*/ 

/** @name Serial initialization data */
/*@{*/ 
/** Write data byte (see under section labled 'Serial Interface' in the camera
    specification document). */ 
#define WRITE  	0X20 	

/** The address of the camera (with the read bit switched on). */
#define READ  	0X21

/** The address of the Setup 0 register in the VV 6300.  This register needs
    to be accessed in order to set the camera to 8 wire parallel output mode.*/
#define SETUP_0 0x10
/** The address of the Setup 4 register in the VV 6300.  This register needs
	to be accessed in order to set the QCK in free running mode and enable FST. */
#define SETUP_1 0x11
#define SETUP_4 0x14
/** The address of the Clock Divisor register.  This register need to be 
    accessed in order to change the clock divisor value.*/
#define CLK_DIV	0X25
/*@}*/ 


/** @name Camera initialization functions */
/*@{*/      
/** The VV 6300 defaults to transmitting in 4 wire parallel mode and its 
  * default clock divisor is 1.  For interfacing with the eyebot we require
  * the camera to transmit via 8 parallel wires and we also require the camera
  * to assume a clock divisor of 8.  This function initializes the camera to
  * these specific settings for the eyebot.
  */
int CMOSCAMInit (void);

/** Generates a start condition for sending serial data */
void startCondition (void);

/** Generates a stop condition for finalising serial data transmission */
void stopCondition (void);

/** Sends data through the serial interface of the eyebot to the VV 6300.  
  * The information is split into three main parts: a read/write byte, an 
  * address byte and a data byte.  
  */
int writeMessage (BYTE index, BYTE data);

/** Outputs a byte to the serial line.*/
int sendByte (BYTE value);

/** Reads a byte from the serial line.*/
BYTE readByte (void);

/** Set the SDA pin low */
void SDA_clear (void);

/** Set the SDA pin high -- should only be used without
	the camera connected */
void SDA_set (void);

/** Read the value of the SDA pin*/
int SDA_read (void);

/** Set the SCL pin low */
void SCL_clear (void);

/** Set the SCL pin high */
void SCL_set (void);

void acknowledge (void);

/** Function to display an error message and quit the program */
void error (char *string);

#define FR_START 0
#define FR_RUNNING 1
#define FR_DONE 2

struct eyecam_frame{
      struct eyecam_frame *next;
      int v_scale;
      int h_scale;
      int v_pos;
      int h_pos;
      int v_size;
      int h_size;
      short int *red;
      short int *blue;
      short int *green;
      short int *green2;
      int state;
      int fifo_ef;
      int fifo_ff;
      int fifo_hf;
      int error;
};

struct eyecam_frame *eyecam_frame_alloc(int);
/* main for vision system driver */
int eyecam_go(struct eyecam_frame *);

struct eyecam_frame *eyecam_get();
int eyecam_kickme();
int eyecam_status();
int empty_fifo();
void eyecam_irq(void);
void eyecam_read(void);
int testme(void);
int eyecam_finish(void);

#define EYECAM yes
#endif
