#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <pthread.h>
#include <time.h>
#include <errno.h>

#include "forms.h"
#include "protos.h"
#include "kern.h"
#include "eyesim.h"

#define MAX_TCB 1000

BYTE ThreadMode = 255;
int NumOfThreads;
struct tcb threads_tcb[MAX_TCB];
pthread_t threads_t[MAX_TCB];
struct sem threads_sem[MAX_TCB];

int NumOfQueue;
struct tcbq *ThreadsInQueue;

int indexThread(void);
void thread_start(struct tcb *curtcb);
void run_routine(void (*fn)(void));
static void forms_redraw(void);


int OSMTInit (BYTE mode) {
  NumOfThreads = 0;
  NumOfQueue = 0;
  if (mode == COOP)
    fprintf(stderr, "COOP mode not implemented, Default to PREEMPT\n");
  ThreadMode = PREEMPT;
  forms_redraw();
  return 0;
}


struct tcb *OSSpawn(char *name, int code, int stksiz, int pri, int uid) {
  if (NumOfThreads == MAX_TCB) {
    fprintf(stderr,"Maximum threads spawned exceeded\n");
    return (struct tcb *)NULL;
  }
  threads_tcb[NumOfThreads].next = (struct tcb*)NULL;
  if(NumOfThreads==0) 
    threads_tcb[NumOfThreads].prev = (struct tcb*)NULL;
  else
    threads_tcb[NumOfThreads].prev = &threads_tcb[NumOfThreads-1];
  threads_tcb[NumOfThreads].semq = (struct tcb*)NULL;
  threads_tcb[NumOfThreads].stksiz = stksiz;
  threads_tcb[NumOfThreads].stackbase = (char *)calloc(stksiz,sizeof(char));
  if(pri != MIN_PRI)
    fprintf(stderr, "Priority threads not implemented, default to MINPRI\n");
  threads_tcb[NumOfThreads].pri = code;   /* Use to store start code addr */
  threads_tcb[NumOfThreads].stat = TASK_NEW;
  threads_tcb[NumOfThreads].name = (char *)malloc(strlen(name));
  memcpy(threads_tcb[NumOfThreads].name, name, strlen(name));
  threads_tcb[NumOfThreads].uid = uid;
  OSSemInit(&threads_sem[NumOfThreads],0); 
  if (pthread_create(&threads_t[NumOfThreads], NULL, (void*)&thread_start, \
			&threads_tcb[NumOfThreads]) == EAGAIN) {
    fprintf(stderr,"Unable to spawn new thread\n");
    forms_redraw();
    return (struct tcb *)NULL;
  }
  NumOfThreads++;
  forms_redraw();
  return threads_tcb+NumOfThreads-1;
}


int OSReady(struct tcb *thread) {
  int i=0, index = -1; 

  if (thread == (struct tcb *)NULL) {
    fprintf(stderr, "OSReady: Thread not spawn\n");
    forms_redraw();
    return -1;
  }
  while (i<NumOfThreads)
    if((int)thread==(int)&threads_tcb[i]) {
      index = i;
      i = NumOfThreads;
    } else
      i++;      
  if(index == -1) {
    fprintf(stderr,"OSReady: Thread not found\n");
    return -1;
  }
  if(threads_tcb[index].stat == TASK_READY) {
    fprintf(stderr,"OSReady: Threads running already\n");
    return 0;
  }      
  threads_tcb[index].stat = TASK_READY;
  OSSemV(&threads_sem[index]);
  forms_redraw();
  return 0;
}


int OSSuspend(struct tcb *thread) {
  int i = 0, index = -1;

  if (thread == (struct tcb *)NULL) {
    fprintf(stderr, "OSSuspend: Thread not spawn\n");
    forms_redraw();
    return -1;
  }
 
  while (i<NumOfThreads)
    if((int)thread==(int)&threads_tcb[i]) {
      index = i;
      i = NumOfThreads;
    } else
      i++;
  if (pthread_equal(pthread_self(), threads_t[index])) {
    thread->stat = TASK_SUSPEND;
    OSSemV(&threads_sem[index]);
    return 0;
  } 
  fprintf(stderr,"Cannot suspend non-current threads\n");
  forms_redraw();
  return -1;
}


int OSReschedule() {
  if(sched_yield() == -1) {
    fprintf(stderr,"Unable to yield\n");
    printf("OSReschedule: Unable to yield\n");
    forms_redraw();
    return -1;
  }
  forms_redraw();
  return 0;
}


int OSYield() {
  int i;

  if((i=indexThread())==-1) {
    fprintf(stderr,"OSYield: critical error\n");
    return -1;
  }
  threads_tcb[i].stat = TASK_SUSPEND;  
  OSSemP(&threads_sem[i]);
  OSReschedule();
  forms_redraw();
  return 0;
}


int OSRun(struct tcb *thread) {
  OSReady(thread);
  OSReschedule();
  forms_redraw();
  return 0;
}


int OSGetUID(struct tcb *thread) {
  int i = 0;

  if (thread == (struct tcb *)NULL) {
    if((i=indexThread())==-1) {
      fprintf(stderr,"OSYield: critical error\n");
      forms_redraw();
      return -1;
    }
    forms_redraw();
    return threads_tcb[i].uid;
  }
  forms_redraw();
  return thread->uid;
}


int OSKill(struct tcb *thread) {
  int i = 0, index = -1;
  void *thread_status;

  if (thread == (struct tcb *)NULL) {
    if((i=indexThread())==-1) {
      fprintf(stderr,"OSKill: critical error\n");
      forms_redraw();
      return -1;
    }
    pthread_exit(NULL);
    forms_redraw();
    return 0;
  }
  while (i<NumOfThreads)
    if (pthread_equal(pthread_self(), threads_t[i])) {
      index = i;
      i = NumOfThreads;
    } else
      i++;

  if(index==-1) {
    fprintf(stderr,"OSKill: Threads not found\n"); 
    forms_redraw();
    return -1;
  }
  pthread_cancel((int)&threads_t[index]);
  i = pthread_join((int)threads_t, &thread_status);
  if (i != 0 && i != ESRCH) {
    fprintf(stderr,"Thread joined failed\n");
    forms_redraw();
    return -1;
  }
  forms_redraw();
  return 0;
}


int OSExit(int code)
{
  forms_redraw();
  pthread_exit((void*)code);
  return 0;
}


int OSPanic(char *msg) {
  fprintf(stderr,"%s", msg);
  forms_redraw();
  return 0;
}


int OSSleep(int n) {
  time_t waitSec;
  unsigned short waitMilli;
  struct timeb nowtime;

  ftime(&nowtime);
  waitSec = n / 100 + nowtime.time;
  waitMilli = (n % 100) * 10 + nowtime.millitm;
  if(waitMilli >= 1000)  {
    waitSec++;
    waitMilli = waitMilli % 1000;
  }
  while(((nowtime.time-waitSec) < 0) || (nowtime.millitm < waitMilli))  {
    sched_yield();
    ftime(&nowtime);
  }

  forms_redraw();
  return 0;
}


int OSForbid() {
  fprintf(stderr,"OSForbid not implemented yet\n");
  forms_redraw();
  return 0;
}


int OSPermit() {
  fprintf(stderr,"OSPermit not implemented yet\n");
  forms_redraw();
  return 0;
}


/**************************  Thread Routine  **************************/

static void
forms_redraw(void)
{
  OSSemP(&sf);
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  OSSemV(&sf);
}

int indexThread() {
  int i = 0, index = -1;
  
  while(i<NumOfThreads) 
    if(pthread_equal(pthread_self(), threads_t[i])) {
      index = i;
      i = NumOfThreads;
    } else
      i++;
  return index;
}

void run_routine(void (*fn)(void)) {
  fn();
  forms_redraw();
}


void thread_start(struct tcb* curtcb) {
  int code = curtcb->pri,i;

  curtcb->pri = MIN_PRI;
  if((i=indexThread()) == -1) 
    fprintf(stderr,"Threads not spawn\n");
  else {
    OSSemP(&threads_sem[i]);
    run_routine(code);
  }
  forms_redraw();
}
