/*-----------------------------------------------------------------------------
|  Filename: camsim.c
|
|  Author: Nicholas Tay (tay-nt@ee.uwa.edu.au)
|	   { Adapted from xfqcam.c - written by Michael Rudolph }
|
|  Description: Routines that enable and use the camera
|
+------------------------------------------------------------------------------
|  Functions:
|
| int CAMInit (void); 			| Reset and initialize Quickcam
| int CAMGetFrame ( image *buf);	| Read an image from a b/w cam
| int CAMGetColFrame ( colimage *buf);  | Read an image from a color cam
| int CAMSet (int bright,int para1,int para2);
|					| Set camera hardware parameters
| int CAMGet (int *bright,int *offsetORhue,int *contrastORsaturation);
|					| Get camera hardware parameters
| int CAMMode ( int mode); 		| Set the display to the given mode
L____________________________________________________________________________*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <forms.h> 

#include "LCDDisp.h"
#include "protos.h"
#include "improc.h"
#include "qcam.h"

/* CAMInit
 *****************************************************************************
 *	Input: 	NONE
 *	Output:	returncode
 *		255 = no camera connected
 *		254 = camera init error
 *		1 - 15 = b/w camera version
 *		16 - 31 = color camera version
 ****************************************************************************/

int CAMInit()
{
    int privatecmap = 0;

    if(!lcd_initialised)
        InitialiseLCD();
    if (QC_init() == -1)  {
        fprintf(stderr,"quickcam already in use ?! \n");
	fprintf(stderr,"(remove /tmp/.quickcam and try again)\n");
	return (254);
    }
    if (_QCConfig.quickcam_port != 0)
    {
        if(_QCConfig.quickcam_port != 0x278 &&
           _QCConfig.quickcam_port != 0x378 &&
           _QCConfig.quickcam_port != 0x3bc)
        {
            fprintf(stderr, "Invalid port address\n");
            return (255);
        }

        ioperm(_QCConfig.quickcam_port,3,1);

    }
    else
    {
        _QCConfig.quickcam_port = QC_find();
        if (_QCConfig.quickcam_port < 0)
        {
            fprintf(stderr,"Can't get IO perms\n");
            return(255);
        }
        else if(_QCConfig.quickcam_port == 0)
        {
            fprintf(stderr,"Can't find a camera\n");
            return (255);
        }
    }

    QC_reset(_QCConfig.quickcam_port);

    CalcXferMode();

    xDepth = fl_state[fl_get_vclass()].depth;

    if (dbg) fprintf(stderr, "X color depth = %d bits\n", xDepth);

    if(xDepth == 8 || xDepth == 4)
        RawToX = RawTo8BitX;
    else if (xDepth == 24)
        RawToX = RawTo24BitX;
    else
    {
        fprintf(stderr, "LCD only works in 8/24 bit X depths\n");
        fprintf(stderr, "This diplay has a %d bit color depth\n", xDepth);
        QC_exit();
	return (254);
    }
    screen_num = DefaultScreen(disp);
    gc = XCreateGC(disp, FL_ObjWin(fd_eyebot->LCDDisplay), 0, &values);

    if (privatecmap)
    {
        cmap = XCreateColormap(disp,
                               win,
                               DefaultVisual(disp, screen_num),
                               AllocNone);

        XSetWindowColormap(disp, win, cmap);
    }
    else
        cmap = DefaultColormap(disp, screen_num);

    xqc_createpalette(cmap);

    ximage = NULL;
    NewImage(_QCConfig.xsize, _QCConfig.ysize);

    /* Attempt to get permission to access IO ports.  Must be root */

    setuid(getuid());

    QC_reset(_QCConfig.quickcam_port);
    QC_set_all();
    scrolling = scrolling | (GRAPHICS >> 1);

    return 0;
}

/* CAMGetFrame(image *buf)
 *****************************************************************************
 *	Input: 	NONE
 *	Output: (buf) Pointer containing the image data
 ****************************************************************************/

int CAMGetFrame(image *buf)  {
    GetImage(*buf);

    return 0;
}

/* CAMGetColFrame(colimage *buf)
 *****************************************************************************
 *	Input:	NONE
 *	Output: (buf) Pointer to a color image
 ****************************************************************************/

int CAMGetColFrame(colimage *buf) {
  return 0;
}

/* CAMSet(int bright, int para1, int para2)
 *****************************************************************************
 *	Input:	(bright) a value for brightness (0-255)
 *		(para1)  a value for whitebalance (0-255)
 *		(para2)  a value for contrast (0-255)
 *	Output: NONE
 ****************************************************************************/

int CAMSet(int bright, int para1, int para2)  {

  _QCConfig.brightness   = bright;
  _QCConfig.whitebalance = para1;
  _QCConfig.contrast	 = para2;
  QC_set_all();

  return 0;
}

/* CAMGet(int *bright, int *offset, int *contrast)
 ****************************************************************************
 * 	Input:  (bright) a pointer to the storing place for camera brightness
 *              (para1) a pointer to the storing place for whitebalance
 *              (para2) a pointer to the storing place for contrast 
 *      Output: (bright) the current brightness value
 *              (para1) the current offset (b/w cam)/hue (color cam) value
 *              (para2) the current contrast 
 ****************************************************************************/

int CAMGet(int *bright, int *offset, int *contrast) {

  bright[0]   = _QCConfig.brightness;
  offset[0]   = _QCConfig.whitebalance;
  contrast[0] = _QCConfig.contrast;

  return 0;
}

/* CAMMode(int mode)
 ****************************************************************************
 *	Input:  (mode) the camera mode you want
 *              Valid values are: (NO)AUTOBRIGHTNESS
 *      Output: NONE
 ***************************************************************************/

int CAMMode(int mode)  {

    return 0;
}
