#include <stdio.h>
#include <stdlib.h> 
#include <forms.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "helpscreen.h"
#include "eyesim.h"
#include "LCDDisp.h"
#include "protos.h"


static const char * const keywords[] = {
  "key1",
  "key2",
  "key3",
  "key4",
  "world",
  "param",
  "zoom",
  "unzoom",
  "center",
  "pause",
  "help",
  "exit",
  "reset",
  "robot",
  "key",
  "lcd",
  "cam",
  "vw",
  "improc",
  "image",
  "system",
  "audio",
  "psd",
  "servo",
  "motor",
  "bump",
  "ad",
  "TEMP",
  NULL  };

static const char * const improcText[] = {
"Image Processing:",
"=================",
"",
"int IPLaplace (image *src, image *dest);", 
"  Input:      (src) source b/w image",
"  Output:     (dest) destination b/w image",
"  Semantics:  The Laplace operator is applied to the source image",
"              and the result ist written to the destination image", 
"",
"int IPSobel (image *src, image *dest);",
"  Input:      (src) source b/w image",
"  Output:     (dest) destination b/w image",
"  Semantics:  The Sobel operator is applied to the source image",
"              and the result ist written to the destination image",
"", 
"int IPDither (image *src, image *dest);",       
"  Input:      (src) source b/w image",
"  Output:     (dest) destination b/w image",
"  Semantics:  The Dithering operator with a 2x2 pattern is applied", 
"              to the source image and the result ist written to the ",
"              destination image", 
"",
"int IPDiffer (image *current, image *last, image *dest);",
"  Input:      (current) the current b/w image",
"              (last) the last read b/w image",
"  Output:     (dest) destination b/w image",
"  Semantics:  The Laplace operator is applied to the source image",
"              and the result ist written to the destination image", 
"",
"int IPColor2Grey (colimage *src, image *dest);", 
"  Input:      (src) source color image",
"  Output:     (dest) destination b/w image",
"  Semantics:  The Laplace operator is applied to the source image",
"              and the result ist written to the destination image ",
NULL
};

static const char * const keyText[] = {
"Key Functions:",
"==============",
"",
"int KEYGetBuf (char *buf);",  
"  Input:      (buf) a pointer to a character buffer",
"  Output:     (buf) the keycode is written into the buffer",
"              Valid keycodes are: KEY1,KEY2,KEY3,KEY4 (keys from ",
"              left to right)     ",
"  Semantics:  Wait for a keypress and store the keycode into the buffer",
"",
"int KEYGet (void);",
"  Input:      NONE",
"  Output:     (returncode) the keycode of a pressed key is returned",
"              Valid keycodes are: KEY1,KEY2,KEY3,KEY4 (keys from left",
"              to right)              ",
"  Semantics:  Wait for a keypress and return keycode ",
"",
"int KEYRead (void);", 
"  Input:      NONE",
"  Output:     (returncode) the keycode of a pressed key is returned or 0",
"              Valid keycodes are: KEY1,KEY2,KEY3,KEY4 (keys from left",
"              to right) or 0 for no key.",
"  Semantics:  Read keycode and return it. Function does not wait.", 
"",
"int KEYWait (int excode);", 
"  Input:      (excode) the code of the key expected to be pressed",
"              Valid keycodes are: KEY1,KEY2,KEY3,KEY4 (keys from left ",
"              to right) or ANYKEY.",
"  Output:     NONE",
"  Semantics:  Wait for a specific key" ,
NULL
};

static const char * const lcdText[] = {
"LCD Functions:",
"==============",
"",
"int LCDClear( void );", 
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Clear the LCD", 
"",
"int LCDPutChar (char char);",
"  Input:      (char) the character to be written",
"  Output:     NONE",
"  Semantics:  Write the given character to the current cursor position",
"              and increment cursor position", 
"",
"int LCDSetChar ( int row,int column,char char); ",
"  Input:      (char) the character to be written",
"              (column) the number of the column ",
"              Valid values are: 0-15",
"              (row) the number of the row",
"              Valid values are: 0-6",
"  Output:     NONE",
"  Semantics:  Write the given character to the given display position",
"",
"int LCDPutString ( char *string);", 
"  Input:      (string) the string to be written",
"  Output:     NONE",
"  Semantics:  Write the given string to the current cursor position",
"              and increment cursor position", 
"",
"int LCDSetString ( int row,int column,char *string);",    
"  Input:      (string) the string to be written",
"              (column) the number of the column",
"              Valid values are: 0-15",
"              (row) the number of the row",
"              Valid values are: 0-6",
"  Output:     NONE",
"  Semantics:  Write the given string to the given display position",
"",
"int LCDMode ( int mode);", 
"  Input:      (mode) the display mode you want",
"              Valid values are: (NON)SCROLLING|(NO)CURSOR",
"  Output:     NONE",
"  Semantics:  Set the display to the given mode",
"              SCROLLING: the display will scroll up one line, when the",
"              right botton corner ist reached and the new cursor",
"              position will be the first column of the now blank bottom",
"              line",
"              NONSCROLLING: display output will resume in the top left",
"              corner when the bottom right corner is reached",
"              NOCURSOR: the blinking hardware cursor is not displayed at",
"              the current cursor position",
"              CURSOR: the blinking hardware cursor is displayed at the",
"              current cursor position",
"",
"int LCDSetPos (int row, int column); ", 
"  Input:      (column) the number of the column ",
"              Valid values are: 0-15",
"              (row) the number of the row",
"              Valid values are: 0-6",
"  Output:     NONE",
"  Semantics:  Set the cursor to the given position",
"",
"int LCDPutGraphic ( image *buf);", 
"  Input:      (buf) pointer to a greyscale image (80*60 pixel)",
"  Output:     NONE",
"  Semantics:  Write the given graphic b/w to the display",
"              it will be written starting in the top left corner ",
"              down to the menu line.", 
"",
"int LCDPutImage ( BYTE *buf);", 
"  Input:      (buf) pointer to a b/w image (128*64 pixel)",
"  Output:     NONE",
"  Semantics:  Write the given graphic b/w to the hole display",
"",
"int LCDMenu (char *string1, char *string2, char *string3,char *string4);",
"  Input:      (string1) menu entry above key1",
"              (string2) menu entry above key2",
"              (string3) menu entry above key3 ",
"              (string4) menu entry above key4",
"              Valid Values are a string with max 4 characters, which ",
"              clears the menu entry and writes the new one or \"\" (NULL)",
"  Output:     NONE",
"  Semantics:  Fill the menu line with the given menu entries",
"",
"int LCDMenuI (int pos, char *string);",
"  Input:      (pos) number of menu entry to be exchanged (1..4)",
"              (string) menu entry above key <pos> a string with max 4",
"              characters",
"  Output:     NONE",
"  Semantics:  Overwite the menu line entry at position pos with the ",
"              given string",
"",
"int LCDSetPixel (int row,int col,BOOL val);", 
"  Input:      (val) pixel operation code",
"              Valid codes are:        0 = clear pixel",
"                                      1 = set pixel",
"                                      2 = invert pixel",
"              (column) the number of the column ",
"              Valid values are: 0-127",
"              (row) the number of the row",
"              Valid values are: 0-63",
"  Output:     NONE",
"  Semantics:  Apply the given operation to the given pixel position",
"",
"int LCDInvertPixel (int row, int col);  ",
"  Input:      (column) the number of the column ",
"              Valid values are: 0-127",
"              (row) the number of the row",
"              Valid values are: 0-63",
"  Output:     NONE",
"  Semantics:  Invert the pixel at the given pixel position",
"",
"int LCDGetPixel (int row, int col);", 
"  Input:      (column) the number of the column ",
"              Valid values are: 0-127",
"              (row) the number of the row",
"              Valid values are: 0-63",
"  Output:     (returncode) the value of the pixel",
"              Valid values are:   1 for set pixel",
"                                  0 for clear pixel",
"  Semantics:  Return the value of the pixel at the given position",
"",
"int LCDLine(int x1, int y1, int x2, int y2, int col);",
"  Input:      (x1,y1) (x2,y2) and color",
"  Output:     NONE",
"  Semantics:  Draw a line from (x1,y1) to (x2,y2) using the Bresenham",
"              Algorithm. top left is 0,0 - bottom right is 127,63",
"              color: 0 white",
"                     1 black",
"                     2 negate image contents",
"",
"int LCDArea(int x1, int y1, int x2, int y2, int col);",
"  Input:      (x1,y1) (x2,y2) and color    ",
"  Output:     NONE",
"  Semantics:  Fill rectangular area from (x1,y1) to (x2,y2) it must hold:",
"              x1<x2 AND y1<y2",
"              top left is 0,0 - bottom right is 127,63",
"              color: 0 white",
"                     1 black",
"                     2 negate image contents",
NULL
};

static const char * const camText[] = {
"Quickcam Functions:",
"===================",
"",
"int CAMInit (int zoom);", 
"  Input:      (zoom) zoom factor",
"              Valid Values are: WIDE,NORMAL,TELE",
"  Output:     (returncode) Cameraversion or Errorcode",
"              Valid values are:  255 = nocamera connected",
"                                 254 = camera init error",
"                                0-15 = version of b/w camera",
"                               16-31 = version of color camera ",
"  Semantics:  Reset and initialize connected Quickcam " ,
"",
"int CAMGetFrame ( image *buf); ",
"  Input:      (buf) a pointer to a b/w image",
"  Output:     NONE",
"  Semantics:  Read an image from b/w cam",
"",
"int CAMGetColFrame ( colimage *buf);", 
"  Input:      (buf) a pointer to a color image",
"  Output:     NONE",
"  Semantics:  Read an image from color cam",
"",
"int CAMSet (int bright,int para1,int para2);", 
"  Input:      (bright) a value for the camera brightness",
"              (para1) a value for offset (b/w cam)/hue (color cam)",
"              (para2) a value for contrast (b/w cam)/saturation (color cam)",
"              Valid values are: 0-255",
"  Output:     NONE",
"  Semantics:  Set camera hardware parameters",
"",
"int CAMGet (int *bright,int *offsetORhue,int *contrastORsaturation);", 
"  Input:      (bright) a pointer to the storing place for camera brightness",
"              (para1) a pointer to the storing place for offset(b/w cam)",
"              /hue (color cam)",
"              (para2) a pointer to the storing place for contrast(b/w cam)",
"              /saturation (color cam)",
"  Output:     (bright) the current brightness value",
"              (para1) the current offset (b/w cam)/hue (color cam) value",
"              (para2) the current contrast (b/w cam)/saturation (color cam)"
"              value",
"              Valid values are: 0-255",
"  Semantics:  Get camera hardware parameters",
"",
"int CAMMode ( int mode);", 
"  Input:      (mode) the camera mode you want",
"              Valid values are: (NO)AUTOBRIGHTNESS",
"  Output:     NONE",
"  Semantics:  Set the display to the given mode",
"              AUTOBRIGHTNESS: the brightness value of the camera is",
"              automatically adjusted",
"              NOAUTOBRIGHTNESS: the brightness value is not automatically",
"              adjusted",
NULL
};

static const char * const systemText[] = {
"System Functions:",
"=================",
"",
"misc:",
"-----",
"",
"int OSError(char *msg,int number,BOOL dead);",
"  Input:      (msg) pointer to message",
"              (number) int number",
"              (dead) switch to choose deadend or keywait",
"              Valid values are:  0 = no deadend",
"                                 1 = deadend",
"  Output:     NONE",
"  Semantics:  Print message and number to display then",
"              stop processor (deadend) or wait for key",
"",
"int OSMachineType(void);",
"  Input:      NONE",
"  Output:     Type of used hardware",
"              Valid values are: VEHICLE, PLATFORM, WALKER",
"  Semantics:  Inform the user in which environment the program runs.", 
"",
"interrupts:",
"-----------",
"",
"int OSEnable ( void );", 
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Enable all cpu-interrupts",
"",
"int OSDisable ( void );", 
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Disable all cpu-interrupts",
"",
"variable save to tpuram:",
"------------------------",
"",
"int OSGetVar(int num);",
"  Input:      (num) number of tpupram save location",
"              Valid values are: SAVEVAR1-4 for word saving",
"              SAVEVAR2a-4a/1b-4b for byte saving",
"  Output:     (returncode) the value saved",
"              Valid values are: 0-65535 for word saving ",
"              0-255 for byte saving",
"  Semantics:  Get the value from the given save location",
"",
"int OSPutVar(int num, int value);", 
"  Input:      (num) number of tpupram save location",
"              Valid values are:  SAVEVAR1-4 for word saving",
"              SAVEVAR1a-4a/1b-4b for byte saving",
"              (value) value to be stored",
"              Valid values are: 0-65535 for word saving ",
"              0-255 for byte saving ", 
"  Output:     NONE",
"  Semantics:  Save the value to the given save location",
"",
"multithreading:",
"---------------",
"",
"int OSMTInit(BYTE mode); ",
"  Input:      (mode) operation mode",
"              Valid values are: COOP=DEFAULT,PREEMT",
"  Output:     NONE",
"  Semantics:  Initialize multithreading environment",
"",
"tcb *OSSpawn(char *name,int code,int stksiz,int pri,int uid);",
"  Input:      (name) pointer to thread name",
"              (code) thread start address",
"              (stksize) size of thread stack",
"	       (pri) thread priority",
"              Valid values are: MINPRI-MAXPRI",
"              (uid) thread user id",
"  Output:     (returncode) pointer to initialized thread control block",
"  Semantics:  Initialize new thread, tcb is initialized and inserted in ",
"              scheduler queue but not set to READY",
"",
"int OSReady(struct tcb *thread);",
"  Input:      (thread) pointer to thread control block",
"  Output:     NONE",
"  Semantics:  Set status of given thread to READY",
"",
"int OSSuspend(struct tcb *thread);",
"  Input:      (thread) pointer to thread control block",
"  Output:     NONE",
"  Semantics:  Set status of given thread to SUSPEND",
"",
"int OSReschedule();", 
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Choose new current thread",
"",
"int OSYield();  ",
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Suspend current thread and reschedule",
"",
"int OSRun(struct tcb *thread); ",
"  Input:      (thread) pointer to thread control block",
"  Output:     NONE",
"  Semantics:  READY given thread and reschedule",
"",
"int OSGetUID(thread);  ",
"  Input:      (thread) pointer to thread control block",
"  Output:     (returncode) UID of thread",
"  Semantics:  Get the UID of the given thread",
"", 
"int OSKill(struct tcb *thread); ",
"  Input:      (thread) pointer to thread control block",
"  Output:     NONE",
"  Semantics:  Remove given thread and reschedule",
"",
"int OSExit(int code); ",
"  Input:      (code) exit code",
"  Output:     NONE",
"  Semantics:  Kill current thread with given exit code and message",
"",
"int OSPanic(char *msg);",
"  Input:      (msg) pointer to message text",
"  Output:     NONE",
"  Semantics:  Dead end multithreading error, print message to display",
"              and stop processor",
"",
"int OSSleep(int n);",
"  Input:      (n) number of 1/100 secs to sleep",
"  Output:     NONE",
"  Semantics:  Let current thread sleep for n*1/100 seconds",
"",
"int OSForbid(); ",
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  disable thread switching in preemtive mode",
"",
"int OSPermit();", 
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  enable thread switching in preemtive mode",
"",
"In the functions described above the parameter \"thread\" can always be ",
"a pointer to a tcb or 0 for current thread.",
"",
"semaphores:",
"-----------",
"",
"int OSSemInit(struct sem *sem,int val);",
"  Input:      (sem) pointer to a semaphore",
"              (val) start value",
"  Output:     NONE",
"  Semantics:  Initialize semaphore with given start value",
"",
"int OSSemP(struct sem *sem);       ",
"  Input:      (sem) pointer to a semaphore",
"  Output:     NONE",
"  Semantics:  Do P operation",
"",
"int OSSemV(struct sem *sem);     ",
"  Input:      (sem) pointer to a semaphore",
"  Output:     NONE",
"  Semantics:  Do V operation",
"",
"timer:",
"------",
"int OSSetTime(int hrs,int mins,int secs);     ",
"  Input:      (hrs) value for hours",
"              (mins) value for minutes",
"              (secs) value for seconds",
"  Output:     NONE",
"  Semantics:  Set system clock to given time",
"",
"int OSGetTime(int *hrs,int *mins,int *secs,int *ticks);",
"  Input:      (hrs) pointer to int for hours"
"              (mins) pointer to int for minutes",
"              (secs) pointer to int for seconds",
"              (ticks) pointer to int for ticks",
"  Output:     (hrs) value of hours",
"              (mins) value of minutes",
"              (secs) value of seconds",
"              (ticks) value of ticks",
"  Semantics:  Get system time, one second has 100 ticks",
"",
"int OSShowTime(void);",
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Print system time to display",
"",
"int OSGetCount(void);",
"  Input:      NONE",
"  Output:     (returncode) number of 1/100 seconds since last reset",
"  Semantics:  Get the number of 1/100 seconds since last reset",
"",
"int OSWait (int n);",
"  Input:      (n) time to wait",
"  Output:     NONE",
"  Semantics:  Busy loop for n*1/100 seconds", 
"",
"timer-irq:",
"----------",
"",
"TimerHandle OSAttachTimer(int scale, TimerFnc function);",
"  Input:      (scale) prescale value for 100Hz Timer (1 to ...)",
"              (TimerFnc) function to be called periodically",
"  Output:     (TimerHandle) handle to referrence the IRQ-slot",
"              A value of 0 indicates an error due to a full list(max. 16).",
"  Semantics:  Attach a irq-routine (void function(void)) to the irq-list.",
"              The scale parameter adjusts the call frequency (100/scale Hz)",
"              of this routine to allow many different applications.",
"",
"int OSDetachTimer(TimerHandle handle);",
"  Input:      (handle) handle of a previous installed timer irq",
"  Output:     0 = handle not valid",
"              1 = function successfully removed from timer irq list",
"  Semantics:  Detach a previously installed irq-routine from the irq list.",
NULL};

/*
download and rs232:
-------------------
int OSDownload(char *name,int *bytes,int baud,int handshake,int interface);     **)
	Input:          (name) pointer to programm name array
        			(bytes) pointer to bytes transferred int
		        	(baud) baud rate selection
			        Valid values are: SER9600,SER19200,SER38400,SER57600,
			        SER115200(only SERIAL2-3)
        			(handshake) handshake selection
		        	Valid values are: NONE,RTSCTS
        			(interface): serial interface
        			Valid values are: SERIAL1-3
	Output:         (returncode)
	                 0 = no error, download incomplete - call again
	                99 = download complete
					 1 = receive timeout error
					 2 = receive status error
					 3 = send timeout error
					 5 = srec checksum error
					 6 = user canceled error
					 7 = unknown srecord error
					 8 = illegal baud rate error
					 9 = illegal startadr. error
                    10 = illegal interface
	Semantics:      Load user program with the given serial setting
        			and get name of program
		        	this functions must be called in a loop until
        			the returncode is !=0. In the loop the bytes 
        			that have been transfered already can be calculated
        			from the bytes that have been transfered in this
        			round

int OSInitRS232(int baud,int handshake,int interface);
	Input:          (baud) baud rate selection
			        Valid values are: SER9600,SER19200,SER38400,SER57600,
			        SER115200(only SERIAL2-3)
        			(handshake) handshake selection
		        	Valid values are: NONE,RTSCTS
        			(interface) serial interface
        			Valid values are: SERIAL1-3
	Output:         (returncode)
                     0 = ok
					 8 = illegal baud rate error
                    10 = illegal interface
	Semantics:      Initialize rs232 with given setting

int OSSendRS232(char *chr,int interface);     
	Input:          (chr) pointer to a character
		        	(interface) serial interface
			        Valid values are: SERIAL1-3
	Output:         (returncode)     
	                 0 = good
					 3 = send timeout error
                    10 = illegal interface
	Semantics:      Send a character over rs232

int OSRecvRS232(char *buf,int interface);         
	Input:          (buf) pointer to a character array
        			(interface) serial interface
		        	Valid values are: SERIAL1-3
	Output:         (returncode)     
	                 0 = good
					 1 = receive timeout error
					 2 = receive status error
                    10 = illegal interface
	Semantics:      Receive a character over rs232

int OSFlushInRS232(int interface);
	Input:      	(interface) serial interface
		        	Valid values are: SERIAL1-3
	Output:         (returncode)     
	                 0 = good
                    10 = illegal interface
	Semantics:      resets status of receiver and flushes its FIFO
        			very useful in NOHANDSHAKE-mode to bring the FIFO in a
		        	defined condition before starting to receive

int OSFlushOutRS232(int interface);
	Input:      	(interface) serial interface
		        	Valid values are: SERIAL1-3
	Output:         (returncode)     
	                 0 = good
                    10 = illegal interface
	Semantics:      flushes the trasnmitter-FIFO
        			very useful to abort current transmission to host
		        	(ex: in the case of a not responding host)

int OSCheckInRS232(int interface);
	Input:      	(interface) serial interface
		        	Valid values are: SERIAL1-3
	Output:         (returncode) the number of chars currently
        			available in FIFO
	Semantics:      useful to read out only packages of a certain size

int OSCheckOutRS232(int interface);
	Input:      	(interface) serial interface
		        	Valid values are: SERIAL1-3
	Output:         (returncode) the number of chars currently waiting
			        in FIFO
	Semantics:      useful to test if the host is receiving properly
        			or to time transmission of packages in the speed the
		        	host can keep up with

int USRStart();                                                                 **)
	Input:          NONE
	Output:         NONE
	Semantics:      Start loaded user programm,

int USRResident(char *name, BOOL mode);                                         **)
	Input:          (name) pointer to name array
			(mode) mode
			Valid values are: SET,GET
	Output:         NONE
	Semantics:      Make loaded user program reset resistant
			SET     save startaddress and programmname
			GET     restore startaddress and programmname

**) this function must not be used in user programs !!!!
*/

static const char * const audioText[] = {
"Audio Functions:",
"================",
"",
"int AUPlaySample(char* sample);",
"  Input:      (sample) pointer to sample data",
"  Output:     (returncode) playfrequency for given sample",
"              0 if unsupported sampletype",
"  Semantics:  Plays a given sample (nonblocking)",
"              supported formats are:",
"              WAV or AU/SND  (8bit, pwm or mulaw)",
"              5461, 6553, 8192, 10922, 16384, 32768 (Hz)",
"",
"int AUCheckSample(); ",
"  Input:      NONE",
"  Output:     FALSE while sample is playing",
"  Semantics:  nonblocking test for sample end",
"",	
"int AUTone(int freq, int msec);",
"  Input:      (freq) tone frequency",
"              (msecs) tone lenght",
"  Output:     NONE",
"  Semantics:  Plays tone with given frequency for the given time ",
"              (nonblocking) supported formats are:",
"              freq = 65 Hz to 21000 Hz",
"              msecs = 1 msec to 65535 msecs",
"",
"int AUCheckTone(); ",
"  Input:      NONE",
"  Output:     FALSE while tone is playing",
"  Semantics:  nonblocking test for tone end",
"", 
"int AUBeep();",
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  BEEP!",
"",
"int AURecordSample(BYTE* buf, long len, long freq); ",
"  Input:      (buf) pointer to buffer",
"              (len) bytes to sample + 28 bytes header",
"              (freq) desired samplefrequency",
"  Output:     (returncode) real samplefrequency",
"  Semantics:  Samples from microphone into buffer with given",
"              frequency (nonblocking)",
"              Recordformat: AU/SND (pwm) with unsigned 8bit samples",
"",
"int AUCheckRecord();     ",
"  Input:      NONE",
"  Output:     FALSE while recording",
"  Semantics:  nonblocking test for record end",
"",	
"int AUCaptureMic();",
"  Input:      NONE",
"  Output:     (returncode) microphone value (10bit)",
"  Semantics:  Get microphone input value" ,
NULL
};

static const char * const psdText[] = {
"PSD Functions:",
"==============",
"",
"PSDHandle PSDInit(DeviceSemantics semantics);",
"  Input:      (semantics) unique definition for desired PSD (see hdt.h)",
"  Output:     (returncode) unique handle for all further operations",
"  Semantics:  Initialize single PSD with given semantics",
"              Up to 8 PSDs can be initialised",
"",
"int PSDRelease();",
"  Input:      NONE",
"  Output:     NONE",
"  Semantics:  Stops all measurings and releases all initialised PSDs",
"",
"int PSDStart(PSDHandle bitmask, BOOL cycle);      ",
"  Input:      (bitmask) sum of all handles to which parallel measuring",
"              should be applied",
"              (cycle)   TRUE  = continous measuring",
"                        FALSE = single measuring",
"  Output:     (returncode) status of start-request",
"                -1 = error (false handle)",
"                0 = ok",
"                1 = busy (another measuring blocks driver)",
"  Semantics:  Starts a single/continous PSD-measuring (ca. 60ms per shot)",
"",
"int PSDStop();",
"  Input:      NONE",
"  Output:     NONE"
"  Semantics:  Stops actual continous PSD-measuring after completion of",
"              the current shot",
"",
"BOOL PSDCheck();",
"  Input:      NONE",
"  Output:     (returncode) TRUE if a valid result is available",
"  Semantics:  nonblocking test if a valid PSD-result is available",
"",
"int PSDGet(PSDHandle handle);",
"  Input:      (handle) handle of the desired PSD",
"              0 for timestamp of actual measure-cycle",
"  Output:     (returncode) actual distance in mm", 
"  Semantics:  Delivers actual timestamp or distance measured by the",
"              selected PSD ",
"",
"int PSDGetRaw(PSDHandle handle);",
"  Input:      (handle) handle of the desired PSD",
"              0 for timestamp of actual measure-cycle",
"  Output:     (returncode) actual raw-data ",
"  Semantics:  Delivers actual timestamp or raw-data measured by the",
"              selected PSD " ,
NULL 
};

static const char * const servoText[] = {
"Servo/Motor Functions:",
"======================",
"",
"ServoHandle SERVOInit(DeviceSemantics semantics);",
"  Input:      (semantics) semantic (see hdt.h)",
"  Output:     (returncode) ServoHandle",
"  Semantics:  Initialize given servo",
"",
"int SERVORelease (ServoHandle handle)",
"  Input:      (handle) sum of all ServoHandles which should be released ",
"  Output:     (returncode) 0 = ok",
"                           0x11110000 = totally wrong handle",
"  Semantics:  Release given servos",
"",
"int SERVOSet (ServoHandle handle,int angle);",
"  Input:      (handle) sum of all ServoHandles which should be set parallel",
"              (angle) servo angle",
"              Valid values: 0-255",
"  Output:     (returncode)  0 = ok",
"                           -1 = error wrong handle",
"  Semantics:  Set the given servos to the same given angle", 
"",
"MotorHandle MOTORInit(DeviceSemantics semantics);",
"  Input:      (semantics) semantic (see hdt.h)",
"  Output:     (returncode) MotorHandle",
"  Semantics:  Initialize given motor",
"",
"int MOTORRelease (MotorHandle handle)",
"  Input:      (handle) sum of all MotorHandles which should be released ",
"  Output:     (returncode) , 0 = ok",
"  Semantics:  Release given motor",
"",
"int MOTORDrive (MotorHandle handle,int speed);",
"  Input:      (handle) sum of all MotorHandles which should be driven",
"              (speed) motor speed in percent",
"              Valid values: -100 - 100 (full backward to full forward)",
"              0 for full stop",
"  Output:     (returncode) 0 = ok",
"  Semantics:  Set the given motors to the same given speed",
"",
"QuadHandle QuadInit(DeviceSemantics semantics);",
"  Input:      (semantics) semantic",
"  Output:     (returncode) QuadHandle or 0 for error",
"  Semantics:  Initialize given Quadrature-Decoder (up to 8 decoder are",
"              possible)",
"",
"int QuadRelease(QuadHandle handle);",
"  Input:      (handle) sum of decoder-handles to be released",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Release one or more Quadrature-Decoder ",
"",
"int QuadReset(QuadHandle handle);",
"  Input:      (handle) sum of decoder-handles to be reseted",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Reset one or more Quadrature-Decoder ",
"",
"int QuadRead(QuadHandle handle);",
"  Input:      (handle) ONE decoder-handle",
"  Output:     32bit counter-value (0 to 2^32-1)",
"              a wrong handle will ALSO result in an 0 counter-value!!",
"  Semantics:  Read actual Quadratue-Decoder counter",
"",
"DeviceSemantics QUADGetMotor(DeviceSemantics semantics);",
"  Input:      (handle) ONE decoder-handle",
"  Output:     semantic of the corresponding motor",
"              0 = wrong handle",
"  Semantics:  Get the semantic of the corresponding motor",
"",
"float QUADODORead(QuadHandle handle);",
"  Input:      (handle) ONE decoder-handle",
"  Output:     meters since last odometer-reset",
"  Semantics:  Get the distance from the last resetpoint of a single motor!", 
"              It is not the overall meters driven since the last reset!",
"              It is just the nr of meters left to go back to the startpoint.",
"              Usefull to implement a PID-control",
"",
"int QUADODOReset(QuadHandle handle);",
"  Input:      (handle) sum of decoder-handles to be reseted",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Resets the simple odometer(s) to define the startpoint " ,
NULL
};

static const char * const vwText[] = {
"V-Omega Interface:",
"==================",
"",
"VWHandle VWInit(DeviceSemantics semantics, int Timescale);",
"  Input:      (semantics) semantic",
"              (Timescale) prescale value for 100Hz IRQ (1 to ...)",
"  Output:     (returncode) VWHandle or 0 for error",
"  Semantics:  Initialize given VW-Driver (only 1 can be initialized!)",
"              The motors and encoders are automatically reserved!!",
"              The Timescale allows to adjust the tradeoff between",
"              accuracy (scale=1, update at 100Hz) and speed(scale>1, ",
"              update at 100/scale Hz).",
"",
"int VWRelease(VWHandle handle);",
"  Input:      (handle) VWHandle to be released",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Release VW-Driver, stop motors ",
"",
"int VWSetSpeed(VWHandle handle, meterPerSec v, radPerSec w);",
"  Input:      (handle) ONE VWHandle",
"              (v) new linear speed",
"              (w) new rotation speed",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Set the new speed: v(m/s) and w(rad/s not degree/s)",
"",
"int VWGetSpeed(VWHandle handle, SpeedType* vw);",
"  Input:      (handle) ONE VWHandle",
"              (vw) pointer to record to store actual v, w values",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Get the actual speed: v(m/s) and w(rad/s not degree/s)",
"",
"int VWSetPosition(VWHandle handle, meter x, meter y, radiant phi);",
"  Input:      (handle) ONE VWHandle",
"              (x) new x-position",
"              (y) new y-position",
"              (phi) new heading",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Set the new position: x(m), y(m) phi(rad not degree)",
"",
"int VWGetPosition(VWHandle handle, PositionType* pos);",
"  Input:      (handle) ONE VWHandle",
"              (pos) pointer to record to store actual position (x,y,phi)",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Get the actual position: x(m), y(m) phi(rad not degree)",
"",
"int VWStartControl(VWHandle hand, float Vv, float Tv, float Vw, float Tw);",
"  Input:      (hand) ONE VWHandle",
"              (Vv) the paramter for the proportional component of ",
"              the v-controller",
"              (Tv) the paramter for the integrating component of",
"              the v-controller",
"              (Vw) the paramter for the proportional component of",
"              the w-controller",
"              (Tv) the paramter for the integrating component of",
"              the w-controller",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Enable the PI-controller for the vw-interface and set ",
"              the parameters.  As default the PI-controller is deactivated",
"              when the vw-interface is initialised. The controller tries to",
"              keep the desired speed (set with VWSetSpeed) stable by adapting",
"              the energy of the involved motors. The parameters for the",
"              controller have to be choosen carefully!  ",
"              The formula for the controller is:",
"                                                t",
"                    new(t) = V*(diff(t) + 1/T * [diff(t)dt )",
"                                                0 ",
"                      V: a value usually around 1.0",
"                      T: a value usually between 0 and 1.0",
"              After enabling the controller the last set speed (VWSetSpeed)"
"              is taken as the speed to be held stable.",
"",
"int VWStopControl(VWHandle handle);",
"  Input:      (handle) ONE VWHandle",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Disable the controller immediately. The vw-interface continues",
"              normally with the last valid speed of the controller.",
"",
"int VWGetPosition(VWHandle handle, PositionType* pos);",
"  Input:      (handle) ONE VWHandle",
"              (pos) pointer to record to store actual position (x,y,phi)",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  Get the actual position: x(m), y(m) phi(rad not degree)",
"",
"int VWDriveStraight(VWHandle handle, meter delta, meterpersec v);",
"  Input:      (handle) ONE VWHandle",
"              (delta)  distance to drive in m (pos. -> forward)",
"                                              (neg. -> backward)",
"              (v)      speed to drive with",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  drives distance \"delta\" with speed v straight ahead (forward",
"              or backward) any subsequent call of VWDriveStraight, -Turn,",
"              -Curve or VWSetSpeed while this one is still being executed,",
"              results in an immediate interruption of this command",
"",
"int VWDriveTurn(VWHandle handle, radiant delta, radPerSec w);",
"  Input:      (handle) ONE VWHandle",
"              (delta)  degree to turn in radiant (pos. -> counter-clockwise)",
"                                                 (neg. -> clockwise)",
"              (w)      speed to turn with",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  turns about \"delta\" with speed w on the spot (clockwise or",
"              counter-clockwise).  Any subsequent call of VWDriveStraight,",
"              -Turn, -Curve or VWSetSpeed while this one is still being",
"              executed, results in an immediate interruption of this command", 
"",
"int VWDriveCurve(VWHandle han, meter del_l, radiant del_phi, meterpersec v);",
"  Input:      (han)     ONE VWHandle",
"              (del_l)   length of curve_segment to drive in m (pos. -> forward)",
"                                                              (neg. -> backward)",
"              (del_phi) degree to turn in radiant (pos. -> counter-clockwise)",
"                                                  (neg. -> clockwise)",
"              (v)       speed to drive with",
"  Output:     0 = ok",
"              -1 = error wrong handle",
"  Semantics:  drives a curve segment of length \"delta_l\" with overall",
"              vehicle turn of \"delta_phi\" with speed v (forw. or backw. /",
"              clockw. or counter-clockw.). any subsequent call of", 
"              VWDriveStraight, -Turn, -Curve or VWSetSpeed while this one is",
"              still being executed, results in an immediate interruption",
"              of this command",
"",
"float VWDriveRemain(VWHandle handle);",
"  Input:      (handle) ONE VWHandle",
"  Output:     0.0 = previous VWDriveX command has been completed",
"              any other value = remaining distance to goal ",
"  Semantics:  remaining distance to goal set by VWDriveStraight, -Turn",
"              (for -Curve only the remaining part of delta_l is reported) ",
"",
"int VWDriveDone(VWHandle handle);",
"  Input:      (handle) ONE VWHandle",
"  Output:     -1 = error wrong handle",
"              0 = vehicle is still in motion",
"              1 = previous VWDriveX command has been completed",
"  Semantics:  checks if previous VWDriveX() command has been completed",
"",
"int VWDriveWait(VWHandle handle);",
"  Input:      (handle) ONE VWHandle",
"  Output:     -1 = error wrong handle",
"              0 = previous VWDriveX command has been completed",
"  Semantics:  blocks the calling process until the previous VWDriveX()",
"              command has been completed"  ,
NULL
};

static const char * const bumpText[] = {
"Bumper/IR Functions:",
"====================",
"",
"BumpHandle BUMPInit(DeviceSemantics semantics);",
"  Input:      (semantics) semantic",
"  Output:     (returncode) BumpHandle or 0 for error",
"  Semantics:  Initialize given bumper (up to 16 bumpers are possible)",
"",
"int BUMPRelease(BumpHandle handle);",
"  Input:      (handle) sum of bumper-handles to be released",
"  Output:     (returncode)",
"              0 = ok",
"              error (nothing is released):",
"  Semantics:  Release one or more bumper      ",
"",
"int BUMPCheck(BumpHandle handle, int* timestamp);",
"  Input:      (handle) ONE bumper-handle",
"              (timestamp) pointer to an int where the timestamp is placed",
"  Output:     (returncode) ",
"              0 = bump occured, in *timestamp is now a valid stamp",
"              -1 = no bump occured or wrong handle, *timestamp is cleared",
"  Semantics:  Check occurence of a single bump and return the timestamp(TPU).",
"              The first bump is recorded and held until BUMPCheck is called.",
"",
"IRHandle IRInit(DeviceSemantics semantics);",
"  Input:      (semantics) semantic",
"  Output:     (returncode) IRHandle or 0 for error",
"  Semantics:  Initialize given IR-sensor (up to 16 sensors are possible)",
"",
"int IRRelease(IRHandle handle);",
"  Input:      (handle) sum of IR-handles to be released",
"  Output:     (returncode) 0 = ok",
"  Semantics:  Release one or more IR-sensors",
"",
"int IRRead(IRHandle handle);",
"  Input:      (handle) ONE IR-handle",
"  Output:     (returncode) ",
"              0/1 = actual pinstate of the TPU-channel",
"              -1 = wrong handle",
"  Semantics:  Read actual state of the IR-sensor"  ,
NULL
};

/*
OutLatch Functions:
===================

BYTE OSReadInLatch(int latchnr);
	Input:          (latchnr) number of desired Inlatch (range: 0..3)
	Output:         actual state of this inlatch
	Semantics:      reads contents of selected inlatch

BYTE OSWriteOutLatch(int latchnr, BYTE mask, BYTE value);
	Input:          (latchnr) number of desired Outlatch (range: 0..3)
			(mask)    and-bitmask of pins which should be cleared
			(inverse!)
			(value)   or-bitmask of pins which should be set
	Output:         previous state of this outlatch
	Semantics:      modifies an outlatch and keeps global state consistent
			example: OSWriteOutLatch(0, 0xF7, 0x08); sets bit4
			example: OSWriteOutLatch(0, 0xF7, 0x00); clears bit4

BYTE OSReadOutLatch(int latchnr);
	Input:          (latchnr) number of desired Outlatch (range: 0..3)
	Output:         actual state of this outlatch
	Semantics:      reads global copy of outlatch

ParallelPort Functions:
=======================

BYTE OSReadParData();
	Input:          NONE
	Output:         actual state of the 8bit dataport
	Semantics:      reads contents of parallelport (active high)

void OSWriteParData(BYTE value);
	Input:          (value) new output-data
	Output:         NONE
	Semantics:      writes out new data to parallelport (active high)

BYTE OSReadParSR();
	Input:          NONE
	Output:         actual state of the 5 statuspins
	Semantics:      reads state of the 5 statuspins active-high! (BUSY(4), 
			ACK(3), PE(2), SLCT(1), ERROR(0)): 

void OSWriteParCTRL(BYTE value);
	Input:          (value) new ctrl-pin-output (4bits)
	Output:         NONE
	Semantics:      writes out new ctrl-pin-states active high! (SLCTIN(3), INT(2), AUTOFDXT(1), STROBE(0))

BYTE OSReadParCTRL();
	Input:          NONE
	Output:         actual state of the 4 ctrl-pins
	Semantics:      reads state of the 4 ctrl-pins active-high! (SLCTIN(3), INT(2), AUTOFDXT(1), STROBE(0))  */

static const char * const adText[] = {
"AD-Converter Functions:",
"=======================",
"",
"int OSGetAD(int channel);",
"  Input:      (channel) desired AD-channel range: 0..15",
"  Output:     (returncode) 10 bit sampled value",
"  Semantics:  Captures one single 10bit value from specified AD-channel",
"",
"int OSOffAD(int mode);",
"  Input:      (mode) 0 = full powerdown",
"              1 = fast powerdown",
"  Output:     none",
"  Semantics:  Powers down the 2 AD-converters (saves energy)",
"              A call of OSGetAD awakens the AD-converter again"  ,
NULL
};
 
void return_cb(FL_OBJECT *ob, long data)  {
  int i=0, j=0;
  BOOL stillsearching = TRUE;
  pthread_mutex_lock(&sem);
  sscanf(fl_get_input(help->inputfield), "%s", compareinput);
  pthread_mutex_unlock(&sem);
  for(i=0;i<strlen(compareinput);i++)
    compareinput[i] = tolower(compareinput[i]);
  i = 0;
  while((keywords[i] != NULL) && stillsearching) {
    if(strstr(compareinput,keywords[i]) != NULL)
      stillsearching = FALSE;
    i++;
  }
  if (dbg) fprintf(stderr,"%s %d\n",compareinput,i);
  pthread_mutex_lock(&sem);
  if(i != 19)
    fl_hide_object(help->inputfield);
  pthread_mutex_unlock(&sem);
  ClearScreen();
  switch(i) {
    case 1:
    case 2:
    case 3:
    case 4:
        KEY_cb(fd_eyebot->KEYLCD[i-1],(long)(pow(4,i-1)));
        break;
    case 5:
        B_World();
        break;
    case 6:
        B_Param();
        break;
    case 7:
        B_Zoom();
        break;
    case 8:
        B_Unzoom();
        break;
    case 9:
        if (dbg) fprintf(stderr,"**********DUD\n");
        menuCENTER(view->Center,0);
        break;
    case 10:
        printf("******\n");
        B_Pause(view->Pause,0);
        break;
    case 11:
        B_Help(view->Help,0);
        break;
    case 12:
        B_Exit();
        break;
    case 13:
        RESET_cb(fd_eyebot->RESET,0);
        break;
    case 14:
        break;
    case 15:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; keyText[j] != NULL; j++)
          fl_add_browser_line(help->showText,keyText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 16:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; lcdText[j] != NULL; j++)
          fl_add_browser_line(help->showText,lcdText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 17:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; camText[j] != NULL; j++)
          fl_add_browser_line(help->showText,camText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 18:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; vwText[j] != NULL; j++)
          fl_add_browser_line(help->showText,vwText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 19:
    case 20:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; improcText[j] != NULL; j++)
          fl_add_browser_line(help->showText,improcText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 21:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; systemText[j] != NULL; j++)
          fl_add_browser_line(help->showText,systemText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 22:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; audioText[j] != NULL; j++)
          fl_add_browser_line(help->showText,audioText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 23:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; psdText[j] != NULL; j++)
          fl_add_browser_line(help->showText,psdText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 24:
    case 25:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; servoText[j] != NULL; j++)
          fl_add_browser_line(help->showText,servoText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 26:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; bumpText[j] != NULL; j++)
          fl_add_browser_line(help->showText,bumpText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    case 27:
	pthread_mutex_lock(&sem);
        fl_clear_browser(help->showText);
        for(j = 0; adText[j] != NULL; j++)
          fl_add_browser_line(help->showText,adText[j]);
        fl_show_object(help->showText);
	pthread_mutex_unlock(&sem);
        break;
    default:
	pthread_mutex_lock(&sem);
        fl_set_object_label(help->title,"\n\nSearch Failed!");
        fl_set_object_label(help->text,"\n\n\n\n\n\n\n\n\n\n\n\n\nText specified not found\n\nSearch Again?");
        fl_set_input(help->inputfield,"");
	pthread_mutex_unlock(&sem);
        return;
  }

}

