/* ----------------------------------------------------------------------------
| Filename: lcdsim.c
|
| Author:       Nicholas Tay     
| Description: 
| Display output routines (128x64)
|
| LCD_.... routines are callable from C 
-------------------------------------------------------------------------------
| Functions:
| int LCDClear( void );                         | clears the display
| int LCDPutChar ( char char);                  | writes a character and adjusts
|                                               | position                      
| int LCDSetChar (int row,int column,char char);| writes a character to the 
|						| given
|                                               | position
| int LCDPutString ( char *string);             | writes a string and adjusts 
|						| position
| int LCDSetString ( int row, int column, char *string);  
|						| writes a string to the 
|						| given position
| int LCDMode ( int mode);                      | switches display mode
| int LCDSetPos (int row, int column);          | sets posistion
| int LCDPutGraphic ( image *buf );             | writes graphic
| int LCDPutImage ( BYTE *buf );                | writes graphic
| int LCDMenu (char *, char *, char *,char *);  | writes the menu line
| int LCDMenuI(int *pos, char *);               | writes just one string into 
|						| the menu line
| int LCDSetPixel (int row,int col,BOOL val);   | sets pixel to value= 0, 1 or 
|						| 2 for invert
| int LCDInvertPixel (int row, int col);        | invert pixel
| int LCDGetPixel (int row, int col);           | gets pixel value, 1 or 0
|
| LCDLine and LCDArea are in lcd.c
--------------------------------------------------------------------------------
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "forms.h"
#include "LCDDisp.h"
#include "viewscreen.h"
#include "protos.h"

int cursorhor = 0, cursorver = 0;
void ClearText(void) {

  if(first_robot() || (robot_count <= 1)) {
    pthread_mutex_lock(&sem);
    fl_set_object_lcol(fd_eyebot->TEXTSPACE,FL_COL1);
    fl_set_object_label(fd_eyebot->TEXTSPACE,"");
    fl_set_object_lcol(fd_eyebot->TEXTSPACE,FL_BLACK);
    pthread_mutex_unlock(&sem);
  }
  
}

void ClearMenu(void) {
  int i;

  if(first_robot() || (robot_count <= 1)) {
    pthread_mutex_lock(&sem);
    for(i = 0; i < 4; i++)  {
      fl_set_object_lcol(fd_eyebot->MENU[i],FL_COL1);
      fl_set_object_label(fd_eyebot->MENU[i],"");
      fl_set_object_lcol(fd_eyebot->MENU[i],FL_BLACK);
    }
    pthread_mutex_unlock(&sem);
  }
  
}

void CreateDisplay(void) {
  int i;
  char *temp;
  
    temp = (char *)calloc((MAXPOS + 40) * 7, sizeof(char));
    for(i = 0; i <= 6; i++) {
      if((scrolling & GRAPHICS) != 0)
	strcat(temp,"                     ");
      strcat(temp,text[i]);
    }
    ClearText();
    if((scrolling & CURSOR) != 0)  {
      i = 25;
      if((scrolling & GRAPHICS) != 0)
	i = 25 + 21*11;
      pthread_mutex_lock(&sem);
      fl_winset(FL_ObjWin(fd_eyebot->LCDDisplay));
      fl_rectf(i+cursorhor*11,87+cursorver*18,10,2,FL_COL1);
      fl_rectf(i+lcdpos*11,87+lcdline*18,10,2,FL_BLUE); 
      pthread_mutex_unlock(&sem);
      cursorhor = lcdpos;
      cursorver = lcdline;
    }
      pthread_mutex_lock(&sem);
      fl_set_object_label(fd_eyebot->TEXTSPACE,temp);
      fl_check_forms();
      fl_redraw_object(view->screen);
      pthread_mutex_unlock(&sem);
}
  
/* LCD_Clear:     
		| ------------------------------------------------------
		| Description: 
		| clears the memory of the LCD module and sets the 
		| cursor position to the upper left corner
		| ------------------------------------------------------
		| Input: NONE
		| Output: NONE
		| --------------------------------------------------- */

int LCDClear(void)  {

  int i, j;
  if(!lcd_initialised)
    InitialiseLCD(); 
  if(first_robot() || (robot_count <= 1)) {
    for(i=0;i<7;i++)
      strncpy(text[i], "\n",MAXPOS + 5);
    scrolling = scrolling & (GRAPHICS - 1);
    lcdpos = 0;
    lcdline = 0;
    for(i = 0; i < 64; i++)
      for(j = 0; j < 128; j++)
	lcdpixel[i][j] = 0;
    fl_winset(FL_ObjWin(fd_eyebot->LCDDisplay));
    fl_rectf(16, 70, 290, 168, FL_COL1);
    LCDMenu("","","","");
    CreateDisplay();
  }
  return 0;
 
}


/* LCD_Mode:
		| ------------------------------------------------------
		| Description: 
		| switches the LCD mode (NON)SCROLLING, (NO)CURSOR
		| 
		| ------------------------------------------------------
		| Input: d1     bit0 =0 noscrolling =1 scrolling
		|               bit1 =0 nocursor =1 cursor
		| Output: NONE
		| ---------------------------------------------------- */


int LCDMode(int buf)  {

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    scrolling = scrolling & GRAPHICS;
    scrolling = scrolling | buf;
  }
  return 0;

}

/* LCD_Menu:               
		| ------------------------------------------------------
		| Description: 
		| writes the LCD menu line, consisting of 4 entries
		| (one for each key) max 4 characters each
		| ------------------------------------------------------
		| Input:        a1      pointer to leftmost string
		|               a2      pointer to next string
		|               a3      pointer to next string
		|               a4      pointer to rightmost string     
		|       
		| Output: NONE
		| ---------------------------------------------------- */
		
int LCDMenu(char *string1, char *string2, char *string3, char *string4)  {

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if((strlen(string1) > 4) || (strlen(string2) > 4) || (strlen(string3) > 4)
    	|| (strlen(string4) > 4))
      return 1;
    if(strlen(string1) > 0) {
      pthread_mutex_lock(&sem);
      fl_set_object_lcol(fd_eyebot->MENU[0],FL_COL1);
      fl_set_object_label(fd_eyebot->MENU[0],"");
      fl_set_object_lcol(fd_eyebot->MENU[0],FL_BLACK);
      fl_set_object_label(fd_eyebot->MENU[0], string1);
      pthread_mutex_unlock(&sem);
    }
    if(strlen(string2) > 0) {
      pthread_mutex_lock(&sem);
      fl_set_object_lcol(fd_eyebot->MENU[1],FL_COL1);
      fl_set_object_label(fd_eyebot->MENU[1],"");
      fl_set_object_lcol(fd_eyebot->MENU[1],FL_BLACK);
      fl_set_object_label(fd_eyebot->MENU[1], string2);
      pthread_mutex_unlock(&sem);
    }
    if(strlen(string3) > 0) {
      pthread_mutex_lock(&sem);
      fl_set_object_lcol(fd_eyebot->MENU[2],FL_COL1);
      fl_set_object_label(fd_eyebot->MENU[2],"");
      fl_set_object_lcol(fd_eyebot->MENU[2],FL_BLACK);
      fl_set_object_label(fd_eyebot->MENU[2], string3);
      pthread_mutex_unlock(&sem);
    }
    if(strlen(string4) > 0) {
      pthread_mutex_lock(&sem);
      fl_set_object_lcol(fd_eyebot->MENU[3],FL_COL1);
      fl_set_object_label(fd_eyebot->MENU[3],"");
      fl_set_object_lcol(fd_eyebot->MENU[3],FL_BLACK);
      fl_set_object_label(fd_eyebot->MENU[3], string4);
      pthread_mutex_unlock(&sem);
    }
    OSSemP(&sf);
    pthread_mutex_lock(&sem);
    fl_check_forms();
    pthread_mutex_unlock(&sem);
    OSSemV(&sf);
  }
  return 0; 
}	



/* LCD_MenuI:               
		| ------------------------------------------------------
		| Description: 
		| writes just one string to the LCD menu line at pos i (1..4)
		| max 4 characters
		| ------------------------------------------------------
		| Input:        a1      pointer to string
   	        |               d0      position (1..4)
		|       
		| Output: NONE
		| --------------------------------------------------- */

int extern LCDMenuI(int pos, char *string)  {

  char **strings;
  int i;

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if(pos > 4) 
      return 1; 
    strings = (char **)calloc(4,sizeof(char *));
    for(i=0;i<4;i++)  {
      strings[i] = (char *)calloc(4,sizeof(char));
      strcpy(strings[i],fd_eyebot->MENU[i]->label);
    }
    strcpy(strings[pos-1],string);
    LCDMenu(strings[0], strings[1], strings[2], strings[3]);
  }
  return 0;
  
}
			
/* LCD_SetChar:            
		| ------------------------------------------------------
		| Description: 
		| writes a character to the given position and does
		| not change cursor position
		| ------------------------------------------------------
		| Input:        d0      ASCII charakter code
		|               d1      row
		|               d2      column
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDSetChar(int row, int col, char ch) {

  int tem;

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if((row > 6) || (col > MAXPOS)) {
      fprintf(stderr,"Coordinates out of range!!\n");
      return 1;
    }
    tem = strlen(text[row]) - 1;
    text[row][tem] = '\0';
    while (strlen(text[row]) < col)   
      strcat(text[row], " ");
    text[row][col] = ch;
    if(tem <= col) {
      text[row][col+1] = '\n';
      text[row][col+2] = '\0';
    } else {
      text[row][tem] = '\n';
      text[row][tem+1] = '\0';
    } 
    CreateDisplay();
  }
  return 0;
  
}


/* LCD_SetString:  
		| ------------------------------------------------------
		| Description: 
		| writes a string to the given position and does
		| not change cursor position
		| ------------------------------------------------------
		| Input:        a0      pointer to zero terminated string
		|               d1      row
		|               d2      column
		|       
		| Output: NONE
		| --------------------------------------------------- */
int LCDSetString(int row, int col, char *string)  {

  int temp, i, j = MAXPOS;

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if((row > 6) || (col > MAXPOS)) {
      fprintf(stderr,"Coordinates out of range!!\n");
      return 1;
    }
    temp = strlen(text[row]) - 1;  
    text[row][temp] = '\0';
    while (strlen(text[row]) <= col)   
      strcat(text[row], " ");
    for(i = 0; i < strlen(string); i++) {
      if((i + col) > j) {
	strcat(text[row], "\n");
	row++;
	j += MAXPOS;
	temp = strlen(text[row]) - 1;
	text[row][temp] = '\0';
      }
    text[row][(i + col) % MAXPOS] = string[i];
    }
    if (((i + col) % MAXPOS) > (temp + 1)) 
      text[row][(i + col) % MAXPOS] = '\0';
    strcat(text[row],"\n");
    CreateDisplay();
  }
  return 0;
  
}


	

/* LCD_PutChar:   
		| ------------------------------------------------------
		| Description: 
		| writes a charakter to the display at the current 
		| cursor position and increments the cursor position by 1
		| special C codes like \n are supported
		| ------------------------------------------------------
		| Input:        d0      ASCII character code
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDPutChar(char ch) {

  char *temp;
  int i;
  
  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    temp = (char *)calloc(2, sizeof(char));
    sprintf(temp, "%c\n", ch);
    if((lcdline == 6) && ((lcdpos >= MAXPOS) || (ch == '\n'))) {
      if((scrolling & SCROLLING) != 0) {
	for(i = 0; i < 6; i++)
	  strcpy(text[i], text[i+1]);
	strncpy(text[6], temp, MAXPOS + 5);
	lcdpos = 0;
	if(ch == '\n') 
	  text[6][1] = '\0';
	else
	  lcdpos = 1;
      } else {
	lcdline = 0;
	for(i = 0; i <= 6; i++)
	  strncpy(text[i], "\n", MAXPOS + 5);
	lcdpos = 0;
	if (ch != '\n') {
	  text[0][0] = '\0';
	  strncpy(text[0], temp, MAXPOS + 5);
	  lcdpos = 1;
	} 
      }
    } else {
      if ((lcdpos >= MAXPOS) || (ch == '\n')) {
	lcdline++;
	lcdpos = 0;
      }
      if((text[lcdline][lcdpos] != '\n') && (text[lcdline][lcdpos] != '\0'))
	text[lcdline][lcdpos] = ch;
      else {
	text[lcdline][lcdpos] = '\0';
	strcat(text[lcdline], temp);
      }
      if(ch == '\n')  
	text[lcdline][1] = '\0';
      else
	lcdpos++;
    }
    CreateDisplay();
  }
  return 0;
  
}

	

		
/* LCD_SetPos:  
		| ------------------------------------------------------
		| Description: 
		| sets the cursor position to the given coordinates
		| ------------------------------------------------------
		| Input:        d1      row
		|               d2      column
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDSetPos(int row, int col)  {

  int temp;

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if((row > 6) || (col > MAXPOS)) {
      fprintf(stderr, "Coordinates out of range!!\n");
      return 1;
    }
    temp = strlen(text[row]) - 1;
    text[row][temp] = '\0';
    while (strlen(text[row]) < col)   
      strcat(text[row], " ");
    strcat(text[row], "\n"); 
    lcdline = row;
    lcdpos = col;
    CreateDisplay();
  }
  return 0;
  
}



/* LCD_PutString: 
		| ------------------------------------------------------
		| Description: 
		| writes a string to the lcd at the current cursor position
		| and adjusts cursor position
		| ------------------------------------------------------
		| Input:        a0      pointer to zero terminated string
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDPutString(char *string)  {

  char *extra, ch = '0', chs = '0';
  int i, j = 0, k = 0, temp, val, total;
  
  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    extra = (char *)calloc(100, sizeof(char));
    total = strlen(string);
    sscanf(string, "%[^\n]", extra);
    val = strlen(extra);
    while(j < total) {
      while (k < val) {
	if((lcdline == 6) && (lcdpos >= MAXPOS)) {
	  if((scrolling & SCROLLING) != 0) {
	    for(i = 0; i < 6; i++)
	      strcpy(text[i], text[i+1]);
	    strncpy(text[6], "\n", MAXPOS + 5);
	  } else {
	    lcdline = 0;
	    for(i = 0; i < 7; i++)
	      strncpy(text[i], "\n", MAXPOS + 5);
	  }
	  lcdpos = 0;
	} else if (lcdpos >= MAXPOS) {
	  lcdline++;
	  lcdpos = 0;
	} else {
	  if (ch != '\n') {
	    k -= lcdpos;
	    temp = strlen(text[lcdline]) -1;
	    if(temp > (lcdpos + strlen(extra))) 
	      chs = text[lcdline][lcdpos + strlen(extra)];
	    text[lcdline][lcdpos] = '\0';
	    strncat(text[lcdline], extra, MAXPOS - lcdpos);
	    if((MAXPOS - lcdpos) < strlen(extra)) 
	      extra = (extra + MAXPOS - lcdpos);
	    lcdpos = strlen(text[lcdline]);
	    if(strlen(text[lcdline]) >= temp) 
	      strcat(text[lcdline], "\n");
	    else 
	      text[lcdline][strlen(text[lcdline])] = chs;
	    k += lcdpos;
	  } else {
	    k++;
	    lcdpos = MAXPOS;
	  }
	}
      }     
      j += k;
      if(j < total) {
	sscanf(string + j, "%c", &ch);
	val = 1;
	if((ch != '\n') && (ch != '\0')) {
	  extra[0] = '\0';
	  sscanf(string + j, "%[^\n]", extra);
	  val = strlen(extra);
	}
	k = 0;
      }
    }
    CreateDisplay();
  }
  return 0;
  
}
  


/* LCD_PutGraphic:  
		| ------------------------------------------------------
		| Description: 
		| writes the inner 80x60 pixels of an 82x62 byte array
		| to the left side of the LCD
		| ------------------------------------------------------
		| Input:        a0      pointer to grey image
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDPutGraphic(image *buf) {

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    ClearText();
    scrolling = scrolling | GRAPHICS;
    OSSemP(&sf);
    pthread_mutex_lock(&sem);
    fl_check_forms();
    pthread_mutex_unlock(&sem);
    OSSemV(&sf);
    RawToX(&buf[0], &ximagemem[0]);
    XPutImage(disp,
	    FL_ObjWin(fd_eyebot -> LCDDisplay),
	    gc,
	    ximage,
	    0,
	    0,
	    fd_eyebot -> LCDDisplay -> x + 4,
	    fd_eyebot -> LCDDisplay -> y + 4,
	    ximage -> width,
	    ximage -> height);
    XFlush(disp);
    OSSemP(&sf);
    pthread_mutex_lock(&sem);
    fl_check_forms();
    pthread_mutex_unlock(&sem);
    OSSemV(&sf);
  }
  return 0;
  
}
		
				

/* LCD_PutImage:  
		| ------------------------------------------------------
		| Description: 
		| writes the complete 128*64 pixels to the LCD
		| ------------------------------------------------------
		| Input:        a0      pointer to bw image
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDPutImage(BYTE *buf)  {

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    ClearText();
    scrolling = scrolling | GRAPHICS;
    RawToX(buf, &ximagemem[0]);
    XPutImage(disp,
	    FL_ObjWin(fd_eyebot -> LCDDisplay),
	    gc,
	    ximage,
	    0,
	    0,
	    fd_eyebot -> LCDDisplay -> x + 4,
	    fd_eyebot -> LCDDisplay -> y + 4,
	    ximage -> width,
	    ximage -> height);
    XFlush(disp);
    OSSemP(&sf);
    pthread_mutex_lock(&sem);
    fl_check_forms();
    pthread_mutex_unlock(&sem);
    OSSemV(&sf);
  }
  return 0;
  
}



/* LCD_SetPixel:
		| ------------------------------------------------------
		| Description: 
		| sets a single pixel on the display to 1 ot 0
		| ------------------------------------------------------
		| Input:        d0      row
		|               d1      column
		|               d2      0,1 or 2
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDSetPixel(int row, int col, BOOL state)  {

  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if((row >63)||(col>127))
      return -1;
    pthread_mutex_lock(&sem);
    fl_winset(FL_ObjWin(fd_eyebot->LCDDisplay));
    pthread_mutex_unlock(&sem);
    switch (state)  {
    case 0 :
      pthread_mutex_lock(&sem);
      fl_rectf(20 + 2 * col, 70 + 2 * row, 2, 2, FL_COL1);
      lcdpixel[row][col] = state;
      pthread_mutex_unlock(&sem);
      break;
    case 1 :
      pthread_mutex_lock(&sem);
      fl_rectf(20 + 2 * col, 70 + 2 * row, 2, 2, FL_BLACK);
      lcdpixel[row][col] = state;
      pthread_mutex_unlock(&sem);
      break;
    case 2 :
      pthread_mutex_lock(&sem);
      if(lcdpixel[row][col] == 1) {
	fl_rectf(20 + 2 * col, 70 + 2 * row, 2, 2, FL_COL1);
	lcdpixel[row][col] = 0;
      } else {
	fl_rectf(20 + 2 * col, 70 + 2 * row, 2, 2, FL_BLACK);
	lcdpixel[row][col] = 1;
      }
      pthread_mutex_unlock(&sem);
      break;
    default :
      fprintf(stderr,"Pixel State out of range!!\n");
      break;
    }
    CreateDisplay();
  }
  return 0;
  
}




/* LCD_InvertPixel:
		| ------------------------------------------------------
		| Description: 
		| inverts a single pixel
		| ------------------------------------------------------
		| Input:        d0      row
		|               d1      column
		|       
		| Output: NONE
		| --------------------------------------------------- */

int LCDInvertPixel(int row, int col)  {
  
  if(!lcd_initialised)
    InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    LCDSetPixel(row, col, 2);
  }
  return 0;
  
}



/* LCD_GetPixel:
		| ------------------------------------------------------
		| Description: 
		| returns true when pixel is see, otherwise false
		| ------------------------------------------------------
		| Input:        d0      row
		|               d1      column
		|       
		| Output: TRUE/FALSE
		| --------------------------------------------------- */

int LCDGetPixel(int row, int col)  {
  
  if(first_robot() || (robot_count <= 1)) {
    return lcdpixel[row][col];
  } else
    return 0;
  
}


/* ----------------------------------------------------------------------- */
/** LCDLine(x1,y1, x2,y2, col)
    Draw a line from x1,y1 to x2,y2
    using the Bresenham Algorithm
    top    left  is   0, 0
    bottom right is 127,63
    color: 0 white
	   1 black
	   2 negate image contents
*/

int LCDLine(int x1, int y1, int x2, int y2, int col)
{ 
  int x,y, deltax,deltay, xchange, ychange;
  int i, error;

    if(!lcd_initialised)
      InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if((x1 == x2) || (y1 == y2)) {
      fl_winset(FL_ObjWin(fd_eyebot->LCDDisplay));
      if(col == 0)
        fl_rectf(20+2*x1,70+2*y1,2+2*(x2-x1),2+2*(y2-y1),FL_COL1);
      else 
        fl_rectf(20+2*x1,70+2*y1,2+2*(x2-x1),2+2*(y2-y1),FL_BLACK);
        if(y1 == y2)  {
          if(col == 2)
            for(i=x1;i<=x2;i++)  {
              if(lcdpixel[y1][i] == 1)
                fl_rectf(20+2*i,20+2*y1,2,2,FL_COL1);
              lcdpixel[y1][i] = !lcdpixel[y1][i];
	    }
        } else {
          if(col == 2)
            for(i=y1;i<=y2;i++) {
              if(lcdpixel[i][x1] == 1)
                fl_rectf(20+2*x1,20+2*i,2,2,FL_COL1);
              lcdpixel[i][x1] = !lcdpixel[i][x1];
            }
        }
      } else {
      x=x1; y=y1;      /* starting point */
      error=0;
      deltax = x2-x1;  /* difference */
      deltay = y2-y1;

      if (deltax < 0) { xchange = -1; deltax = -deltax; } else xchange = 1;
      if (deltay < 0) { ychange = -1; deltay = -deltay; } else ychange = 1;

      if (deltax < deltay) {
	for (i=0; i < deltay+1; i++) {
	  LCDSetPixel(y,x, col);
	  y     += ychange;
	  error += deltax;
	  if (error > deltay) { x += xchange; error -= deltay; }
	}
      }
      else {
	for (i=0; i < deltax+1; i++) {
	  LCDSetPixel(y,x, col);
	  x     += xchange;
	  error += deltay;
	  if (error > deltax) { y += ychange; error -= deltax; }
	}
      }
      LCDSetPixel(y2,x2, col); /* add last pixel to line */
    }
  }
  fl_check_forms();
  return 0;
}


/* ----------------------------------------------------------------------- */
int LCDArea(int x1, int y1, int x2, int y2, int col)
{ 
  int i,j;

  if(!lcd_initialised)
      InitialiseLCD();
  if(first_robot() || (robot_count <= 1)) {
    if(col > 2)
      return -1;
    fl_winset(FL_ObjWin(fd_eyebot->LCDDisplay));
    if(col != 2)  {
      if(col == 0)
	fl_rectf(20+2*x1,70+2*y1,2+2*(x2-x1),2+2*(y2-y1),FL_COL1);
      else 
	fl_rectf(20+2*x1,70+2*y1,2+2*(x2-x1),2+2*(y2-y1),FL_BLACK);
      for(i=y1;i<=y2;i++)
	for(j=x1;j<=x2;j++)
	  lcdpixel[i][j] = col;
    } else {
      fl_rectf(20+2*x1,70+2*y1,2+2*(x2-x1),2+2*(y2-y1),FL_BLACK);
      for (i=y1; i<=y2; i++)
	for (j=x1; j<=x2; j++) {
	  if(lcdpixel[i][j] == 1)  
	    fl_rectf(20+2*i,70+2*j,2,2,FL_COL1);
	  lcdpixel[i][j] = !lcdpixel[i][j];
	}
    }
  }
  return 0;
}
