/*------------------------------------------------------------------------
| Filename: drive.c
|
| Author:       Nicholas Tay, UWA 1997
|
| Description:  Drives EyeBot vehicle
|               using PSD-sensor 
|               Attempts to drive down a corridor, avoiding the walls.
|		( Revised from drive_rc.c )
|
-------------------------------------------------------------------------- */

#include "eyebot.h"
#include <stdio.h>
#include <math.h>


#define GRID_WIDTH 	80   //millimeters
#define GRID_CENTER 	40 
#define GRID_DIMENSIONS 80
#define PSD_MAX		280 
#define PSD_LIMIT	900
#define OFFSET_RIGHT	0
#define OFFSET_LEFT	0
#define OFFSET_FRONT	0

#ifndef PI
#define PI 3.1415926535
#endif

#ifndef ROBOT_RADIUS
#define ROBOT_RADIUS 80
#endif

/* --------------------= VEHStop =---------------------- */

/** Stops the robot.

	@param 	vw_	The VW handle
	@author Nicholas Tay, CIIPS 1998		*/
void VEHStop(VWHandle vw_)
{
  VWSetSpeed(vw_, 0, 0);
}

/* --------------------= Check_Map =--------------------- */

/** Checks adjacent cells for a match in (value_).

	@param 	map_	The map grid
	@param 	pos_	The index of the grid
	@param 	value_	The value of the cell that the routine tries to match
	@return TRUE, 	If a match is found
	@author Nicholas Tay, CIIPS 1998
	@see	Display_MAP	*/
int Check_Map(char *map_, int pos_, int value_)  {
  int i, j, temp = 0;

  for(i = -1; i < 2; i++)
    for(j = -1; j < 2; j++)
      if(map_[pos_ - i * GRID_DIMENSIONS + j] == value_)
        temp = 1;
  return temp;
}

/* --------------------= GetSegment =----------------------- */

int GetSegment(float vals_)  {
  return vals_ * 1000.0 + GRID_WIDTH * GRID_CENTER + 100;
}


/* --------------------= GetMapPos =------------------------ */

/** Obtains the array index for a particular grid cell.

	@param 	temp	Current position of the robot
	@param	dist	Distance received from the PSD sensor
	@param	offset_	The angle offset (Depends on which PSD used)
	@param  x	The array index
	@param  y 	The array index
	@author Nicholas Tay, CIIPS 1998
	@see	Find_Obstacle	*/
void GetMapPos(PositionType temp, int dist, float offset_, int *x, int *y)  {
  float a, b;

  a = temp.x + (float)dist*cos(temp.phi + offset_) / 1000.0;
  b = temp.y + (float)dist*sin(temp.phi + offset_) / 1000.0;
  *x = ((int)(a * 1000) / GRID_WIDTH) + GRID_CENTER;
  *y = ((int)(-b * 1000) / GRID_WIDTH) + GRID_CENTER;
  if(a < 0)
    *x -= 1;
  if(b > 0)
    *y -= 1;
}

/* --------------------= Angle_Dist =--------------------------- */

/** Finds the difference between two radiants.

	@param	x_	The first radiant
	@param 	y_	The second radiant
	@return	The angle difference in radians.  The angle is positive
	@author Nicholas Tay, CIIPS 1998  	*/
radians Angle_Dist(radians x_, radians y_)  {
  if(x_ < 0)
    x_ += 2 * PI;
  if(y_ < 0)
    y_ += 2 * PI;
  if(y_ < x_)
    y_ += 2 * PI;
  return (y_ - x_);
}

/* ---------------------= GetLocation =--------------------------- */

/** Obtains the location of a cell in the grid in terms of its robot position.

	@param  target	The position (PositionType) of the cell
	@param 	index_	The cell array index
	@return The position of the center of the cell
	@author Nicholas Tay, CIIPS 1998  	*/
void GetLocation(PositionType *target_, int index_)  {
  int x, y;

  y = index_ / GRID_DIMENSIONS;
  x = index_ % GRID_DIMENSIONS;
  target_->x = ((float)x - (float)GRID_CENTER + 0.5) * (float)GRID_WIDTH 
		/ 1000.0; 
  target_->y = ((float)y - (float)GRID_CENTER + 0.5) * (float)GRID_WIDTH
		/ -1000.0; 
}

/* ----------------------= GetPosition =---------------------------- */

/** VWGetPosition equivalent.  Obtains the position of the robot.  
	However, this function adds the robot position to the grid map.

	@param	vw_ 	The VW handle for the robot
	@param	pos	The position (PositionType) of the robot
	@param  map_	The grid for storage
	@return The position of the robot	
	@author Nicholas Tay, CIIPS 1998	*/
int GetPosition(VWHandle vw_, PositionType *pos_, char *map_)  {
  int i = 0, j = 0, a, b;

  VWGetPosition(vw_, pos_);
  GetMapPos(*pos_, 0, 0, &i, &j);
  if((map_[j * GRID_DIMENSIONS + i] != 1) && 	
		(map_[j * GRID_DIMENSIONS + i] != 5))
    map_[j * GRID_DIMENSIONS + i] = 0;
  for(a = -1; a < 2; a++)
    for (b = -1; b < 2; b++)
      if(map_[(j + a) * GRID_DIMENSIONS + i + b] == 6)
        map_[(j + a) * GRID_DIMENSIONS + i + b] = 0;
  return j * GRID_DIMENSIONS + i;
}

/* ------------------= Get_Dist =------------------------- */

/** Obtains the distance to the nearest obstacle using the right PSD sensor.
	Records the obstacle and all spaces between the obstacle and the 
	robot into the grid.
	Also checks the left PSD sensor for obstacles.

	@param	vw_	The VW Handle
	@param	psd_	The right PSD sensor
	@param 	psdLeft	The left PSD sensor
	@param  ir_r_	The right infra-red sensor
	@param	map_	The grid for storage
	@return The distance to the obstacle using the right PSD sensor
	@author Nicholas Tay, CIIPS 1998	*/	
int Get_Dist(VWHandle vw_, PSDHandle psd_, PSDHandle psdLeft, 
	IRHandle ir_r_, char *map_) {
  int i, j, k, x = 0, y = 0, obstacle_, dist;
  PositionType pos_n_;

/*  if(IRRead(ir_r_) == 0)
    obstacle_ = 40; */
  dist = PSDGet(psdLeft) - OFFSET_LEFT;
  VWGetPosition(vw_, &pos_n_);
  if(dist < 250)  {
      GetMapPos(pos_n_, dist + ROBOT_RADIUS, PI/2, &x, &y);
      if((x >= 0) && (x < GRID_DIMENSIONS) && (y >= 0) && (y < GRID_DIMENSIONS))
        if((map_[y * GRID_DIMENSIONS + x] != 0) &&
              (map_[y * GRID_DIMENSIONS + x]!=1))
          if(!Check_Map(&map_[0], y * GRID_DIMENSIONS + x, 1))
            map_[y * GRID_DIMENSIONS + x] = 4;
      if(dist > 80) {
        GetMapPos(pos_n_, ROBOT_RADIUS, PI/2, &x, &y);
        if((x >= 0) && (x < GRID_DIMENSIONS) && (y >= 0) && 
		(y < GRID_DIMENSIONS))
          map_[y * GRID_DIMENSIONS + x] = 0;
      }
  } else {
    GetMapPos(pos_n_, ROBOT_RADIUS + 250, PI/2, &x, &y);
    if((x >= 0) && (x < GRID_DIMENSIONS) && (y >= 0) && (y < GRID_DIMENSIONS))
      if((map_[y * GRID_DIMENSIONS + x] != 0) &&
            (map_[y * GRID_DIMENSIONS + x]!=1))
    	if(!Check_Map(&map_[0], y * GRID_DIMENSIONS + x, 1))
          if(!Check_Map(&map_[0], y * GRID_DIMENSIONS + x, 0))
      	    map_[y * GRID_DIMENSIONS + x] = 6;
  }
  obstacle_ = PSDGet(psd_) - OFFSET_RIGHT;
  VWGetPosition(vw_, &pos_n_);
  if(obstacle_ < 250)  {
    dist = obstacle_;
    for(k = 0; k < 5; k++)  {
      GetMapPos(pos_n_, dist + ROBOT_RADIUS, -PI/2, &x, &y);
        if((x >= 0) && (x < GRID_DIMENSIONS) && (y >= 0) && 
		(y < GRID_DIMENSIONS))
        if(dist == obstacle_) {
 	  if(map_[y * GRID_DIMENSIONS + x] != 5)
            map_[y * GRID_DIMENSIONS + x] = 1;
          for(i = -1; i < 2; i++)
            for(j = -1; j < 2; j++)
              if((map_[(y + i) * GRID_DIMENSIONS + x + j] == 4) ||
	              (map_[(y + i) * GRID_DIMENSIONS + x + j] == 6))
	        map_[(y + i) * GRID_DIMENSIONS + x + j] = 3;
        
        
        } else if((map_[y * GRID_DIMENSIONS + x] != 1) 
		&& (map_[y * GRID_DIMENSIONS + x] != 5))
              	map_[y * GRID_DIMENSIONS + x] = 0;
      dist -= obstacle_ / 4;
    }
  }
  return obstacle_;
}

/* ------------------= DesiredAngle =----------------------- */

/** Obtains the orientation required for the robot to face the target position.

	@param 	vw_	The VW handle
	@param  current The current position of the robot
	@param  pos_n	The target destination for the robot
	@return The orientation required for the robot (radians)
	@author Nicholas Tay, CIIPS 1998	*/
float DesiredAngle(VWHandle vw_, PositionType current,  PositionType pos_n)  {
  float angle = 0;

  if(current.x == pos_n.x)
    if(pos_n.y >= current.y)
      return  PI / 2;
    else
      return -PI / 2;
  angle = atan((pos_n.y - current.y) / fabs(pos_n.x - current.x));
  if(pos_n.x < current.x)
    angle = PI - angle;
  return angle;
}  

/* ----------------= PathTrack =-------------------- */

/** Performs a robot navigation routine to reach a destination.
	Includes code for obstacle avoidance and boundary following.

	@param  vw_	The VW handle
	@param  psd_x	The PSD Handles for the left, front and right PSDs
	@param  target	The target cell in the grid
	@param  map_	The grid map used for storage
	@author Nicholas Tay, CIIPS 1998    	*/
void PathTrack(VWHandle vw_, PSDHandle *psd_x, int target, char *map_)  {
  PositionType dest_n, pos_n_;
  SpeedType stopped_;
  int i, j, dist_ = 0, prev_ = 0, x1, x2, y1, y2;


  GetLocation(&dest_n, target);
  LCDPutString("Navigating to point\n");
  do {
    VWSetSpeed(vw_, 0, 0.2);
    VWGetPosition(vw_, &pos_n_);
  } while(Angle_Dist(pos_n_.phi, DesiredAngle(vw_, pos_n_, dest_n)) 
	> 0.03);
  VEHStop(vw_);
  do {
    VWSetSpeed(vw_, 0.2, 0);
    GetPosition(vw_, &pos_n_, &map_[0]);
    VWGetSpeed(vw_, &stopped_);
    if(((PSDGet(psd_x[1]) - OFFSET_FRONT) < 80) || 
	((stopped_.v < 0.05) && (stopped_.w == 0))) {
      fprintf(stderr,"Initiallising wall-following\n");
      if((PSDGet(psd_x[0]) - OFFSET_LEFT) < (PSDGet(psd_x[2]) - OFFSET_RIGHT))  {
        //Wall-follow left wall

	fprintf(stderr,"Wall-following Left wall\n");
	VWSetSpeed(vw_, 0, -0.2);
	do {
	  prev_ = dist_;
          OSWait(1);
          dist_ = PSDGet(psd_x[0]) - OFFSET_LEFT; 
        } while (prev_ >= dist_);
        VEHStop(vw_);
        GetPosition(vw_, &pos_n_, &map_[0]);
        while((Angle_Dist(pos_n_.phi, DesiredAngle(vw_, pos_n_, dest_n)) > 
                0.05) && (Angle_Dist(DesiredAngle(vw_, pos_n_, dest_n),
		pos_n_.phi) > 0.05) && (fabs(pos_n_.x - dest_n.x) +
		fabs(pos_n_.y - dest_n.y) > 0.16)) {

          dist_ = PSDGet(psd_x[0]) - OFFSET_LEFT;
	  GetMapPos(pos_n_, ROBOT_RADIUS, 0, &x1, &y1);
    	  GetMapPos(pos_n_, GRID_WIDTH + ROBOT_RADIUS, 0, &x2, &y2);

          VWGetSpeed(vw_, &stopped_);
          if((stopped_.v < 0.05) && (stopped_.w == 0))
            VWSetSpeed(vw_, 0, -0.3);
    	  else if((map_[y1 * GRID_DIMENSIONS + x1] == 5) ||
        	(map_[y2 * GRID_DIMENSIONS + x2] == 5)) 
	    VWSetSpeed(vw_, 0, -0.3);
          else if((PSDGet(psd_x[1]) - OFFSET_FRONT) < 50)
          {
            VEHStop(vw_);
            VWDriveTurn(vw_, -PI/4, 0.4);
            while(!VWDriveDone(vw_))
            prev_ = PSDGet(psd_x[0]) - OFFSET_LEFT;
            while((dist_ = PSDGet(psd_x[0]) - OFFSET_LEFT) <= prev_)
            {
              VWSetSpeed(vw_, 0, -0.30);
              prev_ = dist_;
            }
          } else {
	    GetMapPos(pos_n_, ROBOT_RADIUS, 0, &x1, &y1);
    	    GetMapPos(pos_n_, GRID_WIDTH + ROBOT_RADIUS, 0, &x2, &y2);
    	    if((map_[y1 * GRID_DIMENSIONS + x1] == 5) ||
        	(map_[y2 * GRID_DIMENSIONS + x2] == 5)) 
	       VWSetSpeed(vw_, 0.15, 0); 
            else if((PSDGet(psd_x[0]) - OFFSET_LEFT) > 100)
            {
              VWSetSpeed(vw_, 0.05, 0.55);
              dist_ = prev_;
            }
            else if(dist_ > prev_)
              VWSetSpeed(vw_, 0.15, 0.2);
            else if(dist_ < prev_)
              VWSetSpeed(vw_, 0.15, -0.2);
            else VWSetSpeed(vw_, 0.15, 0);
          }
          prev_ = dist_;
          GetPosition(vw_, &pos_n_, &map_[0]);
        }
      } else {
        //Wall-follow right wall
       
	VWSetSpeed(vw_, 0, 0.2);
        do {
	  prev_ = dist_;
          OSWait(1);
          dist_ = PSDGet(psd_x[2]) - OFFSET_RIGHT; 
        } while (prev_ >= dist_);
        VEHStop(vw_);
        GetPosition(vw_, &pos_n_, &map_[0]);
        while((Angle_Dist(pos_n_.phi, DesiredAngle(vw_, pos_n_, dest_n)) >
                0.05) && (Angle_Dist(DesiredAngle(vw_, pos_n_, dest_n),
		pos_n_.phi) > 0.05) && (fabs(pos_n_.x - dest_n.x) +
		fabs(pos_n_.y - dest_n.y) > 0.16)) {
  
  	  dist_ = PSDGet(psd_x[2]) - OFFSET_RIGHT;
          GetMapPos(pos_n_, ROBOT_RADIUS, 0, &x1, &y1);
          GetMapPos(pos_n_, GRID_WIDTH + ROBOT_RADIUS, 0, &x2, &y2);

	  VWGetSpeed(vw_, &stopped_);
  	  if((stopped_.v < 0.05) && (stopped_.w == 0))
      	    VWSetSpeed(vw_, 0, 0.3);
    	  else if((map_[y1 * GRID_DIMENSIONS + x1] == 5) ||
            	(map_[y2 * GRID_DIMENSIONS + x2] == 5)) 
	    VWSetSpeed(vw_, 0, 0.3);
  	  else if((PSDGet(psd_x[1]) - OFFSET_FRONT) < 50)
          {
    	    VEHStop(vw_);
    	    VWDriveTurn(vw_, PI/4, 0.4);
    	    while(!VWDriveDone(vw_))
      	    prev_ = PSDGet(psd_x[2]) - OFFSET_RIGHT;
    	    while((dist_ = PSDGet(psd_x[2]) - OFFSET_RIGHT) <= prev_)
      	    {
      	      VWSetSpeed(vw_, 0, 0.30);
      	      prev_ = dist_;
      	    }
    	  }
          else {
            GetMapPos(pos_n_, ROBOT_RADIUS, -PI/2, &x1, &y1);
    	    GetMapPos(pos_n_, GRID_WIDTH + ROBOT_RADIUS, -PI/2, &x2, &y2);
    	    if((map_[y1 * GRID_DIMENSIONS + x1] == 5) ||
        	(map_[y2 * GRID_DIMENSIONS + x2] == 5)) 
	      VWSetSpeed(vw_, 0.15, 0);
  	    else if((PSDGet(psd_x[2]) - OFFSET_RIGHT) > 100)  
            {
              VWSetSpeed(vw_, 0.05, -0.55);
              dist_ = prev_;
            }
  	    else if(dist_ > prev_)  
      	      VWSetSpeed(vw_, 0.15, -0.2);
  	    else if(dist_ < prev_)
      	      VWSetSpeed(vw_, 0.15, 0.2);
            else VWSetSpeed(vw_, 0.15, 0);
 	  }
  	  prev_ = dist_;
          GetPosition(vw_, &pos_n_, &map_[0]);
        }
      }
    }
  } while(fabs(pos_n_.x - dest_n.x)  + 
		fabs(pos_n_.y - dest_n.y)  > 0.16);

  VEHStop(vw_);
  for(i = -1; i < 2; i++)
    for(j = -1; j < 2; j++)
      if(map_[target + GRID_DIMENSIONS * i + j] == 6)
	map_[target + GRID_DIMENSIONS * i + j] = 0;
  
}

int Find_Obstacle(VWHandle vw_, PSDHandle x_, IRHandle ir_, char *map_)  {
  int mindist_ = 999, obstacle_, p_x, p_y, x = 0, y = 0;
  radians angle_ = 0;
  PositionType pos_n_, stop_;
  SpeedType    stopped_;

  VWDriveTurn(vw_, 2*PI, 0.35);
  while (!VWDriveDone(vw_)) {
    GetPosition(vw_, &pos_n_, &map_[0]);
    GetMapPos(pos_n_, 0, 0, &p_x, &p_y);
    obstacle_ = PSDGet(x_) - OFFSET_FRONT;
    if(IRRead(ir_) == 0)
      obstacle_ = 35;
    if(obstacle_ < PSD_MAX)  {
      GetMapPos(pos_n_, obstacle_ + ROBOT_RADIUS, 0, &x, &y);
      if((x >= 0) && (x < GRID_DIMENSIONS) && (y >= 0) && (y < GRID_DIMENSIONS))
        if((map_[y * GRID_DIMENSIONS + x] != 0) && 
		(map_[y * GRID_DIMENSIONS + x]!=1)) 
          if(!Check_Map(&map_[0], y * GRID_DIMENSIONS + x, 1))
          {
            map_[y * GRID_DIMENSIONS + x] = 4;
            if(obstacle_ < mindist_)  {
              mindist_ = obstacle_;
              angle_ = pos_n_.phi;
            }

        } else if (obstacle_ > PSD_LIMIT)  {
          GetMapPos(pos_n_, PSD_MAX + ROBOT_RADIUS, 0, &x, &y);
          if((map_[y * GRID_DIMENSIONS + x] != 0) &&
                (map_[y * GRID_DIMENSIONS + x]!=1))
            if(!Check_Map(&map_[0], y * GRID_DIMENSIONS + x, 0))
	      map_[y * GRID_DIMENSIONS + x] = 6;
        }
      } 
    
  } 
  VEHStop(vw_);

fprintf(stderr,"MINDIST = %d\n",mindist_);

  if(mindist_ == 999)
    return 0;
  
  GetPosition(vw_, &stop_, &map_[0]);
  VWDriveTurn(vw_, -Angle_Dist(angle_, stop_.phi), PI / 3);
  while(!VWDriveDone(vw_));


  VWSetSpeed(vw_, 0.2, 0);
  do {
    GetPosition(vw_, &stop_, &map_[0]);

    VWGetSpeed(vw_, &stopped_);
  } while(((PSDGet(x_) - OFFSET_FRONT) > 80) && (stopped_.v > 0.05));
  VEHStop(vw_);
  return 1;
}


void Wall_Follow_Right(VWHandle vw_, PSDHandle *psd_, IRHandle ir_f_,
	IRHandle ir_r_, char *map_, XSegment *segs_, int *count_) 
{
  int 		dist_ = 0, prev_ = -1, looped_ = 0, x1, y1, x2, y2, turn_ = 0;
  int		val_ = 0;
  radians	orient_ = 0;
  SpeedType     stopped_;
  PositionType  pos_n_, start_;

  VWDriveTurn(vw_, PI / 2, 0.4);
  while(!VWDriveDone(vw_))
    val_ = Get_Dist(vw_, psd_[2], psd_[0], ir_r_, &map_[0]);
  VEHStop(vw_);
   
  VWSetSpeed(vw_, 0.2, 0);
  GetPosition(vw_, &pos_n_, &map_[0]);
  GetPosition(vw_, &start_, &map_[0]);
  if(*count_ < GRID_DIMENSIONS*10) {
    segs_[*count_].x1 = GetSegment(start_.x) +
	(val_ + ROBOT_RADIUS) * cos(start_.phi - PI/2); 
    segs_[*count_].y1 = GetSegment(start_.y) +
	(val_ + ROBOT_RADIUS) * sin(start_.phi - PI/2); 
  }

  while((looped_ == 0) || ((fabs(start_.x - pos_n_.x) > 0.08) || 
				(fabs(start_.y - pos_n_.y) > 0.08)))  {
    GetMapPos(pos_n_, ROBOT_RADIUS, 0, &x1, &y1);
    GetMapPos(pos_n_, GRID_WIDTH + ROBOT_RADIUS, 0, &x2, &y2);
    VWGetSpeed(vw_, &stopped_);
    dist_ = Get_Dist(vw_, psd_[2], psd_[0], ir_r_, &map_[0]);

    if((stopped_.v < 0.05) && (stopped_.w == 0))
      VWSetSpeed(vw_, 0, 0.3);
    else if((map_[y1 * GRID_DIMENSIONS + x1] == 5) ||
	(map_[y2 * GRID_DIMENSIONS + x2] == 5)) {
      VWSetSpeed(vw_, 0, 0.4);
      if((turn_ == 0) && (*count_ < 10*GRID_DIMENSIONS))  {
        segs_[*count_].x2 = GetSegment(pos_n_.x) +
                (ROBOT_RADIUS + dist_) * cos(pos_n_.phi - PI/2) +
		turn_ * ROBOT_RADIUS * cos(pos_n_.phi);
        segs_[*count_].y2 = GetSegment(pos_n_.y) +
                (ROBOT_RADIUS + dist_) * sin(pos_n_.phi - PI/2) +
		turn_ * ROBOT_RADIUS * sin(pos_n_.phi);
        *count_ += 1;
      }
      turn_ = 1;
    } else if((PSDGet(psd_[1]) - OFFSET_FRONT) < GRID_WIDTH)
    {
      VEHStop(vw_);
      VWSetSpeed(vw_, 0, 0.30);
      orient_ = pos_n_.phi - PI / 5.0;
      while((dist_ = Get_Dist(vw_, psd_[2], psd_[0], ir_r_, &map_[0])) >= prev_)
      {
        prev_ = dist_;
        if((dist_ <= 100) && (turn_ == 0) && 
		(*count_ < GRID_DIMENSIONS*10)) {
          GetPosition(vw_, &pos_n_, &map_[0]);
          if(Angle_Dist(orient_, pos_n_.phi) >= (PI / 12.0))   {
	    orient_ = pos_n_.phi;
            segs_[*count_].x2 = GetSegment(pos_n_.x) +
		(ROBOT_RADIUS + dist_) * cos(pos_n_.phi - PI/2);
            segs_[*count_].y2 = GetSegment(pos_n_.y) +  
		(ROBOT_RADIUS + dist_) * sin(pos_n_.phi - PI/2);
            *count_ += 1;
            segs_[*count_].x1 = GetSegment(pos_n_.x) + 
		(ROBOT_RADIUS + dist_) * cos(pos_n_.phi - PI/2);
            segs_[*count_].y1 = GetSegment(pos_n_.y) +
		(ROBOT_RADIUS + dist_) * sin(pos_n_.phi - PI/2);
	  }
	}
      }
      orient_ = pos_n_.phi - PI / 5.0;
      while((dist_ = Get_Dist(vw_, psd_[2], psd_[0], ir_r_, &map_[0])) <= prev_)
      {
        prev_ = dist_;
        if((dist_ <= 100) && (turn_ == 0) &&
		(*count_ < GRID_DIMENSIONS*10)) {
          GetPosition(vw_, &pos_n_, &map_[0]);
          if(Angle_Dist(orient_, pos_n_.phi) >= (PI / 10.0))   {
            orient_ = pos_n_.phi;
            segs_[*count_].x2 = GetSegment(pos_n_.x) +
                (ROBOT_RADIUS + dist_) * cos(pos_n_.phi - PI/2);
            segs_[*count_].y2 = GetSegment(pos_n_.y) +
                (ROBOT_RADIUS + dist_) * sin(pos_n_.phi - PI/2);
            *count_ += 1;
            segs_[*count_].x1 = GetSegment(pos_n_.x) +
                (ROBOT_RADIUS + dist_) * cos(pos_n_.phi - PI/2);
            segs_[*count_].y1 = GetSegment(pos_n_.y) +
                (ROBOT_RADIUS + dist_) * sin(pos_n_.phi - PI/2);
          }
        }
      }
      VEHStop(vw_);
      turn_ = 0;
    }
    else {
      GetMapPos(pos_n_, ROBOT_RADIUS, -PI/2, &x1, &y1);
      GetMapPos(pos_n_, GRID_WIDTH + ROBOT_RADIUS, -PI/2, &x2, &y2);
      if((map_[y1 * GRID_DIMENSIONS + x1] == 5) ||
             (map_[y2 * GRID_DIMENSIONS + x2] == 5))
	VWSetSpeed(vw_, 0.15, 0);
      else if(IRRead(ir_r_) == 0)
        VWSetSpeed(vw_, 0, 0.2);
      else if((PSDGet(psd_[2]) - OFFSET_RIGHT) > 95)  
      {
        if((prev_ <= 95) && (turn_ == 0) && (*count_ < GRID_DIMENSIONS*10))  { 
          GetPosition(vw_, &pos_n_, &map_[0]);
          segs_[*count_].x2 = GetSegment(pos_n_.x) +
		(ROBOT_RADIUS + prev_) * cos(pos_n_.phi - PI/2);
          segs_[*count_].y2 = GetSegment(pos_n_.y) +  
		(ROBOT_RADIUS + prev_) * sin(pos_n_.phi - PI/2);
          *count_ += 1;
          segs_[*count_].x1 = GetSegment(pos_n_.x) + 
		(ROBOT_RADIUS + prev_) * cos(pos_n_.phi - PI/2);
          segs_[*count_].y1 = GetSegment(pos_n_.y) +
		(ROBOT_RADIUS + prev_) * sin(pos_n_.phi - PI/2);
        }
        VWSetSpeed(vw_, 0.15, -1.8);
      }
      else if((prev_ != -1) && (dist_ < prev_))
        VWSetSpeed(vw_, 0.15, 0.2);
      else if((prev_ != -1) && (dist_ > prev_))  
        VWSetSpeed(vw_, 0.15, -0.2);
      else 
	VWSetSpeed(vw_, 0.15, 0);	
    }
    prev_ = dist_;
    GetPosition(vw_, &pos_n_, &map_[0]);
    if((fabs(start_.x - pos_n_.x) > 0.1) || (fabs(start_.y - pos_n_.y) > 0.1))
      looped_ = 1;
  }
  VEHStop(vw_);
  if(*count_ < 10*GRID_DIMENSIONS) {
    segs_[*count_].x2 = GetSegment(start_.x) + 
	(ROBOT_RADIUS + val_) * cos(start_.phi-PI/2); 
    segs_[*count_].y2 = GetSegment(start_.y) + 
	(ROBOT_RADIUS + val_) * sin(start_.phi-PI/2); 
    *count_ += 1;
  }

}

void LCD_MAP(VWHandle vw_, char *map_)  {
  int 		i = 0, j = 0, x, y;
  PositionType 	pos_n_;

  VWGetPosition(vw_, &pos_n_);
  GetMapPos(pos_n_, 0, 0, &x, &y);
  if(x < 64)
    x = 64;
  else if(x > (GRID_DIMENSIONS - 64))
    x = GRID_DIMENSIONS - 64;
  if(y < 32)
    y = 32;
  else if(y > (GRID_DIMENSIONS - 32))
    y = GRID_DIMENSIONS - 32;
  LCDClear();
  while((j < 64) && ((y + j - 32) < GRID_DIMENSIONS)) {
    i = 0;
    while((i < 128) && ((x + i - 64) < GRID_DIMENSIONS)) {
      if((x + i - 64) >= 0)
	if((y + j - 32) >= 0) 
	  if(map_[(y + j - 32)*GRID_DIMENSIONS+(x + i - 64)] == 1)
            LCDSetPixel(j, i, 1);
      i++;
    }
    j++;
  }
  
}

void Display_MAP(char *map_, XSegment *list, int value)  {
  int i, j;
  
  for(i = 0; i < GRID_DIMENSIONS; i++) {
    for(j = 0; j < GRID_DIMENSIONS; j++)
      switch(map_[i * GRID_DIMENSIONS + j])  {
        case 0:
          fprintf(stderr," ");
          break;
        case 1:
 	  fprintf(stderr,"@");
          break;
        case 3:
          fprintf(stderr,":");
          break;
        case 4:
          fprintf(stderr,"X");
          break;
        case 5:
          fprintf(stderr,"*");
          break;
        case 6:
 	  fprintf(stderr,".");
          break;
        default:
          fprintf(stderr,"-");
          break;
        }
    fprintf(stderr,"\n");
  }
  for(i=0;i< value;i++)
    fprintf(stderr,"%d %d  %d %d \n",list[i].x1,list[i].y1,
	list[i].x2,list[i].y2);
} 

int Get_Target(char *map_)  {
  
  int i = 0;
 
  while (i < (GRID_DIMENSIONS*GRID_DIMENSIONS))  {
    if((map_[i] == 6) || (map_[i] == 4))
      return i;
    i++;
  }
  return -1;
}

/* ----------------------------------------------------- */



int main ()
{

  /* -----------= Variable declarations =-------------- */
  VWHandle vw;
  XSegment data[GRID_DIMENSIONS*10];
  int count = 0, i;
  char     map[GRID_DIMENSIONS*GRID_DIMENSIONS] = {1};

  PSDHandle psd[3];
  /* BumpHandle bump_l, bump_r;*/
  IRHandle ir_rf, ir_rm;  

  
  /* - - - - - - -= PSD Handler =- - - - - - - - */
  
  for(i=GRID_DIMENSIONS;i<GRID_DIMENSIONS*GRID_DIMENSIONS;i++)
    map[i] = 3;
  for(i=0;i<GRID_DIMENSIONS;i++) {
    map[i] = 5;
    map[GRID_DIMENSIONS*GRID_DIMENSIONS - 1 - i] = 5;
    map[GRID_DIMENSIONS*i] = 5;
    map[GRID_DIMENSIONS*i +GRID_DIMENSIONS-1] = 5;
  }
  LCDMode(SCROLLING|NOCURSOR);
  LCDMenu("","","","STOP");

  psd[2] = PSDInit(PSD_RIGHT);
  if(psd[2] == 0)
    {
      LCDPutString("PSDInit Error!\n");
      OSWait(20); 
      exit(1);
    }
  psd[0] = PSDInit(PSD_LEFT);
  if(psd[0] == 0)
    {
      LCDPutString("PSDInit Error!\n");
      OSWait(20); exit(1);
    }
  psd[1] = PSDInit(PSD_FRONT);
  if(psd[1] == 0)
    {
      LCDPutString("PSDInit Error!\n");
      OSWait(20); exit(1);
    }
  PSDStart(psd[0], TRUE);
  PSDStart(psd[1], TRUE);
  PSDStart(psd[2], TRUE);

  
  /* - - - - - - -= IR Handler =- - - - - - - - - - */

    if ((ir_rf = IRInit(IR_RF)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(20); exit(1);
    }
    if ((ir_rm = IRInit(IR_RM)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(20); exit(1);
    }

 /* 
    if ((ir_rf = IRInit(IR_RF)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(200); return;
    }
    if ((ir_lf = IRInit(IR_LF)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(200); return;
    }*/

    
  /* - - - - - - - - - - - - - - - - - - - - - */

  vw = VWInit(VW_DRIVE,1); /* init v-omega interface */
  if(vw == 0)
    {
      LCDPutString("VWInit Error!\n");
      OSWait(200); exit(1);
    }

  VWSetPosition(vw, 0.0, 0.0, 0.0);
  VWStartControl(vw,7,0.3,7,0.1);


  /* main action loop */ 

  while (TRUE)  {
    LCDPutString("Scanning...\n");
    if(!Find_Obstacle(vw, psd[1], ir_rf, &map[0]))  {
      i = Get_Target(&map[0]);
      Display_MAP(&map[0], &data[0], count);
      if(i != -1)  {
        PathTrack(vw, &psd[0], i, &map[0]);
      } else {
        printf("\nMap Generation Finished\n");
        Display_MAP(&map[0], &data[0], count);
        KEYWait(ANYKEY);
	exit(0);
      }
    } else {
      LCDPutString("Wall-following\n");
      Wall_Follow_Right(vw, &psd[0], ir_rf, ir_rm, &map[0], &data[0], &count);
      LCD_MAP(vw, &map[0]);
      Display_MAP(&map[0], &data[0], count);
      AUBeep();
      KEYWait(ANYKEY);
    }

  } 
  PSDRelease();
  VWRelease(vw); /* exit driver */
  
}










