/*
 * Elliot Nicholls <nicho-ej@ee.uwa.edu.au> 1997/98
 * Revised by Nicholas Tay
 *
 * This file contains the VW-interface functions for the eyebot simulator
 ********************************************************************************/

/* 	$Id: vw.c,v 1.4 2000/01/07 09:16:31 pere Exp $	 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <forms.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include "protos.h"
#include "eyesim.h"
// #include "multi.h"
 
SpeedType robotSpeed[MAX_ROBOTS];
VWHandle hiddenVW[MAX_ROBOTS];
struct timezone z;
BOOL vwCommand[MAX_ROBOTS], vwInitiate[MAX_ROBOTS];
float DistEst_[MAX_ROBOTS], AngleEst_[MAX_ROBOTS];
PositionType robotView[MAX_ROBOTS];
PositionType pos[MAX_ROBOTS];
FILE *f;
int robot_count = 1;
void main();
void ContinuousPause();

/* cVWState.vwHandle = 0; */

VWHandle VWInit(DeviceSemantics semantics, int Timescale)
/* Input:	(semantics) semantic
 *		(Timescale) prescale value for 100Hz IRQ (1 to ...)
 * Output:	(returncode) VWHandle or 0 for error
 * Semantics:	Initialize given VW-Driver (only 1 can be initialized!)
 *		The motors and encoders are automatically reserved!!
 *		The Timescale allows to adjust the tradeoff between
 *		accuracy (scale=1, update at 100Hz) and speed(scale>1, update
 *		at 100/scale Hz).
 ********************************************************************************/
{
  
    if(!lcd_initialised)
      InitialiseLCD(); 
    if(robot_count != robot_id.Num_robot)
      ThreadFork();

    /* set zoom to full out */
    while((cState.map.w > (cState.w*cState.zoom)) || (cState.map.h > (cState.h*cState.zoom)))
      cState.zoom += 0.5; 
    cState.map.x = 0;
    cState.map.y = 0;
    pthread_mutex_lock(&sem);
    if(!fl_form_is_visible(view->viewscreen))  
      fl_show_form(view->viewscreen, FL_PLACE_FREE, FL_FULLBORDER, "View");
    if(robot_id.Num_robot > 1)
      fl_set_object_label(view->errorText,"\nFirst Robot has hit obstacle !!!\n");
    pthread_mutex_unlock(&sem);
    if (hiddenVW[FIndex(pthread_self())] != 0)
      return 0;
    hiddenVW[FIndex(pthread_self())] = (rand() % 9) + 1;
    VWSetPosition(hiddenVW[FIndex(pthread_self())], 0, 0, 0);
    return hiddenVW[FIndex(pthread_self())];
}


int VWRelease(VWHandle handle)
/* Input:	(handle) VWHandle to be released
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Release VW-Driver, stop motors
 ********************************************************************************/
{
  if(hiddenVW[FIndex(pthread_self())] != handle) {
    return -1;
  } else {
    hiddenVW[FIndex(pthread_self())] = 0;
    robot_count--;
    return 0;
  }  
}
     

int VWSetSpeed(VWHandle handle, meterPerSec v, radPerSec w) {
/* Input:	(handle) ONE VWHandle
 *              (v) new linear speed
 *              (w) new rotation speed
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Set the new speed: v(m/s) and w(rad/s not degree/s)
 *****************************************************************************/

  if(handle != hiddenVW[FIndex(pthread_self())])
    return -1;
  DistEst_[FIndex(pthread_self())] = 0;
  AngleEst_[FIndex(pthread_self())] = 0;
  robotSpeed[FIndex(pthread_self())].v = v;
  robotSpeed[FIndex(pthread_self())].w = w;
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return 0;
}

/* Input:	(handle) ONE VWHandle
 *              (vw) pointer to record to store actual v, w values
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Get the actual speed: v(m/s) and w(rad/s not degree/s)
 ********************************************************************************/

int VWGetSpeed(VWHandle handle, SpeedType* vw) {
  PositionType nowPos;
  struct timeval nowtime, nexttime;
  unsigned long timeP = 0;

  if(handle != hiddenVW[FIndex(pthread_self())]) 
    return -1;
  nowPos.x = robot[FIndex(pthread_self())].x;
  nowPos.y = robot[FIndex(pthread_self())].y;
  nowPos.phi = robot[FIndex(pthread_self())].radians;
  gettimeofday(&nowtime,&z);
  OSWait(5);
  gettimeofday(&nexttime,&z);
  timeP = 1000*(nexttime.tv_sec-nowtime.tv_sec) + (nexttime.tv_usec - nowtime.tv_usec)/1000;
  vw->v = sqrt((robot[FIndex(pthread_self())].x-nowPos.x)*(robot[FIndex(pthread_self())].x-nowPos.x) +
 	  (robot[FIndex(pthread_self())].y-nowPos.y)*(robot[FIndex(pthread_self())].y-nowPos.y)) / timeP;
  if(robotSpeed[FIndex(pthread_self())].v < 0)
    vw->v = -vw->v;
  vw->w = (robot[FIndex(pthread_self())].radians-nowPos.phi)*1000.0/timeP;
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return 0; 
  
}

/* Input:	(handle) ONE VWHandle
 *              (x) new x-position
 *              (y) new y-position
 *              (phi) new heading
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Set the new position: x(m), y(m) phi(rad not degree)
 ********************************************************************************/
int VWSetPosition(VWHandle handle, meter x, meter y, radians phi)  {
  float lengt_, ang_e = 0;

  if(handle != hiddenVW[FIndex(pthread_self())])  {
    fprintf(stderr,"Wrong VWHandle\n");
    return -1;
  }
/*
  robotStartPos[FIndex(pthread_self())].x = robot[FIndex(pthread_self())].x - x*1000;
  robotStartPos[FIndex(pthread_self())].y = robot[FIndex(pthread_self())].y - y*1000;
  robotStartPos[FIndex(pthread_self())].radians = robot[FIndex(pthread_self())].radians - phi;
  robotView[FIndex(pthread_self())].x = x*1000;
  robotView[FIndex(pthread_self())].y = y*1000; */

  x = -x;
  y = -y;
  lengt_ = sqrt(x*x + y*y);
  if(x == 0)
    if(y >= 0)
      ang_e = PI/2;
    else
      ang_e = -PI/2;
  else {
    ang_e  = atan(y/fabs(x));
    if(x < 0)
      ang_e = PI - ang_e;
  }
  robotStartPos[FIndex(pthread_self())].radians = 
	phi - robot[FIndex(pthread_self())].radians;
  robotStartPos[FIndex(pthread_self())].x   = robot[FIndex(pthread_self())].x 
	+ lengt_ * cos(robotStartPos[FIndex(pthread_self())].radians - ang_e)
	* 1000;
  robotStartPos[FIndex(pthread_self())].y   = robot[FIndex(pthread_self())].y 
	+ lengt_ * sin(robotStartPos[FIndex(pthread_self())].radians - ang_e)
	* 1000;

  pthread_mutex_lock(&sem);
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem); 
  return 0;
}

/* Input:	(handle) ONE VWHandle
 *              (pos) pointer to record to store actual position (x,y,phi)
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Get the actual position: x(m), y(m) phi(rad not degree)
 ********************************************************************************/

int VWGetPosition(VWHandle handle, PositionType* pos) {
  float ldiff, rdiff, xdiff, ydiff;

  if(handle != hiddenVW[FIndex(pthread_self())])
    return -1;
/*  
  ldiff = sqrt((robot[FIndex(pthread_self())].x-robotStartPos[FIndex(pthread_self())].x)*(robot[FIndex(pthread_self())].x
                -robotStartPos[FIndex(pthread_self())].x)
          + (robot[FIndex(pthread_self())].y-robotStartPos[FIndex(pthread_self())].y)*(robot[FIndex(pthread_self())].y
                -robotStartPos[FIndex(pthread_self())].y)); 
  rdiff = robot[FIndex(pthread_self())].radians - robotStartPos[FIndex(pthread_self())].radians;
  ang_ = (float)atan(fabs(Y_rob_[FIndex(pthread_self())]-robotStartPos[FIndex(pthread_self())].y)/
                fabs(X_rob_[FIndex(pthread_self())]-robotStartPos[FIndex(pthread_self())].x));
  xdiff_ = X_rob_[FIndex(pthread_self())]-robotStartPos[FIndex(pthread_self())].x;
  ydiff_ = robotStartPos[FIndex(pthread_self())].y - Y_rob_[FIndex(pthread_self())];
  pos->x = (xdiff_*cos(robotStartPos[FIndex(pthread_self())].radians) + ydiff_*sin(robotStartPos[FIndex(pthread_self())].radians))/1000.0;
  pos->y = (-xdiff_*sin(robotStartPos[FIndex(pthread_self())].radians) + ydiff_*cos(robotStartPos[FIndex(pthread_self())].radians))/1000.0;
  pos->phi = rdiff; 
  */

  xdiff = robot[FIndex(pthread_self())].x - 
	robotStartPos[FIndex(pthread_self())].x;
  ydiff = robot[FIndex(pthread_self())].y - 
	robotStartPos[FIndex(pthread_self())].y;
  ldiff = sqrt(xdiff * xdiff + ydiff * ydiff); 
  if(xdiff == 0)
    if(ydiff >= 0)
      rdiff = PI/2;
    else 
      rdiff = -PI/2;
  else {
    rdiff = atan(ydiff / abs(xdiff));
    if (xdiff < 0)
      rdiff = PI - rdiff;
  }

  pos->phi = robot[FIndex(pthread_self())].radians + 
	robotStartPos[FIndex(pthread_self())].radians;
  while (pos->phi > PI)
    pos->phi -= 2*PI;
  while (pos->phi < -PI)
    pos->phi += 2*PI;
  pos->x   = (float)((float)ldiff * cos(rdiff -
	robotStartPos[FIndex(pthread_self())].radians)) / 1000.0;
  pos->y   = -(float)((float)ldiff * sin(rdiff -
	robotStartPos[FIndex(pthread_self())].radians)) / 1000.0;
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return 0;
}

int VWStartControl(VWHandle handle, float Vv, float Tv, float Vw, float Tw)  {
/* Input:	(handle) ONE VWHandle
 *              (Vv) the paramter for the proportional component of the v-controller
 *              (Tv) the paramter for the integrating component of the v-controller
 *              (Vw) the paramter for the proportional component of the w-controller
 *              (Tv) the paramter for the integrating component of the w-controller
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Enable the PI-controller for the vw-interface and set the parameters.
 *              As default the PI-controller is deactivated when the vw-interface is 
 *              initialised. The controller tries to keep the desired speed (set with
 *              VWSetSpeed) stable by adapting the energy of the involved motors.
 *              The parameters for the controller have to be choosen carefully!
 *              The formula for the controller is:
 *                                          t
 *              new(t) = V*(diff(t) + 1/T * [diff(t)dt )
 *                                          0 
 *              V: a value usually around 1.0
 *              T: a value usually between 0 and 1.0
 *              After enabling the controller the last set speed (VWSetSpeed) is taken
 *              as the speed to be held stable.
 ********************************************************************************/
  if(vwInitiate[FIndex(pthread_self())] || (handle != hiddenVW[FIndex(pthread_self())]))  
    return -1;
  vwInitiate[FIndex(pthread_self())] = TRUE;
  return 0;

}

int VWStopControl(VWHandle handle)  {
/* Input:	(handle) ONE VWHandle
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Disable the controller immediately. The vw-interface continues normally
 *              with the last valid speed of the controller.
 ********************************************************************************/
  if((!vwInitiate[FIndex(pthread_self())]) || (handle != hiddenVW[FIndex(pthread_self())]))
    return -1;
  vwInitiate[FIndex(pthread_self())] = FALSE;
  return 0;
}

int VWGetPosition(VWHandle handle, PositionType* pos);
/* Input:	(handle) ONE VWHandle
 *		(pos) pointer to record to store actual position (x,y,phi)
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Get the actual position: x(m), y(m) phi(rad not degree)
 ********************************************************************************/

/* Input:	(handle) ONE VWHandle
 *              (delta)  distance to drive in m (pos. -> forward)
 *                                              (neg. -> backward)
 *              (v)      speed to drive with
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   drives distance "delta" with speed v straight ahead (forward or backward)
 *              any subsequent call of VWDriveStraight, -Turn, -Curve or VWSetSpeed 
 *              while this one is still being executed, results in an immediate interruption
 *              of this command
 ********************************************************************************/
int VWDriveStraight(VWHandle handle, meter delta, meterPerSec v){

  if(handle != hiddenVW[FIndex(pthread_self())])
    return -1;
  if(delta < 0)  {
    v = -v;
    //delta = -delta;
  }
  AngleEst_[FIndex(pthread_self())] = 0;
  DistEst_[FIndex(pthread_self())] = fabs(delta);
  robotSpeed[FIndex(pthread_self())].v = v;
  robotSpeed[FIndex(pthread_self())].w = 0.0000;
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return 0;  
}

/* Input:	(handle) ONE VWHandle
 *              (delta)  degree to turn in radians (pos. -> counter-clockwise)
 *                                                 (neg. -> clockwise)
 *              (w)      speed to turn with
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   turns about "delta" with speed w on the spot (clockwise or counter-clockwise)
 *              any subsequent call of VWDriveStraight, -Turn, -Curve or VWSetSpeed 
 *              while this one is still being executed, results in an immediate interruption
 *              of this command
 ********************************************************************************/

int VWDriveTurn(VWHandle handle, radians delta, radPerSec w){

  if (handle != hiddenVW[FIndex(pthread_self())])
    return -1;
  if(delta < 0)  {
    w = -w;
    delta = -delta;
  }
  robotSpeed[FIndex(pthread_self())].v = 0.00;
  robotSpeed[FIndex(pthread_self())].w = w;
  DistEst_[FIndex(pthread_self())] = 0;
  AngleEst_[FIndex(pthread_self())] = fabs(delta);
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return 0;
}

/* Input:	(handle)    ONE VWHandle
 *              (delta_l)   length of curve_segment to drive in m (pos. -> forward)
 *                                                                (neg. -> backward)
 *              (delta_phi) degree to turn in radians (pos. -> counter-clockwise)
 *                                                    (neg. -> clockwise)
 *              (v)         speed to drive with
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   drives a curve segment of length "delta_l" with overall vehicle turn of "delta_phi"
 *              with speed v (forw. or backw. / clockw. or counter-clockw.).      
 *              any subsequent call of VWDriveStraight, -Turn, -Curve or VWSetSpeed 
 *              while this one is still being executed, results in an immediate interruption
 *              of this command
 ********************************************************************************/

int VWDriveCurve(VWHandle handle, meter delta_l, radians delta_phi, meterPerSec v)  {
  float timeToGo = 0;

  if(handle != hiddenVW[FIndex(pthread_self())])  {
    fprintf(stderr,"Wrong VW handle\n");
    return -1;
  }
  if(delta_l<0)  {
    delta_l = -delta_l;
    v = -v;
  }
  AngleEst_[FIndex(pthread_self())] = fabs(delta_phi);
  DistEst_[FIndex(pthread_self())] = fabs(delta_l);
  timeToGo = fabs(delta_l/v);
  robotSpeed[FIndex(pthread_self())].v = v;
  robotSpeed[FIndex(pthread_self())].w = delta_phi/timeToGo;
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return 0;
  
}

/* Input:	(handle) ONE VWHandle
 * Output:      0.0 = previous VWDriveX command has been completed
 *              any other value = remaining distance to goal 
 * Semantics:   remaining distance to goal set by VWDriveStraight, -Turn
 *              (for -Curve only the remaining part of delta_l is reported)
 ********************************************************************************/
float VWDriveRemain(VWHandle handle){
  unsigned long timeToGo;
  struct timeval timenow;

  if(hiddenVW[FIndex(pthread_self())] != handle)  {
    fprintf(stderr,"Wrong handle\n");
    return -1;
  }
  gettimeofday(&timenow,&z);
  timeToGo = (timeEstimate[FIndex(pthread_self())].tv_sec - timenow.tv_sec)*1000 + (timeEstimate[FIndex(pthread_self())].tv_usec - timenow.tv_usec)/1000;
  if(!vwCommand[FIndex(pthread_self())])
    return 0.0;
  pthread_mutex_lock(&sem);
  fl_check_forms();
  pthread_mutex_unlock(&sem);
  if(robotSpeed[FIndex(pthread_self())].v != 0)
    return (fabs(robotSpeed[FIndex(pthread_self())].v*timeToGo));
  else
    return (fabs(robotSpeed[FIndex(pthread_self())].w*timeToGo));
}


/* Input:	(handle) ONE VWHandle
 * Output:      -1 = error wrong handle
 *              0 = vehicle is still in motion
 *              1 = previous VWDriveX command has been completed
 * Semantics:   checks if previous VWDriveX() command has been completed
 ********************************************************************************/
int VWDriveDone(VWHandle handle){
  if(handle != hiddenVW[FIndex(pthread_self())])  {
    fprintf(stderr,"ERROR: wrong handle\n");
    return -1;
  }
  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);
  return ((DistEst_[FIndex(pthread_self())] == 0.0000) && (AngleEst_[FIndex(pthread_self())] == 0.0000));
}

/* Input:	(handle) ONE VWHandle
 * Output:      -1 = error wrong handle
 *              0 = previous VWDriveX command has been completed
 * Semantics:   blocks the calling process until the previous VWDriveX() command has been completed
 ********************************************************************************/
int VWDriveWait(VWHandle handle) {

  if(handle != hiddenVW[FIndex(pthread_self())])
    return -1;
  while ((DistEst_[FIndex(pthread_self())] != 0.0000) || (AngleEst_[FIndex(pthread_self())] != 0.0000)) {
    pthread_mutex_lock(&sem);
    fl_check_forms();
    fl_redraw_object(view->screen);
    pthread_mutex_unlock(&sem);
  }
  pthread_mutex_lock(&sem);
  fl_check_forms();
  pthread_mutex_unlock(&sem);
  return 0;
}

int
VWStalled (VWHandle handle)
{
  fprintf(stderr, "VWStalled(0x%x) stub called\n", handle);

  pthread_mutex_lock(&sem);
  fl_check_forms();
  fl_redraw_object(view->screen);
  pthread_mutex_unlock(&sem);

  return 0;
}
