/*------------------------------------------------------------------------
| Filename: drive.c
|
| Author:       Nicholas Tay, UWA 1997
|
| Description:  Drives EyeBot vehicle
|               using PSD-sensor 
|               Attempts to drive down a corridor, avoiding the walls.
|		( Revised from drive_rc.c )
|
-------------------------------------------------------------------------- */

#include "protos.h"

#include "keys.h"
#include "lcd.h"
#include "cam.h"
#include "types.h" 
#include "const.h" 
#include "kern.h"
#include "vw.h"
#include "hdt.h"

#include <stdio.h>
#include <math.h>

#define PI 3.1415926535
#define GRID_WIDTH 80   //millimeters
#define GRID_CENTER 35 
#define GRID_DIMENSIONS 70
#define PSD_MAX 300

#ifndef ROBOT_RADIUS
#define ROBOT_RADIUS 80
#endif

/* --------------------= VEHStop =---------------------- */

void VEHStop(VWHandle vw_)
{
  VWSetSpeed(vw_, 0, 0);
}

int Check_Map(char *map_, int pos_, int value_)  {
  int i, j, temp = 0;

  for(i=-1;i<2;i++)
    for(j=-1;j<2;j++)
      if(map_[pos_-i*GRID_DIMENSIONS+j] == value_)
        temp = 1;
  return temp;
}

radians Angle_Dist(radians x_, radians y_)  {
  if(x_ < 0)
    x_ += 2*PI;
  if(y_ < 0)
    y_ += 2*PI;
  if(y_ < x_)
    y_ += 2*PI;
  return (y_ - x_);
}

void GetAngle(PositionType *target_, int index_)  {
  int x, y;

  y = index_ / GRID_DIMENSIONS;
  x = index_ % GRID_DIMENSIONS;
  target_->x = ((float)x - (float)GRID_CENTER + 0.5)*(float)GRID_WIDTH/1000.0; 
  target_->y = ((float)y - (float)GRID_CENTER + 0.5)*(float)GRID_WIDTH/-1000.0; 
}


int GetPosition(VWHandle vw_, PositionType *pos_, char *map_)  {
  int i = 0, j = 0;

  VWGetPosition(vw_, pos_);
  i = ((int)(pos_->x*1000) / GRID_WIDTH) + GRID_CENTER;
  j = ((int)(-pos_->y*1000) / GRID_WIDTH) + GRID_CENTER;
  if(pos_->x < 0)  
    i--;
  if(pos_->y > 0)
    j--;
  if(map_[j*GRID_DIMENSIONS+i] != 1)
    map_[j*GRID_DIMENSIONS+i] = 0;
  return j*GRID_DIMENSIONS+i;
}

float DesiredAngle(VWHandle vw_, PositionType pos_n, char *map_)  {
  PositionType current;
  float angle = 0;

  GetPosition(vw_, &current, &map_[0]);
  if(current.x == pos_n.x)
    if(pos_n.y >= current.y)
      return  PI/2;
    else
      return -PI/2;
  angle = atan((pos_n.y - current.y) / fabs(pos_n.x - current.x));
  if(pos_n.x < current.x)
    angle = PI - angle;
  return angle;
}  

void PathTrack(VWHandle vw_, PSDHandle *psd_, int target, char *map_)  {
  PositionType dest_n, pos_n_;
  int i, j;

  GetAngle(&dest_n, target);
  printf("%f angle to point\n",DesiredAngle(vw_, dest_n, &map_[0]));
  printf("%f %f\n",dest_n.x, dest_n.y);
  do {
    VWSetSpeed(vw_, 0, 0.2);
    VWGetPosition(vw_, &pos_n_);
  } while(Angle_Dist(pos_n_.phi, DesiredAngle(vw_, dest_n, &map_[0])) > 0.03);
  VEHStop(vw_);
  do {
    VWSetSpeed(vw_, 0.2, 0);
    VWGetPosition(vw_, &pos_n_);
fprintf(stderr,"%f %f\n",pos_n_.x,pos_n_.y);
  } while((pos_n_.x - dest_n.x) * (pos_n_.x - dest_n.x) + 
		(pos_n_.y - dest_n.y) * (pos_n_.y - dest_n.y) > 0.01);
  VEHStop(vw_);
  for(i = -1; i < 2; i++)
    for(j = -1; j < 2; j++)
      if(map_[target + GRID_DIMENSIONS * i + j] == 6)
	map_[target + GRID_DIMENSIONS * i + j] = 0;
  
}

int Find_Obstacle(VWHandle vw_, PSDHandle *x_, char *map_)  {
  int mindist_ = 999, i, obstacle_, x1 = 0, y1 = 0;
  radians angle_ = 0, value_ = 0;
  meter x = 0.0, y = 0.0;
  PositionType pos_n_, start_;

  VWGetPosition(vw_, &start_);
  VWSetSpeed(vw_, 0, 0.4);
  do {
    GetPosition(vw_, &pos_n_, &map_[0]);
    for(i=0;i<3;i++)  {
      value_ = pos_n_.phi;
      if(i == 0)
        value_ += PI/2;
      else if(i == 2)
        value_ -= PI/2;
      obstacle_ = PSDGet(x_[i]);
      if(obstacle_ < PSD_MAX)  {
        x = pos_n_.x + (float)(obstacle_+ROBOT_RADIUS)*cos(value_)/1000.0;
        y = pos_n_.y + (float)(obstacle_+ROBOT_RADIUS)*sin(value_)/1000.0;
        x1 = ((int)(x*1000) / GRID_WIDTH) + GRID_CENTER;
        y1 = ((int)(-y*1000) / GRID_WIDTH) + GRID_CENTER;
        if(x < 0)
          x1--;
        if(y > 0)
          y1--;
        if((map_[y1*GRID_DIMENSIONS+x1] != 0) && 
		(map_[y1*GRID_DIMENSIONS+x1]!=1)) 
          if(!Check_Map(&map_[0], y1*GRID_DIMENSIONS+x1, 1))
          
          {
          map_[y1*GRID_DIMENSIONS+x1] = 4;
          if(obstacle_ < mindist_)  {
            mindist_ = obstacle_;
            angle_ = value_;
          }
        }

      } else {
	x = pos_n_.x + (float)(PSD_MAX+ROBOT_RADIUS)*cos(value_)/1000.0;
        y = pos_n_.y + (float)(PSD_MAX+ROBOT_RADIUS)*sin(value_)/1000.0;
        x1 = ((int)(x*1000) / GRID_WIDTH) + GRID_CENTER;
        y1 = ((int)(-y*1000) / GRID_WIDTH) + GRID_CENTER;
        if(x < 0)
          x1--;
        if(y > 0)
          y1--;

        if((map_[y1*GRID_DIMENSIONS+x1] != 0) &&
                (map_[y1*GRID_DIMENSIONS+x1]!=1))
          if(!Check_Map(&map_[0], y1*GRID_DIMENSIONS+x1, 1))
	    map_[y1*GRID_DIMENSIONS+x1] = 6;
      }
    } 
    
  } while(Angle_Dist(start_.phi, pos_n_.phi) < PI);
  VEHStop(vw_);

  if(mindist_ == 999)
    return 0;
  
  GetPosition(vw_, &start_, &map_[0]);
  if(start_.phi < 0)
    start_.phi += 2*PI;
  if(angle_ < 0)
    angle_ += 2*PI;
  VWDriveTurn(vw_, -Angle_Dist(angle_, start_.phi), PI/3);
  while(!VWDriveDone(vw_));


  VWSetSpeed(vw_, 0.2, 0);
  while(PSDGet(x_[1]) > 80) 
    GetPosition(vw_, &start_, &map_[0]);
  VEHStop(vw_);
  return 1;
}

int Get_Dist(VWHandle vw_, PSDHandle psd_, PSDHandle psdLeft, char *map_) {
  int i, j, k, x = 0, y = 0, obstacle_, dist;
  float x1, y1;
  PositionType pos_n_;

  obstacle_ = PSDGet(psd_);
  dist = PSDGet(psdLeft);
  GetPosition(vw_, &pos_n_, &map_[0]);
  if(dist < 200)  {
      x1 = pos_n_.x + (float)(dist+ROBOT_RADIUS)*cos(pos_n_.phi + PI/2)/1000.0;
      y1 = pos_n_.y + (float)(dist+ROBOT_RADIUS)*sin(pos_n_.phi + PI/2)/1000.0;
      x = ((int)(x1*1000) / GRID_WIDTH) + GRID_CENTER;
      y = ((int)(-y1*1000) / GRID_WIDTH) + GRID_CENTER;
      if(x1 < 0)
        x--;
      if(y1 > 0)
        y--;
      if((map_[y*GRID_DIMENSIONS+x] != 0) &&
              (map_[y*GRID_DIMENSIONS+x]!=1))
        if(!Check_Map(&map_[0], y*GRID_DIMENSIONS+x, 1))
          map_[y*GRID_DIMENSIONS+x] = 4;
  }
  if(obstacle_ < 200)  {
    dist = obstacle_;
    for(k=0; k<5; k++)  {
      x1 = pos_n_.x + (float)(dist+ROBOT_RADIUS)*cos(pos_n_.phi - PI/2)/1000.0;
      y1 = pos_n_.y + (float)(dist+ROBOT_RADIUS)*sin(pos_n_.phi - PI/2)/1000.0;
      x = ((int)(x1*1000) / GRID_WIDTH) + GRID_CENTER;
      y = ((int)(-y1*1000) / GRID_WIDTH) + GRID_CENTER;
      if(x1 < 0)
        x--;
      if(y1 > 0)
        y--;
      if(dist == obstacle_) {
        map_[y*GRID_DIMENSIONS+x] = 1;
        for(i=-1;i<2;i++)
          for(j=-1;j<2;j++)
            if((map_[y*GRID_DIMENSIONS+x+i*GRID_DIMENSIONS+j] == 4) ||
	            (map_[y*GRID_DIMENSIONS+x+i*GRID_DIMENSIONS+j] == 6))
	      map_[y*GRID_DIMENSIONS+x+i*GRID_DIMENSIONS+j] = 3;
        
      }
      if((map_[y*GRID_DIMENSIONS+x] != 1) && (dist != obstacle_))
        map_[y*GRID_DIMENSIONS+x] = 0;
      dist -= obstacle_/4;
    }
  }
  return obstacle_;
}

void Wall_Follow_Right(VWHandle vw_, PSDHandle *psd_, char *map_, XSegment *segs_) 
{
  int dist_ = 0, prev_ = -1, looped_ = 0;
  SpeedType    stopped_;
  PositionType pos_n_, start_;

  VWDriveTurn(vw_, PI/2, 0.4);
  while(!VWDriveDone(vw_))
    Get_Dist(vw_, psd_[2], psd_[0], &map_[0]);
  VEHStop(vw_);
   
  VWSetSpeed(vw_, 0.2, 0);
  GetPosition(vw_, &pos_n_, &map_[0]);
  GetPosition(vw_, &start_, &map_[0]);
  while((looped_ == 0) || ((fabs(start_.x - pos_n_.x) > 0.05) || 
				(fabs(start_.y - pos_n_.y) > 0.05)))  {
  VWGetSpeed(vw_, &stopped_);
  dist_ = Get_Dist(vw_, psd_[2], psd_[0], &map_[0]);

  if((stopped_.v < 0.05) && (stopped_.w == 0))
    {
fprintf(stderr,"stalled\n");
      VWSetSpeed(vw_, 0, 0.3);
    }
  else if(PSDGet(psd_[1]) < 60)
    {
    VEHStop(vw_);
    VWDriveTurn(vw_, PI/4, 0.4);
    while(!VWDriveDone(vw_))
      prev_ = Get_Dist(vw_, psd_[2], psd_[0], &map_[0]);
    while((dist_ = PSDGet(psd_[2])) <= prev_)
      {
      VWSetSpeed(vw_, 0, 0.30);
      prev_ = dist_;
      }
    }
  else if(PSDGet(psd_[2]) > 100)  
    {
fprintf(stderr,"Cant find corner %f \n", 0.2/(float)prev_*1000.0);
    VWSetSpeed(vw_, 0.15, -1.6);
    dist_ = prev_;
    }
  else if((prev_ != -1) && (dist_ > prev_))  
  //else if(dist_ > 50)
    {
      VWSetSpeed(vw_, 0.15, -0.15);
    }
  else if((prev_ != -1) && (dist_ < prev_))
  //else if(dist_ < 50)
    {
      VWSetSpeed(vw_, 0.15, 0.15);
    }
  else VWSetSpeed(vw_, 0.15, 0);
  prev_ = dist_;
  GetPosition(vw_, &pos_n_, &map_[0]);
  if((fabs(start_.x - pos_n_.x) > 0.05) || (fabs(start_.y - pos_n_.y) > 0.08))
    looped_ = 1;
}
VEHStop(vw_);

}

void Display_MAP(char *map_)  {
  int i, j;
  
  for(i=0;i<GRID_DIMENSIONS;i++) {
    for(j=0;j<GRID_DIMENSIONS;j++)
      switch(map_[i*GRID_DIMENSIONS+j])  {
        case 0:
          fprintf(stderr," ");
          break;
        case 1:
 	  fprintf(stderr,"@");
          break;
        case 3:
          fprintf(stderr,":");
          break;
        case 4:
          fprintf(stderr,"X");
          break;
        case 6:
 	  fprintf(stderr,".");
          break;
        default:
          fprintf(stderr,"-");
          break;
        }
    fprintf(stderr,"\n");
  }
} 

int Get_Target(char *map_)  {
  
  int i = 0;
 
  while (i < (GRID_DIMENSIONS*GRID_DIMENSIONS))  {
    if((map_[i] == 6) || (map_[i] == 4))
      return i;
    i++;
  }
  return -1;
}

/* ----------------------------------------------------- */



int main ()
{

  /* -----------= Variable declarations =-------------- */
  VWHandle vw;
  PositionType back;
  XSegment data[100];
  int count = 0, i;
  char     map[GRID_DIMENSIONS*GRID_DIMENSIONS] = {1};

  PSDHandle psd[3];
  /* BumpHandle bump_l, bump_r;*/
  /*IRHandle ir_lm, ir_rm, ir_rf, ir_lf;  */

  
  /* - - - - - - -= PSD Handler =- - - - - - - - */
  
  for(i=0;i<GRID_DIMENSIONS*GRID_DIMENSIONS;i++)
    map[i] = 3;
  LCDMode(SCROLLING|NOCURSOR);
  LCDMenu("","","","STOP");

  psd[2] = PSDInit(PSD_RIGHT);
  if(psd[2] == 0)
    {
      LCDPutString("PSDInit Error!\n");
      OSWait(20); 
      exit(1);
    }
  psd[0] = PSDInit(PSD_LEFT);
  if(psd[0] == 0)
    {
      LCDPutString("PSDInit Error!\n");
      OSWait(20); exit(1);
    }
  psd[1] = PSDInit(PSD_FRONT);
  if(psd[1] == 0)
    {
      LCDPutString("PSDInit Error!\n");
      OSWait(20); exit(1);
    }
  PSDStart(psd[0], TRUE);
  PSDStart(psd[1], TRUE);
  PSDStart(psd[2], TRUE);

  
  /* - - - - - - -= IR Handler =- - - - - - - - - - */

 /* 
    if ((ir_lm = IRInit(IR_LM)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(200); return;
    }
    if ((ir_rm = IRInit(IR_RM)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(200); return;
    }
    if ((ir_rf = IRInit(IR_RF)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(200); return;
    }
    if ((ir_lf = IRInit(IR_LF)) == 0)  {
    LCDPutString("IR Init Error!\n");
    OSWait(200); return;
    }*/

    
  /* - - - - - - - - - - - - - - - - - - - - - */

  vw = VWInit(VW_DRIVE,1); /* init v-omega interface */
  if(vw == 0)
    {
      LCDPutString("VWInit Error!\n");
      OSWait(200); exit(1);
    }

  VWSetPosition(vw, 0.0, 0.0, 0.0);
  VWStartControl(vw,7,0.3,7,0.1);


  /* main action loop */ 

  while (TRUE)  {
    if(!Find_Obstacle(vw, &psd[0], &map[0]))  {
      i = Get_Target(&map[0]);
      printf("The map space is %d\n\n",i);
      Display_MAP(&map[0]);
      if(i != -1)  {
        PathTrack(vw, &psd[0], i, &map[0]);
      } else
        printf("\nMap Generation Finished\n");
    } else {
      Wall_Follow_Right(vw, &psd[0], &map[0], &data[0]);
    Display_MAP(&map[0]);
      VWGetPosition(vw,&back);
      printf("%f %f %f\n\n",back.x,back.y,back.phi);  
      AUBeep();
      KEYWait(ANYKEY);
    }

  } 
  PSDRelease();
  VWRelease(vw); /* exit driver */
  
}










