#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <pthread.h>
#include <time.h>
#include <errno.h>
#include <assert.h>

#include "kern.h"
#include "protos.h"
#include "eyesim.h"
#include "debugprint.h"

#define MAX_TCB 1000

/*
 * OSSemV and P don't work if this is 255 therefore i made it 254, no
 * other reason. (Daniel Hoey)
 */
static BYTE ThreadMode = 254;

struct thread_info
{
  struct tcb tcb;
  struct sem sem;
  pthread_t id;
};
static int NumOfThreads = 0;
static struct thread_info threads[MAX_TCB];
static pthread_mutex_t listlock = PTHREAD_MUTEX_INITIALIZER;

void check_forms_redraw_screen(void);
void add_robot_thread(int robotid, pthread_t new_thread);

static __inline__ void
lock(void)
{
  if ( -1 == pthread_mutex_lock(&listlock) )
    DBG(0, "lock(listlock) failed");
}

static __inline__ void
unlock(void)
{
  if ( -1 == pthread_mutex_unlock(&listlock) )
    DBG(0, "unlock(listlock) failed");
}

/**************************  Thread Routine  **************************/

static int
indexThread(void)
{
  int index;
  for (index=0; index<NumOfThreads; index++)
    if (pthread_equal(pthread_self(), threads[index].id))
      break;
  return (index == NumOfThreads) ? -1 : index;
}

static void *
thread_start(void *data)
{
  struct tcb* curtcb = data;
  void (*code)(void) = (void (*)(void))curtcb->pri;
  int i;

  lock();
  curtcb->pri = MIN_PRI;
  if((i=indexThread()) == -1)
    DBG(0,"Thread %ld not spawn", pthread_self());
  else
    {
      unlock();
      OSSemP(&threads[i].sem);
      code();
      lock();
      threads[i].id = -1;
    }
  unlock();
  check_forms_redraw_screen();
  return NULL;
}

/**************************  Multithreading  **************************/

int
OSMTStatus (void)
{
  return ThreadMode;
}

int
OSMTInit (BYTE mode)
{
  static int firstcall = 1;

  if (!firstcall)
    return 0;
  firstcall = 0;

  InitialiseLCD();
  if(robot_count != robot_id.Num_robot)
      ThreadFork();

  if (mode == COOP)
    DBG(0, "COOP mode not implemented, Default to PREEMPT");
  ThreadMode = PREEMPT;
  check_forms_redraw_screen();
  return 0;
}


struct tcb *
OSSpawn(char *name, int code, int stksiz, int pri, int uid)
{
  lock();
  if (NumOfThreads == MAX_TCB)
    {
      DBG(0,"Maximum threads spawned exceeded");
      unlock();
      return (struct tcb *)NULL;
    }
  threads[NumOfThreads].tcb.next = (struct tcb*)NULL;
  if(NumOfThreads==0)
    threads[NumOfThreads].tcb.prev = (struct tcb*)NULL;
  else
    threads[NumOfThreads].tcb.prev = &threads[NumOfThreads-1].tcb;
  threads[NumOfThreads].tcb.semq = (struct tcb*)NULL;
  threads[NumOfThreads].tcb.stksiz = stksiz;
  threads[NumOfThreads].tcb.stackbase = (char *)calloc(stksiz,sizeof(char));
  if(pri != MIN_PRI)
    DBG(0, "Priority threads not implemented, default to MINPRI");
  threads[NumOfThreads].tcb.pri = code;   /* Use to store start code addr */
  threads[NumOfThreads].tcb.stat = TASK_NEW;
  threads[NumOfThreads].tcb.name = (char *)malloc(strlen(name));
  memcpy(threads[NumOfThreads].tcb.name, name, strlen(name));
  threads[NumOfThreads].tcb.uid = uid;
  OSSemInit(&threads[NumOfThreads].sem,0);
  if (0 != pthread_create(&threads[NumOfThreads].id, NULL, &thread_start,
                          &threads[NumOfThreads].tcb))
    {
      DBG(0,"Unable to spawn new thread");
      unlock();
      check_forms_redraw_screen();
      return (struct tcb *)NULL;
    }
  add_robot_thread(OSMachineID(), threads[NumOfThreads].id);
  NumOfThreads++;
  unlock();
  check_forms_redraw_screen();
  return &threads[NumOfThreads-1].tcb;
}


int
OSReady(struct tcb *thread)
{
  int i=0, index = -1;

  if (thread == (struct tcb *)NULL)
    {
      DBG(0, "OSReady: Thread not spawn");
      check_forms_redraw_screen();
      return -1;
    }
  lock();
  while (i<NumOfThreads)
    if((int)thread==(int)&threads[i].tcb)
      {
        index = i;
        i = NumOfThreads;
      }
    else
      i++;
  unlock();
  if (index == -1)
    {
      DBG(0,"OSReady: Thread not found");
      return -1;
    }
  if (threads[index].tcb.stat == TASK_READY)
    {
      DBG(0,"OSReady: Threads running already");
      return 0;
    }
  lock();
  threads[index].tcb.stat = TASK_READY;
  OSSemV(&threads[index].sem);
  unlock();
  check_forms_redraw_screen();
  return 0;
}


int
OSSuspend(struct tcb *thread)
{
  int i = 0, index = -1;

  if (thread == (struct tcb *)NULL)
    {
      DBG(0, "OSSuspend: Thread not spawn");
      check_forms_redraw_screen();
      return -1;
    }

  lock();
  while (i<NumOfThreads)
    if((int)thread==(int)&threads[i].tcb)
      {
        index = i;
        i = NumOfThreads;
      }
    else
      i++;
  if (pthread_equal(pthread_self(), threads[index].id))
    {
      thread->stat = TASK_SUSPEND;
      OSSemV(&threads[index].sem);
      unlock();
      return 0;
    }
  unlock();
  DBG(0,"Cannot suspend non-current threads");
  check_forms_redraw_screen();
  return -1;
}


int
OSReschedule(void)
{
  if(sched_yield() == -1)
    {
      DBG(0,"Unable to yield");
      check_forms_redraw_screen();
      return -1;
    }
  check_forms_redraw_screen();
  return 0;
}


int
OSYield(void)
{
  int i;

  lock();
  i=indexThread();
  if (i ==-1)
    {
      unlock();
      DBG(0,"OSYield: critical error");
      return -1;
    }
  threads[i].tcb.stat = TASK_SUSPEND;
  OSSemP(&threads[i].sem);
  unlock();
  OSReschedule();
  check_forms_redraw_screen();
  return 0;
}


int
OSRun(struct tcb *thread)
{
  OSReady(thread);
  OSReschedule();
  check_forms_redraw_screen();
  return 0;
}


int
OSGetUID(struct tcb *thread)
{
  int i = 0;

  check_forms_redraw_screen();

  if (thread == (struct tcb *)NULL)
    {
      int uid;
      lock();
      i = indexThread();
      if(i==-1)
        {
          unlock();
          DBG(0,"critical error");
          return -1;
        }
      uid = threads[i].tcb.uid;
      unlock();
      return uid;
    }
  return thread->uid;
}


int
OSKill(struct tcb *thread)
{
  int i = 0, index = -1;
  void *thread_status;
  pthread_t id;

  if (thread == (struct tcb *)NULL)
    {
      lock();
      i=indexThread();
      unlock();
      if (i==-1)
        {
          DBG(0,"critical error");
          check_forms_redraw_screen();
          return -1;
        }
      pthread_exit(NULL);
      check_forms_redraw_screen();
      return 0;
    }
  lock();
  index = indexThread();
  id = threads[index].id;
  unlock();
  if (-1==index)
    {
      DBG(0,"Threads not found");
      check_forms_redraw_screen();
      return -1;
    }
  pthread_cancel(id);
  i = pthread_join(id, &thread_status);
  if (i != 0 && i != ESRCH)
    {
      DBG(0,"Thread joined failed");
      check_forms_redraw_screen();
      return -1;
    }
  check_forms_redraw_screen();
  return 0;
}


int
OSExit(int code)
{
  check_forms_redraw_screen();
  pthread_exit((void*)code);
  return 0;
}


int
OSPanic(char *msg)
{
  DBG(0,"OS Panic: %s", msg);
  check_forms_redraw_screen();
  return 0;
}


int
OSSleep(int n)
{
  struct timeval t, waitSec, nowtime;
  struct timezone z;

  gettimeofday(&t,&z);
  waitSec.tv_sec = n / 100 + t.tv_sec;
  waitSec.tv_usec = (n *10000) + t.tv_usec;
  if (waitSec.tv_usec >= 1000000)
    {
      waitSec.tv_sec++;
      waitSec.tv_usec = waitSec.tv_usec % 1000000;
    }
  do
    {
      sched_yield();
      gettimeofday(&nowtime,&z);
      check_forms_redraw_screen();
    }
  while ((nowtime.tv_sec<=(waitSec.tv_sec+1)) && ((nowtime.tv_sec<waitSec.tv_sec)||(nowtime.tv_usec<waitSec.tv_usec)));
  return 0;
}


int
OSForbid(void)
{
  STUB();
  return 0;
}


int
OSPermit(void)
{
  STUB();
  return 0;
}


/**************************  Semaphores  **************************/

/* Detect if semaphore is initialized */
#define SEM_MAGIC 0x0F23B97

int
OSSemInit (struct sem *sem, int val)
{
  if(sem==NULL)
    return -1;

  if ( SEM_MAGIC == sem->magic  )
    {
      DBG(0,"Semaphore already initialized");
    }

  sem->count = val;
#if no_sem_owner
  sem->owner = -1;
#endif
  pthread_mutex_init(&sem->mutex, NULL);
  pthread_cond_init(&sem->cond, NULL);
  sem->magic = SEM_MAGIC;

  DBG(1, "robot %d initializing semaphore %d to %d",
      OSMachineID(), (int)sem, sem->count);
  fflush(NULL);

  return 0;
}


int
OSSemV(struct sem *sem)
{
  /* No semaphore structure, bad semaphore or not in MT mode */
  if(sem==NULL || sem->magic != SEM_MAGIC || 255 == ThreadMode)
    {
      DBG(0,"No sem, bad magic(%x)  or no MT mode(%d)",
              sem->magic, ThreadMode);
      return -1;
    }

  if ( -1 == pthread_mutex_lock(&sem->mutex) )
    {
      DBG(0,"Unable to pthread_mutex_lock");
      return -1;
    }
  sem->count++;
#if no_sem_owner
  sem->owner = -1;
#endif

  DBG(2, "robot %d unlocking semaphore %d = %d", OSMachineID(),
      (int)sem, sem->count);

  if ( -1 == pthread_mutex_unlock(&sem->mutex) )
    {
      DBG(0,"Unable to pthread_mutex_unlock");
      return -1;
    }
  if ( -1 == pthread_cond_signal(&sem->cond) )
    {
      DBG(0,"Unable to pthread_cond_signal");
      return -1;
    }
  sched_yield();

  return 0 ;
}


int
OSSemP(struct sem *sem)
{
  /* No semaphore structure, bad semaphore or not in MT mode */
  if(sem==NULL || sem->magic != SEM_MAGIC || 255 == ThreadMode)
    {
      DBG(0,"No sem, bad magic(%x)  or no MT mode(%d)",
              sem->magic, ThreadMode);
      return -1;
    }

  if ( -1 == pthread_mutex_lock(&sem->mutex) )
    {
      DBG(0,"Unable to pthread_mutex_lock");
      return -1;
    }

#if no_sem_owner
  if (sem->owner == pthread_self())
    {
      DBG(0, "Already got the lock!");
      fflush(NULL);
      abort();
      goto unlock;
    }
#endif
  while(sem->count <= 0)
    {
      if (-1 == pthread_cond_wait(&sem->cond, &sem->mutex) )
        {
          DBG(0,"Unable to pthread_cond_wait ");
          return -1;
        }
    }
  sem->count--;
#if no_sem_owner
  sem->owner = pthread_self();
#endif

  DBG(2, "robot %d locking semaphore %d = %d", OSMachineID(),
      (int)sem, sem->count);

unlock:
  if (-1 == pthread_mutex_unlock(&sem->mutex) )
    {
      DBG(0,"Unable to pthread_mutex_unlock");
      return -1;
    }
  return 0;
}
