/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-05-01
 *
 * EyeSim 3D camera simulation using Open Inventor API.
 *
 * This program is copylefted.  Check the GPL from Free Software
 * Fundation to learn the terms.
 */

#include <Inventor/SoDB.h>
#include <Inventor/SoInput.h>
#include <Inventor/SoInteraction.h>
#include <Inventor/SbBox.h>
#include <Inventor/nodekits/SoNodeKit.h>
#include <Inventor/nodes/SoDirectionalLight.h>
#include <Inventor/nodes/SoPerspectiveCamera.h>
#include <Inventor/nodes/SoSeparator.h>
#include <Inventor/nodes/SoTexture2.h>
#include <Inventor/nodes/SoTranslation.h>
#include <Inventor/nodes/SoRotationXYZ.h>
#include <Inventor/nodes/SoScale.h>
#include <Inventor/SoOffscreenRenderer.h>

#include <iostream>
#include <fstream>
#include <string>
#include <map>
#include <algorithm>
#include <assert.h>
#include <stdlib.h>
#include <pthread.h>

#include "types.h"
#include "vw.h"     // PositionType
#include "eyesim.h" // Segment

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/shape.h>
#include <Imlib.h>

extern "C" {
  int CAMInit(int zoom);
  int CAMGetFrame(image *buf);
  int CAMGetColFrame(colimage *buf, int convert);
  int CAMSet(int bright, int para1, int para2);
  int CAMGet(int *bright, int *para1, int *para2);
  int CAMMode(int mode);
  int CAMRelease(void);
  void eyesim_cam_addobject(int id);
  void eyesim_cam_removeobject(int id);
  void eyesim_cam_newmap(Segment *map, int count);

  int OSMachineID(void);
  int get_robot_position(int id, PositionType* pos);
  int get_obstacle_position(int id, PositionType* pos);
  char *eyesim_find_file(const char *filename);
  Window eyesim_show_image(Window window, int robotid, BYTE *ibuffer,
			   int width, int height,
			   int swidth, int sheight,
			   int xoff, int yoff,
			   int flipped);
}

#define BALLID 0x10
#define CAMSIM 0x12  /* Color camera, 0x10 = Quickcam, 0x11 = EyeCam */

static string vrmldir = "vrml/";

static pthread_mutex_t ivlock = PTHREAD_MUTEX_INITIALIZER;
static SoSeparator *root = NULL;
static SoSeparator *robotroot = NULL;
static SoSeparator *maproot = NULL;

class RobotInfo {
public:
  SoTranslation *translation;  /* Pointer to robot translation */
  SoRotationXYZ *rotationxyz;  /* Pointer to robot rotation */
  SoPerspectiveCamera *camera; /* Pointer to robot camera */
  SoSeparator   *root;         /* Toplevel robot root */
  Window        window;        /* Display last camera snapshot here */
  RobotInfo(SoTranslation *_translation,
	    SoRotationXYZ *_rotationxyz, 
	    SoPerspectiveCamera *_camera,
	    SoSeparator *_root)
    : translation(_translation),
      rotationxyz(_rotationxyz),
      camera(_camera),
      root(_root), window(0)
  {};

  /* Update robot position and orientation in the 3D world. */
  void moveto(PositionType &pos)
    {
      translation->translation.setValue(SbVec3f(pos.x, 0, pos.y));
      rotationxyz->angle.setValue(pos.phi-M_PI_2);
    }
};

typedef map<int, RobotInfo> RobotMap;
typedef RobotMap::value_type ValuePair;
typedef RobotMap::iterator RobotIter;

/* Map robot IDs to RobotInfo object */
static RobotMap robots;

/*****f* CAMSim3D/eyesim_show_image
 * NAME
 *   eyesim_show_image
 * FUNCTION
 *   Display image in the window given, or create a new window with
 *   this image and return the reference to this new window.  Flip
 *   image vertically if requested.
 ****** */
Window
eyesim_show_image(Window window, int robotid, BYTE *ibuffer,
		  int width, int height, 
		  int swidth, int sheight,
		  int xoff, int yoff,
		  int flipped)
{
  static Display *disp = NULL;
  static ImlibData *imlibdata;
  ImlibImage *im;

  if (NULL == disp)
    {
      /* Connect to the default Xserver */
      disp=XOpenDisplay(NULL);
      /* Immediately afterwards Intitialise Imlib */
      imlibdata=Imlib_init(disp);
    }
  
  /* Load the image specified as the first argument */
  im=Imlib_create_image_from_data(imlibdata, ibuffer, NULL, width, height);
  if (flipped)
    Imlib_flip_image_vertical(imlibdata, im);

  /* Render the original 24-bit Image data into a pixmap of size w * h */
  Imlib_render(imlibdata,im,swidth,sheight);

  if (0 == window)
    {
      XSetWindowAttributes attr;
      /* Create a Window to display in */
      window = XCreateWindow(disp, DefaultRootWindow(disp),
			     0, 0, swidth, sheight, 0, imlibdata->x.depth,
			     InputOutput, imlibdata->x.visual, 0, &attr);
      XSelectInput(disp,window,StructureNotifyMask);

      string title = string("Robot ") + (char)('0' + robotid);
      XStoreName(disp, window, title.c_str());

      /* Put the Image pixmap in the window */
      Imlib_apply_image(imlibdata,im,window);
      /* Actually display the window */
      XMapWindow(disp,window);
    }
  else /* Update image */
    Imlib_paste_image(imlibdata,im,window, xoff,yoff,swidth,sheight);

  /* Free some memory */
  Imlib_free_pixmap(imlibdata,Imlib_move_image(imlibdata,im));
  Imlib_kill_image(imlibdata, im);

  /* Synchronise with the Xserver */
  XSync(disp,False);
  return window;
}

static SoSeparator *
load(const string &file)
{
  SoSeparator *object;
  SoInput in;
  char *fullpath = eyesim_find_file(file.c_str());
  cerr << "find_file() found " << fullpath << endl;
  if (NULL == fullpath)
    return NULL;
  else
    {
      in.openFile(fullpath);
      free(fullpath);
      object = SoDB::readAll(&in);
      return object;
    }
}

static SoPerspectiveCamera *
new_eyecam(void)
{
  SoPerspectiveCamera *camera = new SoPerspectiveCamera;
  assert(camera);
  camera->position.setValue(0, 0.07, -0.03);
  // camera->orientation.setValue(0.0,0.0,-1.0,0.0);
  camera->nearDistance.setValue(0.05); /* 5 cm */
  camera->heightAngle.setValue(45/180.0 * M_PI);
  return camera;
}

void
eyesim_cam_addobject(int id)
{
  string file;
  if (0 <= id && id < BALLID)
    file = vrmldir + "robot.wrl";
  else
    file = vrmldir + "ball.wrl";

  cout << "New object " << id << " file=" << file << endl;

  if (NULL == root)
    {
      cerr << __FUNCTION__ << "("<<id<<") root==NULL!" << endl;
      return;
    }

  SoTranslation *translation = new SoTranslation();
  assert(NULL != translation);

  SoRotationXYZ *rotationxyz = new SoRotationXYZ();
  assert(NULL != rotationxyz);
  rotationxyz->axis.setValue(SoRotationXYZ::Y);

  SoSeparator *object = new SoSeparator;
  assert(object);

  SoSeparator * fileroot = load(file);
  if (NULL == fileroot)
    return;

  object->addChild(translation);
  object->addChild(rotationxyz);

  object->addChild(fileroot);

  SoPerspectiveCamera *camera = new_eyecam();
  object->addChild(camera);

  robots.insert(ValuePair(id, RobotInfo(translation, rotationxyz, camera,
					object)));

  if ( -1 == pthread_mutex_lock(&ivlock) ) {
    cerr << "lock(ivlock) failed" << endl;
    return;
  }

  assert(NULL != robotroot);
  robotroot->addChild(object);

  if ( -1 == pthread_mutex_unlock(&ivlock) ) {
    cerr << "unlock(ivlock) failed" << endl;
    return;
  }
}

void
eyesim_cam_removeobject(int id)
{
  cerr << __FUNCTION__ << "(" << id << ") stub!" << endl;
}

static SoDirectionalLight *
white_directional_light(float x, float y, float z)
{
  SoDirectionalLight * light = new SoDirectionalLight;
  light->intensity.setValue(1); /* Full intensity */
  light->color.setValue(1, 1, 1); /* White */
  light->direction.setValue(x, y, z);
  return light;
}

int
CAMInit(int zoom)
{
  if (NULL != root)
    return CAMSIM;

  string lightsfile = vrmldir + "lights.wrl";

  SoDB::init();
  SoNodeKit::init();
  SoInteraction::init();

  SoSeparator * lightsroot = load(lightsfile);
  if (NULL == lightsroot)
    return -1;

  root = new SoSeparator;
  assert(root);
  root->ref();

  /* XXX Try to get the light correct... */
  root->addChild(white_directional_light(-1,-0.5,-1));
  root->addChild(white_directional_light(-1,-0.5, 1));
  root->addChild(white_directional_light( 1,-0.5,-1));
  root->addChild(white_directional_light( 1,-0.5, 1));
  
  root->addChild(lightsroot);

  if (NULL == maproot)
    {
      maproot = new SoSeparator;
      root->addChild(maproot);
    }

  robotroot = new SoSeparator;
  root->addChild(robotroot);

  return CAMSIM;
}

/*****f* CAMSim3D/update_position
 * NAME
 *   update_position
 * FUNCTION
 *   Query current robot position and update Open Inventor database.
 ******
 */
static void
update_position(ValuePair &object)
{
  int id = object.first;
  PositionType pos;
  if (0 < id && id < BALLID)
    get_robot_position(id, &pos);
  else
    get_obstacle_position(id-BALLID, &pos);
  object.second.moveto(pos);
}

/*****f* CAMSim3D/CAMGetColFrame
 * NAME
 *   CAMGetColFrame
 * FUNCTION
 *   Move camera in position, take snapshot and return view as
 *   colimage.  Rendering at double resolution and picking different
 *   row/column for different pixel parts to simulate EyeCam.
 ****** */
int
CAMGetColFrame(colimage *view, int convert)
{
  if (NULL == root)
    {
      static int firsttime = 1;
      if (firsttime)
	{
	  firsttime = 0;
	  cerr << "Called CAMGetColFrame() before CAMInit()!" << endl;
	}
      return -1;
    }
  int robotid = OSMachineID();
  RobotInfo *robot = &(robots.find(robotid)->second);
  SbViewportRegion vp(imagecolumns*2, imagerows*2);

  if (NULL == robot->root) /* Unknown robot */
    {
      cerr << "Unregistered robot " << robotid
	   << " called CAMGetColFrame()" << endl;
      eyesim_cam_addobject(robotid);
      robot = &(robots.find(robotid)->second);
      return -1;
    }

  for_each(robots.begin(), robots.end(), update_position);

  SbVec3f campos = robot->camera->position.getValue() +
    robot->translation->translation.getValue();

  SoSeparator *toplevel = new SoSeparator;
  toplevel->ref();

  SoPerspectiveCamera *camera = new_eyecam();
  camera->position.setValue(campos);
  camera->orientation.setValue(robot->rotationxyz->getRotation());
  toplevel->addChild(camera);

  if ( -1 == pthread_mutex_lock(&ivlock) ) {
    cerr << "lock(ivlock) failed" << endl;
    return -1;
  }

  toplevel->addChild(root);

  SoOffscreenRenderer renderer(vp);
  SbColor bgcolor(0.5, 0.5, 0.5); /* grey */

  renderer.setViewportRegion(vp);
  renderer.setBackgroundColor(bgcolor);
  renderer.setComponents(SoOffscreenRenderer::RGB);
  assert(SoOffscreenRenderer::RGB == renderer.getComponents());

  if (!renderer.render(toplevel))
    {
      cerr << "Failed to generate texture" << endl;
      return -1;
    }

  toplevel->unref();

  if ( -1 == pthread_mutex_unlock(&ivlock) ) {
    cerr << "unlock(ivlock) failed" << endl;
    return -1;
  }

  switch (renderer.getComponents())
    {
    case SoOffscreenRenderer::RGB:
      {
	typedef unsigned char renderbuffer[imagerows*2][imagecolumns*2][3];
	renderbuffer *native = (renderbuffer *)renderer.getBuffer();
	for (int row=0; row < imagerows; row++)
	  for (int column=0; column < imagecolumns; column++)
	    { /* Fake misaligned EyeCam colors */
	      BYTE red   = (*native)[row*2+1][column*2  ][0];
	      BYTE green = (*native)[row*2+1][column*2  ][1];
	      BYTE blue  = (*native)[row*2  ][column*2+1][2];
	      if (convert)
		{
		  image *gview = (image*)view;
		  (*gview)[row][column] = (red + green + green + blue) >> 6;
		}
	      else
		{
		  (*view)[row][column][0] = red;
		  (*view)[row][column][1] = green;
		  (*view)[row][column][2] = blue;
		}
	    }
	robot->window = eyesim_show_image(robot->window, robotid,
					  renderer.getBuffer(),
					  imagecolumns*2, imagerows*2,
					  imagecolumns*2, imagerows*2,
					  0, 0, 0);
      }
      break;
    default:
      cerr << "Unsupported renderer format!" << endl;
      return -1;
      break;
    }

  return 0;
}

int
CAMGetFrame(image *buf)
{
  return CAMGetColFrame((colimage*)buf, 1);
}

int
CAMSet(int bright, int para1, int para2)
{
  /* Ignore new setting */
  return 0;
}

int
CAMGet(int *bright, int *para1, int *para2)
{
  /* Fake settings */
  *bright = 0;
  *para1  = 0;
  *para2  = 0;
  return 0;
}

int
CAMMode(int mode)
{
  /* Ignore new mode */
  return 0;
}

int
CAMRelease(void)
{
  if ( -1 == pthread_mutex_lock(&ivlock) ) {
    cerr << "lock(ivlock) failed" << endl;
    return -1;
  }

  if (NULL != root)
    root->unref();
  root = NULL;
  robotroot = NULL;

  if ( -1 == pthread_mutex_unlock(&ivlock) ) {
    cerr << "unlock(ivlock) failed" << endl;
    return -1;
  }
  return 0;
}

static SoSeparator *
object_translate_scale(SoSeparator *object, float x, float y,
		       float width, float height, float depth,
		       float thickness)
{
  SoSeparator *sep = new SoSeparator;
  SoTranslation *pos = new SoTranslation;
  SoScale *scale = new SoScale;

  SbVec3f position(x, height/2, y);
  pos->translation.setValue(position);

  if (0 == width)
    width = thickness;
  if (0 == height)
    height = thickness;
  if (0 == depth)
    depth = thickness;

  scale->scaleFactor.setValue(fabs(width), fabs(height), fabs(depth));

  sep->addChild(pos);
  sep->addChild(scale);
  sep->addChild(object);
  return sep;
}

/*****f* CAMSim3D/eyesim_cam_newmap
 * NAME
 *   eyesim_cam_newmap
 * FUNCTION
 *   Given a list of map segments (lines), build a world with these
 *   segments as walls, and a surface under all of them.
 ******
 */
void
eyesim_cam_newmap(Segment *map, int count)
{
  string wallfile = vrmldir + "wall.wrl";
  static SoSeparator *wall = NULL;

  string floorfile = vrmldir + "floor.wrl";
  static SoSeparator *floor = NULL;

  float thickness = 0.001; /* meter */
  float height = 0.10;     /* meter */

  if (NULL == root)
    {
      cerr << __FUNCTION__ << "() root==NULL!" << endl;
      CAMInit(0);
      if (NULL == root)
	return;
    }
  if (NULL == map)
    {
      cerr << __FUNCTION__ << "(NULL, "<<count<<")!" << endl;
      return;
    }
  if (NULL == maproot)
    {
      maproot = new SoSeparator;
      root->addChild(maproot);
    }
  else
    maproot->removeAllChildren();

  
  if (NULL == wall)
    wall = load(wallfile);
  if (NULL == floor)
    floor = load(floorfile);

  if (NULL == wall || NULL == floor)
    return;

  /* Make sure they don't go away */
  wall->ref();
  floor->ref();

  SbXfBox3f bbox;
  for (int i = 0; i < count; i++)
    {
      float width = (map[i].x1 - map[i].x2)/1000.0;
      float depth = (map[i].y1 - map[i].y2)/1000.0;
      float cx = map[i].x2/1000.0 + width/2.0;
      float cy = map[i].y2/1000.0 + depth/2.0;

      maproot->addChild(object_translate_scale(wall, cx, cy,
					       width, height, depth,
					       thickness));
      bbox.extendBy(SbVec3f(cx,0,cy));
    }

  /* Place floor below all the walls */
  SbVec3f _min, _max;
  bbox.getBounds(_min, _max);

  SbVec3f size = _max - _min;
  SbVec3f center = _min + size/2;

  maproot->addChild(object_translate_scale(floor, center[0], center[2],
					   size[0], size[1], size[2],
					   thickness));
}

#if defined(TEST)

char *
eyesim_find_file(const char *filename)
{
  FILE *fp = fopen(filename, "r");
  if (NULL != fp)
    {
      fclose(fp);
      return strdup(filename);
    }
  else
    return NULL;
}

static int curid = 0;
int
OSMachineID(void)
{
  return curid;
}

static inline double scale_rand(double scale) {return scale*rand()/RAND_MAX;}

/* Place robot at random position */
int
get_robot_position(int id, PositionType* pos)
{
  pos->x = scale_rand(2.0) - 1.0;
  pos->y = scale_rand(2.0) - 1.0;
  pos->phi = scale_rand(2*M_PI);
  return 0;
}

int
get_obstacle_position(int id, PositionType* pos)
{
  return get_robot_position(id, pos);
}

/*
 * Read colimage, storing the pixels as PPM in filename.  If filename
 * is "-", print image on stdout.
 * Returns 0 on success and < 0 on error.
 */
static int
ppm_save_colimage(const char *filename, colimage cimg)
{
  FILE *fp;
  int row, column;

  if (NULL == filename)
    return -2; /* Filename missing */

  if (0 == strcmp("-", filename))
    fp = stdout;
  else
    fp = fopen(filename, "w");
  if (NULL == fp)
    return -4;

  fprintf(fp, "P3\n");
  fprintf(fp, "%u %u\n", imagecolumns, imagerows);
  fprintf(fp, "%u\n", 255);

  for (row = 0 ; row < imagerows; row++)
    for (column = 0 ; column < imagecolumns; column++)
      {
	fprintf(fp, "%u %u %u \n",
		cimg[row][column][0], /* red */
		cimg[row][column][1], /* green */
		cimg[row][column][2]  /* blue */
		);
      }


  if (0 != fclose(fp))
    return -7; /* I don't know */

  return 0;
}

int
main(int argc, char *argv[])
{
  int robots = 2;
  colimage view;

  srand(time(NULL));

  CAMInit(0 /* NORMAL */);

  for (curid = 1; curid <= robots; curid++)
    eyesim_cam_addobject(curid);

  eyesim_cam_addobject(BALLID); /* Ball */

  for (int loop = 0; loop < 5; loop++)
    for (curid = 1; curid <= robots; curid++)
      {
	string filename = string("r") + (char)('0'+curid) + ".ppm";
	CAMGetColFrame(&view, 0);
	ppm_save_colimage(filename.c_str(), view);
      }

  eyesim_cam_exit();
}
#endif
