/*
 * Load and keep track of a two-dimentional map of walls.  Used when
 * drawing the two-D map of the world, and when generating the 3D
 * world for the camera simulation.
 */

#include <math.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>

#include "eyesim.h"
#include "debugprint.h"

/* The current map */
static int NumSeg = 0;
static Segment *wld = NULL;

Segment *
get_current_map(int *count)
{
  *count = NumSeg;
  return wld;
}

static void
remove_current_map(void)
{
  NumSeg = 0;
  if (NULL != wld)
    free(wld);
  wld = NULL;
}

/*****f* EyeSim/set_robot_start_position
 *
 * Update initial start position for robot id to the coordinates and
 * heading given in pos.  Used when loading new maps.
 *****
 */
void
set_robot_start_position(int id, Robot *pos)
{
  int index = id - 1;
  DBG(1, "New start pos (%d,%d,%f) for robot %d", pos->x, pos->y, pos->radians,
      id);
  X_rob_[index] = pos->x;
  Y_rob_[index] = pos->y;
  robot[index] = *pos;
}

/*
 * Not quite sure what this function does, but it was repeated in both
 * Maze() and World(), so I moved the code into a separate function.  */

static void
post_process_map()
{
  Segment tmpSeg;
  int cnt, pop, pushed;

  for(cnt=2; cnt<NumSeg; cnt++)
    {
      pop = 1;
      while(pop != -1)
        if (wld[pop].x1 < wld[cnt].x1)
          pop = (cnt-pop > 1)? pop+1 : -1;
        else
          {
            memcpy(&tmpSeg, wld+cnt, sizeof(Segment));
            for(pushed=cnt; pushed>pop; pushed--)
              memcpy(wld+pushed, wld+pushed-1, sizeof(Segment));
            memcpy(wld+pop, &tmpSeg, sizeof(Segment));
            pop = -1;
          }
    }
  memcpy(wld,wld+1,sizeof(Segment));

  for(cnt=0; cnt<NumSeg; cnt++)
    DBG(2, "POS(%d) %ld %ld - %ld %ld",
        cnt, wld[cnt].x1, wld[cnt].y1, wld[cnt].x2, wld[cnt].y2);

  while((cState.map.w < (cState.w*cState.zoom))
        || (cState.map.h < (cState.h*cState.zoom)))
    cState.zoom -= 0.5;
  while((cState.map.w > (cState.w*cState.zoom))
        || (cState.map.h > (cState.h*cState.zoom)))
    cState.zoom += 0.5;

  if (cState.map.x < 0)
    cState.map.x = 0;
  else if((cState.map.x+cState.w*cState.zoom) > cState.map.w)
    cState.map.x = cState.map.w - cState.w*cState.zoom;

  if (cState.map.y < 0)
    cState.map.y = 0;
  else if((cState.map.y+cState.h*cState.zoom) > cState.map.h)
    cState.map.y = cState.map.h - cState.h*cState.zoom;

  VW_SetPosition(1,0,0,0);
  sim_x = -1;
  eyesim_cam_newmap(wld, NumSeg);
}

void
World(char *file)
{
  char fdata[50], tmp[50];
  char ch;
  int stacklimit = 100;
  Point *stack = NULL;

  int cnt, pop;
  int pushed = 0;
  int tmpx, tmpy;
  FILE *fp;

  if (file == (char *)NULL)
    return;
  if ((fp = fopen(file,"r")) == NULL)
    {
      DBG(0,"Cannot open %s", file);
      return;
    }

  if (NULL == stack)
    stack = malloc(sizeof(*stack) * stacklimit);
  assert(NULL != stack);

  remove_obstacles();
  remove_current_map();
  while (!feof(fp))
    {
      fscanf(fp, "%[^\n]", fdata);
      if (!feof(fp))
        fscanf(fp, "%c", &ch);
      if (*fdata != '\0')
        {
          cnt = 0;
          while (isspace(fdata[cnt]))
            cnt++;
          if (fdata[cnt] != ';')
            {
              if (!strncmp(fdata+cnt, "width", 5))
                sscanf(fdata+cnt, "%s %ld", tmp, &cState.map.w);
              else if (!strncmp(fdata+cnt, "height", 6))
                sscanf(fdata+cnt, "%s %ld", tmp, &cState.map.h);
              else if ((!strncmp(fdata+cnt, "push", 4))
                       ||(!strncmp(fdata+cnt,"change",6)))
                {
                  assert(pushed < stacklimit);
                  sscanf(fdata+cnt, "%s %ld %ld %hd", tmp,
                         &stack[pushed].x, &stack[pushed].y,
                         &stack[pushed].theta);
                  pushed++;
                }
              else if (!strncmp(fdata+cnt, "pop", 3))
                {
                  pushed--;
                  if (pushed < 0)
                    {
                      DBG(0,"Error in format: Excess Pop");
                      return;
                    }
                }
              else if (!strncmp(fdata+cnt, "position", 8))
                {
                  double angle;
                  Robot pos;
                  sscanf(fdata+cnt, "%s %d %d %lf", tmp,
                         &pos.x, &pos.y, &angle);
                  pos.radians = DegToRad(angle);
                  pos.y = cState.map.h - pos.y; /* XXX Why is this here? */

                  set_robot_start_position(1, &pos);

                  cState.map.x = pos.x - cState.w/2*cState.zoom;
                  cState.map.y = pos.y - cState.h/2*cState.zoom;
                }
              else if (isdigit(fdata[cnt]) || (fdata[cnt] == '-'))
                {
                  if (NumSeg)
                    wld = (Segment *)realloc(wld, (NumSeg+1)*sizeof(Segment));
                  else
                    {
                      NumSeg++;
                      wld = (Segment *)malloc(2*sizeof(Segment));
                    }
                  sscanf(fdata+cnt, "%ld %ld %ld %ld", &wld[NumSeg].x1, &wld[NumSeg].y1, \
                         &wld[NumSeg].x2, &wld[NumSeg].y2);
                  for(pop=pushed-1; pop>=0; pop--)
                    {
                      tmpx = wld[NumSeg].x1;
                      tmpy = wld[NumSeg].y1;
                      wld[NumSeg].x1 = (int)(stack[pop].x
                                             + tmpx*cos(DegToRad(stack[pop].theta))
                                             + tmpy*sin(DegToRad(stack[pop].theta)));
                      wld[NumSeg].y1 = (int)(stack[pop].y
                                             + tmpy*cos(DegToRad(stack[pop].theta))
                                             - tmpx*sin(DegToRad(stack[pop].theta)));
                      tmpx = wld[NumSeg].x2;
                      tmpy = wld[NumSeg].y2;
                      wld[NumSeg].x2 = (int)(stack[pop].x
                                             + tmpx*cos(DegToRad(stack[pop].theta))
                                             + tmpy*sin(DegToRad(stack[pop].theta)));
                      wld[NumSeg].y2 = (int)(stack[pop].y
                                             + tmpy*cos(DegToRad(stack[pop].theta))
                                             - tmpx*sin(DegToRad(stack[pop].theta)));
                    }
                  if (wld[NumSeg].x2 < wld[NumSeg].x1)
                    {
                      tmpx = wld[NumSeg].x1;
                      tmpy = wld[NumSeg].y1;
                      wld[NumSeg].x1 = wld[NumSeg].x2;
                      wld[NumSeg].y1 = cState.map.h-wld[NumSeg].y2;
                      wld[NumSeg].x2 = tmpx;
                      wld[NumSeg].y2 = cState.map.h-tmpy;
                    }
                  else
                    {
                      wld[NumSeg].y1 = cState.map.h-wld[NumSeg].y1;
                      wld[NumSeg].y2 = cState.map.h-wld[NumSeg].y2;
                    }
                  NumSeg++;
                }
              else
                {
                  DBG(0,"Error in format: %s", fdata);
                }
            }
        }
      *fdata = '\0';
    }
  fclose(fp);
  free(stack);
  post_process_map();
  fl_redraw_object(eyesim_view->screen);
}


void
Maze(char *file)
{
  int s, size;
  char ch;
  int i, j;
  int w = 0, h = 0, num_robs = 0;
  char **asc = NULL;
  FILE *fp;
  char tmp[75];

  if (file == (char *)NULL)
    return;
  if ((fp = fopen(file,"r")) == NULL)
    {
      DBG(0,"Cannot open %s", file);
      return;
    }
  remove_obstacles();
  remove_current_map();
  size = 180;
  while (!feof(fp))
    {
      fscanf(fp, "%[^\n]", tmp);
      if (!feof(fp))
        fscanf(fp, "%c", &ch);
      if (isdigit(*tmp))
        {
          size = atoi(tmp);
          size /= 2;
        }
      else
        {
          s = strlen(tmp)+1;
          if(h)
            asc = (char **)realloc(asc,(h+1)*sizeof(char *));
          else
            asc = (char **)malloc(sizeof(char *));
          asc[h] = (char *)malloc(sizeof(char)*s);
          memcpy(asc[h],tmp,s);
          if (s > w)
            w = s;
          h++;
        }
      *tmp = '\0';
    }
  cState.map.w = w*size*1.5;
  cState.map.h = h*size*2;
  for(i=0;i<h;i++)
    {
      s = strlen(asc[i]);
      asc[i] = (char *)realloc(asc[i],w*sizeof(char));
      if (w-1>s)
        {
          for(j=s;j<w;j++)
            asc[i][j] = ' ';
          asc[i][w-1] = '\0';
        }
    }
  i = 0;
  while(i<h)
    {
      j = 0;
      while(j<w-1)
        {
          switch(asc[i][j])
            {
            case '_':
              if ((j%2)&&((j==1)||((j>1)&&(asc[i][j-2]!='_')&&(asc[i][j-2]!='S')
                                   &&(asc[i][j-2]!='s'))))
                {
                  s = j+2;
                  while((s<w-2)&&((asc[i][s]=='_')||(asc[i][s]=='S')||(asc[i][s]=='s')))
                    s+=2;
                  if (NumSeg)
                    wld = (Segment *)realloc(wld,(NumSeg+1)*sizeof(Segment));
                  else
                    {
                      NumSeg++;
                      wld = (Segment *)malloc(2*sizeof(Segment));
                    }
                  wld[NumSeg].x1 = (j-1)*size+size;
                  wld[NumSeg].y1 = i*size*2+size;
                  wld[NumSeg].x2 = (s-1)*size+size;
                  wld[NumSeg].y2 = wld[NumSeg].y1;
                  NumSeg++;
                }
              break;
            case '|':
              if ((i>0)&&(asc[i-1][j] != '|'))
                {
                  s = i+1;
                  while((s<h)&&(asc[s][j] == '|'))
                    s++;
                  if (NumSeg)
                    wld = (Segment *)realloc(wld,(NumSeg+1)*sizeof(Segment));
                  else
                    wld = (Segment *)malloc(sizeof(Segment));
                  wld[NumSeg].x1 = j*size+size;
                  wld[NumSeg].y1 = (i-1)*size*2+size;
                  wld[NumSeg].x2 = wld[NumSeg].x1;
                  wld[NumSeg].y2 = (s-1)*size*2+size;
                  NumSeg++;
                }
              break;
            case 'U':
            case 'D':
            case 'L':
            case 'R':
            case 'S':
              if ((j%2)&&((j==1) ||((j>1)&&(asc[i][j-2]!='_'))))
                {
                  s = j+2;
                  while((s<w-2)&&(asc[i][s]=='_'))
                    s+=2;
                  if (NumSeg)
                    wld = (Segment *)realloc(wld,(NumSeg+1)*sizeof(Segment));
                  else
                    {
                      NumSeg++;
                      wld = (Segment *)malloc(2*sizeof(Segment));
                    }
                  wld[NumSeg].x1 = (j-1)*size+size;
                  wld[NumSeg].y1 = i*size*2+size;
                  wld[NumSeg].x2 = (s-1)*size+size;
                  wld[NumSeg].y2 = wld[NumSeg].y1;
                  NumSeg++;
                }
            case 'u':
            case 'd':
            case 'l':
            case 'r':
            case 's':
              if((num_robs < robot_count) || (robot_count == 0))
                {
                  Robot pos;
                  pos.x = size*j+size;
                  pos.y = (i)*size*2;
                  if((asc[i][j] == 'r') || (asc[i][j] == 'R'))
                    pos.radians = 0;
                  else if((asc[i][j] == 'L') || (asc[i][j] == 'l'))
                    pos.radians = M_PI;
                  else if((asc[i][j] == 'd') || (asc[i][j] == 'D'))
                    pos.radians= -M_PI/2;
                  else
                    pos.radians = M_PI/2;
                  num_robs++;
                  set_robot_start_position(num_robs, &pos);
                }
              break;
            case 'o':
              new_obstacle(size*j+size, (i)*size*2);
              break;
            case 'F':
            case 'f':
            case '.':
            case ' ':
              break;
            default :
              fprintf(stderr,"Undefined symbol %c\n",asc[i][j]);
              break;
            }
          ++j;
        }
      i++;
    }
  fclose(fp);
  post_process_map();
  fl_redraw_object(eyesim_view->screen);
}
