#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <forms.h>
#include <math.h>
#include <ctype.h>
#include <sys/time.h>
#include <unistd.h>
#include <pthread.h>
#include <assert.h>

#include "LCDDisp.h"
#include "protos.h"
#include "eyesim.h"
#include "helpscreen.h"
#include "debugprint.h"

/** array of all robot ids. */
robot_id_type robot_id;

pthread_cond_t pause_button;

static pthread_attr_t thread_attr;

/* Keep track of the pthread ids of multi-threaded robots */
struct robot_thread_map {
  int id;
  pthread_t thread;
};
static int extra_thread_count= 0;
static struct robot_thread_map *extra_thread = NULL;

/** first_robot.
  this function determines whether this is the first robot started.
  returns 1 for first robot -- 0 else.
  also initialises multi-robot data.
  @author Braunl, 1998.
*/
int
first_robot(void)
{
  /** make sure init code is only done once. */
  static int first_time = 1;
  /** id of first robot thread. */
  static pthread_t first_id;
  if(first_time) {
    first_time = 0;
    first_id = pthread_self();

    /* assume single robot unless specified otherwise */
    pthread_mutex_init(&semview, NULL);
  }
  return pthread_equal(first_id, pthread_self());
}

/* Associate an extra thread id to the given robot.  Simple unsorted
   linear list for now.
*/
void
add_robot_thread(int robotid, pthread_t new_thread)
{
  fprintf(stderr, "add_robot_thread(robotid=%d, new_thread=%ld)\n",
          robotid, new_thread);
  assert (NO_THREAD != robot_id.index[robotid-1]);
  extra_thread = realloc(extra_thread,
                         (extra_thread_count+1)*sizeof(*extra_thread));
  extra_thread[extra_thread_count].id = robotid;
  extra_thread[extra_thread_count].thread = new_thread;
  extra_thread_count++;
}

/*
 * Return the current threads robot index [0 .. MAX_ROBOTS-1].
 */
int
robotindex(void)
{
  pthread_t ind = pthread_self();
  int i;
  for (i=0; i<robot_id.Num_robot; i++)
    if(robot_id.index[i] == ind)
      return i;

  for (i = 0; i < extra_thread_count; i++)
    if (extra_thread[i].thread == ind)
      {
        int robotindex = extra_thread[i].id-1;
        if (robotindex < robot_id.Num_robot
            && NO_THREAD != robot_id.index[robotindex])
          return robotindex;
        else
          {
            DBG(0, "ERROR in robotindex(%ld): thread still running, but robot %d is dead!  Exiting\n", ind, extra_thread[i].id);
            abort();
          }
      }
  DBG(0, "ERROR in robotindex(%ld): Unknown thread id.  Exiting\n", ind);
  abort();
}

/** Init_Multi
  multi-robot data initialisations.
  @author Minty, 1998.
*/
int Init_Multi()
{
  int i,j;

  keystatus = 0;
  scrolling = 2;
  lcdline = 0;
  lcdpos = 0;
  startPSD_ = 0;
  endPSD_ = 1000;
  IRrange = 50;

  for(i=0; i<MAX_ROBOTS; i++)
  {
    vw_robots[i].command = FALSE;
    vw_robots[i].stalled = FALSE;
    vw_robots[i].initiate = FALSE;
    vw_robots[i].hiddenhandle = 0;
    vw_robots[i].angleest = 0.0;
    vw_robots[i].distest = 0.0;
    bumperAngle[i] = 0.0;

    X_rob_[i] = 100 + ROBOT_RADIUS*4*i + cState.map.y;
    Y_rob_[i] = 100 + ROBOT_RADIUS + cState.map.y;
    robot[i].x = X_rob_[i];
    robot[i].y = Y_rob_[i];
    robot[i].radians = 0.0;

    VW_SetPosition(i+1, 0,0,0);
  }

  for (i=1; i< MAX_ROBOTS; i++)  /* DUPLICATE FROM FIRST ROBOT */
  {
    for (j=0; j< MaxNumOfPSD;  j++)
    {
      PSD_Sensor[i][j] = PSD_Sensor[0][j];
      PSDSen[i][j].t   = PSDSen[0][j].t;
      PSDSen[i][j].a   = PSDSen[0][j].a;
      PSDSen[i][j].d   = PSDSen[0][j].d;
      PSDSen[i][j].semantic  = PSDSen[0][j].semantic;
    }

    for (j=0; j< MaxNumOfIR;   j++)
    {
      IR_Sensor[i][j] = IR_Sensor[0][j];
      IRSen[i][j].t   = IRSen[0][j].t;
      IRSen[i][j].a   = IRSen[0][j].a;
      IRSen[i][j].d   = IRSen[0][j].d;
      IRSen[i][j].semantic  = IRSen[0][j].semantic;
    }

    for (j=0; j< MaxNumOfBUMP; j++)
    {
      BUMP_Sensor[i][j] = BUMP_Sensor[0][j];
      BUMPSen[i][j].t   = BUMPSen[0][j].t;
      BUMPSen[i][j].a   = BUMPSen[0][j].a;
      BUMPSen[i][j].d   = BUMPSen[0][j].d;
      BUMPSen[i][j].semantic  = BUMPSen[0][j].semantic;
    }

  }
  return 1;
}

void *
main_thread_starter(void *ptr)
{
  struct sem *wait = (struct sem*)ptr;
  char *argv[] = {"EyeSim", NULL};
  extern int main(int argc, char *argv[]);

  if (0 != OSSemV(wait)) /* Unlock starting thread */
    DBG(0, "OSSemv(wait) failed");


  DBG(0, "Calling main() in robot %d.", OSMachineID());

  main(1, argv);
  DBG(0, "Return from main() in robot %d.", OSMachineID());
  return NULL;
}


int
start_new_robot(int id)
{
  struct sem wait;
  int index;
  int err;

  if (-1 == id)
    id = ++robot_id.Num_robot;
  index = id-1;

  robot_count++;
  IRinput[index] = calloc(50,sizeof(char));

  DBG(1, "Robot %d placed on the screen (num robots = %d)\n",
      id, robot_id.Num_robot);

  eyesim_cam_addobject(id);


  if (0 != OSSemInit(&wait, 0))
    DBG(0, "OSSemInit(&wait) failed");

  if (0 != (err = pthread_create(&(robot_id.index[index]), &thread_attr,
                                 &main_thread_starter, &wait)))
    {
      DBG(0, "Failed (%d) to start robot %d's main thread!", err, id);
      return 0;
    }

  DBG(0, "called pthread_create(), waiting for thread to start\n");
  if (0 != OSSemP(&wait)) /* Wait for thread to start */
    DBG(0, "OSSemP(&wait) failed");

  return id;
}

void ThreadFork()
{
  int i, id;

  for(i=0; i<robot_count; i++)
    if(pthread_equal(robot_id.index[i], pthread_self()))
      return;

  while((cState.map.w > (cState.w*cState.zoom)) || (cState.map.h > (cState.h*cState.zoom)))
    {
      cState.zoom += 0.5;
    }
  cState.map.x = 0;
  cState.map.y = 0;


  if(first_robot())
  {
    robot_count = 1;
    robot_id.index[0] = pthread_self();
    eyesim_cam_addobject(1);

    if(!fl_form_is_visible(eyesim_view->viewscreen))
      fl_show_form(eyesim_view->viewscreen, FL_PLACE_FREE, FL_FULLBORDER, "View");

    pthread_cond_init(&pause_button, NULL); /* init cond */
    pthread_attr_init(&thread_attr);
    pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
    //pthread_attr_setschedpolicy(&thread_attr, SCHED_FIFO);
    //pthread_attr_setinheritsched(&thread_attr, PTHREAD_INHERIT_SCHED);

    /*
     * This should be the main thread for the first robot.  Fork of
     * the threads for the rest of the robots as well.
     */
    DBG(0, "robot_count=%d Num_robot=%d", robot_count, robot_id.Num_robot);
    for(id=2; id<=robot_id.Num_robot; id++)
      start_new_robot(id);
  }
}


