/*
 * Elliot Nicholls <nicho-ej@ee.uwa.edu.au> 1997/98
 * Revised by Nicholas Tay
 *
 * This file contains the VW-interface functions for the eyebot simulator
 ****************************************************************************/

/*      $Id: vw.c,v 1.13 2000/08/03 10:28:59 pere Exp $         */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <forms.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include "protos.h"
#include "eyesim.h"
#include "debugprint.h"
#include "viewscreen.h"

int robot_count = 1;

struct vw_robot_info vw_robots[MAX_ROBOTS];

VWHandle
VWInit(DeviceSemantics semantics, int Timescale)
/* Input:       (semantics) semantic
 *              (Timescale) prescale value for 100Hz IRQ (1 to ...)
 * Output:      (returncode) VWHandle or 0 for error
 * Semantics:   Initialize given VW-Driver (only 1 can be initialized!)
 *              The motors and encoders are automatically reserved!!
 *              The Timescale allows to adjust the tradeoff between
 *              accuracy (scale=1, update at 100Hz) and speed(scale>1, update
 *              at 100/scale Hz).
 ****************************************************************************/
{
  struct vw_robot_info *vw;
  InitialiseLCD();
  if(robot_count != robot_id.Num_robot)
    ThreadFork();

  vw = &vw_robots[robotindex()];

  /* set zoom to full out */
  while((cState.map.w > (cState.w*cState.zoom))
        || (cState.map.h > (cState.h*cState.zoom)))
    cState.zoom += 0.5;
  cState.map.x = 0;
  cState.map.y = 0;
  display_screen();

  if(robot_id.Num_robot > 1)
    {
      OSSemP(&sf);
      fl_set_object_label(eyesim_view->errorText,
                          "\nFirst Robot has hit obstacle !!!\n");
      OSSemV(&sf);
    }
  if (vw->hiddenhandle != 0)
    return 0;
  vw->hiddenhandle = (rand() % 9) + 1;
  VWSetPosition(vw->hiddenhandle, 0, 0, 0);
  TRACK();
  return vw->hiddenhandle;
}

int
VWRelease(VWHandle handle)
/* Input:       (handle) VWHandle to be released
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Release VW-Driver, stop motors
 ****************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(vw->hiddenhandle != handle)
    return -1;
  else
    {
      vw->hiddenhandle = 0;
      robot_count--;
      return 0;
    }
}

int
VWSetSpeed(VWHandle handle, meterPerSec v, radPerSec w)
/* Input:       (handle) ONE VWHandle
 *              (v) new linear speed
 *              (w) new rotation speed
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Set the new speed: v(m/s) and w(rad/s not degree/s)
 ****************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(handle != vw->hiddenhandle)
    return -1;
  vw->distest = 0;
  vw->angleest = 0;
  vw->speed.v = v;
  vw->speed.w = w;
  check_forms_redraw_screen();
  return 0;
}

int
VWGetSpeed(VWHandle handle, SpeedType* vwp)
/* Input:       (handle) ONE VWHandle
 *              (vw) pointer to record to store actual v, w values
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Get the actual speed: v(m/s) and w(rad/s not degree/s)
 ****************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  PositionType nowPos;
  struct timeval nowtime, nexttime;
  unsigned long timeP = 0;

  if(handle != vw->hiddenhandle)
    return -1;
  nowPos.x = robot[robotindex()].x;
  nowPos.y = robot[robotindex()].y;
  nowPos.phi = robot[robotindex()].radians;
  gettimeofday(&nowtime,NULL);
  OSWait(5);
  gettimeofday(&nexttime,NULL);
  timeP = 1000*(nexttime.tv_sec-nowtime.tv_sec)
          + (nexttime.tv_usec - nowtime.tv_usec)/1000;
  vwp->v = sqrt((robot[robotindex()].x-nowPos.x)
               * (robot[robotindex()].x-nowPos.x)
               + (robot[robotindex()].y-nowPos.y)
               * (robot[robotindex()].y-nowPos.y)) / timeP;
  if(vw->speed.v < 0)
    vwp->v = -vwp->v;
  vwp->w = (robot[robotindex()].radians-nowPos.phi)*1000.0/timeP;
  check_forms_redraw_screen();
  return 0;

}

int
VW_SetPosition(int robotid, meter x, meter y, radians phi)
{
  int robotindex = robotid -1;
  float lengt_, ang_e = 0;
  x = -x;
  y = -y;
  lengt_ = sqrt(x*x + y*y);
  if(x == 0)
    if(y >= 0)
      ang_e = M_PI_2;
    else
      ang_e = -M_PI_2;
  else
    {
      ang_e  = atan(y/fabs(x));
      if(x < 0)
        ang_e = M_PI - ang_e;
    }
  robotStartPos[robotindex].radians = phi - robot[robotindex].radians;
  robotStartPos[robotindex].x  = robot[robotindex].x
                                 + lengt_ * cos(robotStartPos[robotindex].radians - ang_e) * 1000;
  robotStartPos[robotindex].y  = robot[robotindex].y
                                 + lengt_ * sin(robotStartPos[robotindex].radians - ang_e) * 1000;
  return 0;
}

int
VWSetPosition(VWHandle handle, meter x, meter y, radians phi)
/* Input:       (handle) ONE VWHandle
 *              (x) new x-position
 *              (y) new y-position
 *              (phi) new heading
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Set the new position: x(m), y(m) phi(rad not degree)
 ***************************************************************************/
{
  int retval;
  if(handle != vw_robots[robotindex()].hiddenhandle)
    {
      fprintf(stderr,"Wrong VWHandle\n");
      retval = -1;
    }
  else
    retval = VW_SetPosition(OSMachineID(), x, y, phi);
  TRACK();
  redraw_screen();
  return retval;
}

/*****f* VWSim/VW_GetPosition
 * NAME
 *   GetPosition
 * FUNCTION
 *   Internal function to find position of robot with id from 1 to 15.
 ******
 */
int
VW_GetPosition(int robotid, PositionType* pos)
{
  float ldiff, rdiff, xdiff, ydiff;
  int robotindex = robotid - 1;

  xdiff = robot[robotindex].x - robotStartPos[robotindex].x;
  ydiff = robot[robotindex].y - robotStartPos[robotindex].y;
  ldiff = sqrt(xdiff * xdiff + ydiff * ydiff);
  if(xdiff == 0)
    {
      if(ydiff >= 0)
        rdiff = M_PI_2;
      else
        rdiff = -M_PI_2;
    }
  else
    {
      rdiff = atan(ydiff / abs(xdiff));
      if (xdiff < 0)
        rdiff = M_PI - rdiff;
    }

  pos->phi = robot[robotindex].radians + robotStartPos[robotindex].radians;
  while (pos->phi > M_PI)
    pos->phi -= 2*M_PI;
  while (pos->phi < -M_PI)
    pos->phi += 2*M_PI;

  pos->x   = (ldiff
              * cos(rdiff - robotStartPos[robotindex].radians)) / 1000.0;
  pos->y   = (ldiff
              * sin(rdiff - robotStartPos[robotindex].radians)) /-1000.0;
  return 0;
}

int
VWGetPosition(VWHandle handle, PositionType* pos)
/* Input:       (handle) ONE VWHandle
 *              (pos) pointer to record to store actual position (x,y,phi)
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Get the actual position: x(m), y(m) phi(rad not degree)
 ***************************************************************************/
{
  int retval;

  if(handle != vw_robots[robotindex()].hiddenhandle)
    retval = -1;
  else
    retval = VW_GetPosition(OSMachineID(), pos);
  check_forms_redraw_screen();
  return retval;
}

int
VWStartControl(VWHandle handle, float Vv, float Tv, float Vw, float Tw)
/* Input:       (handle) ONE VWHandle
 *              (Vv) the paramter for the proportional component of the v-controller
 *              (Tv) the paramter for the integrating component of the v-controller
 *              (Vw) the paramter for the proportional component of the w-controller
 *              (Tv) the paramter for the integrating component of the w-controller
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   Enable the PI-controller for the vw-interface and set the parameters.
 *              As default the PI-controller is deactivated when the vw-interface is
 *              initialised. The controller tries to keep the desired speed (set with
 *              VWSetSpeed) stable by adapting the energy of the involved motors.
 *              The parameters for the controller have to be choosen carefully!
 *              The formula for the controller is:
 *                                          t
 *              new(t) = V*(diff(t) + 1/T * [diff(t)dt )
 *                                          0
 *              V: a value usually around 1.0
 *              T: a value usually between 0 and 1.0
 *              After enabling the controller the last set speed (VWSetSpeed) is taken
 *              as the speed to be held stable.
 ***************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(vw->initiate || (handle != vw->hiddenhandle))
    return -1;
  vw->initiate = TRUE;
  return 0;

}

int
VWStopControl(VWHandle handle)
  /* Input:       (handle) ONE VWHandle
   * Output:      0 = ok
   *              -1 = error wrong handle
   * Semantics:   Disable the controller immediately. The vw-interface
   *              continues normally with the last valid speed of the
   *              controller.
   ***************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if((!vw->initiate) || (handle != vw->hiddenhandle))
    return -1;
  vw->initiate = FALSE;
  return 0;
}

int
VWDriveStraight(VWHandle handle, meter delta, meterPerSec v)
/* Input:       (handle) ONE VWHandle
 *              (delta)  distance to drive in m (pos. -> forward)
 *                                              (neg. -> backward)
 *              (v)      speed to drive with
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   drives distance "delta" with speed v straight ahead (forward
 *              or backward) any subsequent call of VWDriveStraight, -Turn,
 *              -Curve or VWSetSpeed while this one is still being executed,
 *               results in an immediate interruption of this command
 ***************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(handle != vw->hiddenhandle)
    return -1;
  if(delta < 0)
    {
      v = -v;
      //delta = -delta;
    }

  vw->angleest = 0;
  vw->distest = fabs(delta);
  vw->speed.v = v;
  vw->speed.w = 0.0;
  check_forms_redraw_screen();
  return 0;
}

int
VWDriveTurn(VWHandle handle, radians delta, radPerSec w)
/* Input:       (handle) ONE VWHandle
 *              (delta)  degree to turn in radians (pos. -> counter-clockwise)
 *                                                 (neg. -> clockwise)
 *              (w)      speed to turn with
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   turns about "delta" with speed w on the spot (clockwise or
 *              counter-clockwise) any subsequent call of VWDriveStraight,
 *              -Turn, -Curve or VWSetSpeed while this one is still being
 *              executed, results in an immediate interruption of this command
 ****************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if (handle != vw->hiddenhandle)
    return -1;
  if(delta < 0)
    {
      w = -w;
      delta = -delta;
    }
  TRACK();
  vw->speed.v = 0.00;
  vw->speed.w = w;
  vw->distest = 0;
  vw->angleest = fabs(delta);
  check_forms_redraw_screen();
  return 0;
}

int
VWDriveCurve(VWHandle handle, meter delta_l, radians delta_phi, meterPerSec v)
/* Input:       (handle)    ONE VWHandle
 *              (delta_l)   length of curve_segment to drive in m (pos. -> forward)
 *                                                                (neg. -> backward)
 *              (delta_phi) degree to turn in radians (pos. -> counter-clockwise)
 *                                                    (neg. -> clockwise)
 *              (v)         speed to drive with
 * Output:      0 = ok
 *              -1 = error wrong handle
 * Semantics:   drives a curve segment of length "delta_l" with overall vehicle turn of "delta_phi"
 *              with speed v (forw. or backw. / clockw. or counter-clockw.).
 *              any subsequent call of VWDriveStraight, -Turn, -Curve or VWSetSpeed
 *              while this one is still being executed, results in an immediate interruption
 *              of this command
 ***************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  float timeToGo = 0;

  if(handle != vw->hiddenhandle)
    {
      fprintf(stderr,"Wrong VW handle\n");
      return -1;
    }
  if(delta_l<0)
    {
      delta_l = -delta_l;
      v = -v;
    }
  vw->angleest = fabs(delta_phi);
  vw->distest = fabs(delta_l);
  timeToGo = fabs(delta_l/v);
  vw->speed.v = v;
  vw->speed.w = delta_phi/timeToGo;
  check_forms_redraw_screen();
  return 0;

}

float
VWDriveRemain(VWHandle handle)
/* Input:       (handle) ONE VWHandle
 * Output:      0.0 = previous VWDriveX command has been completed
 *              any other value = remaining distance to goal
 * Semantics:   remaining distance to goal set by VWDriveStraight, -Turn
 *              (for -Curve only the remaining part of delta_l is reported)
 ***************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  unsigned long timeToGo;
  struct timeval timenow;

  if(vw->hiddenhandle != handle)
    {
      fprintf(stderr,"Wrong handle\n");
      return -1;
    }
  gettimeofday(&timenow,NULL);
  timeToGo = (timeEstimate[robotindex()].tv_sec - timenow.tv_sec)*1000
             + (timeEstimate[robotindex()].tv_usec - timenow.tv_usec)/1000;
  if(!vw->command)
    return 0.0;

  check_forms();

  if(vw->speed.v != 0)
    return (fabs(vw->speed.v*timeToGo));
  else
    return (fabs(vw->speed.w*timeToGo));
}


/* Input:       (handle) ONE VWHandle
 * Output:      -1 = error wrong handle
 *              0 = vehicle is still in motion
 *              1 = previous VWDriveX command has been completed
 * Semantics:   checks if previous VWDriveX() command has been completed
 ***************************************************************************/
int VWDriveDone(VWHandle handle)
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(handle != vw->hiddenhandle)
    {
      fprintf(stderr,"ERROR: wrong handle\n");
      return -1;
    }
  check_forms_redraw_screen();
  return ((vw->distest == 0.0) && (vw->angleest == 0.0));
}

int
VWDriveWait(VWHandle handle)
/* Input:       (handle) ONE VWHandle
 * Output:      -1 = error wrong handle
 *              0 = previous VWDriveX command has been completed
 * Semantics:   blocks the calling process until the previous VWDriveX()
 *              command has been completed
 ****************************************************************************/
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(handle != vw->hiddenhandle)
    return -1;
  while ((vw->distest != 0.0) || (vw->angleest != 0.0))
    {
      check_forms_redraw_screen();
    }
  check_forms();
  return 0;
}

int
VWStalled (VWHandle handle)
{
  struct vw_robot_info *vw = &vw_robots[robotindex()];
  if(handle != vw->hiddenhandle)
    return -1;
  check_forms_redraw_screen();
  return vw->stalled;
}
