/*
 * Author: Petter Reinholdtsen <pere@td.org.uit.no>
 * Date:   2000-03-29
 * 
 * Eyebot compass driver prototype for analog compass sensor no. 1655
 * from Dinsmore Instruments Company.
 */

#include <eyebot.h>

#include <math.h>
#include <stdio.h>

typedef struct
{
  short version;
  BYTE  curve1;
  BYTE  curve2;
} analog_compass_type;


/*
 * To compensate for slow response to channel changes on the A/D
 * converter, we read a few times to be sure the value has stabilized.
 */
int
myGetAD(int channel)
{
  int i;
  for (i = 0; i < 3500; i++)
    OSGetAD(channel);
  return OSGetAD(channel);
}

/* HDT entry */
analog_compass_type compass = {0, 2, 3};

typedef int CompassHandle;
struct compass_info {
  CompassHandle handle;
  int min1, min2;
  int max1, max2;
  int n1, n2; /* north */
  int s1, s2; /* south */
  int o1, o2; /* origo */

  int state;  /* calibration state */
} acompass;

CompassHandle
COMPASS_Init(DeviceSemantics semantics)
{
  /* Fill in default values */
  acompass.handle = 1;
  acompass.max1 = 0;
  acompass.max2 = 0;
  acompass.min1 = 1024;
  acompass.min2 = 1024;

  acompass.n1 = 0;
  acompass.n2 = 1024;
  acompass.s1 = 1024;
  acompass.s2 = 0;
  acompass.o1 = 512;
  acompass.o2 = 512;

  return acompass.handle;
}

int
COMPASS_Start(CompassHandle handle, BOOL cycle)
{
  if (handle != acompass.handle)
    return -1;
  /* Get a few values to initialize min and max */
  COMPASS_Get(handle);
  COMPASS_Get(handle);
  return 0;
}

int
COMPASS_Check(CompassHandle handle)
{
  if (handle != acompass.handle)
    return 0;
  return 1;
}

int
COMPASS_Stop(CompassHandle handle)
{
  return 0;
}

int
COMPASS_Release(CompassHandle handle)
{
  return 0;
}

static int
COMPASS_Update(int *min, int *max, int v)
{
  int changes = 0;
  if (v > *max)
    {
      *max = v;
      changes = 1;
    }
  if (v < *min)
    {
      *min = v;
      changes = 1;
    }
  return changes;
}

static void
COMPASS_UpdateMinMax(CompassHandle handle, int v1, int v2)
{
  if (handle != acompass.handle)
    return;

  /* If min or max changed, recalculate origo */
  if (COMPASS_Update(&acompass.min1, &acompass.max1, v1))
    acompass.o1 = acompass.min1 + (acompass.max1 - acompass.min1)/2;

  if (COMPASS_Update(&acompass.min2, &acompass.max2, v2))
    acompass.o2 = acompass.min2 + (acompass.max2 - acompass.min2)/2;
}

#define limit (0.25)
radians
use_the_best(int x, int y, radians ax, radians ay)
{
  radians angle;
  if (1 - limit < x || -1 + limit > x) /* X is useless, use Y */
    angle = ay;
  else if (1 - limit < y || -1 + limit > y) /* Y is useless, use X */
    angle = ax;
  else /* We can use both, take the average... :-) */
    angle = (ax+ay) / 2;
  return angle;
}

int
COMPASS_Get(CompassHandle handle)
{
  int v1, v2;
  static const double rad_to_deg = 180.0 / M_PI;
  v1 = myGetAD(compass.curve1);
  v2 = myGetAD(compass.curve2);
  COMPASS_UpdateMinMax(handle, v1, v2);
  printf("1: %d<%d<%d \n", acompass.min1, acompass.o1, acompass.max1);
  printf("2: %d<%d<%d \n", acompass.min2, acompass.o2, acompass.max2);

  {
    double x, y;
    radians angle, ax, ay;
    x = (v1 - acompass.o1) / (float) (acompass.o1 - acompass.min1);
    y = (v2 - acompass.o2) / (float) (acompass.o2 - acompass.min2);
    ay = asin(y);
    ax = acos(x);

    printf("(%4.2fx%4.2f)\n =(%3d,%3d)\n", x, y,
	   (int)(ax * rad_to_deg), (int)(ay * rad_to_deg));

    /* calculate quadrant */
    if (x > 0 && y > 0) /* 0 - 90 */
      return (int) (use_the_best(x,y,ax,ay) * rad_to_deg);
    else if (x < 0 && y > 0) /* 90- 180 */
      return (int) (use_the_best(x,y,ax,M_PI-ay) * rad_to_deg);
    else if (x < 0 && y < 0) /* 180 - 270 */
      return (int) (use_the_best(x,y,2*M_PI-ax,M_PI - ay) * rad_to_deg);
    else /* (x > 0 && y < 0) */ /* 270 - 360 */
      return (int) (use_the_best(x,y,2*M_PI-ax,ay) * rad_to_deg);
  }
}

int
COMPASS_Calibrate(CompassHandle handle, int mode)
{
  if (handle != acompass.handle)
    return -1;

  return 0;
}

int
main(int argc, char *argv[])
{
  CompassHandle h;
  LCDPutString("Analog compass\n");

  LCDMenu(" ", " ", " ", "END");

  h = COMPASS_Init(COMPASS);
  COMPASS_Start(h, 1);

  while (KEY4 != KEYRead())
    {
      int heading;
      LCDSetPos(2, 0);
      heading = COMPASS_Get(h);
      LCDSetPos(6, 0);
      LCDPrintf("H: %4d  ", heading);
    }

  COMPASS_Stop(h);
  COMPASS_Release(h);
  return 0;
}
