/*
 * Filename:	nav.c
 *
 * Author:	Elliot Nicholls <nicho-ej@ciips.ee.uwa.edu.au>, UWA 1997
 *
 * Description:	Drives EyeBot vehicle using front IR sensors and PSD sensor
 *		Will navigate with semi-intelligence
 *	        
 ****************************************************************************/

/* 	$Id: nav-1proc.c,v 1.1.1.1 1999/12/20 11:26:24 pere Exp $	 */

#ifndef lint
static char vcid[] = "$Id: nav-1proc.c,v 1.1.1.1 1999/12/20 11:26:24 pere Exp $";
#endif /* lint */

#define PI 3.1415926535 

#include "librobi/librobi.h"
#include "libimpro/improc.h"
#include "librobi/protos.h"

#include "keys.h"
#include "lcd.h"
#include "cam.h"
#include "types.h" 
#include "const.h" 
#include "kern.h"
#include "vw.h"
#include "hdt.h"
#include "hdt_sem.h"

#include <stdio.h>
#include <unistd.h>
#include <math.h>

#define Vv 7.0
#define Tv 0.3
#define Vw 7.0
#define Tw 0.1
#define PSD_LIMIT_L 250
#define PSD_LIMIT_H 800
#define SPD_FWD 0.2
#define SPD_REV -0.1
#define PSD_TURN_CORRECT 12 /* degrees */
#define BACK_OFF_DIST 0.30 /* meters */

/*
 * Global variables
 **********************************************************************/
VWHandle vw;
static PSDHandle psd;
IRHandle irLF, irRF, irLM, irRM, irLB, irRB, irBA;
int lastTurnLeft = 0;
SpeedType s;

/*
 * Declared functions to stop stupid compile-time errors :-)
 **********************************************************************/
void PSDTurnParallel(void);
void irEvade(void);

/*
 * Function to stop the vehicle
 **********************************************************************/

void VWStop(VWHandle vw) {
  VWSetSpeed(vw, 0, 0);
  return;
}

/*
 * Function to turn the vehicle by number of degrees
 * Function uses VWDriveTurn and blocks
 **********************************************************************/
 
void VWTurn(VWHandle vw, float degrees, float speed, BOOL block) {

  float radians;

  radians = degrees * PI / 180;
  VWDriveTurn(vw, radians, speed);
  if(block == TRUE) {
    do {
    } while(VWDriveDone(vw) == 0);
  }
  return;
}


int SysInitialise(void) {

  /*
   * unbuffer stdout/in for printf
   **********************************************************************/
  
  setvbuf(stdout,NULL,_IONBF,0); 
  setvbuf(stdin,NULL,_IONBF,0);  
  LCDMode(SCROLLING|NOCURSOR);

  /*
   * Initialise PSD controller
   **********************************************************************/

  psd = PSDInit(PSD_FRONT);
  if(psd == 0) {
    LCDPutString("PSDInit Error!\n");
    OSWait(200);
    return 0;
  }

  if(PSDStart(psd, TRUE)) {
    LCDPutString("PSD busy!\n");
    return 0;
  }

  /*
   * Initialise VW interface and get initial position data
   **********************************************************************/

  vw = VWInit(VW_DRIVE,1);
  if(vw == 0)
    {
      LCDPutString("VWInit Error!\n");
      OSWait(200);
      return 0;
    }
  VWStartControl(vw, Vv, Tv, Vw, Tw);

  /*
   * Initialise IR sensors
   **********************************************************************/

  irLF = IRInit(IR_LF);
  if(irLF == 0) {
    LCDPutString("IRInit error (LF)\n");
    return 0;
  }
  irRF = IRInit(IR_RF);
  if(irRF == 0) {
    LCDPutString("IRInit error (RF)\n");
    return 0;
  }
  irLM = IRInit(IR_LM);
  if(irLM == 0) {
    LCDPutString("IRInit error (LM)\n");
    return 0;
  }
  irRM = IRInit(IR_RM);
  if(irRM == 0) {
    LCDPutString("IRInit error (RM)\n");
    return 0;
  }
  AUBeep();

  return 1;
}

/*
 * Release all handles before exiting program
 **********************************************************************/

void SysRelease(void) {

  PSDRelease();
  VWRelease(vw);
  IRRelease(irLF + irRF + irLM + irRM);
  return;
  
}

/*
 * Routine to handle driving forwards
 **********************************************************************/

void DriveForward(void) {
  PositionType begin;
  int psdObject, irDetect;

  VWGetPosition(vw, &begin);
  
  do {
    VWSetSpeed(vw, SPD_FWD, 0);

    OSWait(30);
    
    /*
     * Check to see if there is any PSD/IR action
     **********************************************************************/
    do {
      if(VWStall() == 1) {
	VWDriveStraight(vw, BACK_OFF_DIST, SPD_REV);
	if(lastTurnLeft == 1) VWTurn(vw, 90, 1.0, TRUE);
	else VWTurn(vw, -90, 1.0, TRUE);
	VWSetSpeed(vw, SPD_FWD, 0);
      }
      psdObject = PSDGet(psd);
      irDetect = IRDetect();
    } while((psdObject > PSD_LIMIT_L) && (irDetect == 0));
    
    if(psdObject < PSD_LIMIT_L) {
      PSDTurnParallel();
    }
    if(irDetect == 1) {
      irEvade();
    }
  } while(1 == 1);
  
  return;
}

/*
 * Check to see if robot is stalled
 **********************************************************************/

int VWStall(void) {
  SpeedType s;
  VWGetSpeed(vw, &s);
  if((s.v == 0) && (s.w == 0)) {
    return 1;
  }
  return 0;
}
			  
/*
 * Attempt to evade objects using IR sensors
 **********************************************************************/

void irEvade() {
  if(IRRead(irLM) == 0) {
    VWSetSpeed(vw, SPD_FWD, -1.0);

    do {
      if(VWStall() == 1) {
	VWStop(vw);
	VWSetSpeed(vw, SPD_REV, -1.0);
      }
    } while((IRRead(irLM) == 0) && (PSDGet(psd) > PSD_LIMIT_L));
    VWSetSpeed(vw, SPD_FWD, 0.0);
    return;
  }
  if(IRRead(irRM) == 0) {
    VWSetSpeed(vw, SPD_FWD, 1.0);
    
    do {
      if(VWStall() == 1) {
	VWStop(vw);
	VWSetSpeed(vw, SPD_REV, 1.0);
      }
    } while((IRRead(irRM) == 0) && (PSDGet(psd) > PSD_LIMIT_L));
    VWSetSpeed(vw, SPD_FWD, 0.0);
    return;
  }

  /*
   * Back off if too close
   **********************************************************************/
   
  if(IRRead(irLF) == 0 || IRRead(irRF) == 0) {
    VWStop(vw);
    OSWait(30);
    VWSetSpeed(vw, SPD_REV, 0.0);
    do {
    } while(IRRead(irLF) == 0 || IRRead(irRF) == 0);
    VWStop(vw);
    PSDTurnParallel();
    return;
  }
}

/*
 * Detect object using IR sensors
 **********************************************************************/

int IRDetect(void) {
  if(IRRead(irLF) == 0) return 1;
  if(IRRead(irRF) == 0) return 1;
  if(IRRead(irLM) == 0) return 1;
  if(IRRead(irRM) == 0) return 1;
  return 0;
}

/*
 * Turn parallel to wall using PSD sensor on front of eyebot
 * Only works effectively with walls and accurate PSD readings
 **********************************************************************/

void PSDTurnParallel(void) {
  int psdStart, psdCurrent;
  float radRemain, radTurned;
  PositionType start, end;
  
  printf("Starting PSDTurnParallel\n");
  VWStop(vw);
  psdStart = PSDGet(psd);
  if(psdStart > PSD_LIMIT_L) return;
  VWGetPosition(vw, &start);

  printf("Got starting position\n");
  if(lastTurnLeft == 1) {
    VWSetSpeed(vw, 0.0, 1.0);
  } else {
    VWSetSpeed(vw, 0.0, -1.0);
  }

  printf("Calling OSWait\n");
  OSWait(40);
  psdCurrent = PSDGet(psd);

  printf("Entering test loop\n");
  if(psdCurrent > psdStart) {

    printf("In loop...\n");
    do {
      printf("Waiting\n");
      /* VWGetSpeed(vw, &s);*/
      printf("%d, %d\n", s.v, s.w);
    } while(PSDGet(psd) < PSD_LIMIT_H);
    VWStop(vw);
    psdCurrent = PSDGet(psd);
    
    VWGetPosition(vw, &end);
    if(lastTurnLeft == 1) {
      radTurned = end.phi - start.phi;
      if(radTurned < 0) radTurned += 2*PI;
    } else {
      radTurned = start.phi - end.phi;
      if(radTurned < 0) radTurned += 2*PI;
    }

    radRemain = asin((psdStart * sin(radTurned)) / sqrt(sqr(psdStart) + sqr(psdCurrent) - (2 * psdStart * psdCurrent * cos(radTurned))));
    radRemain = radRemain + (PSD_TURN_CORRECT * PI / 180);

    printf("Calling VWDriveTurn\n");
    if(lastTurnLeft == 1) {
      VWDriveTurn(vw, radRemain, 1.0);
    } else {
      VWDriveTurn(vw, -radRemain, 1.0);
    }

    do {
      printf("Waiting to finish\n");
    } while(VWDriveDone(vw) == 0);

  } else {
    VWStop(vw);
    if(lastTurnLeft == 1) lastTurnLeft = 0;
    else lastTurnLeft = 1;
    printf("Going a level deeper\n");
    PSDTurnParallel();
  }

  return;
}

/*
 * Square function
 **********************************************************************/

int sqr(int x) {
  return x*x;
}

/*
 * Main routine
 **********************************************************************/

int main(void) {
  int test;
  PositionType start, stop, current;

  if(SysInitialise() == 0) {
    LCDPutString("Init error!\n");
    return;
  }

  LCDPutString("Init OK!\n");

  DriveForward();

  SysRelease();
  return;
}






